#charset "us-ascii"
/* 
 *  Copyright (c) 2005 by Kevin Forchione. All rights reserved.
 *  Based on ideas from Michael J. Roberts on the T3 Newsgroup.
 *   
 *  This file is part of the TADS 3 Services Pack
 *
 *  tsp_mod_registry.t
 *
 *  Defines module registry and dependency classes. 
 *
 *  The PlaceHolder class inherits the superclasses of indicated by its
 *  properties if those objects are present at the time of compilation.
 *
 *  A RegisterModule is an object that registers the name of the module 
 *  in which it is compiled at the time of compilation for debugging.  Coupled
 * with RequiresModule() these macros allow an author to ensure that a 
 * given module is present during compilation for debugging.
 *
 *  A ModuleDependency stores a single dependency in a 'global' 
 *  object's property pointer if and only if that dependency is found 
 *  as a symbol in the global symbol table.
 *
 *  For instance:
 *
 *      ModuleDependency
 *      {
 *          globalStr   = 'myGlobal'
 *          propStr     = 'myDependencyPtr'
 *          dependStr   = 'myDependency'
 *      }
 *
 *  Will store the dependency myDependency in myGlobal's myDependencyPrt
 *  property. This object definition is equivalent having your program
 *  execute the following code:
 *
 *      myGlobal.myDependencyPtr = myDependency;
 *
 *  Each of the 3 properties, if defined, should be defined as a single-quoted
 *  string value. If globalStr is omitted then "self" will be assumed as the 
 * global object. If propStr is omitted then it will be created by appending 
 *  'Ptr' to the depndStr value. Thus the same results could be achieved
 *  for as the above, using the following definition:
 *
 *      ModuleDependency
 *      {
 *          globalStr   = 'myGlobal'
 *          dependStr   = 'myDependency'
 *      }
 *
 * If assertDependency is set to true (the default) then a run-time exception
 * will be thrown during preinit if the dependency wasn't found in the global 
 * symbols table.
 */

#include "tsp_mod_registry.h"

RegisterModule

/*
 *  The PlaceHolder class implements an object that may 
 *  have its superclass replaced by the superclass or superclasses
 *  listed in its forObject property.
 */
class PlaceHolder: PreinitObject
{
    /*
     *  Should be set to the object or a list of objects
     *  for which this object is to be a placeholder for. If
     *  all of the object(s) are present in the compilation then
     *  the placeholder superclass list is replaced with these 
     *  object(s).
     */
    forObject = nil
        
    /*
     * Should be set to the object or a list of objects
     * for which this object is to use as default superclasses if
     * the forObject isn't present during compilation.
     *
     * If all of the object(s) are present in the compilation then
     * the placeholder superclass list is replaced with these 
     * object(s). If these objects aren't found at compilation
     * then the superclass will remain PlaceHolder.
     */
    defaultObject = nil

    execute()
    {
        if (!inheritFromObj(forObject))
            inheritFromObj(defaultObject);
    }
    inheritFromObj(objs)
    {
        local lst;

        lst = [] + objs;
        lst = lst.mapAll({x: dataType(x) == TypeObject ? x : nil});
        if (lst.indexOf(nil) == nil)
        {
            setSuperclassList(lst);
            return true;
        }
        else return nil;
    }
}

/*
 *  Defines the ModuleDependency class
 */
class ModuleDependency: PreinitObject
{
    assertDependency    = true

    globalStr           = nil
    propStr             = nil
    dependStr           = nil
    buildStr            = nil

    execute()
    {
        local gObj, val, ptr, msg, symTab, revSymTab, str;

        /*  store a reference to the global symbol table */
        symTab = t3GetGlobalSymbols();
        
        /*
         *  building a reverse mapping - in other words, building a
         *  LookupTable that uses symbol values as the keys and symbol
         *  strings as the associated values.
         */
        revSymTab = new LookupTable();
        symTab.forEachAssoc({str, sym: revSymTab[sym] = str});

        /*
         *  Convert dependStr to a list if it was a string. 
         */
        dependStr = ([] + dependStr);

        /*
         *  Remove any nil values from our dependStr list
         */
        dependStr -= nil;

        /* check if globalStr, if defined, is a single-quoted string */
        if (propType(&globalStr) != TypeNil && propType(&globalStr) != TypeSString)
        {
            msg = 'ModuleDependency property globalStr must be TypeSString. ';
            throw new ModuleDependencyError(msg);
        }

        /*
         *  Check to see if our dependStr list has any elements
         *  and throw a runtime error if it is empty.
         */
        if (dependStr.length() == 0)
        {
            msg = 'ModuleDependency property dependStr is unassigned. ';
            throw new ModuleDependencyError(msg);
        }

        /*
         *  Iterate over each element in dependStr list
         *  and check that it is TypeSString. If it isn't
         *  then we throw a runtime error.
         */
        foreach (local str in dependStr)
        {
            if (dataType(str) != TypeSString)
            {
                msg = 'ModuleDependency dependStr elements must be TypeSString. ';
                throw new ModuleDependencyError(msg);
            }
        }

        /*
         *  If propStr hasn't been defined and dependStr is 
         *  TypeSString then create propStr by suffixing dependStr
         *  with 'Ptr'. Otherwise check if propStr is a single-quoted 
         *  string.
         */
        if (propType(&propStr) == TypeNil && dependStr.length() == 1)
            propStr = dependStr.car() + 'Ptr';
        else if (propType(&propStr) != TypeSString)
        {
            msg = 'ModuleDependency propStr must be TypeSString. ';
            throw new ModuleDependencyError(msg);
        }

        /*
         *  Get the symbol for globalStr and check if 
         *  it is an object.
         */
        if (globalStr)
        {
            gObj = symTab[globalStr];
            if (dataType(gObj) != TypeObject)
            {
                msg = 'globalStr value must be TypeObject. ';
                throw new ModuleDependencyError(msg);
            }
        }
        else gObj = self;

        /*
         *  Get our symbol for propStr from the global symbols table
         */
        ptr = symTab[propStr];
        if (ptr == nil)
        {
            msg = 'ModuleDependency propStr value not found in 
                global symbol table. ';
            throw new ModuleDependencyError(msg);
        }

        /*
         *  Check if the symbol associated with propStr is a property 
         */
        if (dataType(ptr) != TypeProp)
        {
            msg = 'ModuleDependency propStr value must be TypeProp. ';
            throw new ModuleDependencyError(msg);
        }

        /*
         *  Get the first symbol match for dependStr. If no symbol
         *  was found we continue through the dependStr list until 
         *  we find a match. If no match is found our global object's
         *  property pointer will have a nil value.
         */
        foreach (str in dependStr)
        {
            val = symTab[str];
            if (val)
            {
                /*
                 *  Store the dependency value of the first 
                 *  successful retrieval in our global and 
                 *  break out of the loop
                 */
                gObj.(ptr) = val;
                break;
            }
        }

        /*
         *  If this module dependency asked to assert the dependency
         *  then we will throw a runtime error if no symbol for our 
         *  dependStr list elements was found.
         */
        if (assertDependency && val == nil)
        {
            msg = 'ModuleDependency \"' + propStr + '\" not found';
            throw new ModuleDependencyError(msg);
        }
        
        buildStr = revSymTab[gObj] + '.' + propStr + ' = '; 
        if (str)
            buildStr += str;
        else buildStr += 'nil';
    }
    display()
    {
        "\nModule Dependency: <<buildStr>> ";
    }
}

/* define a runtime error for ModuleDependency */
class ModuleDependencyError: RuntimeError
{
    construct(msg)
    {
        exceptionMessage = msg;

        inherited(0);
    }
}

/*
 *  A module registry class. This module stores the values for 
 *  every module that registers with it during pre-init. Registration
 *  only occurs during compilation for debugging and is bypassed 
 *  with a warning message in the build window explaining that
 *  registry information is unavailable outside of compilation for debugging. 
 */
ModuleRegistry: PreinitObject
{
    modTbl_         = static new Vector(40);
    dbgMode_      = static (t3DebugTrace(T3DebugCheck))
    registryFail_    = nil

    registerModule(str)
    {
        modTbl_.appendUnique([str]);
    }

    hasModule(modStr)
    {
        return (modTbl_.indexOf(modStr) != nil);
    }

    execute()
    {
        /*
         * If we're not in debug mode and we've not already 
         * failed, issue a warning to the compilation that no
         * registry information is available.
         */
        if (!dbgMode_ && !registryFail_)
        {
            tadsSay('\bTSP ModuleRegistry Warning: Not in debug mode.');
            tadsSay('\nTSP ModuleRegistry info unavailable.\b');
            registryFail_ = true;
            return;
        }

        /*
         * Iterate over our ModuleRegistration objects
         */
        for (local o = firstObj(ModuleRegistration); o != nil; o = nextObj(o, ModuleRegistration))
            o.registerModule();
    }
}

/*
 *  A module registration class. Instances of these objects should
 *  be defined in every module that wishes to register itself with 
 *  the ModuleRegistry object. The RegisterModule macro should
 *  be used to indicate that a module is to be registered. 
 */
class ModuleRegistration: object
{
    registerModule()
    {
        local stVec, stObj, modName;

        stVec = getStack();
        stObj = stVec[1];
        
        if (stObj.obj_ && self.ofKind(stObj.obj_)
            && stObj.srcInfo_ != nil)
        {
            modName = parseModName(stObj.srcInfo_.car());
            ModuleRegistry.registerModule(modName);
        }
    }

    /* 
     *  Each instance overrides this to get the stack trace
     *  for its own instance.
     */
    getStack() {}

    /* 
     * Parse the modules name from the stacktrace path
     */
    parseModName(src)
    {
        local f = 1, idx = 0;

        do
        {
            f = src.find('\\', f + 1);
            if (f)
                idx = f;
        }
        while(f);
 
        return src.substr(idx + 1);
   }
}

/*
 *  A module requirement class. If we've compiled for debugging then
 *  each instance of this class will check with the ModuleRegistry object
 *  to determine if its required module has registerd. If not then a runtime
 *  error is thrown.
 *
 *  Requirements should be defined in source modules using the 
 *  RequiresModule() macro.
 */
class ModuleRequirement: PreinitObject
{
    execBeforeMe = [ModuleRegistry]
    modName_     = nil
    dbgMode_      = static (t3DebugTrace(T3DebugCheck))

    execute()
    {
        if (dbgMode_)
        {
            if (!ModuleRegistry.hasModule(modName_))
                throw new ModuleRequirementError(modName_);
        }
    }
}

/*
 *  A base class for module requirement errors
 */
class ModuleRequirementError: RuntimeError
{
    construct(modName)
    {
        exceptionMessage = 'TSP ModuleReq Error - ' + modName + ' required by this compliation';
        inherited(0);
    }
}