#charset "us-ascii"
/* 
 *  Copyright (c) 2005 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Services Pack
 *
 *  tsp_cond.t
 *
 *  TSP Conditional and Case Functions
 */

#include "tads.h"
#include "t3.h"
#include "vector.h"
#include "tsp_mod_registry.h"

RegisterModule

/*
 *  A Conditional is a sort of if-then statement. In 
 *  its simplest form it looks something like this:
 *
 *      Cond({:if-this-is-true}, 
 *              {:then-do-this});
 *
 *  The "if-this-is-true" argument of the Conditional is 
 *  called the predicate. A predicate is basically any
 *  anonymous function that evaluates to a true or nil value. 
 *  
 *  A Conditional may have more than one predicate clause; 
 *  if so, the Conditional will continue until one of the 
 *  predicates proves to be true, and then skip any remaining 
 *  predicate clauses.
 *
 *      Cond({:predicate-1}, 
 *              {:do-this-1},
 *          {:predicate-2),
 *              {:do-this-2},
 *          {:predicate-3}
 *              {:do-this-3});
 *  
 *  If predicate-1 is true then do-this-1 will occur, and the
 *  second and third clauses will be skipped. If predicate-1
 *  is nil, the Conditional will then look at predicate-2, etc.
 *   
 *  The "then-do-something" argument is any anonymous function.
 *  The Conditional will return any value returned by the 
 *  corresponding "then-do-something" argument. If the 
 *  argument doesn't return a value or none of the predicates
 *  evaluate to true, the Conditional returns nil.
 */
Cond([args])
{
    local pred, act, pVal, ret;

    try {
        /*
         *  If the length is not a multiple of 2 then we
         *  throw a conditional argument length exception.
         */
        if (args.length() < 2 || args.length() % 2 != 0)
            throw new CondArgLenError();

        /*
         *  Loop over each pair of predicate and action
         *  arguments. 
         */
        for (local i = 1; i < args.length(); i += 2)
        {
            pred    = args[i];
            act     = args[i+1];

            /*
             *  If the predicate isn't an anonymous
             *  function then we throw a conditional 
             *  argument type exception.
             */
            if (dataType(pred) != TypeObject
                || !pred.ofKind(AnonFuncPtr))
                throw new CondArgTypeError();

            /*
             *  If the action isn't an anonymous
             *  function then we throw a conditional 
             *  argument type exception.
             */
            if (dataType(act) != TypeObject
                || !act.ofKind(AnonFuncPtr))
                throw new CondArgTypeError();

            /*
             * Evaluate the predicate
             */
            pVal = (pred)();

            /*
             *  If the predicate is non-nil / non-zero value
             *  then we execute the associated action anonymous
             *  function and break out of the loop.
             */
            if (pVal)
            {
                ret = (act)();
                break;
            }
        }
    }

    /*
     *  Catch any runtime errors and handle their
     *  display here.
     */
    catch(RuntimeError r)
    {
        "\b<<r.displayException()>>\n";
    }

    /*
     *  Catch any other exception and do nothing
     */
    catch(Exception e)
    {
        // do nothing
    }
    finally
    {
        return ret;
    }
}

/*
 *  A base class for conditional argument length errors
 */
class CondArgLenError: RuntimeError
{
    exceptionMessage = 'Invalid cond argument length.'
    construct()
    {
        inherited(0);
    }
}

/*
 *  A base class for conditional argument type errors
 */
class CondArgTypeError: RuntimeError
{
    exceptionMessage = 'Invalid cond argument type.'
    construct()
    {
        inherited(0);
    }
}

/*
 *  A Case is a sort of switch statement. In 
 *  its simplest form it looks something like this:
 *
 *      Case({:expression},
 *          {:condition-1}, 
 *              {:action-1});
 *
 *  The expression is any anonymous function that
 *  evaluates to a value.
 *
 *  A condition is any anonymous function that 
 *  evaluates to a value. 
 *  
 *  A Case may have more than one condition clause; 
 *  if so, the Case will continue until one of the 
 *  conditions matches the evaluated expression, and 
 *  then skip any  remaining condition clauses.
 *
 *      Case({:expression}, 
 *          {:condition-1),
 *              {:do-this-1},
 *          {:condition-2),
 *              {:do-this-2},
 *          {:condition-3}
 *              {:do-this-3});
 *  
 *  If condition-1 equals expression then do-this-stuff-1 will occur,
 *  and the second and third clauses will be skipped. If condition-1
 *  doesn't match expression, the Case will then look at condition-2, etc.
 *   
 *  The "do-this" argument is any anonymous function. The Case 
 *  will return any value returned by the corresponding "do-this" 
 *  argument. If the argument doesn't return a value or none of 
 *  the conditions evaluate to the expression, the Case returns nil.
 */
Case(expr, [args])
{
    local eVal, cVal, cond, act, ret;

    try {
        /*
         *  If the expression isn't an anonymous
         *  function then we throw a case 
         *  argument type exception.
         */
        if (dataType(expr) != TypeObject
            || !expr.ofKind(AnonFuncPtr))
            throw new CaseArgTypeError();

        /*
         *  If the length is not a multiple of 2 then we
         *  throw a case argument length exception.
         */
        if (args.length() < 2 || args.length() % 2 != 0)
            throw new CaseArgLenError();

        eVal = (expr)();

        /*
         *  Loop over each pair of predicate and action
         *  arguments. 
         */
        for (local i = 1; i < args.length(); i += 2)
        {
            cond    = args[i];
            act     = args[i+1];

            /*
             *  If the condition isn't an anonymous
             *  function then we throw a case 
             *  argument type exception.
             */
            if (dataType(cond) != TypeObject
                || !cond.ofKind(AnonFuncPtr))
                throw new CaseArgTypeError();

            /*
             *  If the action isn't an anonymous
             *  function then we throw a conditional 
             *  argument type exception.
             */
            if (dataType(act) != TypeObject
                || !act.ofKind(AnonFuncPtr))
                throw new CaseArgTypeError();

            cVal = (cond)();

            /*
             *  If the expression equals the condition
             *  then we execute the associated action anonymous
             *  function and break out of the loop.
             */
            if (eVal == cVal)
            {
                ret = (act)();
                break;
            }
        }
    }

    /*
     *  Catch any runtime errors and handle their
     *  display here.
     */
    catch(RuntimeError r)
    {
        "\b<<r.displayException()>>\n";
    }

    /*
     *  Catch any other exception and do nothing
     */
    catch(Exception e)
    {
        // do nothing
    }
    finally
    {
        return ret;
    }
}

/*
 *  A base class for conditional argument length errors
 */
class CaseArgLenError: RuntimeError
{
    exceptionMessage = 'Invalid case argument length.'
    construct()
    {
        inherited(0);
    }
}

/*
 *  A base class for conditional argument type errors
 */
class CaseArgTypeError: RuntimeError
{
    exceptionMessage = 'Invalid case argument type.'
    construct()
    {
        inherited(0);
    }
}

/*
 *  This function is meant to handle cases where
 *  we're checking to see if an object is one of a 
 *  list of object classes. 
 *
 *  We first check to see if obj is not an object 
 *  dataType. If this is the case then we check to see
 *  if there is a direct match within the vals list. This
 *  handles the case of (obj is in (val[1], val[2], val[3])) 
 *
 *  Next we check to see if obj (now treated as an object
 *  dataType) is ofKind(val[n]) for any of the elements of 
 *  the vals list. If we return true for the ofKind() check
 *  we return immediately with true.
 */
isInOfKind(obj, [vals])
{
    local ret;

    /*
     *  We check to see if obj is a member of the list. This
     *  handles cases where obj is nil or where obj directly
     *  matches one of the values.
     */
    ret = vals.indexOf(obj);
    if (ret)
        return true;

    /*
     *  At this point we're only interested in obj
     *  being of TypeObject.
     */
    if (dataType(obj) == TypeObject)
    {
        foreach (local val in vals)
        {
            /*
             *  If the value isn't an object then we'll
             *  bypass the check. A non-object dataType 
             *  was tested for earlier.
             */
            if (dataType(val) != TypeObject)
                continue;

            if (obj.ofKind(val))
                return true;
        }
    }

    return nil;
}

notInOfKind(obj, [vals])
{
    return !isInOfKind(obj, vals...);
}

/*
 *  This function checks that the condition is true for each 
 *  argument triplet. The argument triplet consists of the following:
 *
 *      arg[i]  : whether val is expected to be found in the list
 *      arg[i+1]: the val to be searched for in the list
 *      arg[i+2]: the list of values or a value to be searched or compared against
 *
 *  An example of this syntax would be:
 *  
 *      inVals(true, currBitFlg, [bitFlg1, bitFlg2],
 *              nil, prevBitFlg, [bitFlg3, bitFlg4, bitFlg5])
 *
 *  inVals will return true if each triplet forms a true condition. Otherwise
 *  it will return nil when the first untrue condition is encountered.
 *
 *  So the above example would be equivalent to:
 *
 *      (([bitFlg1, bitFlg2].indexOf(currBitFlg) != nil)
 *          || ([bitFlg3, bitFlg4, bitFlg5].indexOf(currBitFlg) == nil))
 */
inVals([args])
{
    local isIn, val, valList, ret;

    if ((args.length() == 0) || (args.length() % 3 != 0))
        throw new InValsArgLenError();

    for (local i = 1; i <= args.length() - 2; i += 3)
    {
        isIn    = args[i];
        val     = args[i+1];
        valList = [] + args[i+2];

        ret = isInOfKind(val, valList...);

        if ((isIn && !ret)
            ||(!isIn && ret))
            return nil;
    }
 
    return true;
}

/*
 *  A base class for inVals() argument length errors
 */
class InValsArgLenError: RuntimeError
{
    exceptionMessage = 'Invalid inVals argument length.'
    construct()
    {
        inherited(0);
    }
}

modify Collection
{
    /*
     *  Returns true if the collection contains an element 
     *  of dataType or class vals. Otherwise returns nil.
     */
    hasKind([vals])
    {
        foreach (local elm in self)
            if (isInOfKind(elm, vals...))
                return true;

        return nil;
    }

    /*
     *  Returns true if the collection doesn't contain an element 
     *  of dataType or class vals. Otherwise returns nil.
     */
    hasntKind([vals])
    {
        return !hasKind(vals...);
    }
}