#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Prototype.t
 *
 *  Provides a prototypical view of the TADS 3 object model.
 *
 *  By prototypical we mean the definition of the following
 *  inheritance relationships:
 *
 *      ancestor: An object from which this object is derived
 *      parent: An immediate ancestor to this object
 *      decendant: An object which derives from this object
 *      child: An immediate descendant of this object
 */

#include "proteus.h"
#include "bignum.h"
     
modify Object
{
    /*
     *  Returns a list describing prop. The values of the list are:
     *
     *      v[1] - The prop's TypeXxxx property type 
     *      v[2] - The prop's PropDefXxxx property defined value
     *      v[3] - The prop's defining class for this object.
     */
    propDesc(prop, flag)
    {
         local v = new Vector(3);

        /*
         *  If the property is defined directly or inherited 
         *  we note this in element #2 of the vector. If the 
         *  property is in the property list for this object, 
         *  but is not directly defined or inherited, then we
         *  note this as PropDefNonStatic (as a non-static property
         *  of an intrinsic class.
         */
        if (propDefined(prop, PropDefDirectly))
            v[2] = PropDefDirectly;
        else if (propDefined(prop, PropDefInherits))
            v[2] = PropDefInherits;
        else if (getPropList().indexOf(prop) != nil)
            v[2] = PropDefNonStatic;
        else
            v[2] = PropDefNotDefined;

        if (v[2] == PropDefNotDefined)
        {
            v[1] = dataType(prop);
            v[3] = nil;
        }
        else if (v[2] == PropDefNonStatic)
        {
            /* note that this is a non-static property */
            v[1] = TypePropNonStatic;

            /* note this object as the defining class */
            v[3] = self;
        }
        else
        {
            /* note the property type */
            v[1] = propType(prop);

            /* note the defining class */
            v[3] = propDefined(prop, PropDefGetClass);

            /*
             *  If the property doesn't appear in the property
             *  list of the defining class then note this object
             *  as the defining class and that the property is
             *  defined directly by this object.
             *
             *  This handles the case where unnamed internals 
             *  resulting from modifications of intrinsic classes
             *  define the property. These internals are not part
             *  of the firstObj()/nextObj() loop, and are for all
             *  intents and purposes invisible.
             */
            if (v[3].getPropList() == nil)
            {
                if (getPropList().indexOf(prop) == nil)
                {
                    local s;

                    v[2] = PropDefInherits;

                    s = getSuperclassList().car();
                    while(s)
                    {
                        if (s.getPropList().indexOf(prop) != nil)
                            break;

                        s = s.getSuperclassList().car();
                    }
                    v[3] = s;
                }
                else
                {
                    v[2] = PropDefDirectly;
                    v[3] = self;
                }
            }
        }

        switch(flag)
        {
            case PropDescType:
                return v[1];

            case PropDescDefined:
                return v[2];

            case PropDescGetClass:
                return v[3];

            case PropDescInfo:
                return v.toList();

            default:
                return nil;
        }
    }

    /*
     *-------------------------------------------------------------------
     *  ANCESTORS
     *-------------------------------------------------------------------
     */

    /*
     *  Returns a list of all the unique inheritance 
     *  ancestors for this object.
     */
    getAncestorList([args])
    {
        local flgs, superclassList, ancestorList = [];

        flgs = args.car();
        if (flgs == nil)
            flgs = ObjAll;

        /* get this object's superclass list */
        superclassList = getSuperclassList();

        if (!((flgs & ObjAll) != 0))
            if ((flgs & ObjInstances) != 0)
                superclassList = superclassList.subset({x: !x.isClass()});
            else if ((flgs & ObjClasses) != 0)
                superclassList = superclassList.subset({x: x.isClass()});
    
        /* add this object's superclasses to its ancestor list */
        ancestorList = ancestorList.appendUnique(superclassList);

        /*
         *  For each object in the superclass list, get its
         *  ancestorList and add it to this one.
         */
        foreach (local obj in superclassList)
            ancestorList = ancestorList.appendUnique(
                obj.getAncestorList(args...));
        
        return ancestorList; 
    }

    /*
     *  Returns true if this object is an ancestor of obj;
     *  otherwise returns nil.
     */
    isAncestorOf(obj)
    {
        return (obj.getAncestorList().indexOf(self) != nil);
    }

    /*
     *  Get a list of the first ancestors for this object excluding any
     *  in which func returns true. If no function is passed the list
     *  is identical to that returned by getSuperclassList().
     */
    getFirstAncestorList([args])
    {
		local scList, pList = [], func;

        if (args.length() == 0)
        {
            func = new function(o) {return true;};
            args = new function(o) {return nil; };
        }
        else
            func = args.car();

		/* if func returns true, exclude this object from the trait list */
		if (!func(self)) 
			return (pList + self);

		scList = getSuperclassList();
        
        foreach (local sc in scList)
		    pList += sc.getFirstAncestorList(args...);

        return pList;
    }

    /*
     *  Returns true if this object is a first named
     *  ancestor of obj; otherwise returns nil.
     */
    isFirstAncestorOf(obj)
    {
        return (obj.getFirstAncestorList().indexOf(self) != nil);
    }

    /*
     *  Returns a list of the first symbolic ancestors for this object. If 
     *  the object is dynamically created this will be a list of the 
     *  symbolic reference-able object(s) or class(s) from which this 
     *  object is derived. 
     */
    getFirstSymAncestorList()
    {
        local myParent = self;

        return getFirstAncestorList(new function(o) 
        {
            // exclude the parent object
            if (o == myParent)
                return true;

            // exclude a dynamic or anonymous object
            return (o.isDynamOrAnon());
        });
    }

    /*
     *  Returns true if this object is a first symbolic
     *  ancestor of obj; otherwise returns nil.
     */
    isFirstSymAncestorOf(obj)
    {
        return (obj.getFirstSymAncestorList().indexOf(self) != nil);
    }


    /*
     *  Returns a list of the first named symbolic ancestors for this object. 
     *  If the object is dynamically-created or an unnamed internal this 
     *  will be a list of the named symbolic reference-able objects or 
     *  classes from which this object is derived. 
     */
    getFirstNamedAncestorList()
    {
        local myParent = self;

        return getFirstAncestorList(new function(o) 
        {
            // exclude the parent object
            if (o == myParent)
                return true;

            // exclude a dynamic or anonymous object or unnamed internals
            return (o.isDynamOrAnon() || o.isUnnamedInternal());
        });
    }

    /*
     *  Returns true if this object is a first named
     *  ancestor of obj; otherwise returns nil.
     */
    isFirstNamedAncestorOf(obj)
    {
        return (obj.getFirstNamedAncestorList().indexOf(self) != nil);
    }

    /*
     *-------------------------------------------------------------------
     *  PARENTS
     *-------------------------------------------------------------------
     */

    /*
     *  Returns a list of all the inheritance parents (immediate 
     *  ancestors) for this object. This is the same as the intrinsic
     *  getSuperclassList().
     */
    getParentList([args])
    {
        local flgs, lst;;

        flgs = args.car();
        if (flgs == nil)
            flgs = ObjAll;

        lst = getSuperclassList();

        if ((flgs & ObjAll) != 0)
            return lst;
        else if ((flgs & ObjInstances) != 0)
            return lst.subset({x: !x.isClass()});
        else if ((flgs & ObjClasses) != 0)
            return lst.subset({x: x.isClass()});
        else 
            return lst;
    }

    /*
     *  Returns true if this object is a parent of obj;
     *  otherwise returns nil.
     */
    isParentOf(obj)
    {
        return (obj.getParentList().indexOf(self) != nil);
    }

    /*
     *-------------------------------------------------------------------
     *  DESCENDANTS
     *-------------------------------------------------------------------
     */

    /*
     *  Returns a list of all the inheritance descendants
     *  for this object.
     */
    getDescendantList([args])
    {
        local flgs, lst = new Vector(1000);

        flgs = args.car();
        if (flgs == nil)
            flgs = ObjAll;

        lst += getChildList(ObjClasses);

        foreach (local o in lst)
            lst += o.getDescendantList(args...);

        return lst.toList();
    }

    /*
     *  Returns true if this object is a descendant of obj;
     *  otherwise returns nil.
     */
    isDescendantOf(obj)
    {
        return (obj.getDescendantList().indexOf(self) != nil);
    }

    /*
     *  Get a list of the first children for this object excluding any
     *  in which func returns true. If no function is passed the list
     *  is identical to that returned by getChildList().
     */
    getFirstDescendantList([args])
    {
		local dList, pList = [], func;

        if (args.length() == 0)
        {
            func = new function(o) {return true;};
            args = new function(o) {return nil; };
        }
        else
            func = args.car();

		/* if func returns true, exclude this object from the child list */
		if (!func(self)) 
			return (pList + self);

		dList = getChildList(ObjAll);
        
        foreach (local d in dList)
		    pList += d.getFirstDescendantList(args...);

        return pList;
    }

    /*
     *  Returns true if this object is a first descendant
     *  of obj; otherwise returns nil.
     */
    isFirstDescendantOf(obj)
    {
        return (obj.getFirstDescendantList().indexOf(self) != nil);
    }

    /*
     *  Returns a list of the first named children for this object. 
     *  If the object is an unnamed internal this will be a list of the 
     *  symbolic reference-able object(s) or class(s) from which this 
     *  object is derived. 
     */
    getFirstNamedDescendantList()
    {
        local myParent = self;

        return getFirstDescendantList(new function(o) 
        {
            // exclude the parent object
            if (o == myParent)
                return true;

            // exclude unnamed internals, dynamic or anonymous objects
            return (o.isUnnamedInternal() || o.isDynamOrAnon());
        });
    }

    /*
     *  Returns true if this object is a first named descendant
     *  of obj; otherwise returns nil.
     */
    isFirstNamedDescendantOf(obj)
    {
        return (obj.getFirstNamedDescendantList().indexOf(self) != nil);
    }

    /*
     *-------------------------------------------------------------------
     *  CHILDREN
     *-------------------------------------------------------------------
     */

    /*
     *  Returns a list of all the inheritance children (immediate
     *  descendants) for this object.
     */
    getChildList([args])
    {
        local flgs, vec = new Vector(1000);

        flgs = args.car();
        if (flgs == nil)
            flgs = ObjAll;

        for (local o = firstObj(self, flgs); o != nil; o = nextObj(o, self, flgs))
        {
            if (dataType(o) == TypeObject 
                && o.getSuperclassList().indexOf(self))
                vec.append(o);
        }

        return vec.toList();
    }

    /*
     *  Returns true if this object is a child of obj; 
     *  otherwise returns nil.
     */
    isChildOf(obj)
    {
        return (obj.getChildList().indexOf(self) != nil);
    }

    /*
     *-------------------------------------------------------------------
     *  OTHER RELATIONSHIPS
     *-------------------------------------------------------------------
     */

    getObjModList()
    {
        local obj, lst = [];

        if (isUnnamedInternal())
            obj = getFirstNamedDescendantList().car();
        else obj = self;
        
        lst += obj;
        lst += obj.getUnnamedInternalAncestorList();

        return lst;
    }

    /*
     *  Returns true if this object is an object modification 
     *  of obj; otherwise returns nil.
     */
    isObjModOf(obj)
    {
        return (obj.getObjModList().indexOf(self) != nil);
    }

    getUnnamedInternalAncestorList()
    {
        local lst = [], pList, obj;
        
        pList = getParentList();

        obj = pList.car();

        if (obj == nil)
            return [];

        if (!obj.isUnnamedInternal())
            if (isUnnamedInternal())
                return [] + self;
            else
                return [];

        if (isUnnamedInternal())
            lst += self;

        lst += obj.getUnnamedInternalAncestorList();

        return lst;
    }

    /*
     *  Returns true if this object is an object modification 
     *  of obj; otherwise returns nil.
     */
    isUnnamedInternalAncestorOf(obj)
    {
        return (obj.getUnnamedInternalAncestorList().indexOf(self) != nil);
    }

    /*
     *  Returns true if the object is directly
     *  derived from trait or == trait, without any
     *  intervening multiple inheritance structures.
     *  Otherwise returns nil.
     */
    isDirectLineageOfKind(trait)
    {
        local sc;

        if (self == trait)
            return true;

        if (self == Object)
            return nil;

        sc = getSuperclassList();

        if (sc.length() != 1)
            return nil;

        return sc.car().isDirectLineageOfKind(trait);
    }

    /*
     *  Returns true if this object is ofKind(obj) or
     *  is an unnamed internal that resulted from a 
     *  modification of obj. Otherwise returns nil.
     * 
     *  This method catches runtime errors that might be
     *  caused by calls to an object's ofKind() or indexOf()
     *  method.
     */
    ofKindOrUU(obj)
    {
        try
        {
            if (ofKind(obj))
                return true;
            else
            {
                local lst = getObjModList();

                if (lst.indexOf(obj) != nil)
                    return true;
                else
                    return nil;
            }
        }
        catch(RuntimeError e)
        {
            return nil;
        }
    }

    /*
     *  Determine the object that directly-defines 
     *  the property for this object.
     */
    getPropDefObj(prop, [args])
    {
        local indx, orderList;

        args.length() == 0 ? indx = 1 : indx = args.car();

        orderList = getInherStrucList();

        for (local i = indx; i <= orderList.length(); ++i)
            if (orderList[i].getPropList().indexOf(prop))
                return orderList[i];

        return nil;
    }

    /*
     *  Returns the appropriate PropDefXXX value for 
     *  the property. If no object in the inheritance 
     *  tree defines the property then the method 
     *  returns nil.
     */
    getPropDefVal(prop, [args])
    {
        local obj = getPropDefObj(prop, args...);

        if (obj == nil)
            return nil;
        else
            if (obj == self)
                return PropDefDirectly;
            else
                return PropDefInherits;
    }

    /*
     *  This is method is equivalent to ofKindOrUU() except that 
     *  it doesn't rely on the ofKind() method definition. Instead
     *  it uses the object's inheritance order in its determination.
     */
    hasTrait(trait)
    {
        return (getInherStrucList().indexOf(trait) != nil);
    }
}

modify TadsObject
{
    /*
     *  Returns a list of all the inheritance descendants
     *  for this object.
     */
    getDescendantList([args])
    {
        local flgs, vec = new Vector(1000);

        flgs = args.car();
        if (flgs == nil)
            flgs = ObjAll;

        for (local o = firstObj(self, flgs); o != nil; o = nextObj(o, self, flgs))
        {
            vec.append(o);
        }

        return vec.toList();
    }
}