#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of PROTEUS, the TADS 3 Utility Classes Package
 *
 *  ProteusLU.t
 *
 *  Provides an abstract datatype class that encapsulates a lookup table
 *  for use by other proteus modules.
 */

/* include the Proteus header */
#include "proteus.h"

proteusLUPreinit: PreinitObject
{
    /* execute this preinit after the symbol table has been built */
    execBeforeMe = [adv3LibPreinit, symbolTablePreinit, proteusPreinit]

    execute()
    {
        local o;

        o = firstObj(TadsObject, ObjAll);
        while (o != nil)
        {
            foreach (local lookupTable in gProteusLU)
                lookupTable.initialize(o);

            o = nextObj(o, TadsObject, ObjAll);
        }
    }
}

/*
 *  An abstract datatype that encapsulates a lookup Table class for 
 *  use by other proteus modules. The class is a preinit object that 
 *  provides an API for storing and retrieving a list of values 
 *  associated with an object. 
 *
 *  The values list is initially built up during preinit by referencing
 *  the concrete lookup class' &myValuesPtr for each TadsObject that
 *  defines the property as a list.
 *
 *  Values should be added to the table using the property defined by
 *  the concrete lookup table's property referenced by &myAddProperty.
 *  This allows an each concrete lookup table to define a method that
 *  is meaningful for the table and that can be used to validate the 
 *  object / value requirements of the concrete table.
 */
class ProteusLU: object
{
    tab_            = nil
    
    /*
     *  This should be assigned the property pointer for the list of
     *  values to be stored as object / value list pairs in the
     *  encapsulated lookup table.
     */
    myValuesProp     = nil
    
    /*
     *  This should be assigned the property pointer for the method
     *  used by this table to validate and store object / values pairs
     *  in the encapsulated lookup table.
     */
    myAddProp           = nil

    myServiceSuffix  = nil
    
    /*
     *  Returns a list of values stored in the lookup table for this
     *  object. 
     *
     *  The hasMoreValues(obj) should be used in conjunction with this
     *  method to determine when no more values are available for this
     *  object.
     */
    getValues(obj)
    {
        local values;
        
        /* if tab_ is nil then we've not created the lookup table yet */
        if (tab_ == nil)
            throw new TableNotInitializedPLUException(self);
        
        /*
         *  If this object is not a key of the lookup table, throw an
         *  exception.
         */
        if (!tab_.isKeyPresent(obj))
            throw new KeyNotFoundPLUException(self, obj);
            
        values = tab_[obj];
        
        /* 
         *  If lst is not a list datatype throw an exception.
         */
        if (dataType(values) != TypeList)
            throw new ValuesNotFoundPLUException(self, obj);
            
        return values;
    }
    
    /* 
     *  Returns true if obj has a values list in this table. If obj is
     *  not a key of this table, or values is not a list dataype, or if
     *  values is an empty list then returns nil.
     */
    hasMoreValues(obj)
    {
        local values;
        
        try {
            values = getValues(obj);
            return (values.length() != 0);
        }
        catch(PLUException e)
        {
            return nil;
        }
    }
    
    /*
     *  Returns true if the table stores val in its obj / values list
     *  pair; otherwise returns nil.
     */
    hasValue(obj, val)
    {
        local values;
        
        if (!hasMoreValues(obj))
            return nil;
            
        values = getValues(obj);
        
        return (values.indexOf(val) != nil);
    }
    
    /*
     *  This method stores a value in the encapsulated lookup table for 
     *  the object.
     *  
     *  By the time we get here the object value should have met any 
     *  validation checks required by the concrete lookup table and thus
     *  can be safely added to the table.
     *
     *  This method should be called by the concrete lookup's
     *  &addPropPtr method.
     */
    storeValue(obj, val)
    {
        local values = [];

        /* if tab_ is nil then we've not created the lookup table yet */
        if (tab_ == nil)
            throw new TableNotInitializedPLUException(self);
        
        /*
         *  if the object is already a key of the lookup table then we
         *  retrieve its current value list so that we can add this value 
         *  to the list.
         */
        if (tab_.isKeyPresent(obj))
            values += tab_[obj];
            
        /* uniquely add this value to the list */
        values = values.appendUnique([val]);
        
        /* store the list in the lookup table */
        tab_[obj] = values;
    }
    
    /*
     *  Removes the values list for this object. Throws an exception if
     *  obj is not a key of this table.
     *
     *  The hasMoreValues(obj) method can be used in conjunction with
     *  this method to avoid having to handle an exception.
     */
    removeValues(obj)
    {
        /* if tab_ is nil then we've not created the lookup table yet */
        if (tab_ == nil)
            throw new TableNotInitializedPLUException(self);
        
        if (!tab_.isKeyPresent(obj))
            throw new KeyNotFoundPLUException(self, obj);
            
        tab_.removeElement(obj);
    }
    
    /*
     *  Removes val from the values list for this object. Throws an
     *  exception if obj is not a key of this table. Does nothing if val
     *  is not found in the values list for obj.
     */
    removeValue(obj, val)
    {
        local values;
        
        /* if tab_ is nil then we've not created the lookup table yet */
        if (tab_ == nil)
            throw new TableNotInitializedPLUException(self);
        
        if (!tab_.isKeyPresent(obj))
            throw new KeyNotFoundPLUException(self, obj);
            
        /* get the values list for the object */
        values = getValues(obj);
        
        /* remove any occurrences of val from the values list */
        values -= val;
        
        /* store the new values list for this object */
        tab_[obj] = values;
    }

    construct()
    {
        tab_ = new LookupTable();
    }
    
    /*
     *  Executed during Preinit, this method adds each object and its
     *  associated values to the concrete lookup table when the object 
     *  defines the method referenced by the &addPropPtr.
     */
    initialize(obj)
    {
            /* skip all the lookup tables */
            if (!obj.ofKindOrUU(ProteusLU) && !obj.ofKindOrUU(symbolTablePreinit))
            {                
                if (obj.propDefined(myValuesProp)
                && obj.propType(myValuesProp) == TypeList
                && obj.(myValuesProp).length() != 0)
                {
                    foreach (local ele in obj.(myValuesProp))
                        self.(myAddProp)(obj, ele);
        
                    /* 
                     *  This object has been added to the lookup table, we
                     *  set the object's property to nil, which is as close
                     *  to removing it as we can get.
                     */
                    obj.(myValuesProp) = nil;
                }

                /* 
                  *   If the object has node service references
                  *   store them 
                  */
                if (myServiceProp 
                    && obj.propType(myServiceProp) == TypeTrue)
                    storeNestedServices(obj);
            }
    } 
    
    storeNestedServices(obj)
    {
        servicePropertyIter(obj, {x,y : self.(myAddProp)(x,y)});
    }

    servicePropertyIter(obj, objFunc)
    {
        local val, propList = [];

        /* If there's no service suffix, there's no nested services */
        if (myServiceSuffix == nil)
            return;

        /* get a list of properties for this object */
        propList = obj.getInherPropList();

        /* 
          *   Strip out only properties that are suffixed with
          *   myServicesSuffix.
          */
        propList = propList.subset(new function(prop) {
            local str;
            
            str = String.toSString(prop);
            return str.endsWith(myServiceSuffix);
        });
        
        /*
          *   Iterate over each property in the list. For those 
          *   properties that have object property types execute
          *   the objFunc callback function. 
          */
        foreach (local prop in propList)
        {
            if (obj.propType(prop) != TypeObject)
                continue;
            
            val = obj.(prop);
            
            objFunc(obj,val);
        }
    }
}

class PLUException: ProteusException
{
    construct() { inherited(); }
}

class TableNotInitializedPLUException: PLUException
{
    construct(tbl)
    {
        local tblName;
        
        tblName     = String.toSString(tbl);
        
        exceptionMessage = '[' +  tblName + ' lookup table has not
            been initialized.]';
            
        inherited();
    }
}

class KeyNotFoundPLUException: PLUException
{
    construct(tbl, obj)
    {
        local objName, tblName;
        
        objName     = String.toSString(obj);
        tblName     = String.toSString(tbl);
        
        exceptionMessage = '[' + objName + ' is not a key 
            in the ' + tblName + ' lookup table.]';
            
        inherited();
    }
}

class ValuesNotFoundPLUException: PLUException
{
    construct(tbl, obj)
    {
        local objName, tblName;
        
        objName     = String.toSString(obj);
        tblName     = String.toSString(tbl);
        
        exceptionMessage = '[' + objName + ' does not store a values
            list in the ' + tblName + ' lookup table.]';
            
        inherited();
    }
}