#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Observer.t
 *
 *  Provides registration / notification of state-change events.
 */

/* include the proteus lookup table header */
#include "proteus.h"

/*
  *-------------------------------------------------------------------
  *   OBSERVER PROCESS: This section defines an abstract
  *   ObserverProcess class
  *------------------------------------------------------------------
  */

/*
 *  ObserverProcess should be used to register and unregister objects
 *  that are to be notified of events. notify() can be called by author
 *  code.
 */
class ObserverProcess: object
{
    myObservables = []
    
    /*
     *  Register Subject with the Observer Process.
     */
    addObservable(observable)
    {
        gObservables.addObservable(self, observable);
    }
    
    /*
     *  Remove obj from Observer Process registration.
     */
    removeObservable(observable)
    {
        gObservables.removeObservable(self, observable);
    }
    
    /*
     *  Notify registered Observables of event. By default this does
     * nothing and should be defined by the concrete observer 
     * process.
     */
    notify() {}
}

/*
 *  An Exception class for Observer Process.
 */
class ObserverProcessException: ProteusException
{
    construct(obj, interface) 
    {
        local objName, interfName;
        
        objName     = String.toSString(obj);
        interfName  = String.toSString(interface);
        exceptionMessage = '[' + objName + ' does not implement the ' +
            interfName + ' interface.]';
            
        inherited();
    }
}

/*
  *-------------------------------------------------------------------
  *   OBSERVABLE: This section contains code used to setup
  *   Observer Process / Observable pairs in the Observable 
  *   Lookup Table.
  *------------------------------------------------------------------
  */

class ObservableLU: ProteusLU
{
    /* lookup table custom props for Observable */
    luCustomProps(Observable)

    instantiate()
    {
        return new ObservableLU();
    }

    construct()
    {
        inherited();
        gObservables = self;
    }

    /*
       *  Register this observable with this observer process.
       */
    addObservable(observProc, observable)
    {
        if (!observable.implementsIF(ObservableIF))
            throw new ObserverProcessException(observable,
                ObservableIF);

        /* store the ObserverProcess / Observable pair */
        storeValue(observProc, observable);
    }  
    
    /*
     * Unregister observable from this observer processs.
     */
    removeObservable(observProc, observable)
    {
        /* remove the ObserverProcess / Observable pair */
        if (hasValue(observProc, observable))
            removeValue(observProc, observable);
    }   
}

/*
 *  The Observable Interface class
 */
class ObservableIF: Interface
{
    addObserver(observer) {}
    removeObserver(observer) {}
    notifyObserver([args]) {}
}

class Observable: object
{
    /*
     *  This object implements the ObservableIF interface.
     */
    implements(ObservableIF)

    myObservers   = []
    
    /*
     *  Register observer with this observable.
     */
    addObserver(observer)
    {
        gObservers.addObserver(self, observer);
    }
    
    /*
     *  Unregister this observer from this observable.
     */
    removeObserver(observer)
    {
        gObservers.removeObserver(self, observer);
    }   
    
    /*
      *   Notify the observers for this observable of the observable's
      *   event.
      */
    notifyObserver([args])
    {
        local observerList = gObservers.getValues(self);
        
        foreach (local observer in observerList)
            observer.updateEventChange(self, [args]);    
    }
}

/*
 *-------------------------------------------------------------------
 *   OBSERVER: This section contains code used to setup
 *   Observable / Observer pairs in the Observer Lookup Table.
 *------------------------------------------------------------------
 */

class ObserverLU: ProteusLU
{
    /* lookup table custom props for Observer */
    luCustomProps(Observer) 
     
    instantiate()
    {
        return new ObserverLU();
    }

    construct()
    {
        inherited();
        gObservers = self;
    }
    /*
       *  Register this observer for the observable.
       */
    addObserver(observable, observer)
    {
         if (!observer.implementsIF(ObserverIF))
            throw new ObserverProcessException(observer, ObserverIF);
            
        /* store the Observable / Observer pair */
        storeValue(observable, observer);
    }  
    
    /*
     *  Unregister this observer from the observable.
     */
    removeObserver(observable, observer)
    {
        /* remove the Observable / Observer pair */
        if (hasValue(observable, observer))
            removeValue(observable, observer);
    }   
}

/*
 *  The Oberver Interface
 */
class ObserverIF: Interface
{
    updateEventChange(subject, [args]) {}
}

/*
  *   A base class for observers.
  */
class Observer: object
{
    /*
     *  This object implements the ObserverIF interface.
     */
    implements(ObserverIF)
    
    /* by default we do nothing */
    updateEventChange(subject, [args]) {}
}

/*
  *-------------------------------------------------------------------
  *   This section defines classes for a state observer process and
  *   state observer.
  *------------------------------------------------------------------
  */

class StateObserverProcess: ObserverProcess
{
    /*
     *  Notify registered Observables of state-change events.
     */
    notify()
    {
        local bypass, stateImage, delta;
        local observableList = gObservables.getValues(self);
        
        /* bypass list of snapshot-related properties */
        bypass = [&snapshotList, &snapshotReference,
            &snapshotTS];
        
        foreach (local observable in observableList)
        {
            try {
                stateImage = observable.getSnapshot();

                if (!observable.hasStateOf(stateImage, bypass...))
                {
                    delta = observable.getStateDelta(stateImage, bypass...);
                
                    observable.notifyObserver(stateImage, delta);  
                    
                    if (observable.hasMoreSnapshots)
                        observable.removeSnapshot(stateImage);
                }
            }
            /*
             *  We simply catch any errors, but do nothing about them.
             *  If the observable has no snapshot then it's the first
             *  time and we want to pass on to saving its state.
             */
            catch(NoFurtherSnapshotException e) {}
            /*
             *  Each time notify() is called we save the observable's
             *  state.
             */
            finally {
                observable.saveState();
            }
        }
    }
}

/*
 *  StateObserver class defines one method, updateStateChange(), 
 *  which receives two arguments: subject and image, referencing
 *  the current and snapshot subjects providing information on the 
 *  current and previous state of the subject.
 */
class StateObserver: Observer
{
    /* by default we merely display the state delta */
    updateEventChange(subject, [args])
    {
        local image;

        image = args[2];
        
        subject.displayStateDelta(image);
    }
}