#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Interface.t
 *
 *  Provides Interface enforcement.
 */

/* include the proteus lookup table header */
#include "proteus.h"

/*
 *  Determines whether each object that defines an interfaces list
 *  defines the properties and parameters required by each interface.
 */
class InterfaceLU: ProteusLU
{
    /* lookup table custom props for Interface */
    luCustomProps(Interface)
  
    instantiate()
    {
        return new InterfaceLU();
    }

    construct()
    {
        inherited();
        gInterfaces = self;
    }

    /*
     *  Method returns true if the object implements or inherits an
     *  interace; otherwise it returns nil.
     */
    implementsIF(obj, interface)
    {
        local lst, clsList;
        
        /*
         *  If the interface argument is not of kind Interface then we
         *  throw an exception.
         */
        if (!interface.ofKindOrUU(Interface))
            throw new NotAnIFException(self, interface);
            
        /*
         *  If the object is of kind Interface it implements the
         *  interface.
         */
        if (obj.ofKindOrUU(Interface))
            return true;
            
        /*
         *  Build the object's structure list and strip out only the 
         *  classes/objects.
         */
        clsList = obj.getInherSuperclassList();
  
        /*
         *  If any of this object's traits have this
         *  interface stored in the interfaces lookup table then
         *  the object is said to implement the interface.
         *
         *  In other words, if one of the object's superclasses
         *  implements an interface, the object does.
         */
        foreach (local cls in clsList)
        {
            if (hasMoreValues(cls))
            {
                /* get the list of interfaces for the cls */
                lst = getValues(cls);

                /*
                  *   Iterate over the list of interfaces, if any of
                  *   the elements is of kind interface then we 
                  *   return true - the object implements the 
                  *   interface.
                  */
                foreach (local ele in lst)
                {
                    if (ele.ofKindOrUU(interface))
                        return true;
                }
            }
        }
        
        /* this object didn't meet the implementation criteria */
        return nil;
    }
    
    /*
     *  Returns the interfaces for the object from the lookup table 
     */
    getInterfaces(obj) 
    {
        local valList;
        
        try {
            valList = getValues(obj);
            return valList;
        }
        catch(PLUException e)
        {
            /* recast the exception */
            throw new NotFoundIFException(obj);
        }         
    }
    
    /*
     *  This method should be used if adding an interface dynamically to
     *  an object. This method will verify the interface and object
     *  compliance first, before adding it to the lookup table.
     */
    addInterface(obj, interface)
    {
        validateInterfaces(obj, interface);
        validateReqProps(obj, interface);
        storeValue(obj, interface);  
    }
    
    /*
     *  Method throws an exception if the interface is not of kind
     *  Interface or if the interface isn't a class.
     */
    validateInterfaces(obj, interface)
    {
        if (!interface.ofKindOrUU(Interface))
            throw new NotAnIFException(obj, interface);

        if (!interface.isClass())
            throw new NotClassIFException(obj, interface);
    }
    
    /*
     *  Method throws an excpetion if the object's properties do not
     *  comply to the interface's requirements.
     */
    validateReqProps(obj, interface)
    {
        local propList;
        
        /*
         *  Get the interface structure list (Snapshot). This is used to
         *  allow for inherited interface structures. We only build a
         *  Defined Structure list for traits of kind Interface.
         */
        propList = interface.getInherDefPropList(
            new function(obj, prop) {
                if (obj.ofKindOrUU(Interface))
                    return true;
                else return nil;
            });
        
        foreach (local prop in propList)
        {
            local pt, val;

            /*
             *  Ignore the object's reference tag 
             *  and lexicalParent properties.
             */
            if (prop == &objRefTag || prop == &lexicalParent)
                continue;
            
            pt = interface.propType(prop);
            
            switch(pt)
            {
                case TypeCode:
                    /* 
                     *  When the object's property is of type TypeCode, check
                     *  its parameter list against that of the interface
                     *  property. If they're not equal then throw an exception.
                     */
                    if (obj.propType(prop) == TypeCode)
                    {
                        if (interface.getPropParams(prop) !=
                        obj.getPropParams(prop))
                            throw new ParamNotDefinedIFException(obj,
                                interface, prop);
                    }
                    /* the property type for the object is not of TypeCode */
                    else
                    {
                        throw new PropNotDefinedIFException(obj,
                            interface, prop);
                    }
                    break;
                    
                case TypeInt:
                    /* get the string representation for this property */
                    val = String.toSString(prop);
                
                    /*
                     *  If the string reference is not in all caps, throw an
                     *  exception.
                     */
                    if (val != val.toUpper())
                        throw new InvalidConstIFException(interface, prop);
                    
                     /* 
                     *  If the object directly defines this property and
                     *  the object property value is different from the 
                     *  interfaceproperty value then we throw an 
                     *  exception.
                     */
                    if (obj.propDefined(prop, PropDefAny)
                    && obj.(prop) != interface.(prop))
                        throw new ConflictConstIFException(obj, interface, prop);
                   
                    /* add the CONSTANT to the object */
                    obj.(prop) = interface.(prop);
                
                    break;
                    

                default:
                    /* 
                     *  Throw an exception if the interface property is not of
                     *  type TypeCode.
                     */
                    throw new InvalidPropIFException(interface, prop);
            }             
        }
    }    

    servicePropertyIter(obj, objFunc)
    {
        local val, propList = [], sList;

        /* If there's no service suffix, there's no nested services */
        if (myServiceSuffix == nil)
            return;

        /* get a list of properties for this object */
        propList = obj.getInherPropList();

        /* 
          *   Strip out only properties that are suffixed with
          *   myServicesSuffix.
          */
        propList = propList.subset(new function(prop) {
            local str;
            
            str = String.toSString(prop);
            return str.endsWith(myServiceSuffix);
        });
        
        /*
          *   Iterate over each property in the list. For those 
          *   properties that have object property types execute
          *   the objFunc callback function. 
          */
        foreach (local prop in propList)
        {
            if (obj.propType(prop) != TypeObject)
                continue;

            val = obj.(prop);

            /* get the properties directly defined by this instance */
            sList = val.getPropList();
            
            /* remove objRefTag property */
            sList -= &objRefTag;

            /* remove sourceTextOrder property */
            sList -= &sourceTextOrder;

            /* If the object is dynamic or anonymous, get 
             *  the named ancestor for the Interface instance 
             *  and remove any lexicalParent property from its
             *  directly-defined property list.
             */
            if (val.isDynamOrAnon())
            {
                val = val.getFirstNamedAncestorList()[1];
                sList -= &lexicalParent;
            }
            
            /*
              *   Nested object instances must not define their own 
              *   properties.
              */
            if (sList.length() > 0)
                throw new InvalidNestedPropIFException(obj, val, prop);

            /* add the object and static prototype to the table */
            objFunc(obj,val);
        }
    }
}

/* 
 *  A base class for Interface objects
 */
class Interface: object {}

/*
 *  A base class for Interface Exceptions.
 */
class InterfaceException: ProteusException
;

class InvalidPropIFException: InterfaceException
{
    construct(interface, prop)
    {
        local interfName, propName;
        
        interfName  = String.toSString(interface);
        propName    = String.toSString(prop).substr(2);
        
        exceptionMessage = '[Interface ' + interfName + '.'+ propName 
            + ' not a valid datatype for interfaces.]';
            
        inherited.construct();
    }
}

class NotAnIFException: InterfaceException
{
    construct(obj, interface)
    {
        local objName, interfName;
        
        objName     = String.toSString(obj);
        interfName  = String.toSString(interface);
        
        exceptionMessage = '[' + interfName + ' does not 
            inherit from Interface, but is defined as an 
            interface for ' + objName + ']';
            
        inherited.construct();
    }
}

class NotClassIFException: InterfaceException
{
    construct(obj, interface)
    {
        local objName, interfName;
        
        objName     = String.toSString(obj);
        interfName  = String.toSString(interface);
        
        exceptionMessage = '[' + interfName + ' referenced by ' +
            objName + ' is not defined as a class.]';
            
        inherited.construct();
    }
}

class NotFoundIFException: InterfaceException
{
    construct(obj)
    {
        local objName;
        
        objName     = String.toSString(obj);
        
        exceptionMessage = '[' + objName + ' does not implement an 
            interface.]';
            
        inherited.construct();
    }
}

class PropNotDefinedIFException: InterfaceException
{
    construct(obj, interface, prop)
    {
        local objName, interfName, propName;
    
        objName     = String.toSString(obj);
        interfName  = String.toSString(interface);
        propName    = String.toSString(prop);
    
        exceptionMessage = '[Property ' + propName + ' not defined 
            in ' + objName + ', but is required by Interface ' 
            + interfName + ']';
            
        inherited.construct();
    }
}

class ParamNotDefinedIFException: InterfaceException
{
    construct(obj, interface, prop)
    {
        local objName, interfName, propName;
    
        objName     = String.toSString(obj);
        interfName  = String.toSString(interface);
        propName    = String.toSString(prop).substr(2);
    
        exceptionMessage = '[Property parameters required by 
            Interface ' + interfName + '.' + propName + '() not 
            correctly defined for ' + objName + '.' + propName 
            + '()]';
            
        inherited.construct();
    }
}

class InvalidConstIFException: InterfaceException
{
    construct(interface, prop)
    {
        local interfName, propName;
        
        interfName  = String.toSString(interface);
        propName    = String.toSString(prop).substr(2);
        
        exceptionMessage = '[Interface ' + interfName + '.'+ propName 
            + ' does not follow CONSTANT naming convention.]';
            
        inherited.construct();
    }
}

class ConflictConstIFException: InterfaceException
{
    construct(obj, interface, prop)
    {
        local objName, interfName, propName;
        
        objName     = String.toSString(obj);
        interfName  = String.toSString(interface);
        propName    = String.toSString(prop);
        
        exceptionMessage = '[Interface ' + interfName + ' requires '
             + propName + ', which is already directly defined by '
             + objName + ' (possible Interface conflict).]';
             
        inherited.construct();
    }
}

class InvalidNestedPropIFException: Interface
{
    construct(obj, interface, prop)
    {
        local objName, interfName, propName;
        
        objName     = String.toSString(obj);
        interfName  = String.toSString(interface);
        propName    = String.toSString(prop);
        
        exceptionMessage = '[Nested Object instance for ' + interfName 
            + '  defined in ' + objName + ' property ' + propName 
             + ' can only inherit properties from ' + interfName + ' and may
             not directly define any.]';
             
        inherited.construct();
    }
}


/*
 *  Provides methods for determining if this object implements or
 *  inherits a given Interface.
 */
modify TadsObject
{
    implementsIF(interface)
    {
        return gInterfaces.implementsIF(self, interface);
    }
    
    enforcesIF(interface)
    {
        if (!implementsIF(interface))
            throw new NotAnIFException(self, interface);
    }
}