#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of PROTEUS, the TADS 3 Utility Classes Package
 *
 *  delegateObj.t
 *
 *  Creates a dynamic delegation object that an object can delegate work 
 *  to. This object's state is set to that of the delegating object, so 
 *  that the delegated object knows what the caling object knows.
 */

/* include the proteus lookup table header */
#include "proteus.h"

class DelegateObj: object
{
    /*
     *  This is a simplified version of delegationMethod(), which
     *  can take either a delegation object or a list of delegation
     *  objects, defaults to no exclusion list, and to true on caller
     *  synchronization.
     */
    delegateTo(selfObj, delObj, method, [args])
    {
        if (dataType(delObj) == TypeObject)
            delObj = [delObj];

        return delegateMethod(selfObj, delObj, [], DelegateSynch, 
            method, args...);
    }

    delegateMethod(selfObj, dList, eList, synch, method, [args])
    {
        local delObj, val;

        /* create a delegation object */
        delObj = createDelegate(selfObj, dList, eList);

        /* run the method on the delegation object */
        val = delObj.(method)(args...);

        /*
         *  Synchronize this object with the delegation object's 
         *  defined state.
         */
        if (DelegateSynch)
            synchObjToDefStateOf(selfObj, delObj, eList);

        if (synch == DelegateRet)
            /*
             *  Return a list consisting of the delegate 
             *  object and the return value from the method call.
             */
            return [delObj, val];
        else
            /* return the value of the method call */
            return val;
    }

    /* 
     * Creates a delegate object that is an instance of 
     *  the objects listed in dList, but whose state-related 
     *  properties are the same as this object, with the exclusion
     *  of properties listed in eList.
     */
    createDelegate(selfObj, dList, eList)
    {
        local delObj;

        /*
         *  Create a clone of the original bojects. This object
         *  will have the identical set of properties defined in
         *  the originals.
         */
        delObj  = TadsObject.createInstanceOf(dList...);
        synchObjToInherStateOf(delObj, selfObj, eList);

        return delObj;
    }

    /*
     *  Sets this object's state to that of the inherited state of obj, 
     *  with the exception of those attributes listed in exclustion List.
     */
    synchObjToInherStateOf(obj1, obj2, eList)
    {
        local pList;

        pList   = obj2.getInherDefPropList();
        pList   -= eList;

        setStateTo(obj1, obj2, pList);
    }

    /*
     *  Sets this object's state to that of the defined state of obj, 
     *  with the exception of those attributes listed in exclustion List.
     */
    synchObjToDefStateOf(obj1, obj2, eList)
    {
        local pList;

        pList   = obj2.getPropList();
        pList   -= eList;

        setStateTo(obj1, obj2, pList);
    }

    /*
     *  Sets the properties of obj1 that are listed in pList
     *  to those of obj2.
     */
    setStateTo(obj1, obj2, pList)
    {
        pList.forEach(new function(value) 
        {
            local prop;

            prop = value;

            /*
             *  We only update object attributes that are not 
             *  the same as that of obj. 
             */
            if (obj2.propType(prop) not in (TypeDString, TypeCode,
                TypeNativeCode)
                && obj1.(prop) != obj2.(prop))
            {
                obj1.(prop) = obj2.(prop);
            }
         });
    }
}