#charset "us-ascii"

/*
 *   TADS 3 Object Matching. This extension contains functions that provide
 *   complex pattern matching techniques and reduce the need for object
 *   loops.
 *
 *   Copyright 2006, Krister Fundin (fundin@yahoo.com)
 */

#include <tads.h>

/* ---------------------------------------------------------------------- */
/*
 *   Our ModuleID.
 */
ModuleID
    name = 'TADS 3 Object Matching'
    byLine = 'by Krister Fundin'
    htmlByLine = 'by <a href="mailto:fundin@yahoo.com">Krister Fundin</a>'
    version = '3'
;

/* ---------------------------------------------------------------------- */
/*
 *   The following functions provide an interface between the core of this
 *   extension and outside code. They all expect their criteria to be given
 *   in the variable argument list format which is explained in the
 *   documentation.
 */

/*
 *   A function which returns a list of all objects matching a set of
 *   criteria.
 */
all([args])
{
    /* compile the criteria and call our internal counterpart */
    return all_(compileCriteria(args));
}

/*
 *   Count all the object matching a set of criteria.
 */
count([args])
{
    return all(args...).length;
}

/*
 *   Return a random object matching a set of criteria.
 */
any([args])
{
    /* get the matching objects */
    local lst = all(args...);

    /*
     *   return nil if there were no objects in the list; otherwise pick an
     *   object at random and return it
     */
    if (lst == [])
        return nil;
    else
        return rand(lst);
}

/*
 *   Return the object matching a set of criteria AND with the highest value
 *   for a certain property.
 */
most(prop, [args])
{
    /* get all the objects matching the criteria */
    local objs = all(args...);

    /* return the one with the highest value */
    return best(objs, SortDesc, { x: x.(prop) });
}

/*
 *   Return the object matching a set of criteria AND with the lowest value
 *   for a certain property.
 */
least(prop, [args])
{
    /* get all the objects matching the criteria */
    local objs = all(args...);

    /* return the one with the lowest value */
    return best(objs, SortAsc, { x: x.(prop) });
}

/*
 *   Service function: return the best item in a list, using a callback
 *   function to determine the score of each item. By default, this function
 *   uses an 'ascending' order, meaning that the item with the lowest score
 *   will be returned. The constants SortAsc and SortDesc can be used
 *   control this order.
 */
best(lst, descending, func)
{
    local best = nil;
    local bestScore, curScore;

    /* go through the list */
    foreach (local cur in lst)
    {
        /* get the score of the current item */
        curScore = func(cur);

        /* if we're in descending mode, negate the score */
        if (descending)
            curScore = -curScore;

        /* see if it's the first we have or the best so far */
        if (best == nil || curScore < bestScore)
        {
            /* remember this item */
            best = cur;
            bestScore = curScore;
        }
    }

    /* return the best item, or nil if the list was empty */
    return best;
}

/*
 *   A function which determines if an object exists which meets a set of
 *   criteria
 */
exists([args])
{
    /* compile the criteria and call our internal counterpart */
    return exists_(compileCriteria(args));
}

/*
 *   A function which determines if an object meets a set of criteria.
 */
objMeets(obj, [args])
{
    /* compile the criteria and call our internal counterpart */
    return objMeets_(obj, compileCriteria(args));
}

/* ---------------------------------------------------------------------- */
/*
 *   Compile a list of criteria into the internal format.
 */
compileCriteria(args)
{
    local classes = new Vector(3);
    local hasLoopable = nil;

    local notFlag = nil;

    local criteria;
    local i, len;
    local arg, crit;

    /* remember the number of arguments to go through, for efficiency */
    len = args.length;

    /* start by replacing nested 'any' and 'all' criteria with wrappers */
    for (i = 1 ; i <= len ; i++)
    {
        /* see if the current argument is either 'any' or 'all' */
        if (args[i] is in (any, all))
        {
            /*
             *   compile the nested criteria and replace the symbol with a
             *   wrapper
             */
            args[i] = new FuncWrapper(args[i],
                                      compileCriteria(args[i + 1]));

            /* remove the criteria list */
            args = args.removeElementAt(i + 1);
            len--;
        }
    }

    /* create a vector to hold the compiled criteria */
    criteria = new Vector(len);

    /*
     *   when looking for properties, fall back to Thing if we can't find
     *   them in any explicitly mentioned classes
     */
    classes.append(Thing);

    /* go through the arguments one by one */
    for (i = 1 ; i <= len ; i++)
    {
        /* get the current argument */
        arg = args[i];

        /* decide what to do based on its type */
        switch (dataTypeXlat(arg))
        {
            case TypeObject:
                /*
                 *   It's an object. First, check if it's a Collection
                 *   object (a vector or a dynamically created list)
                 */
                if (arg.ofKind(Collection))
                {
                    /* create a list criterion */
                    crit = new ListCriterion(arg);

                    break;
                }

                /* next, check if it's a custom criterion class */
                if (arg.ofKind(Criterion))
                {
                    /* create an instance of the criterion class */
                    crit = arg.createInstance();

                    break;
                }

                /*
                 *   it's some other type of object - assume that it's a
                 *   class and that we should match only instances of it
                 */
                crit = new ClassCriterion(arg);

                /*
                 *   Unless we're in 'not' mode, add the class to our search
                 *   vector. This will give us a better chance of telling
                 *   how many parameters that are needed if we have to
                 *   evaluate a property later on.
                 */
                if (notFlag == nil)
                    classes.prepend(arg);

                break;
            case TypeProp:
                /*
                 *   It's a property pointer. First, try to find out how
                 *   many parameters it takes by looking for the property in
                 *   our search vector of classes. We'll assume zero
                 *   parameters if we can't find the property at all.
                 */
                local params = 0;

                /*
                 *   Go through the search vector. Classes are added to the
                 *   beginning, so we'll check the most recently mentioned
                 *   class first.
                 */
                foreach (local cls in classes)
                {
                    /*
                     *   if this class defines the property, get the
                     *   parameter count and leave this loop
                     */
                    if (cls.propDefined(arg) != nil)
                    {
                        params = cls.getPropParams(arg)[1];
                        break;
                    }
                }

                if (params != 0)
                {
                    /*
                     *   we need paramaters - extract them from args and
                     *   create a method call criterion
                     */
                    crit = new MethodCriterion(arg,
                                   args.sublist(i + 1, params));

                    /* push the counter past the parameters */
                    i += params;
                }
                else
                {
                    /*
                     *   we need no parameters, so use the simpler property
                     *   call criterion
                     */
                    crit = new PropCriterion(arg);
                }

                break;
            case TypeFuncPtr:
                /*
                 *   it's a function - assume that we should call it with
                 *   the candidate object as the sole argument
                 */
                crit = new FuncCriterion(arg);

                break;
            case TypeList:
                /* it's a list - only objects in it should match */
                crit = new ListCriterion(arg);

                break;
            case TypeEnum:
                /* it's an enum - check for 'not' */
                if (arg == not)
                {
                    /* flag that the next criterion should be negated */
                    notFlag = true;

                    /* skip to the next item in the argument list */
                    continue;
                }
                else
                {
                    /* it's nothing we recognize - throw an exception */
                    throw new UnknownCriterionException();
                }
            default:
                /* it's nothing we recognize - throw an exception */
                throw new UnknownCriterionException();
        }

        /* see if this criterion should be negated */
        if (notFlag)
        {
            crit.isNegated = true;
            notFlag = nil;
        }

        /* make a note if this criterion is a loopable one */
        if (crit.isLoopable)
            hasLoopable = true;

        /* append this criterion to our vector */
        criteria.append(crit);
    }

    /*
     *   See if there were any loopable criteria. If not, then we add an
     *   implied class criterion for the Thing class at the start of the
     *   vector.
     */
    if (hasLoopable == nil)
        criteria.prepend(impliedThingCriterion);

    /* return the finished vector */
    return criteria;
}

/*
 *   An exception that is thrown when we find a criterion that we don't know
 *   how to compile.
 */
class UnknownCriterionException: Exception
    displayException()
    {
        "an unknown criterion was encountered";
    }
;

/* ---------------------------------------------------------------------- */
/*
 *   The following functions are for internal use. They expect their
 *   criteria to be given in the form of a vector of Criterion objects, as
 *   returned by compileCriteria().
 */

/*
 *   A function which returns all objects matching a set of criteria.
 */
all_(criteria)
{
    /*
     *   Find the first loopable criterion; this will be the base of the
     *   object loop we perform here. Note that we don't need to check if
     *   there actually is a loopable criterion, since compileCriteria()
     *   guarantees that there will always be at least one, even if it has
     *   to be added implicitly.
     */
    local loopable = criteria.valWhich({ x: x.isLoopable });

    /*
     *   Remove the loopable criterion from the set, since we are going to
     *   loop through all objects that meet it anyway. The syntax we use
     *   here creates a new vector, but that's fine since our caller may
     *   want to reuse the original one.
     */
    criteria -= loopable;

    /* start with an empty set of objects */
    local objs = new Vector(32);

    /* go through the objects matching the loopable criterion */
    loopable.forEachMatchingObj(new function(obj)
    {
        /*
         *   if the object meets the other criteria, as determined by the
         *   objMeets_() function, add it to the set
         */
        if (objMeets_(obj, criteria))
        {
            objs.append(obj);
        }
    });

    /* return the finished set in list form */
    return objs.toList();
}

/*
 *   A function which determines if an object exists that meets a set of
 *   criteria.
 */
exists_(criteria)
{
    /*
     *   Find a loopable criterion. See the comments in all_() for further
     *   information on this part.
     */
    local loopable = criteria.valWhich({ x: x.isLoopable });

    /* remove the loopable criterion */
    criteria -= loopable;

    try
    {
        /* go through the objects matching the loopable criterion */
        loopable.forEachMatchingObj(new function(obj)
        {
            /*
             *   if the object meets the other criteria, as determined by
             *   the objMeets_() function, leave this loop
             */
            if (objMeets_(obj, criteria))
            {
                throw new BreakLoopSignal();
            }
        });
    }
    catch (BreakLoopSignal sig)
    {
        /* this means that we found a matching object */
        return true;
    }

    /* we found no object matching all the criteria */
    return nil;
}

/*
 *   A function which determines if a given objects meets a set of criteria.
 */
objMeets_(obj, criteria)
{
    /* go through the criteria one by one */
    foreach (local cur in criteria)
    {
        /*
         *   stop if the criterion is either met or not met by the object,
         *   depending on whether it is to be negated or not
         */
        if (cur.isMetBy(obj) == cur.isNegated)
            return nil;
    }

    /* all the criteria were met, so return true */
    return true;
}

/* ---------------------------------------------------------------------- */
/*
 *   Internally, all criteria are represented by instances of the Criterion
 *   class, or rather by one its sub-classes. It is up to the individual
 *   criterion to decide whether it is met by a given object.
 */
class Criterion: object
    /* determine whether we are met by the given object */
    isMetBy(obj) { }

    /*
     *   Is this criterion negated? If so, then it is not supposed to be met
     *   by a candidate object.
     */
    isNegated = nil

    /*
     *   Is this a loopable criterion? For some types of criteria, it is
     *   trivial to create a list of all objects that meet it, or at least
     *   to go through them one by one. Such criteria should set this
     *   property to true, since it allows us to search for objects more
     *   efficiently.
     */
    isLoopable = nil

    /*
     *   Invoke a callback function for each object that meets this
     *   criterion. All loopable criteria must define this method.
     */
    forEachMatchingObj(func) { }
;

/*
 *   A Criterion sub-class for loopable criteria.
 */
class LoopableCriterion: Criterion
    /*
     *   in general, even an inherently loopable criterion will only be
     *   loopable if it isn't also negated
     */
    isLoopable = (!isNegated)
;

/*
 *   A criterion which is met if the object belongs to a certain class.
 */
class ClassCriterion: LoopableCriterion
    /* remember our class */
    construct(cls)
    {
        cls_ = cls;
    }

    /* determine whether we are met by the given object */
    isMetBy(obj)
    {
        /* check if the object belongs to our class */
        return obj.ofKind(cls_);
    }

    /*
     *   we can loop through all objects that meet this criterion using a
     *   simple object loop based on our class
     */
    forEachMatchingObj(func)
    {
        local obj;

        /* go through all objects of our class */
        for (obj = firstObj(cls_) ; obj != nil ; obj = nextObj(obj, cls_))
        {
            /* invoke the callback function */
            func(obj);
        }
    }

    cls_ = nil
;

/*
 *   A criterion which is met if a certain property of the object evaluates
 *   to true.
 */
class PropCriterion: Criterion
    /* remember our property pointer */
    construct(prop)
    {
        prop_ = cls;
    }

    /* determine whether we are met by the given object */
    isMetBy(obj)
    {
        /* evaluate the property and see what it says */
        if (obj.(prop_))
            return true;
        else
            return nil;
    }

    prop_ = nil
;

/*
 *   A criterion which is met if a certain method of the object evaluates to
 *   true. If one of the arguments is of the special 'any' type, we resolve
 *   all the objects that meet the sub-criteria and consider ourselves met
 *   if any one of these objects yields a positive result when passed to the
 *   method. For a nested 'all' phrase, we are met only if all the objects
 *   yield a positive result. If two or more arguments use 'any' or 'all',
 *   we try all permutations of matching objects.
 */
class MethodCriterion: Criterion
    /* remember our property pointer and the argument list */
    construct(prop, args)
    {
        prop_ = prop;
        args_ = args;
    }

    /* determine whether we are met by the given object */
    isMetBy(obj)
    {
        /* if we haven't processed the arguments yet, then do so now */
        if (hasProcessedArgs == nil)
            processArgs();

        /* call the method */
        return callMethod(obj, args_);
    }

    /*
     *   Call our method on an object using the given arguments. This is
     *   separated from isMetBy() so that we can use recursion to handle
     *   nested 'any' and 'all' phrases.
     */
    callMethod(obj, args)
    {
        /* see if we have any wrappers */
        local idx = args.indexWhich({ x: dataType(x) == TypeObject
                                         && x.ofKind(FuncWrapper)});

        if (idx != nil)
        {
            /* extract the wrapper */
            local wrapper = args[idx];

            /* extract the arguments before and after the wrapper */
            local firstArgs = args.sublist(1, idx - 1);
            local lastArgs = args.sublist(idx + 1);

            /* see if it's an 'any' or an 'all' wrapper */
            if (wrapper.type_ == any)
            {
                /*
                 *   return true if at least one of the objects meeting the
                 *   nested criteria causes the method to return true
                 */
                foreach (local cur in wrapper.objs)
                {
                    if (callMethod(obj, firstArgs..., cur, lastArgs...))
                        return true;
                }

                return nil;
            }
            else
            {
                /*
                 *   return true only if the method returns true for all the
                 *   objects meeting the nested criteria
                 */
                foreach (local cur in wrapper.objs)
                {
                    if (!callMethod(obj, firstArgs..., cur, lastArgs...))
                        return nil;
                }

                return true;
            }
        }
        else
        {
            /*
             *   we have no wrappers, so call the method with the arguments
             *   we we're passed
             */
            if (obj.(prop_)(args...))
                return true;
            else
                return nil;
        }
    }

    /*
     *   Process the arguments to our method, making sure that all wrappers
     *   have their objects resolved. If this criterion needs to be checked
     *   several times, E.G. during a call to all(), then it would be a
     *   major waste of time to resolve these objects over and over again,
     *   so we make sure that this is only ever done once.
     */
    processArgs()
    {
        /* go through the arguments and process all wrappers */
        foreach (local arg in args_)
        {
            /* see if it's an 'any' or 'all' wrapper */
            if (dataType(arg) == TypeObject && arg.ofKind(FuncWrapper))
            {
                /* resolve the objects matching its criteria */
                arg.resolveObjects();
            }
        }

        /* remember that we have done this */
        hasProcessedArgs = true;
    }

    prop_ = nil
    args_ = []

    hasProcessedArgs = nil
;

/*
 *   A criterion which is met if a call to a certain function with the
 *   object as the sole arguments returns true.
 */
class FuncCriterion: Criterion
    /* remember our function pointer */
    construct(func)
    {
        func_ = func;
    }

    /* determine whether we are met by the given object */
    isMetBy(obj)
    {
        /* call the function and see what it says */
        if (func_()(obj))
            return true;
        else
            return nil;
    }

    func_ = nil
;

/*
 *   A criterion which is met if the object is in a certain list of vector.
 */
class ListCriterion: LoopableCriterion
    /* remember our list */
    construct(lst)
    {
        lst_ = lst;
    }

    /* determine whether we are met by the given object */
    isMetBy(obj)
    {
        /* check if the object is in our list */
        return (lst_.indexOf(obj) != nil);
    }

    /*
     *   all objects met by this criterion are simply all objects in our
     *   list, so looping through them is trivial
     */
    forEachMatchingObj(func)
    {
        /* go through our list */
        foreach (local obj in lst_)
        {
            /* invoke the callback function */
            func(obj);
        }
    }

    lst_ = []
;

/*
 *   A NestedCriterion is a kind of criterion that can be statically defined
 *   and then reused in several searches. It contains a nested set of
 *   criteria, hence the name.
 */
class NestedCriterion: Criterion
    construct()
    {
        /* compile our nested criteria */
        criteria_ = compileCriteria(args_);
    }

    /* determine whether we are met by the given object */
    isMetBy(obj)
    {
        /* test the object against our nested criteria */
        return objMeets_(obj, criteria_);
    }

    /* the argument list that we compile our nested criteria from */
    args_ = []

    /* our compiled criteria */
    criteria_ = nil
;

/*
 *   Define a ClassCriterion for the Thing class. A set of criteria must
 *   contain at least one criterion that is loopable. If the user does not
 *   supply one, then this is the implied default.
 */
impliedThingCriterion: ClassCriterion
    cls_ = Thing
;

/* ---------------------------------------------------------------------- */
/*
 *   A wrapper for a nested 'any' or 'all' inside a method call criteria.
 */
class FuncWrapper: object
    /*
     *   Remember our type and criteria when constructed. The type can
     *   either 'any' or 'all'.
     */
    construct(type, criteria)
    {
        type_ = type;
        criteria_ = criteria;
    }

    /* resolve the objects matching our criteria */
    resolveObjects()
    {
        /* get the objects */
        objs = all_(criteria_);
    }

    objs = []

    type_ = nil
    criteria_ = []
;

/* ---------------------------------------------------------------------- */
/*
 *   The 'not' keyword, defined here as an enum, can be used to signal that
 *   a criterion should be logically negated.
 */
enum not;

