#charset "us-ascii"

/*
 *   TADS 3 Action Report Combiner. This extension provides a way of
 *   combining several consecutive action reports into one, even when the
 *   reports are generated by separate nested actions.
 *
 *   Copyright 2006, Krister Fundin (fundin@yahoo.com)
 */

#include <adv3.h>

/* ---------------------------------------------------------------------- */
/*
 *   Our ModuleID.
 */
ModuleID
    name = 'TADS 3 Action Report Combiner'
    byLine = 'by Krister Fundin'
    htmlByLine = 'by <a href="mailto:fundin@yahoo.com">Krister Fundin</a>'
    version = '4'
;

/* ---------------------------------------------------------------------- */
/*
 *   Create a new TranscriptTransform responsible for applying all rules for
 *   combining reports.
 */
combineReportsTransform: TranscriptTransform
    /* apply this transformation */
    applyTransform(trans, vec)
    {
        /*
         *   We use the ModuleExecObject framework to apply our rules, since
         *   that allows individual rules to decide when they want to be
         *   applied in relation to other rules. We can't supply any
         *   arguments to the classExec() method, however, so we store the
         *   vector of reports temporarily in this object.
         */
        curVec = vec;

        /* apply all the rules */
        CombineReportsRule.classExec();

        /* forget about the vector again */
        curVec = nil;
    }

    /*
     *   when applying our rules, the current vector of reports is
     *   temporarily stored in this property
     */
    curVec = nil
;

/*
 *   Modify the CommandTranscript to include our new transform by default.
 *   Make it the last one, since we want it to apply as directly as possible
 *   to the text that the player would see on the screen
 */
modify CommandTranscript
    transforms_ = static (inherited + combineReportsTransform)
;

/*
 *   A CombineReportsRule is an object that is designed to find certain
 *   sequences of reports and replace them with a single report.
 */
class CombineReportsRule: ModuleExecObject
    /*
     *   Since we're a ModuleExecObject, we will be invoked through our
     *   execute() method. Fetch the report vector from its temporary
     *   storage and pass it on to our applyRule() method.
     */
    execute()
    {
        applyRule(combineReportsTransform.curVec);
    }

    /* apply this rule to a vector of reports */
    applyRule(vec)
    {
        local firstIdx;
        local idx;
        local report;
        local p;

        local pat = pattern;
        local len = pat.length;
        local matchVec = new Vector(len);

        /* keep going until we have tried all possible starting indices */
nextSequence:
        for (firstIdx = 1 ; firstIdx <= vec.length ; firstIdx++)
        {
            /*
             *   Although we want to ignore skippable reports that occur
             *   between two matching reports, we don't want a sequence to
             *   start or end with a skippable report. Thus, the report at
             *   the starting index must not be skippable.
             */
            if (canSkipReport(vec[firstIdx]) != nil)
                continue;

            /* notify that we're trying a new candidate sequence */
            beginSequence();

            /* start with an empty match vector */
            matchVec.setLength(0);

            /* go through the pattern */
            for (p = 1, idx = firstIdx ; p <= len ; p++)
            {
                /* get the next report */
                report = vec[idx++];

                /*
                 *   if the report doesn't match the current pattern item,
                 *   continue to the next sequence
                 */
                if (matchReport(pat[p], report) == nil)
                    continue nextSequence;

                /* check for the 'multi' keyword */
                if (p != len && pat[p + 1] == multi)
                {
                    /*
                     *   create a list to hold the matching reports and add
                     *   the one we have already
                     */
                    local lst = [report];

                    /*
                     *   to avoid adding skippable reports to the end of a
                     *   matching sequence, remember the index after the
                     *   last matching report
                     */
                    local lastIdx = idx;

                    /* try to match the longest possible sequence */
                    while (idx <= vec.length)
                    {
                        /* get the next report */
                        report = vec[idx++];

                        /* see if we want to skip it */
                        if (canSkipReport(report) != nil)
                            continue;

                        /* try to match the report */
                        if (matchReport(pat[p], report) == nil)
                        {
                            /* it was not matched - leave this loop */
                            break;
                        }

                        /* add the matching report to our list */
                        lst += report;

                        /* remember this index */
                        lastIdx = idx;
                    }

                    /* append the list to the match vector */
                    matchVec.append(lst);

                    /* restore the index */
                    idx = lastIdx;

                    /* move the pattern index past the 'multi' keyword */
                    p++;
                }
                else
                {
                    /*
                     *   this was not a 'multi' pattern item, so just add
                     *   the matching report directly to the match vector
                     */
                    matchVec.append(report);
                }

                /* see if we have more pattern items left */
                if (p != len)
                {
                    /*
                     *   We need to match more reports. Move past any
                     *   skippable reports between the one we last matched
                     *   and the candidate for the next pattern item.
                     */
                    do
                    {
                        /*
                         *   if we have exhausted the report vector, then
                         *   there is no report to match the next pattern
                         *   item against, so continue to the next sequence
                         */
                        if (idx > vec.length)
                            continue nextSequence;

                        /* get the next report */
                        report = vec[idx++];
                    }
                    while (canSkipReport(report) != nil);

                    /*
                     *   We have a new candidate. Decrement idx, since it
                     *   should always point to the next report to process.
                     */
                    idx--;
                }
            }

            /*
             *   We have gone through all our pattern items and found
             *   matching reports for all of them. Call our combineReports()
             *   method with the matching reports given in the form of a
             *   chain of arguments.
             */
            report = combineReports(matchVec.toList()...);

            /*
             *   if the return value was nil, then we're not supposed to
             *   combine these reports after all
             */
            if (report != nil)
            {
                /* if we have a string, wrap it in a main command report */
                if (dataType(report) == TypeSString)
                    report = new MainCommandReport(report);

                /*
                 *   Remove the matching sequence from the report vector.
                 *   Note that idx will be pointing to the item after the
                 *   one that was last matched, so decrease it by one.
                 */
                vec.removeRange(firstIdx, idx - 1);

                /*
                 *   insert the new report where the removed sequence
                 *   started
                 */
                vec.insertAt(firstIdx, report);
            }
        }
    }

    /*
     *   Notify that we're trying a new candidate sequence. The purpose of
     *   this entrypoint is to make it easier for a rule to save information
     *   from one report and use it to match other reports later on in a
     *   sequence.
     */
    beginSequence()
    {
        // we do nothing here by default
    }

    /* should we skip a given report? */
    canSkipReport(report)
    {
        /*
         *   By default, we skip all command separators and similar reports
         *   that are only there in order to keep other reports apart. We
         *   also skip marker reports, since they don't actually display
         *   anything.
         */
        return (report.ofKind(CommandSepAnnouncement)
                || report.messageText_ == '<.commandsep>'
                || report.ofKind(GroupSeparatorMessage)
                || report.ofKind(InternalSeparatorMessage)
                || report.ofKind(MarkerReport));
    }

    /* can a pattern item match a given report? */
    matchReport(item, report)
    {
        /*
         *   by default, we assume that the pattern item is a function which
         *   takes a report as the sole argument and returns true for a
         *   succesful match
         */
        return item(report);
    }

    /*
     *   The pattern that we use for matching reports. It should consist of
     *   a number of items, one for each consecutive report in a matching
     *   sequence. An item may also be followed by the 'multi' keyword, in
     *   which it can match more than one report. What an item looks like
     *   depends on the matchReport() method.
     */
    pattern = []

    /*
     *   combine a matching sequence of reports, given as a chain of
     *   arguments, returning either a single report or string to replace
     *   them with or nil to indicate that we don't want to replace the
     *   reports after all
     */
    combineReports(...) { }
;

/*
 *   If the 'multi' keyword follows an item in a rule's pattern list, the
 *   preceding item can match more than one report of the same kind.
 */
enum multi;

/* ---------------------------------------------------------------------- */
/*
 *   A CustomReport is a type of report that is particularly suited for
 *   being combined with other reports. When a CustomReport is created, the
 *   arguments passed to the constructor are automatically stored, so that
 *   they can be used for matching the report in a CombineCustomReportsRule.
 *   The arguments should contain all information that is needed in order to
 *   generate the message.
 *
 *   Since we may want to have different kinds of CustomReports, not only
 *   MainCommandReports, we create this base class as a mix-in. Note though,
 *   that when it's combined with an existing report class, the default
 *   constructor must be overridden so that we don't also invoke the
 *   constructor of the report class with the arguments meant for the
 *   CustomReport constructor, which would be likely to cause run-time
 *   errors.
 */
class CustomReportBase: object
    construct([args])
    {
        /* remember our arguments */
        args_ = args;

        /* continue as usual with our actual message */
        inherited(getMessage(args...));
    }

    /*
     *   Return a string giving the actual message for this report, based on
     *   the given arguments. Use gMessageParams() on any arguments that
     *   represent objects mentioned in the message.
     */
    getMessage(...) { }

    /* the arguments for each instance of this report are stored here  */
    args_ = []
;

/*
 *   Define the actual CustomReport class, which is a custom kind of
 *   MainCommandReport. The constructor is necessary.
 */
class CustomReport: CustomReportBase, MainCommandReport
    construct([args]) { inherited(args...); }
;

/*
 *   Define a CustomReport version of a DefaultCommandReport. To define
 *   custom versions of other report classes, just repeat the same pattern.
 */
class DefaultCustomReport: CustomReportBase, DefaultCommandReport
    construct([args]) { inherited(args...); }
;

/*
 *   A CombineReportsRule sub-class for matching CustomReports. This type of
 *   rule uses a different format for the pattern items. Each item should be
 *   a list, in which the first item is the CustomReport sub-class to match,
 *   and the following items (if any) correspond to the arguments that this
 *   report class uses. Each argument item can be one of the following:
 *
 *   - a string, in which case the value of that argument is remembered. If
 *   several argument items (in the same pattern item, or in different
 *   pattern items) share the same string, then the corresponding report
 *   arguments must also share the same value.
 *
 *   - a string wrapped in a list, in which case all corresponding report
 *   arguments are remembered individually. They need not be the same.
 *
 *   - nil, in which case any value is accepted for that report argument.
 *   The value will not be remembered.
 *
 *   - anything else, in which case the value of the corresponding report
 *   argument must be the same as the argument item.
 *
 *   Instances of this rule should define combineCustomReports() instead of
 *   combineReports(). This method will be passed the values of all the
 *   strings from the pattern, in order of first appearance.
 */
class CombineCustomReportsRule: CombineReportsRule
    /* can a pattern item match a given report? */
    matchReport(item, report)
    {
        /*
         *   the report must belong to the class at the head of the pattern
         *   item
         */
        if (!report.ofKind(item[1]))
            return nil;

        /* get the argument items */
        local args = item.sublist(2);

        /*
         *   Make a copy of our variable vector. We're going to see if a
         *   report matches one of our pattern items. A negative result
         *   won't always mean that the current candidate sequence is
         *   abandoned - this is the case with 'multi' pattern items. The
         *   problem is that the matching can alter our variables, and this
         *   would lead to errors if the report didn't match but we still
         *   got a matching sequence. To get around this, we replace our
         *   current (possibly modified) variable vector with a safe copy
         *   in case the report doesn't match.
         */
        local oldVars = new Vector(vars.length, vars);

        /* go through the argument items */
        for (local i = 1 ; i <= args.length ; i++)
        {
            /*
             *   see if the current argument item matches the value of 
             *   the corresponding report argument
             */
            if (matchArgument(args[i], report.args_[i]) == nil)
            {
                /* 
                 *   it didn't - replace our variable vector with the
                 *   copy we made earlier
                 */
                vars = oldVars;

                /* return nil for failure */
                return nil;
            }
        }
    
        /* all argument items were successfully matched, so return true */
        return true;
    }

    /* does an argument item match the given report argument? */
    matchArgument(arg, val)
    {
        /* if the argument item is nil, then it matches anything */
        if (arg == nil)
            return true;

        /* check for strings and strings wrapped in lists */
        if (dataType(arg) == TypeSString)
        {
            /* it's a string - see if we have tied it to a value yet */
            local var = vars.valWhich({ x: x[1] == arg });

            if (var == nil)
            {
                /*
                 *   it's the first time we see it, so add it to our
                 *   variable vector
                 */
                vars.append([arg, val]);
            }
            else
            {
                /*
                 *   we already have a value for this string, so the report
                 *   argument must contain the same value
                 */
                if (val != var[2])
                    return nil;
            }
        }
        else if (dataType(arg) == TypeList)
        {
            /* it's a string wrapped in a list - extract the string */
            arg = arg[1];

            /* check if we have seen it before */
            local idx = vars.indexWhich({ x: x[1] == arg });

            if (idx == nil)
            {
                /*
                 *   it's a new string - wrap the value from the report
                 *   argument in a list and add it to our variable vector
                 */
                vars.append([arg, [val]]);
            }
            else
            {
                /* add this value to the existing list */
                vars[idx][2] += val;
            }
        }
        else
        {
            /*
             *   the argument item is something else, so just check if it's
             *   the same as the value from the report argument
             */
            if (val != arg)
                return nil;
        }

        /* no objections */
        return true;
    }

    /* notify that we're trying a new candidate sequence */
    beginSequence()
    {
        /* clear our vector of variables */
        vars.setLength(0);
    }

    /*
     *   Combine a matching sequence of reports. We disregard the actual
     *   reports. All necessary information is stored in their arguments,
     *   and we can extract this information using strings as argument items
     *   in our pattern.
     */
    combineReports(...)
    {
        /* extract the values from our variable vector */
        local args = vars.toList().mapAll({ x: x[2] });

        /* clear the variable vector */
        vars.setLength(0);

        /* pass the values on to our combineCustomReports() method */
        return combineCustomReports(args...);
    }

    /*
     *   Combine a sequence of matching CustomReports. The arguments passed
     *   to this method will correspond to the values of the strings used in
     *   our pattern.
     */
    combineCustomReports(...) { }

    /*
     *   Our variable vector. Here we store all the strings from our pattern
     *   and the values tied to them, in a simple [string, value] format.
     *   (We could use a LookupTable, but that would probably be overkill.)
     */
    vars = perInstance(new Vector(5))
;

