/*
** winctls.c -- Contains all the functions which handle the WinTADS window
**              controls and dialogs.
*/

#define NOMINMAX

#include <windows.h>
#include <stdio.h>
#include "wintads.h"
#include "winio.h"
#include "global.h"

#define STATUS_EXTRA_HEIGHT	6
#define MB_TIMEOUT 3		/* Timeout of message in seconds */

LONG APIENTRY MessageBoxWndProc(HWND hwnd, UINT msg, UINT wParam, LONG lParam);
LONG APIENTRY StatusWndProc(HWND hwnd, UINT msg, UINT wParam, LONG lParam);
extern SHORT UpdateInputFont(HDC hdc, BOOL fBold);
extern BOOL CenterWindow (HWND hwndChild, HWND hwndParent);
VOID TweakMacroMenus(VOID);
VOID SetStatusFont(LOGFONT *plf);
static void appendmacro(char *buf, char *src);

static short	fLastMacroMode = -1;/* Stores TRUE or FALSE, depending on whether
								   the menu is set up for macro-use (FALSE) or
								   macro-define (TRUE). Reset to -1 if the macro
								   set changes. */
BOOL			bClearMB = FALSE;	// Should we clear the message box?

static HBRUSH		hbrBtnFace;
HFONT				hfontMB, hfontStatus;
static TEXTMETRIC	tmMBFont, tmStatusFont;
static int			dyMB, cxMBBorder, cyMBBorder, cxFrame, cyFrame, dyField;

short cmdkeylist[12] = {
	VK_F1, VK_F2, VK_F3, VK_F4, VK_F5, VK_F6, VK_F7, VK_F8, VK_F9, VK_F10,
	VK_F11, VK_F12
};

static BOOL fPendingClear;	// Are we about to clear the status line?

/*
** Menu functions-------------------------------------------------------------
*/

BOOL MyEnableMenuItem(HWND hwndMenu, UINT ulID, BOOL fEnabled)
{
	return EnableMenuItem(hwndMenu, ulID, fEnabled ? MF_ENABLED : MF_GRAYED);
}

/* Enable/disable cut/copy if the user hasn't made a selection. Also take
   care of the selections which are only available when the program is
   running */
VOID TweakMenus(VOID)
{
	BOOL	bCanCopy = FALSE, bCanCut = FALSE;
	HWND	hwndMenu = GetMenu(hwndFrame);
	ULONG	i;
	
	xted_option_capability(&bCanCut, &bCanCopy);
	MyEnableMenuItem(hwndMenu, IDM_CUT, bCanCut && fThreadRunning);
	MyEnableMenuItem(hwndMenu, IDM_COPY, bCanCopy && fThreadRunning);
	MyEnableMenuItem(hwndMenu, IDM_PASTE,
		IsClipboardFormatAvailable(CF_TEXT) && fThreadRunning);
	
	MyEnableMenuItem(hwndMenu, IDM_LOAD, !fThreadRunning);
	MyEnableMenuItem(hwndMenu, IDM_SAVE, fThreadRunning);
	MyEnableMenuItem(hwndMenu, IDM_RESTORE, fThreadRunning);
	MyEnableMenuItem(hwndMenu, IDM_BIND, fThreadRunning);
	MyEnableMenuItem(hwndMenu, IDM_UNBIND, fThreadRunning);
	for (i = 0; i < 10, prefsTADS.szRecentGames[i][0] != 0; i++)
		MyEnableMenuItem(hwndMenu, IDM_RG1 + i, !fThreadRunning);
	MyEnableMenuItem(hwndMenu, IDM_UNDO, fThreadRunning);
	MyEnableMenuItem(hwndMenu, IDM_FIND, fThreadRunning);
	MyEnableMenuItem(hwndMenu, IDM_FINDAGAIN, fThreadRunning);
}

// Adjust the "recent games" menu
VOID TweakRecentMenu(VOID)
{
	HWND	hwndMenu = GetMenu(hwndFrame);
	int		i;
	char	s[1000], *p;
	MENUITEMINFO	mii;
	
	// Get the handle of the file submenu
	hwndMenu = GetSubMenu(hwndMenu, 0);

	for (i = IDM_RG1; i <= IDM_RG10; i++)
		RemoveMenu(hwndMenu, i, MF_BYCOMMAND);
	if (validBinding) {
		RemoveMenu(hwndMenu, 7, MF_BYPOSITION);
		return;
	}

	mii.cbSize = sizeof(MENUITEMINFO);
	mii.fMask = MIIM_TYPE | MIIM_ID;
	mii.fType = MFT_STRING;
	mii.dwTypeData = s;
	for (i = 0; i < 10 && prefsTADS.szRecentGames[i][0] != 0; i++) {
		p = prefsTADS.szRecentGames[i] +
			strlen(prefsTADS.szRecentGames[i]) - 1;
		while (*p != '\\' && p != prefsTADS.szRecentGames[i])
			p--;
		if (i == 9)
			sprintf(s, "1&0. %s", p+1);
		else sprintf(s, "&%i. %s", i+1, p+1);
		mii.wID = IDM_RG1 + i;
		mii.cch = strlen(s);
		InsertMenuItem(hwndMenu, 7 + i, TRUE, &mii);
	}
	if (i == 0) {
		strcpy(s, "No .gam files recently loaded");
		mii.wID = IDM_RG1;
		mii.cch = strlen(s);
		InsertMenuItem(hwndMenu, 7, TRUE, &mii);
		MyEnableMenuItem(hwndMenu, IDM_RG1, FALSE);
	}
}

// Add a recently loaded file to the "recent games" menu
VOID AddRecent(char *new)
{
	int	i, j;
	char	title[500], *p;
	
	p = new + strlen(new) - 1;
	while (*p != '\\' && p != new)
		p--;
	if (p != new)
		p++;
	sprintf(title, "WinTADS (%s", p);
	i = strlen(title);
	p = title + i - 1;
	while (*p != '.' && p != title)
		p--;
	if (p != title) {
		*p = ')';
		*(p+1) = 0;
	}
	else {
		title[i] = ')';
		title[i+1] = 0;
	}
	SetWindowText(hwndFrame, title);
	for (i = 0; i < 10 && prefsTADS.szRecentGames[i][0] != 0; i++) {
		if (!stricmp(new, prefsTADS.szRecentGames[i])) {
			for (j = i; j > 0; j--)
				strcpy(prefsTADS.szRecentGames[j],
					prefsTADS.szRecentGames[j-1]);
			strcpy(prefsTADS.szRecentGames[0], new);
			TweakRecentMenu();
			return;
		}
	}
	if (i == 10)
		i--;
	while (i > 0) {
		strcpy(prefsTADS.szRecentGames[i],
			prefsTADS.szRecentGames[i-1]);
		i--;
	}
	strcpy(prefsTADS.szRecentGames[0], new);
	TweakRecentMenu();
}

// Adjust the binding menu
VOID TweakBindingMenu(VOID)
{
	HWND		hwndMenu = GetMenu(hwndFrame);

	// Get the handle of the file submenu
	hwndMenu = GetSubMenu(hwndMenu, 0);
	
	if (validBinding)
		RemoveMenu(hwndMenu, IDM_BIND, MF_BYCOMMAND);
	else RemoveMenu(hwndMenu, IDM_UNBIND, MF_BYCOMMAND);
}

void macrom_dirty_macrolist()
{
	fLastMacroMode = (-1);
	TweakMacroMenus();
}

// Fix the macro menu
VOID TweakMacroMenus(VOID)
{
	int		i;
	char	str[255], tempstr[255], *pStr;
	BOOL	fItems;
	HWND	hwndMenu = GetMenu(hwndFrame);
	SHORT	fMacroMode = FALSE;
	
	// Get the handle of the macro submenu
	hwndMenu = GetSubMenu(hwndMenu, 3);

	if (xtexted_getmodifymode(FALSE) == op_DefineMacro)
		fMacroMode = TRUE;

	if (fMacroMode == fLastMacroMode) return;
	
	fLastMacroMode = fMacroMode;
	for (i = IDM_DEFINE_MACRO; i <= IDM_MACRO12; i++)
		RemoveMenu(hwndMenu, i, MF_BYCOMMAND);

	if (fMacroMode) {		// We're defining a macro
		strcpy(str, "&Cancel Macro Define\tCtrl+D");
		InsertMenu(hwndMenu, 0, MF_BYPOSITION | MF_STRING, IDM_DEFINE_MACRO,
			str);
		// Show every macro
		for (i = 0; i < 12; i++) {
			pStr = keycmdargs[cmdkeylist[i] | keytype_macro];
			if (pStr && pStr[0]) {
				strcpy(tempstr, "Replace ");
				appendmacro(tempstr, pStr);
				sprintf(str, "%s\tF%i", tempstr, i+1);
			}
			else sprintf(str, "Define This Macro\tF%i", i+1);
			AppendMenu(hwndMenu, MF_STRING, IDM_MACRO1 + i, str);
		}
	} else {
		strcpy(str, "&Define Macro\tCtrl+D");
		InsertMenu(hwndMenu, 0, MF_BYPOSITION | MF_STRING, IDM_DEFINE_MACRO, str);
		// Only show the macros which exist
		fItems = FALSE;
		for (i = 0; i < 12; i++) {
			pStr = keycmdargs[cmdkeylist[i] | keytype_macro];
			if (pStr && pStr[0]) {
				fItems = TRUE;
				strcpy(tempstr, "Macro ");
				appendmacro(tempstr, pStr);
				sprintf(str, "%s\tF%i", tempstr, i+1);
				AppendMenu(hwndMenu, MF_STRING, IDM_MACRO1 + i, str);
			}
		}
		if (!fItems) {
			strcpy(str, "No Macros Defined");
			AppendMenu(hwndMenu, MF_STRING | MF_GRAYED, IDM_MACRO1, str);
		}
	}
}

// Set up the macro menu information for the first time
VOID SetupMacroMenus(VOID)
{
	TweakMacroMenus();
}

/* Stick src onto the end of buf, trimming to 20
	characters, enclosing in quotes, skipping weird chars. */
static void appendmacro(char *buf, char *src)
{
	int ix, jx;
	
	ix = strlen(buf);
	
	buf[ix++] = '"';
	
	for (jx=0; src[jx]; jx++) {
		if (jx == 20) {
			buf[ix++] = '.';
			buf[ix++] = '.';
			buf[ix++] = '.';
			break;
		}
		buf[ix++] = src[jx];
	}
	
	buf[ix++] = '"';
	buf[ix] = 0;
}


/*
** Message box functions------------------------------------------------------
*/

/* This function sets the text of the message box.  It prepends two spaces to the
   text for aesthetic purposes. If bTimeout is true, the message will vanish
   after MB_TIMEOUT seconds. */
void SetMessageBoxText(char *pszNewText, BOOL bTimeout)
{
	char	pszBuf[256];

	if (pszNewText)
		sprintf(pszBuf, "  %s", pszNewText);
	else pszBuf[0] = 0;
	SetWindowText(hwndMessageBox, pszBuf);
	InvalidateRect(hwndMessageBox, NULL, TRUE);
	if (bTimeout) {
		SetTimer(hwndFrame, ID_MBTIMER, MB_TIMEOUT * 1000, NULL);
		bClearMB = TRUE;
	}
	else bClearMB = FALSE;
}

/* Initialize the message box. */
BOOL InitMessageBox(HANDLE hInstance)
{
	WNDCLASS	wndclass;
	
	hbrBtnFace = CreateSolidBrush(GetSysColor(COLOR_BTNFACE));
	
	wndclass.style = CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc = (WNDPROC)MessageBoxWndProc;
	wndclass.cbClsExtra = 0;
	wndclass.cbWndExtra = 0;
	wndclass.hInstance = hInstance;
	wndclass.hIcon = NULL;
	wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
	wndclass.hbrBackground = hbrBtnFace;
	wndclass.lpszMenuName = NULL;
	wndclass.lpszClassName = "MessageBox";
	
	if (!RegisterClass(&wndclass)) return FALSE;
	return TRUE;
}

/* Create the message box. hwndParent is the handle to the window which should
   be the message box's parent. */
BOOL CreateMessageBox(HWND hwndParent, HANDLE hInstance)
{
	LOGFONT	lfTmp;
	HDC		hdc;

	hwndMessageBox = CreateWindow("MessageBox", "MessageBox",
		WS_CHILD | WS_BORDER | WS_VISIBLE,
		0, 0, 0, 0,
		hwndParent, (HMENU)ID_MESSAGEBOX, hInstance, NULL);

	if (!hwndMessageBox) return FALSE;

	// Set the text of the message box
	SetMessageBoxText("Welcome to WinTADS", TRUE);

	// Set the font
	hfontMB = CreateFont(14, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0,
		VARIABLE_PITCH | FF_SWISS, "");
	if (GetObject(hfontMB, sizeof(LOGFONT), &lfTmp)) {
		if ((lfTmp.lfPitchAndFamily & VARIABLE_PITCH) &&
			(lfTmp.lfPitchAndFamily & FF_SWISS)) {
		}
		else {
			DisplayInformation("Unable to create unnamed variable pitch swiss font for the message box");
			hfontMB = CreateFont(14, 0, 0, 0, 0, 0, 0, 0,
				0, 0, 0, 0,
				VARIABLE_PITCH | FF_SWISS, "Arial");
		}
	}
	if (!hfontMB)
		DisplayInformation("Unable to Create Message Box Font");
	hdc = GetDC(hwndMessageBox);
	SelectObject(hdc, hfontMB);
	GetTextMetrics(hdc, &tmMBFont);
	cxMBBorder = GetSystemMetrics(SM_CXBORDER);
	cyMBBorder = GetSystemMetrics(SM_CYBORDER);
	cxFrame = 3*cxMBBorder;
	cyFrame = 3*cyMBBorder;
	dyField = tmMBFont.tmHeight + (2*cyMBBorder);
	dyMB = dyField + (2*cyFrame);
	ReleaseDC(hwndMessageBox, hdc);
	return TRUE;
}

// Adjust the size and location of the message box
BOOL AdjustMessageBox(HWND hwnd)
{
	RECT	rect;
	
	GetClientRect(hwnd, &rect);
	MoveWindow(hwndMessageBox, rect.left-cxMBBorder, rect.bottom - dyMB + cyMBBorder,
		rect.right - rect.left + (cxMBBorder*2), dyMB, TRUE);
	return TRUE;
}

// Return the height of the message box
int MessageBoxHeight(HWND hwnd)
{
	RECT rect;

	GetClientRect(hwndMessageBox, &rect);
	return rect.bottom - rect.top;
}

/*
** MessageBoxWndProc processes the message box's WM_PAINT message to put a
** border around the thing.
*/
LONG APIENTRY MessageBoxWndProc(HWND hwnd, UINT msg, UINT wParam, LONG lParam)
{
	switch (msg) {
	  case WM_DESTROY:
		if (hfontMB)
			DeleteObject(hfontMB);
		break;
		
	  case WM_PAINT: {
	  		PAINTSTRUCT ps;
			HDC hdc = BeginPaint(hwnd, &ps);
			RECT rect, border;
			HBRUSH hBrush;
			HFONT hTmp;
			char szText[256];
			int len;

			GetClientRect(hwnd, &rect);

			// Paint the background
			hBrush = CreateSolidBrush(GetSysColor(COLOR_WINDOW));
			border = rect;
			border.bottom = border.top + cyMBBorder;
			FillRect(hdc, &border, hBrush);
			DeleteObject(hBrush);

			// Make the border
			hBrush = CreateSolidBrush(GetSysColor(COLOR_BTNSHADOW));
			border = rect;
			border.top = border.bottom - cyMBBorder;
			FillRect(hdc, &border, hBrush);
			DeleteObject(hBrush);
			
			// Make the inner border
			InflateRect(&rect, -(2*cxMBBorder), -cyMBBorder);
			hBrush = CreateSolidBrush(GetSysColor(COLOR_BTNSHADOW));
			border = rect;
			border.bottom = border.top + cyMBBorder;
			FillRect(hdc, &border, hBrush);
			border = rect;
			border.right = border.left + cxMBBorder;
			FillRect(hdc, &border, hBrush);
			DeleteObject(hBrush);
			
			hBrush = CreateSolidBrush(GetSysColor(COLOR_WINDOW));
			border = rect;
			border.top = border.bottom - cyMBBorder;
			FillRect(hdc, &border, hBrush);
			border = rect;
			border.left = border.right - cxMBBorder;
			FillRect(hdc, &border, hBrush);
			DeleteObject(hBrush);
			
			// Print the text
			if (len = GetWindowText(hwnd, szText, sizeof(szText))) {
				hTmp = SelectObject(hdc, hfontMB);
				SetTextColor(hdc, GetSysColor(COLOR_BTNTEXT));
				SetBkColor(hdc, GetSysColor(COLOR_BTNFACE));
				
				InflateRect(&rect, -(cxMBBorder*2), -cyMBBorder);
				ExtTextOut(hdc, rect.left, rect.top,
					ETO_OPAQUE | ETO_CLIPPED,
					&rect,
					(LPSTR)szText,
					len, NULL);
				SelectObject(hdc, hTmp);
			}
			
			EndPaint(hwnd, &ps);

			return DefWindowProc(hwnd, msg, wParam, lParam);
		}

	  default:
		return( DefWindowProc(hwnd, msg, wParam, lParam) );
	}
	return 0L;
}


/*
** Status window functions----------------------------------------------------
*/

/* SetStatusTextLeft changes the text which is printed on the left side of
   the status bar */
void SetStatusTextLeft(char *pszNewText)
{
	if (pszNewText)
		strcpy(szStatusLeft, pszNewText);
	else szStatusLeft[0] = 0;
	
	InvalidateRect(hwndStatus, NULL, FALSE);
}

// StatusInsertLeft inserts a character at the left side of the status bar
void StatusInsertLeft(unsigned char c)
{
	int i;
	
	// CR or LF means clear the status bar next time
	if (c == '\n' || c == '\r') {
		fPendingClear = TRUE;
		return;
	}

	if (fPendingClear) {
		fPendingClear = FALSE;
		i = 0;
	}
	else i = strlen(szStatusLeft);
	szStatusLeft[i] = c;
	szStatusLeft[i+1] = (char)NULL;
	
	InvalidateRect(hwndStatus, NULL, FALSE);
}

/* SetStatusTextRight changes the text which is printed on the right side of
   the status bar */
VOID SetStatusTextRight(PSZ pszNewText)
{
	if (pszNewText)
		strcpy(szStatusRight, pszNewText);
	else szStatusRight[0] = 0;
	
	InvalidateRect(hwndStatus, NULL, FALSE);
}

// ClearStatus clears out the status
VOID ClearStatus(VOID)
{
	SetStatusTextLeft(NULL);
	SetStatusTextRight(NULL);
}

// Initialize the status window class
BOOL InitStatus(HANDLE hInstance)
{
	WNDCLASS	wndclass;
	
	wndclass.style = CS_HREDRAW | CS_VREDRAW | CS_NOCLOSE;
	wndclass.lpfnWndProc = (WNDPROC)StatusWndProc;
	wndclass.cbClsExtra = 0;
	wndclass.cbWndExtra = 0;
	wndclass.hInstance = hInstance;
	wndclass.hIcon = LoadIcon(hInstance, "WinTADSSmall");
	wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH)(COLOR_WINDOW+1);
	wndclass.lpszMenuName = NULL;
	wndclass.lpszClassName = "StatusBar";
	
	if (!RegisterClass(&wndclass))
		return FALSE;
	return TRUE;
}

/* CreateSeparateStatus creates a status bar which is contained in its own
   window, separate from the main window of TADS/2 */
static BOOL CreateSeparateStatus(HANDLE hInstance)
{
	RECT		rclClient;

	hwndStatus = CreateWindow("StatusBar", "WinTADS Status",
		WS_OVERLAPPEDWINDOW | WS_VISIBLE,
		0, 0, 0, 0,
		NULL, NULL, hInstance, NULL);
	if (hwndStatus == NULL) {
		return FALSE;
	}

	SetStatusFont(&prefsTADS.lfStatus);

	// Set the size of the status bar
	GetWindowRect(hwndFrame, &rclClient);				// Get client window size
	lStatusWidth = rclClient.right - rclClient.left;
	lStatusHeight = tmStatusFont.tmHeight + STATUS_EXTRA_HEIGHT +
		GetSystemMetrics(SM_CYMENU) + cyFrame * 2;

	// Adjust the size & location of the window surrounding the status bar
	rclClient.top -= lStatusHeight;
	if (rclClient.top < 0)
		rclClient.top = 0;
	MoveWindow(hwndStatus, rclClient.left, rclClient.top, lStatusWidth, lStatusHeight, TRUE);

	return TRUE;
}

/* CreateTogetherStatus creates a status bar which is contained within the main
   window of TADS/2 */
static BOOL CreateTogetherStatus(HANDLE hInstance)
{
	RECT		rclClient;

	hwndStatus = CreateWindow("StatusBar", "WinTADS Status",
		WS_CHILD | WS_BORDER | WS_VISIBLE,
		0, 0, 0, 0,
		hwndFrame, (HMENU)ID_STATUS, hInstance, NULL);
	if (!hwndStatus)
		return FALSE;

	SetStatusFont(&prefsTADS.lfStatus);

	// Set the size of the status bar
	GetClientRect(hwndFrame, &rclClient);				// Get client window size
	lStatusWidth = rclClient.right - rclClient.left;
	lStatusHeight = tmStatusFont.tmHeight + STATUS_EXTRA_HEIGHT;
	MoveWindow(hwndStatus, 0, 0, lStatusWidth, lStatusHeight, TRUE);

	return TRUE;
}

// UpdateStatus updates the windows surrounding the status bar
VOID UpdateStatus(VOID)
{
	RECT	rect;
	
	/* Force an update of the windows via the frame. If we're separated, update
	   our own personal frame. */
	GetWindowRect(hwndFrame, &rect);
	PostMessage(hwndFrame, WM_SIZE, SIZE_RESTORED, MAKEWORD(rect.right - rect.left,
		rect.bottom - rect.top));
	InvalidateRect(hwndFrame, NULL, TRUE);
	if (prefsTADS.fStatusSeparated)
		InvalidateRect(hwndStatus, NULL, TRUE);
}

// CreateStatus creates the proper status bar.  It also updates the main menu
BOOL CreateStatus(HANDLE hInstance)
{
	BOOL	rC;
	
	// Let there be status bar
	if (prefsTADS.fStatusSeparated)
		rC = CreateSeparateStatus(hInstance);
	else rC = CreateTogetherStatus(hInstance);

	// Check/uncheck the separated status menu item
	CheckMenuItem(GetMenu(hwndFrame), IDM_SEPARATED_STATUS, 
		(prefsTADS.fStatusSeparated ? MF_CHECKED : MF_UNCHECKED));

	UpdateStatus();
	
	return rC;
}

// SwapStatus switches between a separated status bar and a joined one
BOOL SwapStatus(HANDLE hInstance)
{
	DestroyWindow(hwndStatus);

	prefsTADS.fStatusSeparated = !prefsTADS.fStatusSeparated;

	return CreateStatus(hInstance);
}

// Adjust the size and location of the status window
BOOL AdjustStatusBar(HWND hwnd)
{
	RECT	rect;
	
	GetClientRect(hwnd, &rect);
	lStatusWidth = rect.right - rect.left;
	MoveWindow(hwndStatus, 0, 0, lStatusWidth, lStatusHeight, TRUE);
	return TRUE;
}

// Return the height of the message box
int StatusBarHeight(HWND hwnd)
{
	RECT rect;

	GetClientRect(hwndStatus, &rect);
	return rect.bottom - rect.top;
}

/* SetStatusFont sets up the status bar's font and adjusts its size */
VOID SetStatusFont(LOGFONT *plf)
{
	HDC		hdc;
	RECT	rclClient, rclStatus;
	
	if (hfontStatus)
		DeleteObject(hfontStatus);
	hfontStatus = CreateFontIndirect(plf);
	GetObject(hfontStatus, sizeof(LOGFONT), plf);
	hdc = GetDC(hwndStatus);
	SelectObject(hdc, hfontStatus);
	GetTextMetrics(hdc, &tmStatusFont);
	ReleaseDC(hwndStatus, hdc);
	
	// Set the size of the status bar
	lStatusHeight = tmStatusFont.tmHeight + STATUS_EXTRA_HEIGHT;
	if (prefsTADS.fStatusSeparated) {
		GetWindowRect(hwndFrame, &rclClient);
		lStatusHeight += GetSystemMetrics(SM_CYMENU) + cyFrame * 2;

		// Adjust the size & location of the window surrounding the status bar
		GetWindowRect(hwndStatus, &rclStatus);
		MoveWindow(hwndStatus, rclStatus.left, rclStatus.top, lStatusWidth,
			lStatusHeight, TRUE);
	}
	else
		MoveWindow(hwndStatus, 0, 0, lStatusWidth, lStatusHeight, TRUE);
}

/*
** StatusWndProc processes the status bar's WM_PAINT message to print the two
** sides of the status bar.
*/
LONG APIENTRY StatusWndProc(HWND hwnd, UINT msg, UINT wParam, LONG lParam)
{
	LPMINMAXINFO	mmi;
	
	switch (msg) {
	  case WM_DESTROY:
		if (hfontStatus)
			DeleteObject(hfontStatus);
		break;
		
	  case WM_GETMINMAXINFO:				// Don't let the user adjust the separate
		if (!prefsTADS.fStatusSeparated)	//  status bar's height
			break;
		mmi = (LPMINMAXINFO)lParam;
		mmi->ptMaxSize.y = lStatusHeight;
		mmi->ptMinTrackSize.y = lStatusHeight;
		mmi->ptMaxTrackSize.y = lStatusHeight;
		break;
		break;
		
	  case WM_PAINT:
		{
			HBRUSH hBrush;
			// Rectangles for containing the size of the status bar & text
			RECT	rclStatus, rclLeft, rclRight;

			// Get presentation space handle for drawing
			HDC	hdc;

			// Do the standard drawing
			LONG	rc = DefWindowProc(hwnd, msg, wParam, lParam);
			
			hdc = GetDC(hwnd);

			SelectObject(hdc, hfontStatus);
			SetTextColor(hdc, prefsTADS.ulFore);
			SetBkColor(hdc, prefsTADS.ulBack);

			GetClientRect(hwnd, &rclStatus);
			hBrush = CreateSolidBrush(prefsTADS.ulBack);
			FillRect(hdc, &rclStatus, hBrush);
			DeleteObject(hBrush);
			InflateRect(&rclStatus, -2, -2);

			// Copy these rectangles into rclLeft and rclRight
			memcpy(&rclLeft, &rclStatus, sizeof(RECT));
			memcpy(&rclRight, &rclStatus, sizeof(RECT));

			// Get the size required for the LHS & RHS text
			DrawText(hdc, szStatusLeft, -1, &rclLeft,
				DT_LEFT | DT_SINGLELINE | DT_BOTTOM | DT_CALCRECT);
			DrawText(hdc, szStatusRight, -1, &rclRight,
				DT_RIGHT | DT_SINGLELINE | DT_BOTTOM | DT_CALCRECT);
			rclRight.left += rclStatus.right - rclRight.right;
			rclRight.right = rclStatus.right;

			// Adjust the reported size of the LHS & RHS text
			lStatusLeftWidth = rclLeft.right - rclLeft.left;
			lStatusRightWidth = rclRight.right - rclRight.left;

			// Adjust the sizes if necessary
			if (rclLeft.right >= rclRight.left)
				rclLeft.right = rclRight.left - 2;

			DrawText(hdc, szStatusLeft, -1, &rclLeft,
				DT_LEFT | DT_SINGLELINE | DT_BOTTOM);
			DrawText(hdc, szStatusRight, -1, &rclRight,
				DT_RIGHT | DT_SINGLELINE | DT_BOTTOM);

			ReleaseDC(hwnd, hdc);

			return rc;
		}
		
	  case WM_KEYDOWN:
	  	SetFocus(hwndFrame);
	  	PostMessage(hwndFrame, msg, wParam, lParam);
		return (DefWindowProc(hwnd, msg, wParam, lParam));

	  default:
		return (DefWindowProc(hwnd, msg, wParam, lParam));
	}
	return (DefWindowProc(hwnd, msg, wParam, lParam));
}


/*
** Scroll bar procedures------------------------------------------------------
*/

VOID MyShowScrollBar(BOOL fShow)
{
	ShowScrollBar(hwndClient, SB_VERT, fShow);
	fScrollVisible = fShow;
}

/*
** Dialog procedures----------------------------------------------------------
*/

BOOL APIENTRY FindDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch (msg) {
	  case WM_INITDIALOG:
		CenterWindow(hwnd, hwndFrame);
		CheckDlgButton(hwnd, IDD_CAREABOUTCASE, (fd.fCase ?
			BST_CHECKED : BST_UNCHECKED));
		CheckDlgButton(hwnd, IDD_SEARCHBACKWARDS, (fd.fBack ?
			BST_CHECKED : BST_UNCHECKED));
		SendDlgItemMessage(hwnd, IDD_FIND_TEXT, EM_LIMITTEXT,
			sizeof(fd.findText), 0L);
		SetDlgItemText(hwnd, IDD_FIND_TEXT, fd.findText);
		SendDlgItemMessage(hwnd, IDD_FIND_TEXT, EM_SETSEL,
			0, strlen(fd.findText));
		return TRUE;

	  case WM_COMMAND:
		switch (wParam) {
		  case IDOK:
			GetDlgItemText(hwnd, IDD_FIND_TEXT, fd.findText,
				sizeof(fd.findText));
			fd.fCase = (IsDlgButtonChecked(hwnd, IDD_CAREABOUTCASE) ==
				BST_CHECKED);
			fd.fBack = (IsDlgButtonChecked(hwnd, IDD_SEARCHBACKWARDS) ==
				BST_CHECKED);
		  case IDCANCEL:
			EndDialog(hwnd, (wParam == IDOK));
			return TRUE;
		}
	}
	return FALSE;
}

BOOL APIENTRY StoryWindowDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	BOOL	fRefresh = FALSE, fTemp;
	LONG	lTemp;
	NM_UPDOWN	*nmu;
	
	switch (msg) {
	  case WM_INITDIALOG:
		CenterWindow(hwnd, hwndFrame);
		SendDlgItemMessage(hwnd, IDD_LRSPIN, UDM_SETRANGE, 0,
			(LPARAM)MAKELONG(99, 0));
		SendDlgItemMessage(hwnd, IDD_LRSPIN, UDM_SETPOS, 0,
			(LPARAM)MAKELONG(prefsTADS.marginx, 0));
		SendDlgItemMessage(hwnd, IDD_TBSPIN, UDM_SETRANGE, 0,
			(LPARAM)MAKELONG(99, 0));
		SendDlgItemMessage(hwnd, IDD_TBSPIN, UDM_SETPOS, 0,
			(LPARAM)MAKELONG(prefsTADS.marginy, 0));
		CheckDlgButton(hwnd, IDD_JUSTIFYTEXT, (prefsTADS.fulljustify ?
			BST_CHECKED : BST_UNCHECKED));
		CheckDlgButton(hwnd, IDD_DOUBLESPACE, (prefsTADS.doublespace ?
			BST_CHECKED : BST_UNCHECKED));
		CheckDlgButton(hwnd, IDD_BOLDINPUT, (prefsTADS.fBoldInput ?
			BST_CHECKED : BST_UNCHECKED));
		return TRUE;

	  case WM_NOTIFY:
		switch (wParam) {
		  case IDD_LRSPIN:
			nmu = (NM_UPDOWN *)lParam;
			
			return TRUE;
		}
		return FALSE;

	  case WM_COMMAND:
		switch (wParam) {
		  case IDOK:
			lTemp = GetDlgItemInt(hwnd, IDD_LRMARGINS, NULL, FALSE);
			if (lTemp >= 0 && lTemp <= 99 &&
				lTemp != prefsTADS.marginx) {
				prefsTADS.marginx = (SHORT)lTemp;
				fRefresh = TRUE;
			}
			lTemp = GetDlgItemInt(hwnd, IDD_TBMARGINS, NULL, FALSE);
			if (lTemp >= 0 && lTemp <= 99 &&
				lTemp != prefsTADS.marginy) {
				prefsTADS.marginy = (SHORT)lTemp;
				fRefresh = TRUE;
			}
			fTemp = (IsDlgButtonChecked(hwnd, IDD_JUSTIFYTEXT) ==
				BST_CHECKED);
			if (fTemp != prefsTADS.fulljustify) {
				prefsTADS.fulljustify = fTemp;
				fRefresh = TRUE;
			}
			fTemp = (IsDlgButtonChecked(hwnd, IDD_DOUBLESPACE) ==
				BST_CHECKED);
			if (fTemp != prefsTADS.doublespace) {
				prefsTADS.doublespace = fTemp;
				fRefresh = TRUE;
			}
			fTemp = (IsDlgButtonChecked(hwnd, IDD_BOLDINPUT) ==
				BST_CHECKED);
			if (fTemp != prefsTADS.fBoldInput) {
				prefsTADS.fBoldInput = fTemp;
				UpdateInputFont(hdcClient, fTemp);
				fRefresh = TRUE;
			}
			if (fRefresh) {
			  	xtext_resize(0, 0, (SHORT)(clientRect.right -
			  		clientRect.left), (SHORT)lClientHeight);
				InvalidateRect(hwndClient, NULL, FALSE);
			}
		  case IDCANCEL:
			EndDialog(hwnd, (wParam == IDOK));
			return TRUE;
		}
	}
	return FALSE;
}

BOOL APIENTRY InterpreterDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	BOOL	fRefresh = FALSE, fTemp;
	LONG	lTemp;
	
	switch (msg) {
	  case WM_INITDIALOG:
		CenterWindow(hwnd, hwndFrame);
		SendDlgItemMessage(hwnd, IDD_HLSPIN, UDM_SETRANGE, 0,
			(LPARAM)MAKELONG(99, 2));
		SendDlgItemMessage(hwnd, IDD_HLSPIN, UDM_SETPOS, 0,
			(LPARAM)MAKELONG(prefsTADS.historylength, 0));
		SendDlgItemMessage(hwnd, IDD_BSISPIN, UDM_SETRANGE, 0,
			(LPARAM)MAKELONG(30000, 400));
		SendDlgItemMessage(hwnd, IDD_BSISPIN, UDM_SETPOS, 0,
			(LPARAM)MAKELONG(prefsTADS.buffersize, 0));
		SendDlgItemMessage(hwnd, IDD_BSLSPIN, UDM_SETRANGE, 0,
			(LPARAM)MAKELONG(5000, 1000));
		SendDlgItemMessage(hwnd, IDD_BSLSPIN, UDM_SETPOS, 0,
			(LPARAM)MAKELONG(prefsTADS.bufferslack, 0));
		CheckDlgButton(hwnd, IDD_PAGING, (prefsTADS.paging ?
			BST_CHECKED : BST_UNCHECKED));
		CheckDlgButton(hwnd, IDD_CLEARBYSCROLL,
			(prefsTADS.clearbyscroll ? BST_CHECKED : BST_UNCHECKED));
		return TRUE;

	  case WM_COMMAND:
		switch (wParam) {
		  case IDOK:
			lTemp = GetDlgItemInt(hwnd, IDD_HISTORYLENGTH, NULL, FALSE);
			if (lTemp >= 2 && lTemp <= 99 &&
				lTemp != prefsTADS.historylength) {
				prefsTADS.historylength = (SHORT)lTemp;
				fRefresh = TRUE;
			}
			lTemp = GetDlgItemInt(hwnd, IDD_BUFFERSIZE, NULL, FALSE);
			if (lTemp >= 400 && lTemp <= 30000 &&
				lTemp != prefsTADS.buffersize) {
				prefsTADS.buffersize = lTemp;
				fRefresh = TRUE;
			}
			lTemp = GetDlgItemInt(hwnd, IDD_BUFFERSLACK, NULL, FALSE);
			if (lTemp >= 1000 && lTemp <= 5000 &&
				lTemp != prefsTADS.bufferslack) {
				prefsTADS.bufferslack = lTemp;
				fRefresh = TRUE;
			}
			fTemp = (IsDlgButtonChecked(hwnd, IDD_PAGING) ==
				BST_CHECKED);
			if (fTemp != prefsTADS.paging)
				prefsTADS.paging = fTemp;
			fTemp = (IsDlgButtonChecked(hwnd, IDD_CLEARBYSCROLL) ==
				BST_CHECKED);
			if (fTemp != prefsTADS.clearbyscroll)
				prefsTADS.clearbyscroll = fTemp;
			if (fRefresh) {
			  	xtext_resize(0, 0, (SHORT)(clientRect.right -
			  		clientRect.left), (SHORT)(lClientHeight));
				InvalidateRect(hwndClient, NULL, FALSE);
			}
		  case IDCANCEL:
			EndDialog(hwnd, (wParam == IDOK));
			return TRUE;

		}
	}
	return FALSE;
}

BOOL APIENTRY OptionsDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch (msg) {
	  case WM_INITDIALOG:
		CenterWindow(hwnd, hwndFrame);
		CheckDlgButton(hwnd, IDD_STICKYPATHS, (prefsTADS.fStickyPaths ?
			BST_CHECKED : BST_UNCHECKED));
		CheckDlgButton(hwnd, IDD_STICKYMACROS, (prefsTADS.fStickyMacros ?
			BST_CHECKED : BST_UNCHECKED));
		CheckDlgButton(hwnd, IDD_CLOSEONEND, (prefsTADS.fCloseOnEnd ?
			BST_CHECKED : BST_UNCHECKED));
		CheckDlgButton(hwnd, IDD_CLOSEABORTS, (prefsTADS.fCloseAborts ?
			BST_CHECKED : BST_UNCHECKED));
		return TRUE;

	  case WM_COMMAND:
		switch (wParam) {
		  case IDOK:
			prefsTADS.fStickyPaths =
				(IsDlgButtonChecked(hwnd, IDD_STICKYPATHS) ==
				BST_CHECKED);
			prefsTADS.fStickyMacros =
				(IsDlgButtonChecked(hwnd, IDD_STICKYMACROS) ==
				BST_CHECKED);
			prefsTADS.fCloseOnEnd =
				(IsDlgButtonChecked(hwnd, IDD_CLOSEONEND) ==
				BST_CHECKED);
			prefsTADS.fCloseAborts =
				(IsDlgButtonChecked(hwnd, IDD_CLOSEABORTS) ==
				BST_CHECKED);
		  case IDCANCEL:
			EndDialog(hwnd, (wParam == IDOK));
			return TRUE;
		}
	}
	return FALSE;
}

BOOL APIENTRY AboutDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	extern BOOL APIENTRY DialogProc(HWND, UINT, WPARAM, LPARAM);
	
	switch (msg) {
	  case WM_INITDIALOG:
		CenterWindow(hwnd, hwndFrame);
		return TRUE;

	  case WM_COMMAND:
		switch (wParam) {
		  case IDOK:
		  case IDCANCEL:
			EndDialog(hwnd, TRUE);
			return TRUE;
		  case IDD_LICENSE:
			DialogBox(hInst, MAKEINTRESOURCE(DLG_LICENSE), hwndFrame,
				DialogProc);
			return TRUE;
		}
	}
	return FALSE;
}

BOOL APIENTRY RegistryDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	extern BOOL APIENTRY DialogProc(HWND, UINT, WPARAM, LPARAM);
	
	switch (msg) {
	  case WM_INITDIALOG:
		CenterWindow(hwnd, hwndFrame);
		return TRUE;

	  case WM_COMMAND:
		switch (wParam) {
		  case IDOK:
		  case IDCANCEL:
		  case IDD_LATER:
			EndDialog(hwnd, wParam);
			return TRUE;
		}
	}
	return FALSE;
}
