/*

   == Literal Capture Verb 

   == Copyright (c) 2002, Steve Breslin
 
   ==>>  License:
     _______________________________________________________________
    /                                                               \
    |     You can use this program however you want, but if you     |
    |     decide to distribute anything that uses, #include's or    |
    |     otherwise borrows from this program, you have to send     |
    |     a copy of any improvements or modify-cations you made     |
    |     to this program to its author, Steve Breslin.             |
    |                                                               |
    |     That way, you will help keep the program up to date for   |
    |     everybody else, and everybody else will help keep it      |
    |     up to date for you.                                       |
    |                                                               |
    |     You may redistribute this verbatim or in modified form    |
    |     only if you keep the copyright and license intact.        |
    |                                                               |
    |     Also, feel encouraged to release your source code along   |
    |     with your game, though this isn't a requirement.          |
    |                                                               |
    |     The author can be reached at <versim@hotmail.com>.        |
    \______________________________________________________________*/


literalCaptureVerb: deepverb
    commandStringBuffer = ''
    action(actor) = {
        self.literalAction(actor);
        literalCaptureVerb.commandStringBuffer := '';
    }
;

/*0001 0010 0011 0100 0101 0110 0111 1000 1001 1010 1011 1100 1101*\
\^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\
 \                          Pre-parser                               \
  \___________________________________________________________________\
   \*1110 1111 0000 0001 0010 0011 0100 0101 0110 0111 1000 1001 1010*/

    
// This preparse function is designed to capture input following
// certain verbs -- verbs of class literalCaptureVerb. The
// literalCaptureVerb is just a normal verb with a couple of changes.
//
// The literalCaptureVerb object is capable of storing a string, and
// does so when a string follows such a verb on the command input line.
// The literalCaptureVerb clears this string after it processes the
// action.
//
// The literalCaptureVerb takes care of processing that string, to the
// extent that processing is necessary. If we encounter a
// literalCaptureVerb, only that verb, and not the material which
// follows it on the command line, will be passed to the parser. The
// other material (the captured material) will be handled from within
// the action processing of the verb.

preparse: function(str) {

    /* firstToken is the first token on the command line. we want to
     * check if the first token is a literalCaptureVerb. if it is, we
     * store whatever comes after it, if anything, in the
     * literalCaptureVerb.commandStringBuffer property.
     *
     * tokType is the type of the first token.
     */
    local firstToken, tokType, verbResolved;

    /* get the first token on the command line, and put it in list form
     */
    firstToken := [(parserTokenize(str)[1])];

    /* get a list of token types which the first token can possibly
     * resolve to. (PRSTYP_NOUN, PRSTYP_VERB, etc.)
     */
    tokType := parserGetTokTypes(firstToken);

    /* if it resolves to one or more token types, and resolves to the
     * verb token type, assume it is a verb token.
     */
    if (tokType != [] && intersect ([PRSTYP_VERB], tokType) != []) {

        /* verbResolved is the verb object which resolves from the
         * first token.
         */
        local verbResolved;

        /* at this point, we can resolve the token, as we are assuming
         * it is a verb.
         */
        verbResolved := parserDictLookup(firstToken, [PRSTYP_VERB]);

        /* if the token resolves to a literalCaptureVerb, and material
         * follows the verb (that is, it's not alone on the command
         * line), we want to capture the material that follows it.
         */
        if(isclass(verbResolved[1], literalCaptureVerb) &&
                                                    find(str, ' ')) {

            /* capture the material that follows it to the
             * literalCaptureVerb's commandStringBuffer.
             */ 
            literalCaptureVerb.commandStringBuffer := substr(str,
                                                 find(str, ' '),
                                                 length(str));

        /* we pass only the first token to the parser, and allow the
         * literalCaptureVerb to take care of processing the captured material.
         */
        return firstToken[1];
        }
    }

    /* the first token didn't resolve to a literalCaptureVerb, or there
     * was no material after the literalCaptureVerb on the command
     * line.
     * either way, we proceed to the parsing stage as if nothing
     * happened.
     */
    return true;
}
;
