/* ex:set ts=4 sw=4:
 *
 * sysfuncs.t:	customized version of std.t
 *
 * This module contains the standardized functions that I include into all
 * TADS games.  It contains most of the functions provided by the standard
 * <std.t> but also defines the default hook routines for some bug-fixing
 * and for some rearrangement of output.
 *
 * The module also provides a standard initialization mechanism for other
 * modules which are included.  Rather than need to keep track of what
 * needs to be initialized, the user can concentrate on writing the game.
 *
 * Each module which requires initialization should declare an object of
 * class "initialization".  The function "preinit()" will scan for these
 * objects and will call the "prescan_phase" method in them.  It will also
 * collect all initialization objects which have "init_phase" code methods
 * and keep them.  When "init()" is run, it will first invoke each of these
 * methods; when finished, it will call the function "gameinit()" which
 * should perform all remaining game-specific initialization.
 *
 * This module is Copyright (c) 1994 Jeff Laing.  Permission to use any or all
 * of this code in other TADS games is granted provided credit is given in
 * your "CREDITS" command for this code and that you leave this copyright
 * message in the source whenever distributed.  Please make all changes in
 * an backward compatible manner (where possible) and make them available
 * for free (like I did ;-)
 */
sysfuncsVersion: versionTag
	id="$Id: sysfuncs.t_v 1.3 1994/05/06 09:20:30 jeffl Exp jeffl $\n"
;

/*
 * this module provides the following functions as entry points
 */
preinit:	function;		// run at compile time
init:		function;		// run at startup time
gameinit:	function;		// run by init() as last action

/*
 * called by the parser when the player doesn't type anything
 */
pardon: function
{
}

/*
 * called by the parser to issue a prompt (under various circumstances)
 */
commandPrompt: function( prompt_mode )
{
	switch (prompt_mode) {
	case 0:		// normal command
		"> ";
		break;
	case 1:		// command after invalid word (allowing "oops" to be used)
	case 2:		// disambiguation (after "which x do you mean..." question)
	case 3:		// command after askdo (after "what do you want to x?")
	case 4:		// command after askio
		"\n> ";
		break;
	}
}

/*
 * called whenever the parser encounters an error
 */
parseError: function( errno, str )
{
	switch (errno) {

	// Errors I have tweaked with
	case 2:		// I don't know the word '%s'.
		return('[You don\'t need the word "%s" to play this game]');
	case 9:		// I don't see any %s here
		return('I don\'t see what you are referring to here.' );
	case 23:	// internal error: verb has no action, doAction or ioAction
		return(str+'\nPlease notify the program\'s author of this message.');

	//	Normal error messages
	//	---------------------
	case 1:		// I don't understand the punctuation '%c'.
	case 3:		// The word '%s' refers to too many objects.
	case 4:		// I think you left something out after 'all of'.
	case 5:		// There's something missing after 'both of'
	case 6:		// I expected a noun after 'of'
	case 7:		// An article must be followed by a noun.
	case 8:		// You used 'of' too many times
	case 10:	// You're referring to too many objects with '%s'.
	case 11:	// You're referring to too many objects.
	case 12:	// You can only speak to one person at a time.
	case 13:	// I don't know what you're referring to with '%s'.
	case 14:	// I don't know what you're referring to.
	case 15:	// I don't see what you're referring to.
	case 16:	// I don't see that here.
	case 17:	// There's no verb in that sentence!
	case 18:	// I don't understand that sentence
	case 19:	// There are words after your command I couldn't use.
	case 20:	// I don't know who to use the word '%s' like that.
	case 21:	// There appear to be extra words after your command.
	case 22:	// There seem to be extra words in your command.
	case 24:	// I don't recognize that sentence.
	case 25:	// You can't use multiple indirect objects.
	case 26:	// There's no command to repeat.
		return('['+str+']');

	// Used during disambiguation
	// --------------------------
	case 100:	// Lets try it again:
	case 101:	// Which %s do you mean,
	case 102:	// ,
	case 103:	// or
	case 104:	// ?
		return(str);

	// Used to complain when an object is no good for a verb
	// -----------------------------------------------------
	case 110:	// I don't know how to
		return('['+str);

	case 111:	// 
	case 112:	// anything
	case 113:	// to
	case 114:
		return(nil);

	case 115:	// .
		return(str+']');

	// Used after each object when multiple objects are used.
	// ------------------------------------------------------
	case 120:	// :

	// Used to note objects being used by default
	// ------------------------------------------
	case 130:	// (
	case 131:	// )
	case 132:
		return( nil );

	// Used to ask for an object when one can't be defaulted
	// -----------------------------------------------------
	case 140:	// What do you want to
		return('['+str);	
	case 141:	// it
	case 142:	// to
		return( nil );
	case 143:	// ?
		return( str+']' );

	// Used when an object is unreachable obj.sdesc ":" obj.cantReach
	// --------------------------------------------------------------
	case 200:	// :

		return(nil);
	}
}

/*
 *   darkTravel() is called whenever the player attempts to move from a dark
 *   location into another dark location.  By default, it just says "You
 *   stumble around in the dark," but it could certainly cast the player into
 *   the jaws of a grue, whatever that is...
 */
darkTravel: function
{
    "You stumble around in the dark, and don't get anywhere. ";
}

/*
 * preinit() will scan all objects of class "initialization".  for each
 * object found, it will call "object->preinit_phase".  It will then check
 * for the existance of a method called "init_phase"; if it is found, the
 * object will be added to "global.init_list" for init() to call later.
 */
class initialization: object ;

/*
 * preinit() is called after compiling the game, before it is written
 * to the binary game file.  It performs all the initialization that can
 * be done statically before storing the game in the file, which speeds
 * loading the game, since all this work has been done ahead of time.
 */
preinit: function
{
    local o;

	global.init_list := [];

	for (o:=firstobj(initialization); o<>nil; o:=nextobj(o,initialization)) {

		// add to the list (for init: to call)
		if (proptype(o,&init_phase)=6)
			global.init_list += o;

		// now do whatever we can in advance
		o.preinit_phase;
	}
}

/*
 *	We work through the list of objects which were found at preinit() time
 *	and call any init_phase methods found.
 *
 *	Once processed, we call the "gameinit()" function which performs the
 *	functions traditionally associated with init() (except those which have
 *	already been automatically called)
 */
init: function
{
	local i,len;

	// call all the modules which registered that they wanted initialization
	len := length(global.init_list);
	for (i:=1; i<=len; i++) {
		global.init_list[i].init_phase;
	}

	// start the turncounter daemon here as well.
    setdaemon( turncount, nil );               // start the turn counter daemon

	// now call the game specific one
	gameinit();
}

/*
 *   The numObj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numObj, with its "value"
 *   property set to 621.
 */
numObj: basicNumObj ;	// use default definition from adv.t

/*
 *   strObj works like numObj, but for strings.  So, a player command of
 *     type "hello" on the keyboard
 *   will result in a direct object of strObj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 */
strObj: basicStrObj ;	// use default definition from adv.t

/*
 *   The "global" object is the dumping ground for any data items that
 *   don't fit very well into any other objects.  The properties of this
 *   object that are particularly important to the objects and functions
 *   are defined here; if you replace this object, but keep other parts
 *   of this file, be sure to include the properties defined here.
 *
 */
global: object
    turnsofar = 0                            // no turns have transpired so far
    verbose = nil                             // we are currently in TERSE mode
;

/*
 *   The "version" object defines, via its "sdesc" property, the name and
 *   version number of the game.  Change this to a suitable name for your
 *   game.
 */
version: object
    sdesc = "A TADS Adventure
      \n Developed with TADS, the Text Adventure Development System.\n"
;

/*
 *   The terminate() function is called just before the game ends.  It
 *   generally displays a good-bye message.  The default version does
 *   nothing.  Note that this function is called only when the game is
 *   about to exit, NOT after dying, before a restart, or anywhere else.
 */
terminate: function
{
}
