/*
 * 	museum.t				Release 2 Serial number 991214
 *	by Kevin Forchione		( Lysseus@msn.com )
 *
 *  One of the standard Inform 6 example games, ported to TADS.
 *
 *------------------------------------------------------------
 *
 *	This "game" is a compendium of solved exercises from the 
 *  Inform Designer's Manual.  Room numbers are section numbers 
 *  within the Third Edition.
 *	Many of these exercises are difficult, and so is some of the 
 *  code below.
 *
 *------------------------------------------------------------
 *
 *  This is a port to TADS 2.5.0 of museum.inf, from Inform.  
 *  It is only partial, with most of the rooms and puzzles 
 *  translated into a TADS equivalent. About 2/3 of the game
 *  is here ... from the basement to through the second floor.
 *  
 *  Among things left undone are the entire 3rd floor of the Museum,
 *  which housed listing functions; Room 10a, which handled a compass
 *  directional change; and Dyslexic Dan, who took a wrong turn
 *  and got lost somewhere along the way.
 *
 *  I found this an enjoyable learning experience. Even when the puzzle
 *  solution was laid out in black and white, translation into TADS 
 *  proved challenging and educational.
 *
 *-------------------------------------------------------------
 *
 *	This file requires 
 *      inform.t 4.0
 *      HTML TADS 2.5.1 or later
 *
 *------------------------------------------------------------------------------
 *		Revision History
 *------------------------------------------------------------------------------
 *		99-07-31:	Ported to TADS.
 *      99-11-18:   Added TADS / Inform.t revision history.
 */

#define USE_HTML_STATUS
#define USE_COMMON_SENSE

#include <adv.t>
#include <std.t>
#include <inform.t>
 
#pragma C+

/*
 *  This demonstrates a sackItem. This class 'tidies' away objects in a
 *  player's possessions when they have reached their carrying capacity.
 */
samplesBag: sackItem
	location = Me
	sdesc = "sack"
	adesc =
	{
		"your "; self.sdesc;
	}
	ldesc = 
	{
		"A capacious SACKITEM OBJECT. ";
		pass ldesc;
	}
	whenClosed = "Your samples bag lies on the floor."
	whenOpen = "Your samples bag lies open on the floor."
	noun = 'bag' 'sack'
	adjective = 'samples'		
;

/*==============================================================================
 *	FIRST FLOOR
 *=============================================================================*/

/*------------------------------------------------------------------------------
 *	FOYER
 *----------------------------------------------------------------------------*/

foyer: room
	sdesc = "Foyer"
	ldesc = "The foyer of the magnificent Museum of TADS. A flight of 
		stairs sweeps upward to Chapters IV and V, and corridors
		fan out on this level into Chapter III. These run:
		<br>\tnorth into Room 10 (the Halls of Sense and Direction),
		<br>\tsouth along Corridor 12 (doors) to Room 13 (switches),
		<br>\t<<global.d00>> to Room 11 (the Wonderful World of Containers)
		<br>\tand <<global.d01>> to Room 14 (which houses larger exhibits).
		<P>\t(And a plain ladder descends into the basement; you can also
		step out onto a balcony.)"
	north   = room10
	south   = corridor12
	east    = room11
	west    = room14
	up      = secondFloor
	down    = basement
	out     = balcony
;

map1: fixeditem, wysiwyg
    location = foyer
    noun = 'sign' 'map'
    adjective = 'large' 'wall'
    sdesc = "map"
    initial = "A map of the ground floor is embossed in one wall."
    ldesc = 
    {
        "It's a large map that occupies one entire wall of the 
        foyer. It reads: ";
        pass ldesc;
    }
    headerText =
    [
        'MUSEUM MAP'
        'FIRST FLOOR'
    ]
    bodyText = 
    [
        '                                    (mirror)'
        'Weights          Room 10 ---------- Room 10a'
        ' Room             senses            directions'
        '   |                |'
        'Room 14 --------- Foyer ------------ Room 11'
        '(changing     (you are here)        containers'
        'character)          |'
        '                    |'
        'Room 15 ------- Corridor 12'
        '(MS room)     (door, curator)'
        '(Bible)             |'
        '              Switches Hall 13'
        '                 (chasm)'  
        '                    |'
        '          Far Side of Room 13'
    ]
    footerText =
    [
        '(This Sign Demonstrates The WYSIWYG Class Object, now available with 
        the TADS inform.t library extension.)'
    ]
;

/*----------------------------------------------------------------------
 *	BASEMENT (MUSEUM BASEMENT)
 *---------------------------------------------------------------------*/
	
basement: darkroom
	sdesc = "Museum Basement"
	ldesc = "Once Room 17, this space has been abandoned by the curators, who
		have simply sprayed its walls with moth-repellent and 
		abandoned it. "
	roomPreAction( a, v, d, p, i ) = 
	{
		if ( v == smellVerb )
		{
			"Napthalene. ";
			exit;
		}
	}
	up = foyer
;

basementLight: lightsource, fixeditem
	location = basement
	noun = 'light'
	adjective = 'basement'
	sdesc = "basement light"
;

/*------------------------------------------------------------------------------
 *	ROOM 10 (HALL OF SENSES)
 *----------------------------------------------------------------------------*/

room10: room
	sdesc = 
	{
	    if ( orangeCloud.isIn( self ) )
	        "Wandering In An Orange Cloud";
	    else
	        "Hall of Senses";
	}
	ldesc = "This is Room 10, north of the foyer. If you try going
		further north, a string will be printed instead. The exit
		to the <<global.d00>> has a method attached, and randomly leads either
		to the Directions Room or back to the foyer. 
		There is a trap lever here."
	//
	//  roomPreAction( a, v, d, p, i )
	//
	//  An example of roomPreAction. This code works better here than in
	//  roomAction, because it supercedes any scopePreAction. It also
	//  needs to be here, instead of orangeCloud.scopePreAction, because
	//  we want the orangeCloud to have precedence over all other
	//  objects when it is present in the room.
	//
	roomPreAction( a, v, d, p, i ) =
	{
	    if (!orangeCloud.isIn(self)) return;
	    
	    if (isclass(v, sysverb)) return;

	    if (isclass(v, travelVerb))
	    {
	        "You wander around in circles, choking. ";
	        exitobj;
	    }
	    switch( v ) 
	    {
	        case inspectVerb:
	        case lookVerb:
	            "You can't see for the orange cloud surrounding you. ";
	            break;
	        case smellVerb:
	            if (d == nil || d == self)
	                "Cinnamon? No, nutmeg. ";
	            break;
		    case pullVerb:
		    case pushVerb:
		        if (d == trapLever) return;
	        default:
	            "You grope helplessly in the orange cloud. ";
	    }
        exitobj;
	}
	north = "This string is the <q>north</q> value for this room. "
	south = foyer
	east =
	{
		if ( _rand( 100 ) < 50 ) 
		   return foyer;
		else
		   return room10a;
	}
;

trapLever: fixeditem
    location = room10	
	noun = 'lever'
	adjective = 'trap'
    sdesc = "TRAP lever"
	verDoPush( actor ) = {}
	doPush( actor ) =
	{
		if ( orangeCloud.isIn( room10 ) )
		{
			"Click! The orange cloud vanishes.<P>";
			orangeCloud.moveInto( nil );

		}
		else
		{

			"Clunk! An orange cloud suddenly descends upon you.<P>";
			orangeCloud.moveInto( room10 );
		}
	}
	doSynonym( 'Push' ) = 'Pull'
;

goldWatch: item
	location = room10
	noun = 'watch'
	adjective = 'gold'
	sdesc = "gold watch"
	ldesc = "The watch has no hands, oddly. "
	verDoListento( actor ) = {}
	doListento( actor ) =
	{
		"The watch ticks loudly. ";
	}
	scopePreAction( a, v, d, p, i ) =
	{
	    if (v == listenVerb
	    && d == nil)
	    {
	        self.doListento( a );
	        exitobj;
	    }
	}
;

sprig: item
	location = room10
	noun = 'sprig'
	adjective = 'lavender'
	sdesc = "sprig of lavender"
	verDoSmell( actor ) = {}
	doSmell( actor ) =
	{
		"Lavender-perfume. Ahh! ";
	}
	scopePreAction( a, v, d, p, i ) =
	{
	    if (v == smellVerb
	    && (d == nil || d == self))
	    {
	        self.doSmell( a );
	        exitobj;
	    }
	}
;

orangeCloud: fixeditem
	noun = 'cloud'
	adjective = 'orange'
	sdesc = "orange cloud"
;


/*------------------------------------------------------------------------------
 *	FOR EAST-WEST REFLECTION
 *----------------------------------------------------------------------------*/
 
room10a: room
	sdesc = "Hall of Directions"
	ldesc = "An annexe to Room 10: the main exit is back <<global.d01>>, and there's a
		curious misty mirror on the north wall. Also there's an exit in the
		fourth, <q>hyper</q> dimension. (which is only visible in the weird
		geometry of this room.\b"
	northAction(actor) = 
	{
		"Walking through the mirror is confusing...\b";
        global.eastwestReversed = !global.eastwestReversed;
        actor.location.lookAround(actor, true, true);
	}
	north = nil
	west = room10
	inAction(actor) = "Amazing! Your body corkscrews as you pass."
	in = room11
;

/*------------------------------------------------------------------------------
 *	ROOM 11 (WONDERFUL WORLD OF CONTAINERS
 *----------------------------------------------------------------------------*/

room11: room
	sdesc = "Wonderful World of Containers"
	ldesc = "This is Room 11, <<global.d00>> of the foyer. You notice a typical piece
		of scenery which turns out to be a Surface: a mantelpiece. "
	west = foyer
;

psychiatrist: Actor
	location = room11
	noun = 'doctor' 'psychiatrist' 'psychologist' 'shrink'
	adjective = 'bearded'
	sdesc = "bearded psychiatrist"
	initial = "A bearded psychiatrist has you under observation. "
	iobjCheck( a, v, d, p ) =
	{
	    if (v == inspectVerb) return;
	    
		"He is fascinated by your behaviour, but makes no attempt to
		interfere with it. ";
		exitobj;
	}
	dobjCheck( a, v, i, p ) =
	{
	    self.iobjCheck( a, v, i, p );
	}
	scopePreAction( a, v, d, p, i ) =
	{
		switch( v )
		{
			case takeVerb: 
			case removeVerb:
				"<q>Subject feels lack of <<d.thedesc>>. Suppressed Oedipal
				complex? Mmm.</q>\b";
				break;
		}
	}
	scopePostAction( a, v, d, p, i, status ) =
	{
		switch( v )
		{
			case putVerb: 
			    // checking game-state
			    if (!d.isIn(i))
			        status = true;
				"\b<q>Subject <<status ? "fails to put " : "puts ">> 
				<<d.adesc>> in <<i.thedesc>>.
				<<status ? "Most distressing" : "Interesting">>.</q> ";
					break;
		}
	}
;

mantelpiece: fixeditem, surface
	location = room11
	noun = 'mantel' 'mantle' 'piece' 'mantelpiece' 'mantlepiece'
	adjective = 'mantel' 'mantle'
	sdesc = "mantelpiece"
;

greenBall: item
	location = room11
	noun = 'ball'
	adjective = 'green'
	sdesc = "green ball"
;

redCone: item
	location = room11
	noun = 'cone'
	adjective = 'red'
	sdesc = "red cone"
;

bluePyramid: item
	location = room11
	noun = 'pyramid'
	adjective = 'blue'
	sdesc = "blue pyramid"
;

shoppingBag: container
	location = room11
	noun = 'bag'
	adjective = 'plain' 'shopping'
	maxbulk = 2
	sdesc = "plain shopping bag (which can only hold 2 things) "
;

glassBox: transparentItem
	location = room11
	noun = 'box' 
	adjective = 'glass'
	sdesc = "glass box with a lid"
;

steelBox: openable
	location = room11
	noun = 'box'
	adjective = 'steel'
	sdesc = "steel box with a lid"
;

toothedBag: openable
	location = room11
	noun = 'bag'
	adjective = 'toothed'
	sdesc = "toothed bag"
	ldesc = "A capacious bag with a toothed mouth. "
	doClose( actor ) = 
	{
		"The bag resists all attempts to close it. ";
	}
	verGrab( obj ) =
	{
		"The bag defiantly bites itself shut on your hand until you
		desist. ";
	}
	ioPutIn( actor, dobj ) =
	{
		"The bag wriggles hideously as it swallows
		<<dobj.thedesc>>. ";
		pass ioPutIn;
	}
;

cupboard: fixeditem, lockable
	location = room11
	noun = 'cupboard'
	adjective = 'bolted'
	isopen = nil
	mykey = boltedKey
	sdesc = "bolted cupboard"
	describe = 
	{   
	    "\n\t";
		if ( self.isopen )
		{
			"Bolted up on one wall is an open cupboard. ";
		}
		else
		{
			"A pretty typical shut cupboard is bolted to one
			wall. ";
		}
		return true;
	}
;

boltedKey: keyItem
	location = room11
	noun = 'key'
	adjective = 'bolted'
	sdesc = "bolted key"
;

television:	item
	location = room11
	noun = 'television' 'tv' 'set'
	adjective = 'portable'
	haspower = nil
	sdesc = "television"
	initial = 
	{
		"A portable television set has two sub-objects attached: a 
		power button and a screen. ";
	}
	doPush -> powerButton
;

powerButton: fixeditem
	location = television
	noun = 'button'
	adjective = 'power'
	sdesc = "power button"
	verDoPush( actor ) = {}
	doPush( actor ) =
	{
		if ( television.haspower )
		{
			television.haspower = nil;
		}
		else
		{
			television.haspower = true;
		}
		screen.ldesc;
	}
;

screen: fixeditem
	location = television
	noun = 'screen'
	adjective = 'television' 'tv'
	sdesc = "screen"
	ldesc = 
	{
		if ( television.haspower )
		{
			"The screen writhes with a strange Japanese cartoon. ";
		}
		else
		{
			"The screen is black. ";
		}
	}
;

macrameBag: fixeditem, transparentItem
	location = room11
	noun = 'bag'
	adjective = 'macrame'
	isopen = nil
	sdesc = "macrame bag"
	doOpen( actor ) = "%You% can't open the bag, it's shut tight."
	describe = 
	{
		"\n\tA macrame hangs from the ceiling, shut tight. ";
		if (itemcnt(self.contents)) 
		{
			"Inside you can make out ";
			listcont(self); ". ";
		}
		return true;
	}
	//
	//  This scopePreAction doesn't get much action. This is because
	//  TADS handles accessibility rules as part of the PARSER
	//  RESOLUTION phase, and not the EXECUTION phase. Since the 
	//  box is inside the bag its access is very limited.
	//
	scopePreAction( actor, verb, dobj, prep, iobj ) =
	{
		switch( verb )
		{
			case inspectVerb:
			case searchVerb:
			case listenVerb:
			case listentoVerb:
			case smellVerb:
			    break;
			default:
				if (dobj && dobj.isIn(self))
				{
					"\^<<dobj.thedesc>> is tucked away in the bag. ";
					exitobj;
				}
				if (iobj && iobj.isIn(self))
				{
					"\^<<iobj.thedesc>> is tucked away in the bag. ";
					exitobj;
				}
		}
	}
;

musicBox: item
	location = macrameBag
	noun = 'box'
	adjective = 'music' 'musical'
	sdesc = "music box"
	ldesc = "Attractively lacquered."
	verDoListento( actor ) = {}
	doListento( actor ) =
	{
			"The musical box chimes some Tchaikovsky. ";
	}
	scopePreAction( actor, verb, dobj, prep, iobj ) =
	{
		if (verb == listenVerb)
		{
			self.doListento( actor );
			exitobj;
		}
	}
;

/*------------------------------------------------------------------------------
 *	CORRIDOR 12
 *----------------------------------------------------------------------------*/

corridor12: room
	sdesc = "Corridor 12"
	ldesc = 
	{
		"Runs south of the foyer. A side exit leads <<global.d01>> to Room 15, the
		manuscripts room. <<oakDoor.isopen ? "The oaken door stands open" :
		"The corridor ends in an oaken door">>.";
	}
	north = foyer
	south = oakDoor
	west = room15
;

curator: Actor
	location = corridor12
	noun = 'curator'
	adjective = 'sinister'
  	myfollower = followerCurator
	sdesc = "sinister curator"
	actorDesc =
	{
		if ( self.isawake )
			"The curator is here, casting menacing looks. ";
		else
			"A sinister curator leans against the wall, asleep. ";
	}
	//
	//  actorAction is the TADS equivalent of Inform's orders method
	//
	actorAction( verb, dobj, prep, iobj ) =
	{
		"He rears back with disdain. ";
		exitobj;
	}
	verDoWakeup( actor ) = 
	{
		if ( self.isawake )
			"But he's already awake!";
	}
	doWakeup( actor ) =
	{
		self.isawake = true;
		"He wakes and rouses to his feet, pushes past you and stalks away. ";
		self.moveInto( foyer );
		notify( self, &actorDaemon, 0 );
	}
	//
	//  dobjGen() and iobjGen() serve as the equivalent of Inform's life
	//  method. dobjCheck and iobjCheck can serve this role also.
	//
	dobjGen( a, v, o, p ) =
	{
		if ( self.isawake )
			"He raises a skeletal finger to his lips, <q>Shh!</q> ";
		else
			"The curator only snores. ";
		exit;
	}
	iobjGen( a, v, o, p ) =
	{
	    self.dobjGen( a, v, o, p );
	}
  	actorDaemon =  
  	{    
  		local dir, dirnum, tries, newloc;  
  		
    	for ( tries = 1 ; tries < 50 ; tries++ )    
    	{        
    		dirnum = _rand( 6 );
        	dir = [ &north, &south, &east, &west, &up, &down ][ dirnum ];
        	if ( proptype( self.location, dir ) == 2 /* object */ )        
        	{
            	newloc = self.location.( dir );
            	if ( ! isclass( newloc, room ) ) continue;
            	if ( self.location == parserGetMe().location )
                	"The curator stalks away. ";
            	self.moveInto( newloc );
            	if ( self.location == parserGetMe().location )
                 	"The curator stalks into the room.";
                break;
        	}
    	}
  	}
;

followerCurator: follower
	noun = 'curator'
	adjective = 'sinister'
	myactor = curator
;

oakDoor: door
	noun = 'door'
	adjective = 'oaken' 'oak'
	foundin = [ corridor12 room13 ]
	sdesc = "oaken door"
	ldesc = "Despite appearances, there is no key to this door."
;

/*------------------------------------------------------------------------------
 *	ROOM 13 (SWITCHES HALL)
 *----------------------------------------------------------------------------*/

room13: room
	sdesc = "Switches Hall"
	ldesc = 
	{
		"This is Room 13, at the south end of a long corridor leading north. 
		There is a curious switch upon one wall.
		<<oakDoor.isopen ? "The oaken door stands open" :"The corridor ends in an 
		oaken door">>. Progress further south is blocked by a chasm. A precarious plank 
		bridge spans the chasm.\b";
	}
	north = oakDoor
	south = plankBridge
;

searchlight: lightsource, switchItem
	location = room13
	noun = 'light' 'searchlight'
	adjective = 'gotham' 'city'
	sdesc = "Gotham City searchlight"
	adesc = "<<self.thedesc>>"
	whenOn = "The old city searchlight shines out a bat against 
        the alabaster ceiling of the Museum."
    whenOff = "The old Gotham city searchlight has been lovingly 
                restored and housed here."
	ldesc = "It has some kind of template on it."
;

basementLightSwitch: switchItem
	location = room13
	noun = 'switch'
	adjective = 'basement' 'light'
	isListed = true
	sdesc = "basement light switch"
	doTurnon( actor ) =
	{
		basementLight.islit = true;
		"Click! ";
	}
	doTurnoff( actor ) =
	{
		basementLight.islit = nil;
		"Click! ";
	}
;

plankBridge: door
	isopen = true
	foundin = [ farside room13 ]
	sdesc = "plank bridge"
	ldesc = "Extremely fragile and precarious."
;

/*------------------------------------------------------------------------------
 *	Far Side
 *----------------------------------------------------------------------------*/

farside: room
	sdesc = "Far Side of Chasm in Room 13"
	ldesc = "This side of the chasm is dull and dusty after all."
	north = plankBridge
;

/*------------------------------------------------------------------------------
 *	ROOM 14 (LARGE EXHIBIT ROOM)
 *----------------------------------------------------------------------------*/

room14: room
	sdesc = "Large Exhibits Room"
	ldesc = "This is Room 14, west of the foyer, which gives onto the Weights
		Room to the north."
	north = weightRoom
	east = foyer
	in = weightRoom
;

portrait: fixeditem
	location = room14
	noun = 'portrait' 'picture'
	adjective = 'hypnotic' 'warthog' 'hog'
	sdesc = "hypnotic portrait"
	initial = "A somewhat hypnotic full-length portrait of a warthog
		hangs on the wall."
	doInspect( actor ) =
	{
		hog.doInspect( actor ); 
		"\b...and somehow, so do you...\b";
		switchPlayer(hog, true);
		execCommand(hog, lookVerb);
	}
;

louisxvChair: fixeditem, chairitem
	location = room14
	noun = 'chair'
	adjective = 'louis' 'xv'
	sdesc = "Louis XV chair"
    initial = "A valuable Louis XV chair takes pride of place."
;

redCar: dvehicle
	location = room14
	noun = 'car'
	adjective = 'little' 'red'
	sdesc = "little red car"
    whenOn = "The red car sits here, its engine still running."
    whenOff = "A little red car is parked here."
	ldesc = 
	{	    	    
	    "Large enough to sit inside. Among the controls is a prominent
		on/off switch. The number plate is KAR 1. The car is 
		<<self.isActive ? "on" : "off">>. ";
		showcontcont(self);
 	}
	doTurnon -> carSwitch
	doTurnoff -> carSwitch
	/*
	 *	This method should return true if the direction is valid for the
	 *	vehicle; otherwise it should return nil.
	 */
	validVehDir( v, l ) =
	{
		if ( v == uVerb || v == dVerb )
			return nil;
		else
			pass validVehDir;
	}
	goMessage = "Brmm! Brmm!"
	cantgoMessage = "You can't go that way."
	inactiveMessage = "(The ignition is off at the moment.)"
    verDoMoveN(actor) = {}
    verDoMoveS(actor) = {}
    verDoMoveE(actor) = {}
    verDoMoveW(actor) = {}
    verDoMoveNE(actor) = {}
    verDoMoveNW(actor) = {}
    verDoMoveSE(actor) = {}
    verDoMoveSW(actor) = {}
    doMoveN(actor) = { self.vehMoveDir(actor,nVerb); }
    doMoveS(actor) = { self.vehMoveDir(actor,sVerb); }
    doMoveE(actor) = { self.vehMoveDir(actor,eVerb); }
    doMoveW(actor) = { self.vehMoveDir(actor,wVerb); }
    doMoveNE(actor) = { self.vehMoveDir(actor,neVerb); }
    doMoveNW(actor) = { self.vehMoveDir(actor,nwVerb); }
    doMoveSE(actor) = { self.vehMoveDir(actor,seVerb); }
    doMoveSW(actor) = { self.vehMoveDir(actor,swVerb); }
    vehMoveDir(actor, verb) = 
    {
    	switch(verb)
    	{
    		case nVerb:
    			self.location.north;
    			break;
    		case sVerb:
    			self.location.south;
    			break;
    		case eVerb:
    			self.location.east;
    			break;
    		case wVerb:
    			self.location.west;
    			break;
    		case neVerb:
    			self.location.ne;
    			break;
    		case nwVerb:
    			self.location.nw;
    			break;
    		case seVerb:
    			self.location.se;
    			break;
    		case swVerb:
    			self.location.sw;
    			break;
    		default:
    			"%You% can't seem to do that. "; 
    	}
    }
;

carSwitch: switchItem
	location = redCar
	noun = 'switch'
	sdesc = "large on/off switch"
	doTurnon( actor ) =
	{
		redCar.isActive = true;
		pass doTurnon;
	}
	doTurnoff( actor ) =
	{
		redCar.isActive = nil;
		pass doTurnoff;
	}
;

smallNote: readable
	location = redCar
	noun = 'note'
	adjective = 'small'
	sdesc = "small note"
	initial = "A small note rests on the seat of the little
	    red car."
	ldesc = "!!!! FROBOZ MAGIC CAR COMPANY !!!!\bHello
		Driver!\bInstructions for use:\b\tSwitch on the ignition and off you
		go!\b\tWarranty:\bThis car is guaranteed against all defects for a
		period of 76 milliseconds from date of purchase or until used, whichever
		comes first.\b\bGood Luck!"
;

giant: basicMe
	location = room14
	noun = 'giant'
	sdesc = "Giant"
	adesc = "a <<sdesc>>"
	thedesc = "the <<sdesc>>"
	isHim = true
	initial = "A Giant is here. (And you know the old legend... push a Giant,
		become a Giant!)"
	actorAction( v, d, p, i ) =
	{
		local p;
		p = parserGetMe();
		if ( parserGetMe() != self )
			pass actorAction;
		else
			if ( isclass( v, travelVerb ) ) {}
			else
			switch( v )
			{
			    case boardVerb:
			    case getOutVerb:
				case lookVerb:
				case inspectVerb:
				case iVerb:
				case eatVerb:
				case pushVerb:
				case smellVerb:
				case tasteVerb:
				case touchVerb:
				case takeVerb:
				case dropVerb:
				case quitVerb:
				case saveVerb:
				case sitVerb:
				case restoreVerb:
				case undoVerb:
					break;
				default:
					"Giants don't have such desires. ";
					exit;
			}
	}
	doAttack( actor ) =
	{
		"The Giant looks at %you% with doleful eyes. <q>Me not so bad!</q> ";
	}
	verDoPush( actor ) = {}
	doPush( actor ) =
	{
		if ( actor != self )
		{
			switchPlayer(giant, true);
			execCommand(giant, lookVerb);
		}
		else
		{
			switchPlayer(Me, nil);
			execCommand(Me, lookVerb);
		}
	}
;

/*------------------------------------------------------------------------------
 * CALDERA
 *----------------------------------------------------------------------------*/

caldera: room
	sdesc = "Caldera"
	ldesc = "An old volcanic crater of mud and ash, from which there is nowhere
		to go."
;

hog: basicMe
	location = caldera
	noun = 'hog' 'warthog'
	sdesc = "warthog"
	adesc = "a <<sdesc>>"
	thedesc = "the <<sdesc>>"
	ldesc = "Muddy, sniffing and grunting."
	actorAction( v, d, p, i ) =
	{
		local p;
		p = parserGetMe();
		if ( parserGetMe() != self ) 
			pass actorAction;
		else
			if ( isclass( v, travelVerb ) ) {}
			else
			switch( v )
			{
				case lookVerb:
				case inspectVerb:
				case eatVerb:
				case smellVerb:
				case tasteVerb:
				case touchVerb:
				case quitVerb:
				case saveVerb:
				case restoreVerb:
				case undoVerb:
					break;
				default:
					"Warthogs cant do anything as tricky as that. ";
					exit;
			}
	}
;

redBerry: fooditem
	location = caldera
	noun = 'berry'
	adjective = 'red' 'magic'
	sdesc = "red berry"
    initial = "A magic red berry is half-uncovered in the mud."
	doEat( actor ) =
	{
		"The blood-smell of the berry brings you back, back...\b";
		switchPlayer(Me, nil);
		parserGetMe().location.lookAround(actor, true, true);
		parserGetMe().location.isseen = true;  
	}
	doSynonym( 'Eat' ) = 'Smell' 'Touch' 'Taste'
;

/*------------------------------------------------------------------------------
 *	WEIGHT ROOM
 *----------------------------------------------------------------------------*/

#define CARRYING_STRENGTH 	500;
#define HEAVINESS_THRESHOLD 100;
 
weightRoom: room
	sdesc = "Weights Room"
	ldesc = "This is an annexe, south of Room 14. In here (though nowhere
		else) objects have given weights, and you can carry only a limited total
		weight. (Items from elsewhere in the museum all weigh 10 zobs,
		regardless of what they are.) The longer you carry heavy objects, the
		more tired you become, until you're forced to drop them; your strength
		recovers once you're carrying only a light load."
	enterRoom( actor ) =
	{
		weightMonitor.activate;
		pass enterRoom;
	}
	leaveRoom( actor ) =
	{
		unnotify( weightMonitor, &monitorDaemon );
	}
	south = room14
;

feather: item
	location = weightRoom
	noun = 'feather'
	weight = 1
	sdesc = "feather"
;

ironAnvil: item
	location = weightRoom
	noun = 'anvil'
	adjective = 'iron'
	weight = 300
	sdesc = "iron anvil"
	adesc = "an <<self.sdesc>>"
;

fiveZobWeight: item
	location = weightRoom
	noun = 'weight'
	adjective = 'five' '5' 'zob'
	weight = 5
	sdesc = "five-zob weight"
;

tenZobWeight: item
	location = weightRoom
	noun = 'weight'
	adjective = 'ten' '10' 'zob' 
	weight = 10
	sdesc = "ten-zob weight"
;

twentyZobWeight: item
	location = weightRoom
	noun = 'weight'
	adjective = 'twenty' '20' 'zob'
	weight = 20
	sdesc = "twenty-zob weight"
;

fiftyZobWeight: item
	location = weightRoom
	noun = 'weight'
	adjective = 'fifty' '50' 'zob'
	weight = 50
	sdesc = "fifty-zob weight"
;

hundredZobWeight: item
	location = weightRoom
	noun = 'weight'
	adjective = 'hundred' '100' 'zob'
	weight = 100
	sdesc = "hundred-zob weight"
;

weightof : function( obj )
{
	local c, o, i, t;
	
   	t = obj.weight;
   
   	c = obj.contents;
   	o = car( c );
   	while( o )
   	{
   		t += weightof( o );
   		c = cdr( c );
   		o = car( c );
   	}
   	return t;
}
;

weightMonitor: decoration
	location = weightRoom
	playerStrength = 0
	warningLevel = 5
	sdesc = "weight monitor"
	activate =
	{
		self.playerStrength = CARRYING_STRENGTH;
		notify( self, &monitorDaemon, 0 );
	}
	monitorDaemon =
	{
		local c, cs, w, s, b, bw;
		
		s = self.playerStrength - weightof( parserGetMe() ) +
			HEAVINESS_THRESHOLD;
		if ( s < 0 ) 
			s = 0;
		cs = CARRYING_STRENGTH;
		if ( s > cs )
			s = CARRYING_STRENGTH;
		
		self.playerStrength = s;
		
		if ( s == 0 )
		{
			bw = -1;
			c = parserGetMe().contents;
			b = car( c );
			while( b )
			{
				if ( weightof( b ) > bw )
				{
					bw = weightof( b ); 
					w = b;
				}
				self.playerStrength += bw;
				c = cdr( c );
				b = car( c );
			}
			"\bExhausted with carrying so much, you decide to discard
			<<w.thedesc>>: "; w.moveInto( parserGetMe().location );
		}
			
		w = s / 100;
		if ( w == self.warningLevel ) return;
			
		self.warningLevel = w;
		switch( w )
		{   
            case 3: 
            	"\bYou are feeling a little tired. ";
            	break;
            case 2: 
                "\bYou possessions are weighing you down. ";
                break;
   			case 1: 
                "\bCarrying so much weight is wearing you out. ";
                break;
        	case 0: 
                "\bYou're nearly exhausted enough to drop everything
             	at an inconvenient moment. ";
             	break;
		}
	}
;

/*------------------------------------------------------------------------------
 *	ROOM 15 (MANUSCRIPTS ROOM)
 *----------------------------------------------------------------------------*/

room15: room
	sdesc = "Manuscripts Room"
	ldesc = "This is Room 15, adjoining a corridor to the <<global.d00>>."
	east = corridor12
;

/*
 *	class consultItem: readable
 *	
 *  Defines an object that is a non-Actor in which the player can 
 *  CONSULT or LOOK UP topics of interest, such as a book or 
 *  encyclopedia.
 *
 *	The consultItem is simply a readable with special handling for 
 *	CONSULT and LOOK UP verbs. Processing should be overridden in 
 *	the consultWord() and consultObj() methods using switch() logic.
 */
class consultItem: readable
	isconsultItem = true
	wordlist = []
	verDoConsult( actor ) = {}
	/*
	 *	The player has entered CONSULT BOOK but hasn't given a 
	 *  subject. Here we present a generic message.
	 */
	doConsult( actor ) =
	{
		"%You% search%es% through <<self.thedesc>>, but find%s% 
		nothing of interest.";
	}
	verDoConsultAbout( actor ) = {}
	doConsultAbout( actor, iobj ) =
	{
		if ( iobj != self )
		{
			/*
			 *	An object has been identified by the parser. We 
			 *	call consultObj() which should switch( obj ) through
			 *	various responses.
			 */
			self.consultObj( actor, iobj );
		}
		else if ( length( self.wordlist ) )
		{
			/*
			 *	The dobj == iobj and we have a wordlist. This means
			 *	that the command has gone through parseUnknownXobj().
			 *	We clear out the wordlist each time in order to avoid
			 *	instances of self-referencing.
			 */
			local w = [];
			w = self.wordlist;
			self.wordlist = [];
			self.consultWord( actor, w );
		}
		else
		{
			/*
			 *	We have a case of self-referencing. The player has 
			 *	asked the 'book' to consult itself.
			 *	In this case we produce the generic message.
		 	 */			
			"%You% search%es% through <<self.thedesc>>, but find%s%
			no entry on this. ";
		}
	}
	verIoConsultAbout( actor, dobj ) = {}
	ioConsultAbout( actor, dobj ) =
	{
		dobj.doConsultAbout( actor, self );
	}
	doSynonym( 'ConsultAbout' ) = 'ConsultOn'
	ioSynonym( 'ConsultAbout' ) = 'ConsultOn'
	verDoLookupIn( actor, dobj ) = {}
	verIoLookupIn( actor ) = {}
	ioLookupIn( actor, dobj ) =
	{
		self.doConsultAbout( actor, dobj );
	}
	/*
	 * 	This method is called when the dobj != iobj, which means that 
	 *  we are referencing an object that exists in the game. The 
	 *  method is passed the actor and the object. Switch( obj ) can 
	 *  then be used in the method override to handle each object 
	 *  separately. Here we present a generic message.
	 */
	consultObj( actor, obj ) =
	{
		"%You% consult%s% <<self.thedesc>> about <q><<obj.sdesc>>,</q> but 
		find%s% nothing of interest. ";	
	}
	/*
	 *	This method is called when the dobj == iobj, which means that 
	 *  we are referencing the wordlist created from one of the 
	 *  parseUnknownXobj().
	 *
	 *	The method is passed the actor and the wordlist, which can be 
	 *  handled in the method override with switch( w[ i ] ) 
	 *  processing for each required word in the list. Here we present 
	 *  a generic message.
	 */
	consultWord( actor, w ) =
	{
		local i;
		
		"%You% consult%s% <<self.thedesc>> about <q>";

		for (i = 1 ; i <= length( w ) ; ++i)
		{
			if (i != 1)
   				" ";
			say( w[ i ] );
		}
		"<\q>, but find%s% nothing of interest. ";	
	}
;

/*
 *	bible: consultItem
 *
 *	This object demonstrates one use of consultWord(); using a one- or 
 *	two-word list and formatting a display string. More elaborate editing
 *	would be required to make this fool-proof, but it mimics the Tyndale
 *	Bible found in Museum.t.
 *
 *	The consultObj() method is overriden only to send a general message to the
 *	player if they try to reference anything but 'matthew', 'mark', 'luke' or
 *	'john'.
 */
bible: consultItem
	location = room15
	noun = 'bible' 'book'
	adjective = 'black' 'tyndale'
	sdesc = "Tyndale Bible"
	initial = "A black Bible rests on a spread-eagle lectern."
	ldesc = "A splendid foot-high Bible, which must have survived 
    	the burnings of 1520."
	consultObj( actor, obj ) =
	{
		"There are only the four Gospels. ";
	}
	consultWord( actor, wlst ) =
	{
		local x, w;
		w = wlst[ 1 ];
		
		switch( w )
		{
			case 'matthew':
				x = 'Gospel of St. Matthew';
				break;
			case 'mark':
				x = 'Gospel of St. Mark';
				break;
			case 'luke':
				x = 'Gospel of St. Luke';
				break;
			case 'john':
				x = 'Gospel of St. John';
				break;
			default:
				self.consultObj( nil, nil );
		}
		if ( x )
		{
			if ( length( wlst ) == 1 )
			{
				"You read the "; say( x ); " right through. ";
			}
			else
			{
				local n;
				n = cvtnum( wlst[ 2 ] );
				if ( n && n != true )
				{
					"Chapter "; say( wlst[ 2 ] ); " in the "; say( x ); " is too
					sacred for you to understand now. ";
				}
				else
				{
					"I was expecting a chapter number in "; say( x ); ". ";
				}
			}
		}
	}
;

/*------------------------------------------------------------------------------
 *	BALCONY
 *----------------------------------------------------------------------------*/

#define SUNRISE 360
#define SUNSET 1140

class outside: room
    daystate = 2
;

balcony: outside
	sdesc = "Balcony"
  	ldesc = 
  	{
  	    "An open-air balcony on the cliffside wall of the 
		cathedral-like Museum, hundreds of feet above 
		<<self.daystate == 1 ? "sunlit plains of farms and settlements." :
        "a darkling plain. Dim light from the walls above is 
        the only beacon in this vast night">>.";
  	}
    roomEndCommand = 
    {
        local f, t = timesys.time;
            
        if (t >= SUNRISE
        && t < SUNSET)
            f = 1;

        if (self.daystate == f) return;      
        if (self.daystate == 2) {self.daystate = f; return;}
        self.daystate = f;
        if (f == 1) 
        {
            "\bThe sun rises, illuminating the landscape! ";
            return;
        }
        else
        {
            "\bAs the sun sets, the landscape is plunged into
            darkness. ";
            return;
        }
    }
	in = foyer
   	noexit = "The only way is back inside. "
;

giantClock: distantItem
	location = balcony
	noun = 'clock'
	adjective = 'giant'
	sdesc = "giant clock"
	ldesc = 
	{
		"According to the giant clockface on the wall above, it is
		now "; say(gettimesys(TS_TIME, timesys.time, nil)); ". ";
	}
	scopePostAction( a, v, d, p, io, r ) =
	{
        if ( v == lookVerb )
        {
            "\b";
            self.ldesc;
        }
	}
;

moth: item
	location = balcony
	noun = 'moth'
	sdesc = "moth"
    initial = "A catchable moth flits about."
	scopeEndCommand =
	{
		if ( isclass( parserGetMe().location, darkroom ) 
		&& self.isIn( parserGetMe() ) 
		&& !self.islit)
		{
			self.moveInto( nil );
			"\bAs your eyes try to adjust, you feel a ticklish 
			sensation and hear a tiny fluttering sound. ";
		}
	}
;

class crewmember: Actor;

commanderData: crewmember
	location = balcony
	noun = 'data'
	adjective = 'commander'
	sdesc = "Commander Data"
	adesc = { self.sdesc; }
	thedesc = { self.sdesc; }
	actorAction( v, d, p, i ) =
	{
		"<q>I regret that I am only here to make an example work elsewhere.
		Please proceed upstairs to 10 Forward to see.</q>";
	}
;

/*=============================================================================
 *	SECOND FLOOR
 *============================================================================*/

secondFloor: room
	sdesc = "Mezzanine"
	ldesc = "Halway up the marble staircase of the building, a second floor
		mezzanine which leads south into Grammar Hall (an extension purpose-
		built for the Room 16 exhibition)."
	south = grammarHall
	up = thirdFloor
	down = foyer
;

map2: fixeditem, wysiwyg
	location = secondFloor
  	noun = 'map' 'sign'
  	adjective = 'mezzaniine' 'level'
	sdesc = "map"
    initial = "A map of the mezzanine level is embossed in one wall."
  	ldesc = 
  	{
  		"A map of the mezzanine level is embossed in one wall ";
  		pass ldesc;
  	}
  	headerText =
  	[
  		'MUSEUM MAP'
  		'SECOND FLOOR'
  	]
  	bodyText =
  	[
  		'  Mezzanine'
  		'(you are here)'
  		'     |'
  		'     |'
  		'Grammar Hall 16'
  		'(telepathy, clock)'
  		'(Charlotte, Dan) ------ Ten Forward'
  		'     |                  (replicator)'
  		' Liberator               (computer)'
  		'  Bridge                (tricorder)'
  		'  (zen)'
	]
;

leaflet: readable
	location = secondFloor
	noun = 'leaflet' 'guide'
	adjective = 'sixteen' '16' 'exhibition' 'room'
	sdesc = "room 16 exhibition leaflet"
    initial = "An exhibition leaflet has fallen to the floor."
    ldesc = "Among the rare & fascinating exhibits in Room 16 are
    	Telekinetic (and telepathic) Martha. In the telepathy
    	booth, you can speak to her as though she were in Room 16
    	with you -- but she's not, she is far away. Tell her to
    	<q>look</q> to find out where, and you can also ask her to give
    	you things (by telekinesis).\b
    	Charlotte, who is playing Simon Says. She'll obey
    	instructions you give as long as you preface it with
    	<q>simon says</q>. (Though she only knows how to wave, or how to
    	clap many times (just tell her a number).)\b
    	Dyslexic Dan knows how to take and drop things and how to
    	perform an inventory, but unfortunately confuses Take and
    	Drop orders.\b
    	The alarm clock can be told <q>on</q>, <q>off</q> or a time of day
    	(to set its alarm to)."
;

/*------------------------------------------------------------------------------
 *	GRAMMAR HALL
 *----------------------------------------------------------------------------*/
 
grammarHall: room
 	sdesc = "Grammar Hall"
 	ldesc = "The main exhibit on the second floor: Room 16, south of the 
 		mezzanine. A Jeffreys Tube runs <<global.d00>> into a Star Trek: The Next
 		Generation room, while a hexagonal corridor to the south leads to the
 		bridge of the spaceship <q>Liberator</q>."
 	north = secondFloor
 	south = blakesSeven
 	east = starTrek
;

telepathyBooth: container
    isenterable = true
 	location = grammarHall
 	noun = 'booth'
 	adjective = 'telepathy'
 	sdesc = "telepathy booth"
    initial = "A telepathy booth stands invitingly open: 
           <q>Talk To Telekinetic Martha</q>."
;
 
/*
 *	This preparse object gets the timestring from commands such as "clock, 9:30"
 */
clock_ppo: preparseObj
    ppomethod( cmd ) =
    {
        local ret, grp;       
        cmd = lower(cmd);
                
        ret = parsetime( cmd );
        if (ret)
        {
            local tmpcmd = '';
             
            alarmClock.time = ret[3];           
            /* Restructuring the command. */
            ret = car(ret);            
            if (ret[1] - 1)
            {
                tmpcmd += substr(cmd, 1, ret[1]-1) + ' active. ';
            }                
            if (length(cmd) - ret[1] - ret[2] + 1)
        	{	
             	local tmpcmd2;
             	
             	tmpcmd2 = substr(cmd, ret[1] + ret[2], 
                   length(cmd) - ret[1] - ret[2] + 1);

        		tmpcmd += tmpcmd2;
        	}
            return tmpcmd;
        }
        return true;
    }
;

alarmClock: item
	location = grammarHall
	noun = 'clock'
	adjective = 'alarm'
	time = 0
	isActive = nil
	sdesc = "alarm clock"
	adesc = "an alarm clock"
	ldesc =
	{
		local ret = parsetime( cvtstr(self.time) );
		"The alarm is <<self.isActive ? "on, " : "off, but">> the clock reads
		<<say(gettimesys(TS_TIME, timesys.time, nil))>> and the alarm is set for 
		<<say(gettimesys(TS_TIME, self.time, nil))>>. ";
	}
	grammar( tokenlist, typelist, errnum ) =
	{
		local ret = self.process( tokenlist, typelist, errnum );
		if ( ret ) return ret;
		pass grammar;
	}
	process( tokenlist, typelist, errnum ) =
	{
		if (tokenlist[1] == 'on' || tokenlist[1] == 'active')
			self.doTurnon(self);
		else if (tokenlist[1] == 'off')
			self.doTurnoff(self);
		return true;
	}
    actorAction(v, d, p, i) =
    {
		if (not (v == turnOnVerb || v == turnOffVerb))
		{
           "[Try <q>clock, on</q> or <q>clock, off</q> or 
           <q>clock, 9:30 am</q> to address the clock.]";
           exit;
		}
    }
	dobjCheck( a, v, i, p ) =
	{
		if (not (v == turnOnVerb || v == turnOffVerb))
		{
           "[Try <q>clock, something</q> to address the clock.]";
           exit;
		}		
	}
	verDoTurnoff( actor ) = {}
	doTurnoff( actor ) =
	{
		self.isActive = nil;
		unnotify(self, &daemon);
		"<q>Alarm cancelled.</q> ";
	}
	verDoTurnon( actor ) = {}
	doTurnon( actor ) =
	{
		if (self.isActive)
			unnotify(self, &daemon);
		self.isActive = true;
		notify(self, &daemon, 0);
		"<q>Alarm set.</q> ";
	}
	scopePostAction( actor, verb, dobj, prep, iobj, ret ) =
	{
		switch( verb )
		{
			case iVerb:
				if (actor.isCarrying(self))
					self.ldesc;
				break;
			case lookVerb:
				if (self.isIn(actor.location))
					self.ldesc;
				break;
		}
	}
	daemon =
	{
		if (timesys.time > self.time 
		and timesys.time <= self.time + 3
		and self.isActive)
			"\bBeep! Beep! The alarm goes off. ";
	}
;
  
/*----------------------------------------------------------------------------
 *	BLAKES SEVEN
 *--------------------------------------------------------------------------*/
 
blakesSeven: room
 	sdesc = "Spaceship Liberator: Bridge"
 	ldesc = "The great serried bridge of the alien spaceship Liberator,
 		captured by Blake's group of revolutionary criminals on the
 		run from the oppressive Earth Federation. (Don't worry -- 
 		it's just a convincing replicak, and the museum still lies 
 		back north.)\bThere is an informative plaque here."
 	north = grammarHall
;
 
informativePlaque: fixeditem, wysiwyg
 	location = blakesSeven
 	noun = 'plaque' 'sign'
 	adjective = 'informative'
 	sdesc = "informative plaque"
 	bodyText =
 	[
 		'[Among the commands which Zen was often given by the hard-pressed'
 		'Liberator crew were:'
 		'     zen, scan 360 orbital'
 		'     zen, speed standard by six'
 		'     zen, clear the neutron blasters for firing'
 		'     zen, raise the force wall'
 		'     zen, set course for centauro]'
 	]
;
 
zen: Actor
	location = blakesSeven
	noun = 'zen'
	dobjwordlist = []
	iobjwordlist = []
	sdesc = "zen"
	actorDesc = "Square lights flicker unpredictably across a hexagonal fascia 
           on one wall, indicating that the flight computer Zen is on-line."
	actorAction( verb, dobj, prep, iobj ) =
	{
		switch( verb )
		{
			case setVerb:
				"<q>Confirmed.</q> ";
				if ( prep == forPrep )
				{
					"The ship turns to a new bearing. ";
				}
				caps(); self.phrase( dobjwordlist );
				 " "; verb.sdesc; " "; prep.sdesc; " "; 
				self.phrase( iobjwordlist );
				 ", Captain.";
				 exit;
			case scanVerb:
				"<q>Confirmed.</q> ";
				"The main screen shows a starfield turning 
				through "; say( self.getnum( dobjwordlist ) ); 
				" degrees. ";
				exit;
			case clearVerb: 
				break;
			case raiseVerb:
				if ( dobj != forceWall )
				{
					"<q>Please clarify.</q> ";
					exit;
				}
				break;
			case speedVerb:
				if ( prep == byPrep )
				{
					local n;
					if ( length( self.iobjwordlist ) != 0 )
					{
						n = self.getnum( iobjwordlist );
					}
					else
					{
						n = numObj.value;
					}
					if ( n > 12 )
					{
						"<q>Standard by <<n>> exceeds design tolerances.</q> ";
					}
					else if ( n > 0 )
					{
						"<q>Confirmed.</q> The ship's engines step to standard
						by <<n>>. ";
					}
					else
					{
						"<q>Confirmed.</q> The ship comes to a stop. ";
					}
				}
				exit;
			default:
				"<q>Information. That function is unavailable.</q> ";
				exit;
		}
	}
	verIoClearFor( actor ) = {}
	verDoClearFor( actor, iobj ) = {}
	/*
	 *	We could have as easily handled this in actorAction, but I prefer to use
	 *	the io method.
	 */
	ioClearFor( actor, dobj ) =
	{
		if ( dobj == blasters )
		{
			"<q>Battle-computers on line. 
			Neutron blasters cleared for ";
			self.phrase( iobjwordlist );
			".</q> ";
		}
		else
		{
			"<q>Please clarify.</q> ";
		}
	}
	phrase( wlst ) =
	{
		local c, s;
		c = car( wlst );
		while( c )
		{
			" "; say( c );
			wlst = cdr( wlst );
			c = car( wlst );
		}
	}
	getnum( wlst ) =
	{
		local c, n;
		c = car( wlst );
		while( c )
		{
			c = cvtnum( c );
			if ( datatype( c ) == 1 && c != 0 )
				return c;
			wlst = cdr( wlst );
			c = car( wlst );
		}
		return 0;
	}
;

blasters: fixeditem
	location = blakesSeven
	noun = 'blasters'
	adjective = 'neutron'
	sdesc = "blasters"
	verDoClearFor( actor, iobj ) = {}
;

forceWall: fixeditem
	location = blakesSeven
	noun = 'wall' 'shields'
	adjective = 'force'
	sdesc = "force wall"
	verDoRaise( actor ) = {}
	doRaise( actor ) =
	{
		"<q>Force wall raised.</q> ";
	}
;

/*-----------------------------------------------------------------------------
 *	STAR TREK (10 Forward)
 *---------------------------------------------------------------------------*/
  
picard: crewmember
	noun = 'picard'
	adjective = 'captain'
	sdesc = "Captain Picard"
	adesc = { self.sdesc; }
	thedesc = { self.sdesc; }
;

starTrek: room
	sdesc = "10 Forward"
	ldesc = "The starswept observation lounge forward of deck 10
		of the Starship Enterprise, where many milkshakes have
		been consumed in crisis situations. (It's only a replica,
		of course, and the museum lies back <<global.d01>>.)"
  	west = grammarHall
;

stPlaque: fixeditem, wysiwyg
	location = starTrek
	noun = 'plaque' 'sign'
	adjective = 'informative'
	sdesc = "informative plaque"
	bodyText = 
	[
		'[The computer, called just <q>computer</q>, is'
		'useful for locating crew-members: try <q>computer,'
		'where is commander data</q>, or <q>...captain picard</q>,'
		'or any of the other people who can be found in' 
		'the Museum.'
		' '
		'The tricorder will scan something if you tell'
		'it the name of whom or what you wish to scan.'
		' '
		'The replicator is a superior drinks machine:'
		'for instance, Captain Picard is fond of'
		'saying <q>replicator, tea earl grey</q>.'
		' '
		'There\'s also brandy and distilled water.]'
	]
;

talkable: movableActor
	validActor = { return( self.isVisible( parserGetMe() ) ); }
;

computer: talkable
	location = starTrek
	noun = 'computer'
	sdesc = "computer"
	actorDesc = "The computer, of course, always responds to your voice here."
	actorAction( verb, dobj, prep, iobj ) =
	{
		if ( verb == whereisVerb )
		{
			if ( isclass( dobj, crewmember ) )
			{
				if ( dobj.location == nil )
				{
					"<q>, <<dobj.sdesc>> is no longer aboard this demonstration
					game.</q> ";
					exit;
				}
				else
				{
					"<q>, <<dobj.sdesc>> is in <<dobj.location.sdesc>>.</q> ";
					exit;
				}
			}
			else
				"The computer's only really good for locating the crew. ";
				exit;
		}
		else
		{
			"The computer is too simple.";
			exit;
		}
	}
;

tricorder: talkable
	location = starTrek
	noun = 'tricorder'
	sdesc = "tricorder"
	actorAction( verb, dobj, prep, iobj ) =
	{
		switch( verb )
		{
			case askVerb:
			case showVerb:
			case tellVerb:
				"The tricorder is too simple. ";
				exit;
		}
	}
	verDoPointAt( actor, iobj ) = 
	{
		if ( ! actor.isCarrying( self ) )
			"But %you're% not holding the tricorder! ";
	}
	doPointAt( actor, iobj ) = 
	{
		if ( iobj == parserGetMe() )
		{
			"<q>You radiate life signs.</q> ";
		}
		else if ( isclass( iobj, movableActor ) 
			and ! isclass( iobj, talkable ) )
		{
			caps(); "<<iobj.thedesc>> radiates life signs.</q> ";
		}
		else
		{
			caps(); "<<iobj.thedesc>> radiates no life signs.</q> ";
		}
	}
;

/*
 *	foodReplicator: Actor
 *
 *	The replicator demonstrates the use of alternate grammars using
 *	parseUnknownVerb() in conjunction with parserDictLookup() and 
 *	parserResolveObjects().
 *
 */
foodReplicator: Actor
	location = starTrek
	noun = 'replicator'
	sdesc = "replicator"
	actorDesc = "A replicator (i.e. Star Trek drinks machine occupies a niche in 
		one wall."	
	ldesc = 
	{
		caps(); "<<self.thedesc>> is capable of replicating anything that it is
		familiar with.\b";
		chute.ldesc;
	}
	grammar( tokenlist, typelist, errnum ) =
	{
		local ret = self.process( tokenlist, typelist, errnum );
		if ( ret ) return ret;
		pass grammar;
	}
	process( tokenlist, typelist, errnum ) =
	{
		local i, tlist, objlist = [], resultlist = [], obj;

		/* build valid typelists */
		tlist = self.validsyntax( typelist );

		/* check if we have a valid noun-phrase */
		if ( length( tlist ) == 0 ) 
			return self.errorSyntax;
	
		/* build object list for each typelist */
		for ( i = 1; i <= length( tlist ); i++ )
			objlist += parserDictLookup( tokenlist, tlist[i] );

		/* check if we have any objects */
		if ( length( objlist ) == 0 ) 
			return self.errorLookup;

		/* 
		 *  run each object's verDoReplicate() silently, building an
		 *  objectlist of valid objects
		 */
		resultlist = self.verifyObj( objlist );

		/* formulate error messages based on resultlist */
		if ( length( resultlist ) == 0 )
			if ( length( objlist ) == 1 )
			{
				obj = car( objlist );
				obj.verDoReplicate( self ); 
				return true;
			}
			else
				return self.errorVerifyObjNil( tokenlist );
		else if ( length( resultlist ) == 1 )
			obj = car( resultlist );
		else
		{
		    /*
		     *  Disambiguate multiple objects
		     */
		    resultlist = self.buildlist( length(tokenlist), 
                resultlist );
		    objlist = self.parserResolve( tokenlist, resultlist );

		    if ( objlist[1] == PRS_SUCCESS )
		        obj = car( cdr( objlist ) );
		    else if (objlist[1] == PRSERR_DISAMBIG_RETRY)
            {
                /* run the new command, which is in the second 
                element */
                parserReplaceCommand(objlist[2]);
            }
            else
            {
                /* we were in non-silent mode, so the resolver displayed an error */
                return nil;
            }
		}

		/* run the object's doReplicate() */
		obj.doReplicate( self );

		/* tell actor grammar we've handled this */
		return true;
	}
	/*
	 *	validsyntax( typelist )
	 *	
	 *	Method constructs 'valid' lists of typelists for use by
	 *	parserDictLookup(). The valid lists are composed of one
	 *	or more of the following:
	 *		a. one-word adjective
	 *		b. one-word noun 
	 *	or
	 *		a. multi-word of the format [ a <a> ... n ]
	 *		b. multi-word of the format [ n a <a> ... ]
	 *
	 *	Method returns a list composed of one or two lists if
	 *	valid; otherwise it returns an empty list.
	 */
	validsyntax( typelist ) =
	{
		local i, a = 0, n = length( typelist );
		local newlist = [], templist = [];

		if ( n == 1 )
		{
			if ( typelist[1] & PRSTYP_NOUN )
				templist += [PRSTYP_NOUN];
			newlist += [templist];
			templist = [];
			if ( typelist[1] & PRSTYP_ADJ )
				templist += [PRSTYP_ADJ];
			newlist += [templist];
			templist = [];
		}
		else 
		{
			if ( typelist[1] & PRSTYP_ADJ )
			{
				for ( i = 1; i <= n - 1; i++ )
					if ( typelist[i] & PRSTYP_ADJ ) a++;
				if ( a != n - 1 )
					return nil;
				if ((typelist[n] & PRSTYP_NOUN ) == 0 )
					return nil;
				for ( i = 1; i <= n - 1; i++ )
					templist += PRSTYP_ADJ;
				templist += PRSTYP_NOUN;
			}
			newlist += [templist];
			templist = [];
			if ( typelist[1] & PRSTYP_NOUN )
			{
				for ( i = 2; i <= n; i++ )
					if ( typelist[i] & PRSTYP_ADJ ) a++;
				if ( a != n - 1 )
				return nil;
				templist += PRSTYP_NOUN;
				for ( i = 2; i <= n; i ++ )
					templist += PRSTYP_ADJ;
			}
			newlist += [templist];
		}
 		return newlist;
	}
	/*
	 *	verifyObj( objlist )
	 *	
	 *	Method executes verDoReplicate() silently for each object, 
	 *	building a new object list. verDoReplicate() is used mainly
	 *	for the one error message that should never display -- a 
	 *	check to determine if obj.isanalyzed is true.
	 *
	 *	Method returns a list of all objects that pass the 
	 *	verDoReplicate().
	 */
	verifyObj( objlist ) =
	{
		local i, obj, stat, str, newlist = [];

		obj = car( objlist );
		while( obj )
		{
			stat = outcapture( true );
			obj.verDoReplicate( self );
			str = outcapture( stat );
			if ( length( str ) == 0 )
				newlist += obj;

			objlist = cdr( objlist );
			obj = car( objlist );
		}
		return newlist;	
	}
	buildlist( len, objlist ) =
	{
	    local i, newlist = [ 1 len ];
	    
	    for ( i = 1; i <= length( objlist ); i++)
	    {
	       newlist += objlist[i];
	       newlist += 0;
	    }
	    newlist = [len++] + [newlist];
	    return newlist;
	}
	parserResolve( tokenlist, objlist ) =
	{
	    local resultlist;
	    
	    resultlist = parserResolveObjects(self, 
	        replicateVerb, nil, nil, PRO_RESOLVE_DOBJ, 
	        &verDoReplicate, tokenlist, objlist, nil);

        return resultlist;
	}
	/*
	 *	errorSyntax
	 *
	 *	Method displays an appropriate message when the typelist
	 *	analysis indicates that we do not have a valid typelist.
	 *
	 *	Method returns true, we have handled the error.
	 */
	errorSyntax =
	{
		"\b<<self.thedesc>> replies, <q>I do not understand the
		request.</q> ";
		return true;
	}
	/*
 	 *	errorLookup
	 *	
	 *	Method displays an appropriate message when the dictionary
	 *	lookup didn't return any matching objects.
	 *
	 *	Method returns true, we have handled the error.
	 */
	errorLookup =
	{
		"\b<<self.thedesc>> replies, <q>There is no reference to this
		item in my database.</q> ";
		return true;
	}
	/*
	 *	errorVerifyObjNil( tokenlist )
	 *
	 *	Method displays an appropriate message when all objects 
	 *	from the dictionary lookup have failed the verification
	 *	method.
	 *
	 *	Method returns true, we have handled the error.
	 */
	errorVerifyObjNil( tokenlist ) =
	{
		"\b<<self.thedesc>> reports, <q>There are no entries in my
		database matching your input. Please place a sample of the 
		requested object on the analyzer. ";
		return true;
	}
	/*
	 *	errorVerifyObjMult
	 *
	 *	Method displays an appropriate message when more than one
	 *	object is returned from verifyObj(). 
	 *
	 *	Method returns true, we have handled the error.
	 */
	errorVerifyObjMult =
	{
		"\b<<self.thedesc>> replies, <q>Could you be more 
		specific? ";
		return true;
	}
	actorAction( verb, dobj, prep, iobj ) =
	{
		switch( verb )
		{
			case replicateVerb:
				break;
			case giveVerb:
				"The replicator serves up a cup of <<dobj.sdesc>> which you
				drink eagerly. ";
				exit;
			case askVerb:
			case tellVerb:
				"The replicator has no conversation skill. ";
				break;
			default:
				"The replicator is unable to oblige. You must give it the name
				of a drink it knows about. ";
		}
	}
;

/*
 *	analyzer: surface, fixeditem
 *
 *	This performs the important task of moving an object into nil and providing
 *	several very useful attributes.
 */
analyzer: fixeditem, surface
	location = foodReplicator
	noun = 'analyzer'
	sdesc = "analyzer"
	ioPutOn( actor, dobj ) =
	{
		if ( dobj.isclone )
			"The replicator chirps, <q>The molecular composition of
			this object has already been stored in my database!</q> ";
		else
		{
			dobj.isEquivalent = true;
			dobj.isanalyzed = true;
			dobj.moveInto( nil );
			"The analyzer glows briefly with an intense light and 			
			the object disappears. <q>Molecular composition has now 			
			been memorized for future requests!</q> the replicator 			
			chirps. ";
		}
	}
;

/*
 *	chute: container, fixeditem
 *
 *	This is simply a handy place to put the dynamically-created objects when
 *	they are created.
 */
chute: fixeditem, container
	location = starTrek
	noun = 'chute'
	sdesc = "chute"
;

class copyDrink: fooditem
	isEquivalent = true
	isanalyzed = true
	doEat( actor ) = {fooditem.doDrink( actor );}
	doDrink( actor ) = {fooditem.doEat( actor );}
;

tea: copyDrink
	noun = 'tea'
	adjective = 'earl' 'grey' 'hot'
	sdesc = "Earl Grey tea"
;

brandy: copyDrink
	noun = 'brandy'
	adjective = 'aldebaran'
	sdesc = "Aldebaran brandy"
;

water: copyDrink
	noun = 'water'
	adjective = 'distilled'
	sdesc = "distilled water"
;

/*=============================================================================
 *	SEALED ROOM
 *============================================================================*/

sealedRoom: room
	sdesc = "Sealed Room"
	ldesc = "<q>I'm in a sealed room, like a squash court without a door, maybe six
		or seven yards across.</q> "
    reachInto = {
        if (parserGetMe().isIn(telepathyBooth))
            return grammarHall;
    }
;

fish: item
	location = sealedRoom
	noun = 'fish'
	sdesc = "fish"
;
	
key: keyItem
	location = sealedRoom
	noun = 'key'
	adjective = 'cadmium'
	sdesc = "key"
	doDrop( actor ) =
	{
		self.moveInto( nil );
	}
	doPostAction( a, v, d, p, i, s ) =
	{
	    if (v == dropVerb)
		    "The key smashes into smithereens! ";
	}	    
;

martha: crewmember
	location = sealedRoom
	noun = 'martha'
	isHer = true
	sdesc = "Martha"
	adesc = { self.sdesc; }
	thedesc = { self.sdesc; }
	actorAction( verb, dobj, prep, iobj ) =
	{
		switch( verb )
		{	
			case inspectVerb:
				if ( dobj == sealedRoom )
				{
					self.location.ldesc;
					if ( itemcnt( self.location.contents ) )
					{
						"I can see <<listcont( self.location )>>. ";
					}
					else
					{
						"There's nothing here but me. ";
					}
					exit;
				}
			case lookVerb:
			case dropVerb:
			case takeVerb:
				break;
			case giveVerb:
			    if (iobj == nil)
			    {
			        "<q>I'm afraid you're breaking up... you want me to
			        give <<dobj.thedesc>> to what?</q>";
			        exitobj;
			    }
			    else if (!dobj.isIn(self.location))
			    {
			        "<q>That's beyond my telekinesis.</q>";
			        exitobj;
			    }
			    else if (dobj == self)
			    {
			        "<q>Teleportation's too hard for me.</q>";
			        exitobj;
			    }
				else if (!self.isCarrying(dobj))
				{	
					local ret;
					"(first taking <<dobj.thedesc>>.)\n";
					ret = execCommand( self, takeVerb, dobj, 
						nil, nil, EC_HIDE_SUCCESS );
				    if (ret != EC_SUCCESS || !dobj.isIn(martha)) 
				        exitobj;
				}
                "<q>Here goes...</q> and Martha's telekinetic 
                talents magically bring <<dobj.thedesc>> to 
                your hands.";
                dobj.moveInto(iobj);
				exitobj;
			case askVerb:
				"<q>You're on your own this time.</q> ";
				exit;
			case tellVerb:
				"Martha clucks sympathetically. ";
				exit;
			default:
				"<q>Afraid I can't help you there.</q> ";
				exit;
		}
	}
	validActor = {
	    if (parserGetMe().isIn(telepathyBooth))
	        return true;
	    else
	        pass validActor;
	}
;

/*============================================================================
 *	THIRD FLOOR
 *===========================================================================*/

thirdFloor: room
  	sdesc = "Third Floor"
  	ldesc = "Atop the marble staircase, this third floor 
           foyer fans out into Rooms from Chapter V. These run:
           \n\t\tsouth to Corridor 22 (inventories) 
           and the List Property Office,
           \n\t\t<<global.d00>> to Room 24 (Curious Names),
           \n\t\t<<global.d01>> to Room 28, which is divided by a glass window.
           \b\t\tOnly a very rickety ladder goes further up, through an 
           open trapdoor to the roof."
  	south = corridor22
//  east = room24
//  west = windowW
//  up = roof
  	down = secondFloor
;

map3: fixeditem, wysiwyg
	location = thirdFloor
  	noun = 'map' 'sign'
  	adjective = 'mezzaniine' 'level'
  	sdesc = "map"
    initial = "A map of the third floor is embossed in one wall."
  	ldesc = 
  	{
  		"A map of the mezzanine level is embossed in one wall ";
  		pass ldesc;
  	}
  	headerText =
  	[
  		'MUSEUM MAP'
  		'THIRD FLOOR'
  	]
  	bodyText =
  	[
  		'(glass'
  		'window)'
  		'Room 28 --------- Stairs --------- Room 24'
  		'(you are here)     Names               |'
  		'                     |                 |'
  		'                     |            Museum Cafe'
  		'                Corridor 22'
  		'               (Inventories)'
  		'              /      |        \\'
  		'             /       |         \\'
  		'     Room 23 W -- Room 23 ---- Room 23 E'
  		'     (Listing   List Property  (I Ching)'
  		'     machine)      Office'
	]
;

lexicon: readable
    location = thirdFloor
    noun = 'lexicon'
    sdesc = "lexicon"
    initial = "On the top step is a discarded Lexicon of verbs."
    ldesc = "(The lexicon has the number 27 written on the spine.)\b
        Inside are many familiar verbs that you regularly use
        when navigating through TADS games. There are also
        some special ones, only available in the Museum...\b
        <q>megalook</q> : like <q>look</q> but much fuller;\n
        <q>threefold <noun> <noun> <noun></q> : this doesn't
        actually do anything, but it shows how a verb can
        be parsed which has three objects, not just 0 to 2
        as usual;\n
        <q>settime <time of day></q> : sets the current time of day
        (go out onto the Balcony to see sunrise at 6 AM);\n
        <q>fp <floating-point number></q> : shows how floating-point
        numbers can be parsed;\n
        <q>dial <phone-style number></q> : likewise phone numbers.\b
        Also, in the Museum, <q>lock</q> and <q>unlock</q> are cunningly
        able to presume keys as their second objects."
;

/*----------------------------------------------------------------------
 *  CORRIDOR 22
 *--------------------------------------------------------------------*/
 
corridor22: room
    sdesc = "Corridor 22"
    ldesc = "Runs south of the top floor of the foyer, and leads to Room
        23, such a long room that you can enter it to the southwest,
        south or southeast.\bIn the floor is a curious solid quartz
        window."
    north = thirdFloor
    south = room23
    se = room23e
    sw = room23w
;

quartzWindow: fixeditem, seethruItem
    location = corridor22
    noun = 'window'
    adjective = 'quartz'
    sdesc = "quartz window"
    ldesc = { self.thrudesc; }
    thrudesc =
    {
        "Through the window you can hazily make out the contents of the
        Grammar Hall below...\n";
    }
;

geoffreysBook: readable
    location = corridor22
    noun = 'book'
    adjective = 'old' 'harmless' 'geoffrey\''
    sdesc = "Geoffrey's Book"
    invent = "that harmless old book of Geoffrey's"
    doInspect( actor ) =
    {
        self.invent = "that lethal old book of Geoffrey's";
        delword(self,&adjective,'harmless');
        addword(self,&adjective,'lethal');
        "The apparently harmless book turns fiery hot in your hands ,
        and your eyes are dragged toward the hideous sigils inscribed
        within it... Just in time, you break the gorgon gaze and look
        away\b[From this moment, its inventory entry changes.] ";
    }
;

platinumPyramid: item
    location = corridor22
    noun = 'pyramid'
    adjective = 'platinum'
    sdesc = "platinum pyramid"
    describe = {
        "\bThe platinum pyramid catches the light beautifully. ";
        return true;
    }
    ldesc = "No matter how many times you pick this up and put it 
		down again, it never becomes an ~ordinary~ object for room 
  		description purposes -- it always has a line to itself. "
;

ornateBox: openable
	location = corridor22
	noun = 'box'
	adjective = 'ornate'
	sdesc = "ornate box"
	ldesc = "[What's interesting about the ornate box is that its
		inventory listing overrides its short name entirely,
		and yet its contents will still be inventoried.]"
;

pearl: decoration
	location = ornateBox
	sdesc = "pearl"
	ldesc = "Don't look at me -- I'm irrelevant."
;

/*------------------------------------------------------------------------------
 *	ROOM 23 WEST
 *----------------------------------------------------------------------------*/

room23w: room
	sdesc = "Room 23 <<global.d01>>: Inventory Annexe"
	ldesc = "The <<global.d01>>ern end of Room 23 is a housing for the Listing Machine."
	east = room23
	ne = corridor22
;

room23: room
	sdesc = "Room 23: List Property Office"
	ldesc = "This the List Property Office (it seems that an ~O~ 
        was lost some time ago), at the southern end of the top floor; 
        the management have kindly provided a green baize table, while 
        the rest is presumably the work of patrons.\b
        Room 23 continues <<global.d00>> to <<global.d01>>.",
	north = corridor22
	east = room23e
	west = room23w
;

class star: item
    isEquivalent = true
    noun = 'star'
    ldesc = "A little star of precious-metal, perhaps exotic currency."
    list_together = 'stars'
    plural = 'stars'
    pluraldesc = "stars"
;
class goldStar: star
    adjective = 'gold'
    sdesc = "gold star"
    pluraldesc = "gold stars"
;
class silverStar: star
    adjective = 'silver'
    sdesc = "silver star"
    pluraldesc = "silver stars"
;
class bronzeStar: star
    adjective = 'bronze'
    sdesc = "bronze star"
    pluraldesc = "bronze stars"
;

s1: silverStar
    location = room23
;
s2: silverStar
    location  = room23
;
s3: silverStar
    location = room23
;
s4: silverStar
    location = room23
;
b1: bronzeStar
    location = room23
;
g1: goldStar
    location = room23
;
g2: goldStar
    location = room23
;
g3: goldStar
    location = room23
;

room23e: room
	sdesc = "Room 23 <<global.d00>>"
	nw = corridor22
	west = room23
;

/*=============================================================================
 *	GRAMMAR
 *============================================================================*/

/*	
 *	zenverb: deepverb
 *
 *	This is the heart of the demonstration. zenverbs use the parseUnknownXobj()
 *	methods, creating a corresponding wordlist and returning 'zen' as either 
 *	the dobj or the iobj. In this way it is possible to give zen a completely
 *	bogus direct and indirect object, yet process the verb (i.e. >Zen, scan 360
 *	orbital.) without defining game objects.
 */
class zenverb: deepverb
   	validDoList(actor, prep, iobj) = (nil)
    validDo(actor, obj, seqno) = { return (seqno == 1); }
	parseUnknownDobj(actor, prep, iobj, words) =
	{
		if ( actor == zen )
		{
			/* use our special Zen object for the unknown words */
			actor.dobjwordlist = words;
			return actor;
		}
        else
        {
        	/* 
       		 *   it's not Zen, return nil to use the
         	 *   default system handling 
             */
       		return nil;
    	}
	}
	parseUnknownIobj(actor, prep, dobj, words) =
	{
		if ( actor == zen )
		{
			/* use our special Zen object for the unknown words */
			actor.iobjwordlist = words;
			return actor;
		}
        else
        {
        	/* 
       		 *   it's not Zen, return nil to use the
         	 *   default system handling 
             */
       		return nil;
    	}
	}
;

setVerb: zenverb
	sdesc = "set"
	verb = 'set'
	defaultPrep = forPrep
	ioAction( forPrep ) = 'SetFor'
	ioAction( toPrep ) = 'SetTo'
;



scanVerb: zenverb
	sdesc = "scan"
	verb = 'scan'
	doAction = 'Scan'
;

clearVerb: zenverb
	sdesc = "clear"
	verb = 'clear'
	defaultPrep = forPrep
	ioAction( forPrep ) = 'ClearFor'
;

raiseVerb: zenverb
	sdesc = "raise"
	verb = 'raise'
	doAction = 'Raise'
;

speedVerb: zenverb
	sdesc = "speed"
	verb = 'speed'
	defaultPrep = byPrep
	ioAction( byPrep ) = 'SpeedBy'
;

pointVerb: deepverb
	sdesc = "point"
	verb = 'point'
	defaultPrep = atPrep
	ioAction( atPrep ) = 'PointAt'
;

/*
 *	replicateVerb: deepverb
 *
 *	The replicate verb conflicts uses validDoList() and validDo()
 *  to allow selection of the objects, which would ordinarily not
 *  be accessible. Because we are using the parser, disambiguation
 *  follows normal rules (i.e. "replicate ball" will select 'red
 *  ball' over 'ball bearing' when ambiguous.
 */
replicateVerb: deepverb
	sdesc = "replicate"
	verb = 'replicate'
	doAction = 'Replicate'
	/* logical for verb */
    validDoList(actor, prep, iobj) = (nil)
    /* valid for verb */
    validDo(actor, obj, seqno) = (true)
;

whereisVerb: deepverb
	sdesc = "where is"
	verb = 'where is'
	doAction = 'Whereis'
	/*
	 *	The following statements allow us to find players that are not visible.
	 */
    validDo(actor, obj, seqno) = { return (seqno == 1); }
    validDoList(actor, prep, iobj) = (nil)
;

byPrep: Prep
	preposition = 'by'
	sdesc = "by"
;

forPrep: Prep
	preposition = 'for'
	sdesc = "for"
;

isPrep: Prep
	preposition = 'is'
	sdesc = "is"
;

/*
 *	consultVerb: deepverb
 *	
 *	Allows for CONSULT dobj ABOUT iobj and CONSULT dobj ON iobj 
 *  statements. Because TADS normally disambiguates the indirect 
 *  object first parseUnknownIobj() wouldn't know what the direct 
 *  object is and would require something like the unknownAskIobj 
 *  cited in TADS Version Release 2.2.4. Disambiguating the dobj 
 *  first allows us to make the direct object the same as the 
 *  indirect object. 
 */
consultVerb: deepverb
	sdesc = "consult"
	verb = 'consult'
	doAction = 'Consult'
	ioAction( aboutPrep ) = [disambigDobjFirst] 'ConsultAbout'
	ioAction( onPrep ) = [disambigDobjFirst] 'ConsultOn'
	validIo( actor, obj, seqno ) = { return (seqno == 1); }
    validIoList( actor, prep, dobj ) = (nil)
	/*
 	 *   For "consult," use special handling for unknown words so
 	 *   that the book can respond directly to the unknown words. 
 	 */
	parseUnknownIobj(actor, prep, dobj, words) =
	{
		/* 
		 *  if we're consulting about something, have the consultItem 
		 *  respond 
		 */
		if ( prep == aboutPrep || prep == onPrep )
		{
			/* use our special CONSULT object for the unknown words */
			dobj.wordlist = words;
			return dobj;
		}
        else
        {
        	/* 
       		 *   it's not CONSULT ON, return nil to use the
         	 *   default system handling 
             */
       		return nil;
    	}
	}
;
/*
 *	lookupVerb: deepverb
 *	
 *	Allows for LOOK UP iobj IN dobj statements. Since we are 
 *  disambiguating the direct object we know the indirect object and 
 *  can use it as our direct object, eliminating the need for two 
 *  objects to handle a look up command.
 */
lookupVerb: deepverb
	sdesc = "look up"
	verb = 'look up'
	defaultPrep = inPrep
	ioAction( inPrep ) = 'LookupIn'
	validDo( actor, obj, seqno ) = { return (seqno == 1); }
	parseUnknownDobj( actor, prep, iobj, words ) =
	{
		/* 
		 *  if we're LOOKing UP something, have the consultItem 
		 *  respond 
		 */
		if (prep == inPrep)
		{
			/* use our special CONSULT object for the unknown words */
			iobj.wordlist = words;
			return iobj;
		}
        else
        {
        	/* 
       		 *   it's not LOOKUP IN, return nil to use the
         	 *   default system handling 
             */
       		return nil;
        }
	}
;

/*=============================================================================
 *	INFORM.T MODIFICATIONS
 *============================================================================*/
 
modify thing
	verIoPointAt( actor ) = {}
	ioPointAt( actor, dobj ) = { dobj.doPointAt( actor, self ); }
	/*
 	 *	Modification of thing to allow the replicate verb
 	 */
	verDoReplicate( actor ) =
	{
		if ( actor != foodReplicator )
			"<<actor.thedesc>> is incapable of replicating 
			anything. ";

		else if ( ! self.isanalyzed )
	    {
			caps(); "<<actor.thedesc>> does not know the molecular
			composition of <<self.adesc>>, please place a specimen
			on the analyzer. ";
	    }
		/* Note, this message should never be seen */
		else if ( self.isclone )
			"You cannot make a copy of a copy. ";
	}
	doReplicate( actor ) =
	{
		local x = new self;
		x.isEquivalent = true;
		x.isclone = true;
		x.parentobj = self;
		x.moveInto( chute );
		caps(); "<<actor.thedesc>> replicates <<self.adesc>>, which slowly 
		coalesces inside the chute attached to the machine. ";
		if ( length( x.contents ) != 0 )
			self.replicateCont( x, x.contents );
	}
	replicateCont( obj, oldcont ) =
	{
		local i, x;
		
		obj.contents = [];
		for ( i = 1; i <= length( oldcont ); i++ )
		{
			local o = oldcont[i];
			if ( o.isclone )
				x = new o.parentobj;
			else
				x = new o;
			x.isEquivalent = true;
			x.isclone = true;
			x.parentobj = o;
			x.moveInto( obj );
		}
		
		"<q>Composite replication is now complete.</q> ";
	}
;

modify containment
    dispBeginSdesc = "<B>"
    dispEndSdesc = "</B>"
;

modify basicMe
    ioGiveTo(actor, dobj) =
    {
        "\^<<self.fmtYou>> accept<<self.fmtS>> <<dobj.thedesc>> 
        from <<actor.thedesc>>.";
        dobj.moveInto(parserGetMe());
    }
;

modify travelVerb
    travelDir(actor) = {
        local aPtr = self.dirAction;
        local dPtr = self.dir;
        local ret;
        
        if (global.eastwestReversed)
        {
            switch(self)
            {
                case eVerb:
                    aPtr = wVerb.dirAction;
                    dPtr = wVerb.dir;
                    break;
                case wVerb:
                    aPtr = eVerb.dirAction;
                    dPtr = eVerb.dir;
                    break;
            }
        }
        
        ret = actor.location.(aPtr)(actor);
        "\n";
        if (ret == true) return nil;
        return actor.location.(dPtr); 
    }
;
wakeupVerb: deepverb
	sdesc = "wake"
	verb = 'wake' 'wake up'
	doAction = 'Wakeup'
;

modify story
    starting_location = foyer
    initialise_daemon = {
	    settimesys( '5:00 am', 3, '2000001' );
    }
    title = "MUSEUM OF TADS"
    headline = "An Interactive
	    Translation of the Musem of Inform\nAn Interactive
	    Companion to the Inform Designer's Manual (Third Edition)\nOne of 
	    the standard Inform 6 example games\nRelease
	    2 / Serial number 961111 / Inform v6.15 Library 6/7 D\bTranslated 
	    to TADS by Kevin Forchione\n
        Using Inform.t v4.0\b"
    introduction = "\b\b\bWelcome indeed to the..."
    release = 2
    serial = '991214'
;

modify Me
	maxbulk = 5
	maxweight = 500
;

modify global
    eastwestReversed = nil
    d00 = {
        if (eastwestReversed)
            "west";
        else
            "east";
    }
    d01 = {
        if (eastwestReversed)
            "east";
        else
            "west";
    }
;
