#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <errno.h>

/* define USE_ARTS 1 to use aRts sound server instead of esd */
#define USE_ARTS 0

#if USE_ARTS
#include <artsc/artsc.h>
#else
#include "esd.h"
#endif

static unsigned char* mem = NULL;
static long length = 0;
static int repeat = 1;
static int volume = 65536;

#define BUFSZ 2048
static unsigned char buffer[BUFSZ];




/*** comaand processing ***/

/* command buffer */

#define CMDSZ 256
static char cmdbuf[CMDSZ];
static int cmdpos = 0;

/*
 * process commands piped from Glk
 */
static void handle_command() {
  int num;
  for(;;) {
    int c = fgetc(stdin);
    if (c<0) {
      if (feof(stdin)) {
	/*fprintf(stderr,"HALT\n");*/
	exit(0);
      };
      if (length) return;
      continue;
    }
    if ((c!='\n') && (c!='\0')) {
      cmdbuf[cmdpos++] = c;
      if (cmdpos>=CMDSZ) cmdpos = 0;
      continue;
    }
    cmdbuf[cmdpos] = '\0';
    cmdpos = 0;

    /*fprintf(stderr,"%s\n",cmdbuf);*/

    if (!strncmp(cmdbuf,"READ ",5) && (sscanf(cmdbuf+5,"%d",&num)==1)) {
      if (num>0) {
	length = num;
	return;
      }
    }
    else if (!strncmp(cmdbuf,"VOL ",4) && (sscanf(cmdbuf+4,"%d",&num)==1)) {
      if ((num>=0) && (num<=0x10000)) volume = num;
    }
    else if (!strncmp(cmdbuf,"REP ",4) && (sscanf(cmdbuf+4,"%d",&num)==1)) {
      repeat = num;
    }
  }
}


/* variables for resampling by linear interpolation */
#define INTERPREZ 16384
static int pos = 0;
static int fracpos = 0;
static int sbytes, step;
static int sampval[4];

/* lowpass filter */
static int filtsz;
static int filtx = 0;
static int* filtval;

/* AIFF COMM chunk fileds */
static unsigned short nchan = 0;
static unsigned int nframe = 0;
static unsigned short sampsize = 0;
static int aiffrate = 0;

/* location of SSND data */
static unsigned int ssndbeg = 0, ssndend = 0;

/* default rate */
#if USE_ARTS
static int outrate = 44100;
#else
static int outrate = ESD_DEFAULT_RATE;
#endif

static unsigned short get_ui16() {
  unsigned int x = mem[pos++]<<8;
  x += mem[pos++];
  return x;
}

static unsigned int get_ui32() {
  unsigned int x = mem[pos++]<<8;
  x += mem[pos++];
  x <<= 8;
  x += mem[pos++];
  x <<= 8;
  x += mem[pos++];
  return x;
}

/*
  For the purpose of sampling rate we need not carry out a full IEEE-754
  conversion. Only find the integer part and discard the rest
*/
static int get_extended() {
  int exponent, bexp, i, mant = 0;

  bexp = ((mem[pos] & 0x7f)<<8) + mem[pos+1];
  pos += 2;
  if ((bexp & 0x4000) == 0) {
    fprintf(stderr,"error in rate. Attempting fix\n");
    bexp |= 0x4000;
  }
  exponent = bexp - 0x3ffe;
  mant = 0;
  for( i = 0; i<8; i++ ) {
    if (exponent>=8) {
      mant <<= 8;
      mant += mem[pos++];
      exponent -= 8;
    }
    else if (exponent>0) {
      mant <<= exponent;
      mant |= (mem[pos++]>>(8-exponent));
      exponent = 0;
    }
    else
      pos++;
  }
  return mant;
}

static int process_chunk() {
  unsigned int id, clen;
  id = get_ui32();
  clen = get_ui32();
  if (pos+clen>length)
    return -1;

  if (id == 0x434f4d4d) {
    nchan = get_ui16();
    nframe = get_ui32();
    sampsize = get_ui16();
    aiffrate = get_extended();
    /*fprintf(stderr,"%d %d %d %d\n",nchan, nframe, sampsize, aiffrate);*/
  }
  else if (id == 0x53534e44) {
    int offset = get_ui32();
    pos += 4; /* skip block size */
    ssndbeg = pos + offset;
    ssndend = pos - 8 + clen;
    /* ensure chunk size is even */
    pos += clen - 8;
  }
  else
    pos += clen;
  return clen + 8;
}

static void parse_aiff() {
  int aifflen;

  unsigned int id;
  int tot;
  id = get_ui32();
  if (id != 0x464f524d) {
    fprintf(stderr, "FORM missing\n");
    exit(-1);
  }
  tot = get_ui32();
  id = get_ui32();
  if (id != 0x41494646) {
    fprintf(stderr, "AIFF missing\n");
    exit(-1);
  }
  tot -= 4;
  /*fprintf(stderr,"%d bytes\n",tot+12);*/
  while (tot>=8 && length>pos+8) {
    int clen = process_chunk();
    if (clen<0) break;
    tot -= clen;
  }
  pos = ssndbeg;
  sbytes = (sampsize-1)/8+1;
  step = nchan*sbytes;
}

/*
 resample SSND data to output frequency converting to 16-bit 2-channel
 Do not rely on ESD to resample or it'll sound like crap; ARTS does a
 better job at this */
static int fill_buffer() {
  int bx, i, samp;
  for ( bx = 0; bx <= BUFSZ-4; bx += 4) {
    /* next frame */
    fracpos += aiffrate;
    while (fracpos>outrate) {
      fracpos -= outrate;
      pos += step;
      /* convert to 16-bit 2 channel */

      for( i = 0; i<2; i++ ) {
	unsigned char *p = mem + pos;
	if (i && nchan>1) p += sbytes;
	samp = p[0];
	if (samp & 0x80) samp-=256;
	samp <<= 8;
	if (sbytes>1)
	  samp |= p[1];
	sampval[i] = sampval[i+2];
	sampval[i+2] = samp;
      }
    }
    if (pos >=ssndend) break;

    filtx ++;
    if (filtx>=filtsz) filtx = 0;

    /* for each channel */
    for( i = 0; i<2; i++) {
      int value, fval;
      int j = fracpos*INTERPREZ/outrate;

      /* linear interpolation */
      value = sampval[i+2]*j + sampval[i]*(INTERPREZ-j);
      value /=INTERPREZ;

      /* adjust volume */
      value = value*volume/65535;

      filtval[filtx+i*filtsz] = value;
      fval = 0;
      for( j = 0; j<filtsz; j++)
	fval += filtval[j+i*filtsz];
      fval /= filtsz;

      buffer[bx+i*2] = fval%256;
      buffer[bx+i*2+1] = fval/256;
    }
  }
  return bx;
}

int main(int argc, char* argv[] ) {
#if USE_ARTS
  arts_stream_t arts;
#else
  esd_format_t format;
  int esdsock;
#endif
  int i;

  handle_command();

  /*** load in sound file from stdin ***/
  mem = (unsigned char*)malloc(length);
  if (!mem) {
    fprintf(stderr,"malloc failed\n");
    exit(-1);
  }
  if (!fread(mem,length,1,stdin)) {
    fprintf(stderr,"fread failed\n");
    exit(-1);
  }

  /* Make stdin non-blocking; will be polled for volume and repetition
     commands */
  if (fcntl(fileno(stdin), F_SETFL, O_NONBLOCK)) {
    fprintf(stderr, "fcntl error\n");
    exit(-1);
  }

  parse_aiff();
  if (!ssndbeg) {
    fprintf( stderr, "SSND missing\n");
    exit(-1);
  }
  if (!nchan) {
    fprintf( stderr, "COMM missing\n");
    exit(-1);
  }

  /*fprintf(stderr,"ssndbeg %d ssndend %d\n",ssndbeg, ssndend);*/

#if USE_ARTS
  if (arts_init()) exit(-1);
  arts = arts_play_stream( outrate, 16, 2, "glkmodplay" );
#else
  /** open socket to esd **/
  format = ESD_BITS16 | ESD_STEREO | ESD_STREAM | ESD_PLAY;

  esdsock = esd_play_stream_fallback( format, outrate, NULL, NULL );
  if ( esdsock <= 0 ) 
    exit(-1);
#endif

  /* initialise interpolation buffer */
  for( i=0; i<4; i++)
    sampval[i] = 0;

  /* initialise filter buffer */
  filtsz = outrate/aiffrate;
  if (filtsz<1) filtsz = 1;

  filtval = (int*)malloc(sizeof(int)*filtsz*2);
  for( i=0; i<filtsz*2; i++)
    filtval[i] = 0;

  while(repeat) {
    int bx;

    handle_command();
    bx = fill_buffer();
#if USE_ARTS
    i = arts_write( arts, buffer, bx );
#else
    i = write( esdsock, buffer, bx );
#endif
    if (i<bx) exit(-1);
    if (bx<BUFSZ-4) {
      if (repeat>0 && repeat!=0xffffffff) repeat--;
      if (repeat) {
	pos = ssndbeg;
	fracpos = 0;
      }
    }
  }
#if USE_ARTS
  arts_close_stream(arts);
  arts_free();  
#else
  esd_close(esdsock);
#endif
  printf("FIN\n");
  exit(0);
}

  

