#include <stdio.h>
#include <mikmod.h>
#include <unistd.h>
#include <fcntl.h>

/* define USE_ARTS 1 to use aRts sound server instead of esd */
#define USE_ARTS 0

static MODULE* mod;

static unsigned char* mem = NULL;
static long pos = 0;
static long length = 0;
static BOOL m_eof = 0;
static int repeat = 1;

static MREADER read_mem;


/****** User defined MikMod Reader methods ******/

static BOOL mem_seek(struct MREADER* m, long offset, int whence) {
  long i;
  if (whence == SEEK_SET) {
    i = offset;
  }
  else if (whence == SEEK_CUR) {
    i = pos + offset;
  }
  else if (whence == SEEK_END) {
    i = length;
  }
  if (i<0 || i>length) return 1;
  pos = i;
  m_eof = 0;
  return 0;
}

long mem_tell(struct MREADER* m) {
  return pos;
}

BOOL mem_read(struct MREADER* m, void* buf, size_t count) {
  size_t i = 0;
  while(count--) {
    if (pos>=length) {
      m_eof = 1;
      return 0;
    }
    ((unsigned char*)buf)[i++] = mem[pos++];
  }
  return 1;
}

int  mem_get(struct MREADER* m) {
  if (pos >= length) {
    m_eof = 1;
    return -1;
  }
  return (int) mem[pos++];
}

BOOL mem_eof(struct MREADER* m) {
  return m_eof;
}

/*** comaand processing ***/

/* command buffer */

#define CMDSZ 256
static char cmdbuf[CMDSZ];
static int cmdpos = 0;

/*
 * process commands piped from Glk
 */
static void handle_command() {
  int num;
  for(;;) {
    int c = fgetc(stdin);
    if (c<0) {
      if (feof(stdin)) exit(0);
      if (length) return;
      continue;
    }
    if ((c!='\n') && (c!='\0')) {
      cmdbuf[cmdpos++] = c;
      if (cmdpos>=CMDSZ) cmdpos = 0;
      continue;
    }
    cmdbuf[cmdpos] = '\0';
    cmdpos = 0;

    /*fprintf(stderr,"%s\n",cmdbuf);*/

    if (!strncmp(cmdbuf,"READ ",5) && (sscanf(cmdbuf+5,"%d",&num)==1)) {
      if (num>0) {
	length = num;
	return;
      }
    }
    else if (!strncmp(cmdbuf,"VOL ",4) && (sscanf(cmdbuf+4,"%d",&num)==1)) {
      if ((num>=0) && (num<=0x10000)) md_volume = 128*num/65536;
    }
    else if (!strncmp(cmdbuf,"REP ",4) && (sscanf(cmdbuf+4,"%d",&num)==1)) {
      repeat = num;
    }
  }
}


int main(int argc, char* argv[] ) {
  char* driveroptions;

  /* make custom MikMod reader class */
  read_mem.Seek = mem_seek;
  read_mem.Tell = mem_tell;
  read_mem.Read = mem_read;
  read_mem.Get = mem_get;
  read_mem.Eof = mem_eof;

  handle_command();

  /*** load in MOD file from stdin ***/
  mem = (unsigned char*)malloc(length);
  if (!mem) {
    fprintf(stderr,"malloc failed\n");
    exit(-1);
  }
  if (!fread(mem,length,1,stdin)) {
    fprintf(stderr,"fread failed\n");
    exit(-1);
  }

  /* Register the loaders we want to use:  */
  MikMod_RegisterAllLoaders();

  /* Register the drivers we want to use: */
  MikMod_RegisterAllDrivers();

  md_pansep=128;

  /* if your sound card and CPU can handle it, set this to 44100 */
  md_mixfreq = 22050;

  md_mode|=DMODE_SOFT_MUSIC;
  md_mode|=DMODE_INTERP;
  md_mode|=DMODE_HQMIXER;
  md_mode|=DMODE_16BITS;
  md_mode|=DMODE_STEREO;
  /*md_device = 0;*/
  md_device = MikMod_DriverFromAlias("esd");
  driveroptions = NULL;
#if USE_ARTS
  driveroptions = (char*)malloc(32*sizeof(char));
  sprintf(driveroptions, "pipe=artscat -r %d", md_mixfreq);
  md_device = MikMod_DriverFromAlias("pipe");
#endif
  if (MikMod_Init(driveroptions)) {
    fprintf(stderr,"MikMod_Init failed\n");
    exit(-1);
  }
  if (MikMod_SetNumVoices(16, 16)) {
    fprintf(stderr,"MikMod_SetNumVoices failed\n");
    exit(-1);
  }

  /* Make stdin non-blocking; will be polled for volume and repetition
       commands */
  if (fcntl(fileno(stdin), F_SETFL, O_NONBLOCK)) {
    fprintf(stderr, "fcntl error\n");
    exit(-1);
  }


  while(repeat) {
    if (MikMod_EnableOutput()) {
      fprintf(stderr,"MikMod_EnableOutput failed\n");
      exit(-1);
    }

    mod = Player_LoadGeneric(&read_mem,128,0);
    Player_Start(mod);

    while (Player_Active() && repeat) {
      handle_command();
      MikMod_Update();
    }
    Player_Stop();
    MikMod_DisableOutput();
    
    if (MikMod_Reset(driveroptions)) {
      fprintf(stderr,"MikMod_Reset failed\n");
      exit(-1);
    }
    if (repeat>0 && repeat!=0xffffffff) repeat--;
  }
  MikMod_Exit();
  printf("FIN\n");
  exit(0);
}
