/* gtinput.c: Key input handling
        for GlkDOS, curses.h/allegro implementation of the Glk API
    Designed by L. Ross Raszewski <lraszewski@justice.loyola.edu>
    
    based upon GlkTerm by Andrew Plotkin <erkyrath@netcom.com>
    http://www.eblong.com/zarf/glk/index.html
*/

#include "gtoption.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cscr.h"
#include "glk.h"
#include "glkdos.h"
#include "gtw_grid.h"
#include "gtw_buf.h"

void gt_start_setup();

char kill_buffer[1024];
int killlen=0;
typedef void (*command_fptr)(window_t *win, glui32);

typedef struct command_struct {
    command_fptr func;
    int arg;
} command_t;
/* The idea is that, depending on what kind of window has focus and
    whether it is set for line or character input, various keys are
    bound to various command_t objects. A command_t contains a function
    to call to handle the key, and an argument. (This allows one
    function to handle several variants of a command -- for example,
    gcmd_buffer_scroll() handles scrolling both up and down.) If the
    argument is -1, the function will be passed the actual key hit.
    (This allows a single function to handle a range of keys.) 
   Key values may be 0 to 255, or any of the special KEY_* values
    defined in curses.h. */

/* Keys which are always meaningful. */

void gcmd_shell(window_t *win, glui32 arg)
{
  char *s=getenv("COMSPEC");
  cscr_suspend();
  if (s) system(s);
  else system("command");
  cscr_resume();
  gcmd_win_refresh(win,0);
}
void gcmd_start_setup(window_t *win, glui32 arg)
{
 gt_start_setup();
 gcmd_win_refresh(win,0);
}
void gcmd_screencap(window_t *win, glui32 arg)
{
 cscr_screencap();
 gcmd_win_refresh(win,0);
}

typedef struct {
 int key;
 char *cmd_name;
 command_t cmd;
 }
 key_command;
static int reveal_codes=0;
void gcmd_reveal_codes(window_t *win, glui32 arg)
{
 reveal_codes=!reveal_codes;
}
static int get_arbitrary_key()
{
 char b[5]; int i=0;
 gli_msgin_getline("Keycode: ",b,4, &i);
 if (i)
  return atoi(b);
 return 32; 
}
void gcmd_buffer_arbitrary_line(window_t *win, glui32 arg)
{
 gcmd_buffer_insert_key(win,get_arbitrary_key());
}
void gcmd_grid_arbitrary_line(window_t *win, glui32 arg)
{
 gcmd_grid_insert_key(win,get_arbitrary_key());
}

void gcmd_buffer_arbitrary_key(window_t *win, glui32 arg)
{
 gcmd_buffer_accept_key(win,get_arbitrary_key());
}
void gcmd_command(window_t *win, glui32 arg);

key_command keys_always[]=
{ { KEY_INSERT, "toggle_overwrite", { gcmd_buffer_toggle_overwrite,0}},
  { 18,     "reveal_codes",     { gcmd_reveal_codes,0}},
  { 26,     "shell",            { gcmd_shell, 0 } },
  { '\t',   "change_focus",     { gcmd_win_change_focus, 0}},
  { '\014', "refresh",          { gcmd_win_refresh, 0 }},
  { 295,    "setup",            { gcmd_start_setup, 0 }},
  { 296,    "screencap",        { gcmd_screencap, 0}},
  { 0, NULL, { NULL,0}}
};
key_command keys_grid[] =
{
  { 297,    "exec_command",     { gcmd_command, 0}},
  { 0, NULL, { NULL,0}}
};
key_command keys_grid_key[] =
{
  {-1, "accept_key", { gcmd_grid_accept_key, -1 }},
  { 0, NULL, { NULL,0}}
};
key_command keys_grid_line[] =
{
 { -1, "insert_key", { gcmd_grid_insert_key, -1 }},
 { 15,  "arbitrary_key",     { gcmd_grid_arbitrary_line,0}},
 { KEY_ENTER, "accept_line",   { gcmd_grid_accept_line, 0 }},
 { 13, "accept_line",   { gcmd_grid_accept_line, 0 }},
 { 2, "move_left", { gcmd_grid_move_cursor, gcmd_Left }},
 { KEY_LEFT, "move_left", { gcmd_grid_move_cursor, gcmd_Left }},
 { 6, "move_right", { gcmd_grid_move_cursor, gcmd_Right }},
 { KEY_RIGHT, "move_right", { gcmd_grid_move_cursor, gcmd_Right }},
 { 1, "begin_line", { gcmd_grid_move_cursor, gcmd_LeftEnd }},
 { KEY_HOME, "begin_line", { gcmd_grid_move_cursor, gcmd_LeftEnd }},
 { 5, "end_line", { gcmd_grid_move_cursor, gcmd_RightEnd }},
 { KEY_END, "end_line", { gcmd_grid_move_cursor, gcmd_RightEnd }},
 { 8, "delete_backward", { gcmd_grid_delete, gcmd_Delete }},
 { 4, "delete_forward", { gcmd_grid_delete, gcmd_DeleteNext }},
 { KEY_DELETE, "delete_forward", { gcmd_grid_delete, gcmd_DeleteNext }},
 { '\177', "delete_forward", { gcmd_grid_delete, gcmd_DeleteNext }},
 { 025, "undo_input", { gcmd_grid_delete, gcmd_KillInput }},
 { 013, "kill_line", { gcmd_grid_delete, gcmd_KillLine }},
 { 0, NULL, { NULL,0}}
};
key_command keys_buffer[]=
{
  { 297,    "exec_command",     { gcmd_command, 0}},
 { CTL_HOME, "scroll_to_top", { gcmd_buffer_scroll, gcmd_UpEnd }},
 { CTL_END,  "scroll_to_bottom", { gcmd_buffer_scroll, gcmd_DownEnd }},
 { CTL_UP,   "scroll_up", { gcmd_buffer_scroll, gcmd_Up }},
 { CTL_DOWN, "scroll_down", { gcmd_buffer_scroll, gcmd_Down } },
 { 031, "page_up", { gcmd_buffer_scroll, gcmd_UpPage }},
 { KEY_PAGEUP, "page_up", { gcmd_buffer_scroll, gcmd_UpPage }},
 { 026, "page_down", { gcmd_buffer_scroll, gcmd_DownPage }},
 { KEY_PAGEDN, "page_down", { gcmd_buffer_scroll, gcmd_DownPage }},
 { 0, NULL, { NULL,0}}
};
key_command keys_paging[]=
{ 
 { -1, "more", { gcmd_buffer_scroll, gcmd_DownPage }},
 { 0, NULL, { NULL, 0}}
};
key_command keys_buffer_key[] = 
{
 { -1, "accept", { gcmd_buffer_accept_key, -1 }},
 { 15,     "arbitrary_key",     { gcmd_buffer_arbitrary_key,0}},
 { 0, NULL, { NULL, 0}}
};
key_command keys_buffer_line[] =
{
 { -1, "insert_key", { gcmd_buffer_insert_key, -1 }},
 { 15,  "arbitrary_key",     { gcmd_buffer_arbitrary_line,0}},
 { KEY_ENTER, "accept_line",   { gcmd_buffer_accept_line, 0 }},
 { 13, "accept_line",   { gcmd_buffer_accept_line, 0 }},
 { KEY_LEFT, "move_left", { gcmd_buffer_move_cursor, gcmd_Left }},
 { 2, "move_left", { gcmd_buffer_move_cursor, gcmd_Left }},
 { KEY_RIGHT, "move_right", { gcmd_buffer_move_cursor, gcmd_Right }},
 { 6, "move_right", { gcmd_buffer_move_cursor, gcmd_Right }},
 { KEY_HOME, "begin_line", { gcmd_buffer_move_cursor, gcmd_LeftEnd }},
 { 1, "begin_line", { gcmd_buffer_move_cursor, gcmd_LeftEnd }},
 { KEY_END, "end_line", { gcmd_buffer_move_cursor, gcmd_RightEnd }},
 { 5, "end_line", { gcmd_buffer_move_cursor, gcmd_RightEnd }},
 { 8, "delete_backward", { gcmd_buffer_delete, gcmd_Delete }},
 { KEY_DELETE, "delete_forward", { gcmd_buffer_delete, gcmd_DeleteNext }},
 { '\177', "delete_forward", { gcmd_buffer_delete, gcmd_DeleteNext }},
 { 4, "delete_forward", { gcmd_buffer_delete, gcmd_DeleteNext }},
 { 025, "undo_input", { gcmd_buffer_delete, gcmd_KillInput }},
 { 013, "kill_line", { gcmd_buffer_delete, gcmd_KillLine }},
 { KEY_UP, "history_prev", { gcmd_buffer_history, gcmd_Up }},
 { 020, "history_prev", { gcmd_buffer_history, gcmd_Up }},
 { KEY_DOWN, "history_next", { gcmd_buffer_history, gcmd_Down }},
 { 016, "history_next", { gcmd_buffer_history, gcmd_Down }},
 { 289, "set_macro_1", { gcmd_buffer_setmacro, 0 }},
 { 290, "set_macro_2", { gcmd_buffer_setmacro, 1 }},
 { 291, "set_macro_3", { gcmd_buffer_setmacro, 2 }},
 { 292, "set_macro_4", { gcmd_buffer_setmacro, 3 }},
 { 269, "macro_1", { gcmd_buffer_domacro, 0 }},
 { 270, "macro_2", { gcmd_buffer_domacro, 1 }},
 { 271, "macro_3", { gcmd_buffer_domacro, 2 }},
 { 272, "macro_4", { gcmd_buffer_domacro, 3 }},
 { 479, "buffer_extract",  { gcmd_buffer_paste,0 }},
 { 24,  "buffer_extract",  { gcmd_buffer_paste,0 }},
 { 467, "buffer_wipe", {gcmd_buffer_cutcopy,1 }},
 { 23, "buffer_wipe", {gcmd_buffer_cutcopy,1 }},
 { 477, "buffer_save", { gcmd_buffer_cutcopy,0 }},
 { 19, "buffer_save", { gcmd_buffer_cutcopy,0 }},
 { 17, "set_mark",  { gcmd_buffer_setmark,0 }},
 { 0, NULL, { NULL, 0}}
};

static key_command *key_commands[] = { keys_always, keys_grid, keys_grid_key,
   keys_grid_line, keys_buffer, keys_paging, keys_buffer_key, keys_buffer_line
};
void gcmd_command(window_t *win, glui32 arg)
{
  char buf[64];
  int l[3];
  int i=0,j;
  l[0]=0;
  l[1]=0;
  l[2]=0;
  switch(win->type)
  {
   case wintype_TextGrid:
        l[1]=1;
        if (win->line_request)
         l[2]=3;
        else if (win->char_request)
         l[2]=2;
        break; 
   case wintype_TextBuffer:
        l[1]=4;
        if (win->line_request)
         l[2]=7;
        else if (win->char_request)
         l[2]=6;
        break;
  }
  
  gli_msgin_getline("Execute Command: ",buf,64, &i);
  for(j=0;j<3;j++)
   for(i=0;key_commands[l[j]][i].key!=0;i++)
    if (strcmp(key_commands[l[j]][i].cmd_name,buf)==0)
     {
      key_commands[l[j]][i].cmd.func(win,key_commands[l[j]][i].cmd.arg);
      return;
     }
   gli_msgline_tick("No matching command");
}

static command_t *commands_always(int key)
{
    int i;
    for(i=0;key_commands[0][i].key!=0;i++)
     if (key_commands[0][i].key==key) {
                                if (reveal_codes) gli_msgline_tick(key_commands[0][i].cmd_name);
                                return &key_commands[0][i].cmd;
                                }
    return NULL;
}

/* Keys which are always meaningful in a text grid window. */
static command_t *commands_textgrid(int key)
{
    int i;
    for(i=0;key_commands[1][i].key!=0;i++)
     if (key_commands[1][i].key==key) return &key_commands[1][i].cmd;
    return NULL;
}
/* Keys for char input in a text grid window. */
static command_t *commands_textgrid_char(int key)
{
    return &key_commands[2][0].cmd;
}

/* Keys for line input in a text grid window. */
static command_t *commands_textgrid_line(int key)
{
    int i;
    if (key >= 32 && key < 256 && key != 127) 
        return &key_commands[3][0].cmd;
    for(i=1;key_commands[3][i].key!=0;i++)
     if (key_commands[3][i].key==key) {
                                     if (reveal_codes) gli_msgline_tick(key_commands[3][i].cmd_name);
                                     return &key_commands[3][i].cmd;
                                        }
    return NULL;
}
   

/* Keys which are always meaningful in a text buffer window. Note that
    these override character input, which means you can never type ctrl-Y
    or ctrl-V in a textbuffer, even though you can in a textgrid. The Glk
    API doesn't make this distinction. Damn. */
static command_t *commands_textbuffer(int key)
{
    int i;
    for(i=0;key_commands[4][i].key!=0;i++)
     if (key_commands[4][i].key==key) {
                        if (reveal_codes) gli_msgline_tick(key_commands[4][i].cmd_name);
                        return &key_commands[4][i].cmd;
                        }
    return NULL;
}

/* Keys for "hit any key to page" mode. */
static command_t *commands_textbuffer_paging(int key)
{
  return &key_commands[5][0].cmd;
}

/* Keys for char input in a text buffer window. */
static command_t *commands_textbuffer_char(int key)
{
  return &key_commands[6][0].cmd;
}

/* Keys for line input in a text buffer window. */
static command_t *commands_textbuffer_line(int key)
{
    int i;
    if (key >= 32 && key < 256 && key != '\177')
        return &key_commands[7][0].cmd;
    for(i=1;key_commands[7][i].key!=0;i++)
     if (key_commands[7][i].key==key) {
                                if (reveal_codes) gli_msgline_tick(key_commands[7][i].cmd_name);
                                return &key_commands[7][i].cmd;
                                }
    return NULL;
}


/* Check to see if key is bound to anything in the given window.
    First check for char or line input bindings, then general
    bindings. */
void gcmd_hyperlink(window_t *win, glui32 arg)
{
 win->link_request=FALSE;
 gli_event_store(evtype_Hyperlink,win,arg,0);
}
static command_t *get_link(window_t *win)
{

  static command_t link = { gcmd_hyperlink, 0};
  static command_t cmdrefresh = { gcmd_win_refresh, 0 };
  struct glt_link *c;  
  c=win->str->hlink;
  if (c)
  while(1)
  {
   char buf[80];
   int i;
   sprintf(buf,"%d:%s%s",(int) c->number,c->text,
           c->buflen==LINK_BUFFER_SIZE ? "..." : " ");
   i=gli_msgin_getchar(buf,FALSE);
   link.arg=c->number;
   if (i=='\n' || i=='\r' || i==KEY_ENTER)
    return &link;
   else if (i=='\033' || i=='q' || i=='Q')
    return &cmdrefresh;
   else if (i=='p' || i=='P' || i==KEY_UP)
    if (c->next) c=c->next;
    else c=win->str->hlink;
   else if (c->prev) c=c->prev;
    else while(c->next) c=c->next;

  }

  return &cmdrefresh;
}
void gcmd_get_link(window_t *win, glui32 arg)
{
 command_t *cmd=get_link(win);
 (*cmd->func)(win,cmd->arg);
}

command_t *command_user(window_t *, int, int);
static command_t *commands_window(window_t *win, int key)
{
    command_t *cmd = NULL;
   
    cmd=command_user(win,key,reveal_codes);
    if (!cmd)
    {
    if (win->link_request && key==482)
     cmd = get_link(win);
    else 
    switch (win->type) {
        case wintype_TextGrid:
            cmd = commands_textgrid(key);
            if (!cmd) {
                if (win->line_request)
                    cmd = commands_textgrid_line(key);
                else if (win->char_request)
                    cmd = commands_textgrid_char(key);
            }
            break;
        case wintype_TextBuffer: {
            window_textbuffer_t *dwin = win->data;
            cmd = commands_textbuffer(key);
            if (!cmd) {
                if (dwin->lastseenline < dwin->numlines - dwin->height) {
                    cmd = commands_textbuffer_paging(key);
                }
                if (!cmd) {
                    if (win->line_request)
                        cmd = commands_textbuffer_line(key);
                    else if (win->char_request)
                        cmd = commands_textbuffer_char(key);
                }
            }
            }
            break;
    }
    }
    return cmd;
}

/* Return a string describing a given key. This (sometimes) uses a
    static buffer, which is overwritten with each call. */
static char *key_to_name(int key)
{
    static char kbuf[32];
    
    if (key >= 32 && key < 256) {
        if (key == 127) {
            return "delete";
        }
        kbuf[0] = key;
        kbuf[1] = '\0';
        return kbuf;
    }

    switch (key) {
        case '\t':
            return "tab";
        case '\033':
            return "escape";
        case KEY_DOWN:
            return "down-arrow";
        case KEY_UP:
            return "up-arrow";
        case KEY_LEFT:
            return "left-arrow";
        case KEY_RIGHT:
            return "right-arrow";
        case KEY_HOME:
            return "home";
        case 8:
            return "backspace";
        case KEY_DELETE:
            return "delete-char";
        case KEY_INSERT:
            return "insert-char";
        case KEY_PAGEDN:
            return "page-down";
        case KEY_PAGEUP:
            return "page-up";
        case KEY_ENTER:
            return "enter";
        case KEY_END:
            return "end";
    }

    if (key >= 0 && key < 32) {
        sprintf(kbuf, "ctrl-%c", '@'+key);
        return kbuf;
    }
    sprintf(kbuf, "unknown key (%d)", key);
    return kbuf;
}

/* Handle a keystroke. This is called from glk_select() whenever a
    key is hit. */

void gli_input_handle_key(int key)
{
    command_t *cmd = NULL;
    window_t *win = NULL;
    
    /* First, see if the key has a general binding. */
    if (!cmd) {
        cmd = commands_always(key);
        if (cmd)
            win = NULL;
    }

    /* If not, see if the key is bound in the focus window. */
    if (!cmd && gli_focuswin) {
        cmd = commands_window(gli_focuswin, key);
        if (cmd)
            win = gli_focuswin;
    }
    /* If not, see if there's some other window which has a binding for
        the key; if so, set the focus there. */
    if (!cmd && gli_rootwin) {
        window_t *altwin = gli_focuswin;
        command_t *altcmd = NULL;
        do {
            altwin = gli_window_iterate_treeorder(altwin);
            if (altwin && altwin->type != wintype_Pair) {
                 altcmd = commands_window(altwin, key);
                if (altcmd)
                    break;
            }
        } while (altwin != gli_focuswin);
        if (altwin != gli_focuswin && altcmd) {
            cmd = altcmd;
            win = altwin;
            gli_focuswin = win; /* set the focus */
        }
    }
    
    if (cmd) {
        /* We found a binding. Run it. */
        glui32 arg;
        if (cmd->arg == -1) {
            /* convert from curses.h key codes to Glk, if necessary. */
            switch (key) {
                case '\t': 
                    arg = keycode_Tab;
                    break;
                case '\033':
                    arg = keycode_Escape;
                    break;
                case KEY_DOWN:
                    arg = keycode_Down;
                    break;
                case KEY_UP:
                    arg = keycode_Up;
                    break;
                case KEY_LEFT:
                    arg = keycode_Left;
                    break;
                case KEY_RIGHT:
                    arg = keycode_Right;
                    break;
                case KEY_HOME:
                    arg = keycode_Home;
                    break;
                case '\177': /* delete */
                case '\010': /* backspace */
                case KEY_DELETE:
                    arg = keycode_Delete;
                    break;
                case KEY_PAGEDN:
                    arg = keycode_PageDown;
                    break;
                case KEY_PAGEUP:
                    arg = keycode_PageUp;
                    break;
                case KEY_ENTER:
                case '\015': /* ctrl-M */
                    arg = keycode_Return;
                    break;
                case KEY_END:
                    arg = keycode_End;
                    break;
                default:
                    if (key < 0 || key >= 256) {
                        arg = keycode_Unknown;
                    }
                    else {

#ifdef OPT_NATIVE_LATIN_1
                        arg = key;

#else /* OPT_NATIVE_LATIN_1 */
                        arg = key; 
                        if (!arg && key != '\0')
                            arg = keycode_Unknown;
#endif /* OPT_NATIVE_LATIN_1 */

                    }
                    break;
            }
        }
        else {
            arg = cmd->arg;
        }
        (*cmd->func)(win, arg);
    }
    else {
        char buf[256];
        char *kbuf = key_to_name(key);
        sprintf(buf, "The key <%s> is not currently defined.", kbuf);
        gli_msgline(buf);
    }
}

/* Pick a window which might want input. This is called at the beginning
    of glk_select(). */
void gli_input_guess_focus()
{
    window_t *altwin;
    
    if (gli_focuswin 
        && (gli_focuswin->line_request || gli_focuswin->char_request)) {
        return;
    }
    
    altwin = gli_focuswin;
    do {
        altwin = gli_window_iterate_treeorder(altwin);
        if (altwin 
            && (altwin->line_request || altwin->char_request)) {
            break;
        }
    } while (altwin != gli_focuswin);
    
    if (gli_focuswin != altwin)
        gli_focuswin = altwin;
}
