/*
    DosGlk  --  A Glk implementation for MS-DOS
    Copyright (C) 1998  Matt Kimball

    Permission is hereby granted, free of charge, to any person
    obtaining a copy of this software and associated documentation
    files (the "Software"), to deal in the Software without
    restriction, including without limitation the rights to use, copy,
    modify, merge, publish, distribute, sublicense, and/or sell copies
    of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following condition:
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT.  IN NO EVENT SHALL MATT KIMBALL BE LIABLE FOR ANY
    CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
    CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
    WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <dos.h>

#include "glk.h"
#include "win.h"

static int modal = 0;
static int paged = 0;
                             
static long timer_time = 0;                             
static struct _dostime_t last_timer;
static int lglk_check_timer_event(event_t *event) {
	struct _dostime_t t;
	long difftime;
	                   
	if(timer_time) {
		_dos_gettime(&t);
		                                                   
		/*  This won't work properly if more than one day passes between last_time and
			now.  I can live with that.  */		                                                   	
		difftime = (t.hour - last_timer.hour) * 60 * 60 * 100
						+ (t.minute - last_timer.minute) * 60 * 100 
						+ (t.second - last_timer.second) * 100
						+ (t.hsecond - last_timer.hsecond);
		if(difftime < 0)
			difftime += 24 * 60 * 60 * 100;
			
		if(difftime >= timer_time) {
			last_timer = t;
			event->type = evtype_Timer;
			event->win = 0;
			event->val1 = event->val2 = 0;
			
			return 1; 
		}
		
	}
	
	return 0;
}

long lglk_translate(int c) {
	unsigned char nc;
	
	if(c == 0) {
		nc = getch();
		
		switch(nc) {               
		case 72:
			return keycode_Up;
			
		case 75:
			return keycode_Left;
			
		case 80:
			return keycode_Down;
			
		case 77:		
			return keycode_Right;
			
		case 71:
			return keycode_Home;
		
		case 73:
			return keycode_PageUp;
		
		case 79:
			return keycode_End;
		
		case 81:
			return keycode_PageDown;
			
		case 83:
			return keycode_Delete;
			
		case 59:
		case 60:
		case 61:
		case 62:
		case 63:
		case 64:
		case 65:
		case 66:
		case 67:
		case 68:
			return (long)keycode_Func1 - (long)(nc - 59); 
		
		case 133:
			return keycode_Func11;
		case 134:
			return keycode_Func12;
		
		default:                  
			return 0;
		}	
	}
	              
	if(c == '\r')
		return keycode_Return;
		
	if(c == '\t')
		return keycode_Tab;
		
	if(c == 27)
		return keycode_Escape;
		
	return c;
}

static int lglk_process_char(winid_t id, int inc, event_t *event) {
	win *w; 
	unsigned at;
	long c;
	
	w = lglk_get_window(id);
	c = lglk_translate(inc);
	if(c == 0) 
		return 0;
		
	if(w->last_page < w->out_row) {
		lglk_window_page_win(id);
		lglk_window_sync();
		paged = 1;
		return 0;
	}
	
	if(c == keycode_Tab) {
		if(!modal) {
			lglk_cycle_focus();
			lglk_window_sync();
		}
	
		return 0;
	}
	
	if(w->evrequest == evrequest_Char) {
		w->evrequest = evrequest_None;
		                       
		/*  Backspace is treated as delete for character events  */
		if(c == '\b')
			c = keycode_Delete;
		
		event->type = evtype_CharInput;
		event->win = id;
		event->val1 = c;
		event->val2 = 0;
		
		return 1;
	} else if(w->evrequest == evrequest_Line) {
		if(c == keycode_Return) {
			w->evrequest = evrequest_None;
		
			event->type = evtype_LineInput;
			event->win = id;
			event->val1 = w->linebuf_cur;
			event->val2 = 0;
			
			for(at = 0; at < w->linebuf_cur; at++) {
				lglk_window_putch(id, w->linebuf[at], style_Input);
			}
			lglk_window_putch(id, '\n', style_Input);
			
			return 1;
		} else if(c == '\b') {
			if(w->linebuf_pos) {
				w->linebuf_pos--;
				for(at = (unsigned)w->linebuf_pos; at < w->linebuf_cur; at++)
					w->linebuf[at] = w->linebuf[at + 1];
				w->linebuf_cur--;
			}
			lglk_window_snap_win(id);
		} else if(c == keycode_Delete) {
			if(w->linebuf_cur > w->linebuf_pos) {
				for(at = (unsigned)w->linebuf_pos; at < w->linebuf_cur; at++)
					w->linebuf[at] = w->linebuf[at + 1];
				w->linebuf_cur--;
			}		
			lglk_window_snap_win(id);
		} else if(c == keycode_Home) {
			w->linebuf_pos = 0;
			lglk_window_snap_win(id);
		} else if(c == keycode_End) {
			w->linebuf_pos = w->linebuf_cur;
			lglk_window_snap_win(id);
		} else if(c == keycode_Left) { 
			if(w->linebuf_pos)
				w->linebuf_pos--;
			lglk_window_snap_win(id);
		} else if(c == keycode_Right) {
			if(w->linebuf_pos < w->linebuf_cur)
				w->linebuf_pos++;
			lglk_window_snap_win(id);
		} else if(c == keycode_Down) {
			if(w->top_row < w->out_row) {
				w->top_row++;
				w->need_refresh = 1;
			}
		} else if(c == keycode_Up) {
			if(w->top_row) {
				w->top_row--;
				w->need_refresh = 1;
			}
		} else if(c == keycode_PageDown) {
			w->top_row += w->bounds.b - w->bounds.t + 1;
			if(w->top_row > w->out_row)
				w->top_row = w->out_row;
			w->need_refresh = 1;
		} else if(c == keycode_PageUp) {
			w->top_row -= w->bounds.b - w->bounds.t + 1;
			if((signed)w->top_row < 0) 
				w->top_row = 0;				
			w->need_refresh = 1;
		} else {                         
			if(c >= 32 && c < 256) {
				if(w->linebuf_cur < w->linebuf_max) {
					for(at = (unsigned)w->linebuf_cur; at > w->linebuf_pos; at--) {
						w->linebuf[at] = w->linebuf[at - 1];
					}
					
					w->linebuf[w->linebuf_pos] = (char)c;				
					w->linebuf_pos++;
					w->linebuf_cur++;
				}
			}
			lglk_window_snap_win(id);
		}
	    lglk_window_sync();       
	}
	
	return 0;
}

static int lglk_check_keyboard_event(event_t *event) {
	if(kbhit()) {
		return lglk_process_char(lglk_get_focus(), getch(), event);
	}
	
	return 0;
}

static int lglk_check_mouse_event(event_t *event) {
	return 0;
}

void glk_select(event_t *event) {
	event->type = evtype_None;
	event->win = 0;
	event->val1 = event->val2 = 0;
    
    lglk_window_snap();
	lglk_window_sync();	
	while(event->type == evtype_None) {
		if(lglk_check_timer_event(event))
			return;
		if(lglk_check_keyboard_event(event))
			return;
		if(lglk_check_mouse_event(event))
			return;
	}
}

void lglk_event_getline(char *prompt, char *s, int max) {
	winid_t focus;
	win *w;
	event_t event;
	strid_t oldstream;
	
	int back_request;
	char *back_line;
	glui32 back_max, back_cur, back_pos;
                  
	focus = lglk_get_focus();
	w = lglk_get_window(focus);
	oldstream = glk_stream_get_current();
	glk_stream_set_current(glk_window_get_stream(focus));
                 
	glk_put_string(prompt);
	                                    
	back_request = w->evrequest;
	back_line = w->linebuf;
	back_max = w->linebuf_max;
	back_cur = w->linebuf_cur;
	back_pos = w->linebuf_pos;	                                    
	
	glk_request_line_event(focus, s, max, 0);
	                   
	modal = 1;	                   
	lglk_window_snap();
	lglk_window_sync();
	while(!lglk_check_keyboard_event(&event));
	s[event.val1] = 0;
	modal = 0;

	w->evrequest = back_request;
	w->linebuf = back_line;
	w->linebuf_max = back_max;
	w->linebuf_cur = back_cur;
	w->linebuf_pos = back_pos;	
	
	glk_stream_set_current(oldstream);
}

char lglk_event_getchar(char *prompt) {
	winid_t focus;
	win *w;
	event_t event;
	strid_t oldstream;
	
	int back_request;
                  
	focus = lglk_get_focus();
	w = lglk_get_window(focus);
	oldstream = glk_stream_get_current();
	glk_stream_set_current(glk_window_get_stream(focus));
                 
	if(prompt[0])
		glk_put_string(prompt);
	                                    
	back_request = w->evrequest;
	
	glk_request_char_event(focus);
	          
	modal = 1;
	lglk_window_snap();
	lglk_window_sync();
	while(!lglk_check_keyboard_event(&event));
	modal = 0;

	w->evrequest = back_request;
	glk_stream_set_current(oldstream);
	
	return (char)event.val1;
}

void lglk_event_page() {
	win *w;
	int back_request;
	winid_t focus;
	event_t event;

	focus = lglk_get_focus();
	w = lglk_get_window(focus);

	back_request = w->evrequest;

	glk_request_char_event(focus);

	modal = 1;
	lglk_window_snap();
	lglk_window_sync();
	paged = 0;
	while(!paged) {
		lglk_check_keyboard_event(&event);
	}
	modal = 0;

	w->evrequest = back_request;
}

void glk_select_poll(event_t *event) {
	event->type = evtype_None;
	event->win = 0;
	event->val1 = event->val2 = 0;
                       
    lglk_window_sync();       
	lglk_check_timer_event(event);                              
}

void glk_request_timer_events(glui32 millisecs) {
	timer_time = millisecs / 10;
	_dos_gettime(&last_timer);
}

void glk_request_line_event(winid_t id, void *buf, glui32 maxlen,
    						glui32 initlen) {
    win *w;
            
    w = lglk_get_window(id);
    if(w == NULL)
    	return;
    w->evrequest = evrequest_Line;
    w->linebuf = buf;
    w->linebuf_max = maxlen;
    w->linebuf_cur = initlen;
    w->linebuf_pos = initlen;
}
    						
void glk_request_char_event(winid_t id) {
    win *w;
            
    w = lglk_get_window(id);
    if(w == NULL)
    	return;
    w->evrequest = evrequest_Char;
}

void glk_request_mouse_event(winid_t id) {
	/*  No mouse support yet  */
}

void glk_cancel_line_event(winid_t id, event_t *event) {
	win *w;
	
	w = lglk_get_window(id);
	if(w == NULL)
		return;
		
	event->type = evtype_LineInput;
	event->win = id;
	event->val1 = w->linebuf_cur;
	event->val2 = 0;	      
	
	lglk_window_putch(id, '\n', style_Input);

	w->evrequest = evrequest_None;
}

void glk_cancel_char_event(winid_t id) {
	win *w;
	
	w = lglk_get_window(id);
	if(w == NULL)
		return;
		
	w->evrequest = evrequest_None;
}

void glk_cancel_mouse_event(winid_t id) {
	win *w;
	
	w = lglk_get_window(id);
	if(w == NULL)
		return;
		
	w->evrequest = evrequest_None;
}
