/************************************************************************
*
* Program: GINA System
* Module ginamain.c
* Version 0.1, July 1994.
* Version 0.2, October 1994.
* Version 0.3, February 1995.
* Version 0.4, August 1995.
*
* Copyright 1994, 1995, Jeff Standish.  (jestandi@cs.indiana.edu)
* All rights reserved.
* Permission is hereby granted for unrestricted non-commercial use,
* provided full attribution of source code origin is included.
*
* This module contains the main setup functions and the built-in parser.
*
*************************************************************************
*
* Usage:
* GINAS is designed to be invoked from the command line, with a single
* argument specifying the name of a game file to load.
*
*	ginas [-gm] [-gc #] [gamefile1] [gamefile2] [gamefile3] ...
*
* Additional command line flags (or switches, if you prefer) exist:
*	-gm	This flag turns on the garbage collection messages.
*		(This can also be done by setting *collection-messages*
*		global binding to a true value somewhere in the game file,
*		but this flag has the advantage of immediately turning
*		on the messages, instead of waiting until declared in
*		a LISP file.)  Warning: If *collection-messages* is
*		reset to nil in the game file, this flag gets overriden.
*
*	-gc #	This flag resets the frequency of garbage collection.
*		By default, collection occurs once for every 1000 cons
*		nodes (re)allocated.  Setting this to a larger number will
*		reduce the frequency of garbage collection, if you feel it
*		is eating up too much CPU time.
*
* GINAS can be invoked with no specified game files.  When this happens,
* the system automatically goes into interactive LISP mode.
*
*************************************************************************
*
* Version 0.1:
* Initial construction of world-object interface.
*
* Version 0.2:
* Construction of built-in parser.
*
* Version 0.3:
* Built-in parser rewritten to be recursive, so that it can backtrack
* partial successes to allow for matching all possible combinations of
* objects before giving up with no valid combinations found.
*
* The *no-ginas* flag was redone so that it only prevents the !ginas
* parser command from working if it is defined as being non-nil, rather
* than blocking if it is simply defined (even when nil).  The old way
* permanently blocked access to the interactive interpreter, with no
* way back in after *no-ginas* was defined.
*
* Version 0.4:
* The main game loop was changed so that instead of sequentially parsing
* input, running daemons, and whatnot, the main loop now just invokes
* all daemons, and then takes care of any garbage collection.  It is now
* left to the daemon for the player actor to invoke the parser using the
* (parser...) function, or by writing a new parser in LISP.
*
* Due to the restructuring of the parser, the !ginas command was ripped
* out and moved into the lisp code, leaving it to be inplemented in
* ginalisp.def -- or whatever standard file is used -- and reducing the
* burden of internal declarations.  This also allowed the +player+
* definition to be removed from the internal declarations as well.
*
* The function headers and prototypes were modified to allow for either
* ANSI Standard definitions, or the normal (i.e., the way my compiler
* does it).  If _ANSI_ is defined in ginas.h, then the ANSI Standard
* format is used, otherwise it defaults to "normal".
*
* Additionally, all function prototypes were moved into ginas.h to
* remove the need to keep track of which external functions are used
* and what their prototypes are currently defined to be.
*
* The new parser was further changed to allow for disambiguation of
* multiple object matches.  Instead of just halting when a good object
* combination satisfied the action precondition, all potential object
* combinations are tried, and a log kept of the ones which satisfy
* the precondition.  Then the player/user is prompted for further input
* to disambiguate which object was intended to be referenced.
*
************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include "ginas.h"


	/* externally defined global variables */
extern ACTION *actionlist;
extern IDNODE *nounid, *adjid;
extern IDNODE *initialid, *objnameid;
extern NODEZ  *actorbind, *subjbind, *instrbind, *locatbind;
extern NODEZ  *actorptrbind, *subjptrbind, *instrptrbind, *locatptrbind;
extern NODEZ  *actorquantbind, *subjquantbind, *instrquantbind, *locatquantbind;
extern NODEZ  *locals, *selfbind, *truebind, *garbmesgbind;
extern DAEMON *daemonlist;
extern int    nodetally;
extern OBJECT *objectroot;
extern FILE   *lispfile;
extern int    thischar, nextchar;

extern NODEZ *parserbindlist[];

	/* locally defined global variables */
int killthisdaemon;	/* if set, then kill current daemon when it is done */
int gcsize = 1000;	/* # of nodes allocated before collecting garbage */
int maindone;		/* flag for main loop, if set then exit loop */


	/* some strings for the built-in parser */
#define COMLEN  256
#define MAXWORD 32
char comline[COMLEN], *comwords[MAXWORD];

ACTION *matchedaction;
PARSY  *matchedparsy;



/************************************************************************
*
* main() - Handles displaying the intro message identifying the program,
*	memory initialization, and then parses any command line arguments
*	(normally loading gamefiles).
*
************************************************************************/

#ifdef _ANSI_
int main(int argc, char **argv)
#else
int main(argc, argv)
int  argc;
char **argv;
#endif
{
    int argnum;
    int filesloaded;

	/* intro message */
    printf("GINAS: The Generic Interactive Narrative Authoring System\n");
    printf("Version 0.4\n");
    printf("Copyright 1995, Jeff Standish.  All rights reserved.\n\n\n");

	/* initialize the internal memory structures
	   and see the random number generator */
    initialize_memory();
    set_seed();

	/* parse the command line arguments */
    argnum = 1;
    filesloaded = 0;
    while (argnum < argc) {

	    /* turn on garbarge collection messages */
	if (same_string(argv[argnum], "-gm"))
	    garbmesgbind->value.ptr.cdr = truebind->value.ptr.cdr;

	    /* set frequency of garbage collection by resetting the
	       number of nodes to allocate between collections */
	else if (same_string(argv[argnum], "-gc")) {
	    ++argnum;
	    gcsize = str2int(argv[argnum]);
	}

	   /* load a lisp file */
	else {
	    gina_lisp(argv[argnum], 1);
	    ++filesloaded;
	}

	++argnum;
    }

	/* if no game files were specified on the command line, drop into
	   interactive LISP mode */
    if (filesloaded == 0) {
	printf("No game files specified: dropping into interactive mode.\n");
	gina_lisp(NULL, 1);
    }

	/* otherwise, start up the main game evaluation sequence */
    else
	game_loop();

    return 0;
}



/************************************************************************
*
* game_loop() - Run the user-defined Initialize function, the drop into a
*	loop to evaluate all daemons and perform garbage collection when
*	it is needed.
*
************************************************************************/

#ifdef _ANSI_
void game_loop(void)
#else
game_loop()
#endif
{
    NODEZ *fptr, *head, *ptr;

	/* construct the call to the user-defined function <Initialize> */
    if ((fptr = find_function(initialid)) != NULL) {
	head = new_node();
	head->type = TYPElisthead;
	ptr = new_node();
	ptr->type = TYPEidname;
	ptr->value.idptr = initialid;
	head->value.ptr.car = ptr;
	head = eval_user_func(head, fptr, NULL);
    }

	/* loop through the daemon processing sequence */
    maindone = 0;
    lispfile = stdin;
    thischar = '\n';
    nextchar = '\n';
    while (!maindone) {

	    /* if loop is not done, go and evaluate any active daemons */
	if (!maindone)
	    evaluate_daemons();

	    /* perform garbage collection if enough nodes have been
		allocated since the last time collection was performed */
        if (nodetally >= gcsize)
            collect_garbage(garbmesgbind->value.ptr.cdr != NULL);
    }
}



/************************************************************************
*
* sentence_to_list() - Prompt the user for an input sentence, then
*	parse that sentence into a list of strings, so that each word
*	is a separate string, and that each puntuation mark (except
*	hypens and apostraphies) are separated into individual strings.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *sentence_to_list(void)
#else
NODEZ *sentence_to_list()
#endif
{
    NODEZ *head, *tail, *tmp;
    char  sentence[1024], word[128], *sptr, *wptr;

    head = tail = NULL;

	/* read in the input sentence */
    print_string("\\n");
    printf("> ");
    gets(sentence);

    sptr = sentence;

	/* scan through the sentence, splitting up words into separate
	   strings which are added to the tail of the list of strings
	   which have been thus far accumulated */
    while (*sptr != '\0') {
	wptr = word;

	while (isspace(*sptr))
	    ++sptr;

	if (*sptr == '\0')
	    break;

	if (isalnum(*sptr) || (*sptr == '-') || (*sptr == '\'')) {
	    while (isalnum(*sptr) || (*sptr == '-') || (*sptr == '\'')) {
		*wptr = *sptr;
		++wptr;
		++sptr;
	    }
	}
	else if (*sptr == '"') {
	    *wptr = '\\';
	    ++wptr;
	    *wptr = '"';
	    ++wptr;
	    ++sptr;
	}
	else {
	    *wptr = *sptr;
	    ++wptr;
	    ++sptr;
	}

	*wptr = '\0';

	if (head == NULL)
	    head = tail = new_node();
	else {
	    tail->value.ptr.cdr = new_node();
	    tail = tail->value.ptr.cdr;
	}

	tmp = new_node();
	tmp->type = TYPEstring;
	tmp->value.idptr = find_idstring(word);
	tail->type = TYPElisthead;
	tail->value.ptr.car = tmp;
    }

    return (head);
}



/************************************************************************
*
* parser() - This is the built-in parser.  The current version just sets
*	up a call to matcher(), which does all of the work.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *parser(OBJECT *actorptr, NODEZ *sentence)
#else
NODEZ *parser(actorptr, sentence)
OBJECT *actorptr;
NODEZ  *sentence;
#endif
{
	/* flush the printing buffer, and prompt for a new command */
    new_line();

	/* split up command line into a list of words, and then
	    go and try to match actions against this list */

    return (matcher(actorptr, sentence));
}



/************************************************************************
*
* matcher() - Given an input sentence, try to match it against all
*	defined actions.  If an action is found, match all special
*	wildcard symbols and invoke the action precondition to see
*	which objects are appropriate.  Then, if any ambiguity can be
*	resolved, invoke the action and return its result.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *matcher(OBJECT *actorptr, NODEZ *sentence)
#else
NODEZ *matcher(actorptr, sentence)
OBJECT *actorptr;
NODEZ  *sentence;
#endif
{
    int    patlength, matchedlength, wildcards, matchedwildcards, flag;
    int    count, i, foo;
    ACTION *actionptr;
    NODEZ  *listptr, *sptr, *pattern, *p2ptr, *s2ptr, *p3ptr, *matchedpattern;
    NODEZ  *bindhead, *bindtail, *ptr, *head, *result;
    PARSY  *pptr;

	/* set all of the special parser bindings to nil */
    for (i = 0; i < MAXBIND; ++i)
	parserbindlist[i]->value.ptr.cdr = NULL;

    matchedaction = NULL;
    matchedpattern = NULL;
    matchedlength = -1;
    matchedwildcards = 0;

	/* verify that the given sentence is a valid list of strings */
    sptr = sentence;
    while (sptr != NULL) {
	if (sptr->type != TYPElisthead) {
	    printf("Error: bad sentence list\n");
	    return (NULL);
	}

	s2ptr = sptr->value.ptr.car;
	if ((s2ptr == NULL) || (s2ptr->type != TYPEstring)) {
	    printf("Error: bad sentence list\n");
	    return (NULL);
	}

	sptr = sptr->value.ptr.cdr;
    }

	/* loop through all actions known */
    actionptr = actionlist;
    while (actionptr != NULL) {
	    /* loop through all possible sentence forms for this action */
	listptr = actionptr->words;
	while (listptr != NULL) {
	    if (listptr->type != TYPElisthead) {
		printf("Error: invalid list of word lists: ");
		print_tree(actionptr->words, stdout);
		putchar('\n');
		break;
	    }

	    sptr = sentence;
	    pattern = listptr->value.ptr.car;
	    patlength = 0;
	    wildcards = 0;
	    for (;;) {
		if ((pattern != NULL) && (pattern->type != TYPElisthead)) {
		    printf("Error: invalid sentence pattern: ");
		    print_tree(listptr->value.ptr.car, stdout);
		    printf("\n");
		    break;
		}

		if ((sptr == NULL) && (pattern == NULL)) {
/*
printf("action %s\n", actionptr->idptr->name);
printf("match found: ");
print_tree(listptr->value.ptr.car, stdout);
printf("\nlength = %d, wildcards = %d\n", patlength, wildcards);
*/
		    if ((patlength > matchedlength)
				|| ((patlength == matchedlength)
				    && (wildcards < matchedwildcards))) {
			matchedaction = actionptr;
			matchedpattern = listptr->value.ptr.car;
			matchedlength = patlength;
			matchedwildcards = wildcards;
		    }
		    break;
		}
		else if ((sptr == NULL) || (pattern == NULL))
		    break;
		else {
		    s2ptr = sptr->value.ptr.car;
		    p2ptr = pattern->value.ptr.car;
		    if (p2ptr == NULL) {
			printf("Error: nil value in pattern\n");
			break;
		    }
		    else if (p2ptr->type == TYPEstring) {
			if (same_string(p2ptr->value.idptr->name,
					s2ptr->value.idptr->name)) {
			    pattern = pattern->value.ptr.cdr;
			    sptr = sptr->value.ptr.cdr;
			    ++patlength;
			}
			else
			    break;
		    }
		    else if (p2ptr->type == TYPEidname) {
			++patlength;
			++wildcards;
			p3ptr = pattern->value.ptr.cdr;
			if (p3ptr == NULL) {
			    /*
			    printf("action %s\n", actionptr->idptr->name);
			    printf("match found: ");
			    print_tree(listptr->value.ptr.car, stdout);
			    printf("\nlength = %d, wildcards = %d\n",
						patlength, wildcards);
			    */
			    if ((patlength > matchedlength)
					|| ((patlength == matchedlength)
					  && (wildcards < matchedwildcards))) {
				matchedaction = actionptr;
				matchedpattern = listptr->value.ptr.car;
				matchedlength = patlength;
				matchedwildcards = wildcards;
			    }
			    break;
			}
			else if (p3ptr->type != TYPElisthead) {
			    printf("Error: dotted pair in pattern\n");
			    break;
			}
			p3ptr = p3ptr->value.ptr.car;
			sptr = sptr->value.ptr.cdr;
			if (p3ptr == NULL) {
			    printf("Error: invalid pattern\n");
			    break;
			}
			else if (p3ptr->type == TYPEstring) {
			    flag = 0;
			    while (1) {
				if (sptr == NULL) {
				    flag = 1;
				    break;
				}
				s2ptr = sptr->value.ptr.car;
				if (same_string(p3ptr->value.idptr->name,
						s2ptr->value.idptr->name)) {
				    pattern = pattern->value.ptr.cdr;
				    pattern = pattern->value.ptr.cdr;
				    sptr = sptr->value.ptr.cdr;
				    ++patlength;
				    break;
				}
				else
				    sptr = sptr->value.ptr.cdr;
			    }
			    if (flag)
				break;
			}
			else {
			    printf("Error: invalid pattern\n");
			    break;

			}
		    }
		    else {
			printf("Error: bad symbol in pattern.\n");
			break;
		    }
		}
	    }

	    listptr = listptr->value.ptr.cdr;
	}
	actionptr = actionptr->next;
    }

    if (matchedaction == NULL) {
	printf("no matching pattern found\n");
	return (NULL);
    }

/*
printf("final matched action %s\n", matchedaction->idptr->name);
*/

    sptr = sentence;
    pattern = matchedpattern;
    for (;;) {
	if ((sptr == NULL) || (pattern == NULL))
	    break;
	s2ptr = sptr->value.ptr.car;
	p2ptr = pattern->value.ptr.car;
	if (p2ptr->type == TYPEstring) {
/*
printf("seeing \"%s\" : \"%s\"\n", p2ptr->value.idptr->name,
s2ptr->value.idptr->name);
*/
	    if (same_string(p2ptr->value.idptr->name,
			    s2ptr->value.idptr->name)) {
		sptr = sptr->value.ptr.cdr;
		pattern = pattern->value.ptr.cdr;
	    }
	    else
		break;
	}
	else if (p2ptr->type == TYPEidname) {
	    bindhead = bindtail = NULL;
	    p3ptr = pattern->value.ptr.cdr;
	    if (p3ptr == NULL)
		bindhead = sptr;
	    else {
		bindhead = bindtail = new_node();
		bindhead->type = TYPElisthead;
		bindhead->value.ptr.car = s2ptr;
		sptr = sptr->value.ptr.cdr;
		p3ptr = p3ptr->value.ptr.car;
		for (;;) {
		    if (sptr == NULL)
			break;
		    s2ptr = sptr->value.ptr.car;
		    if (same_string(p3ptr->value.idptr->name,
				    s2ptr->value.idptr->name))
			break;
		    ptr = new_node();
		    ptr->type = TYPElisthead;
		    ptr->value.ptr.car = s2ptr;
		    bindtail->value.ptr.cdr = ptr;
		    bindtail = ptr;
		    sptr = sptr->value.ptr.cdr;
		}
	    }

	    if ((ptr = find_binding(p2ptr->value.idptr)) != NULL)
		ptr->value.ptr.cdr = bindhead;
	    else
		assign_global(p2ptr->value.idptr, bindhead);
/*
printf("pattern: %s = ", p2ptr->value.idptr->name);
print_tree(bindhead, stdout);
printf("\n");
*/
	    pattern = pattern->value.ptr.cdr;
	}
	else
	    break;
    }

    parserbindlist[1]->value.ptr.cdr = obj_to_node(actorptr);

    matchedparsy = NULL;
    bind_search(0);
    count = count_parsy(matchedparsy);

/*
printf("matched %d object combinations\n", count);
*/

    if (count == 0)
	printf("I'm not sure what you're referring to.\n");
    else if (count == 1) {
	for (i = 0; i < MAXBINDOBJ; ++i)
	    parserbindlist[(i*3)+1]->value.ptr.cdr = matchedparsy->obarray[i];
	result = evaluate(matchedaction->after, &flag);
    }
    else {
	for (i = 0; i < MAXBINDOBJ; ++i) {
/*
printf("------> %d\n", i);
pptr=matchedparsy;
while(pptr!=NULL){
 for(flag=0;flag<4;++flag){
  print_tree(pptr->obarray[flag], stdout);printf(" ");
 }
 printf("\n");
 pptr=pptr->next;
}
*/
	    count = count_parsy_objs(matchedparsy, i);
	    if (count > 1) {
		print_string("\\nWhich do you mean, ");
		foo = count;
		pptr = matchedparsy;
		while (pptr != NULL) {
		    if (!is_parsy_seen(pptr->obarray[i], pptr->next, i)) {
			head = new_node();
			head->type = TYPElisthead;
			ptr = new_node();
			ptr->type = TYPEidname;
			ptr->value.idptr = objnameid;
			head->value.ptr.car = ptr;
			ptr = new_node();
			ptr->type = TYPElisthead;
			head->value.ptr.cdr = ptr;
			ptr->value.ptr.car = pptr->obarray[i];
			ptr = evaluate(head, &flag);

			--foo;
			if (foo == 1) {
			    if (count > 2)
				print_string(", or ");
			    else
				print_string(" or ");
			}
			else if (foo > 1)
			    print_string(", ");
		    }
		    pptr = pptr->next;
		}

		print_string("?\\n");

		head = sentence_to_list();
		if (!which_object(i, head))
		    return (matcher(actorptr, head));
	    }
	}

	if (count_parsy(matchedparsy) != 1)
	    printf("I'm still not sure what you are referring to.\n");
	else {
	    for (i = 0; i < MAXBINDOBJ; ++i)
		parserbindlist[(i*3)+1]->value.ptr.cdr
						= matchedparsy->obarray[i];
	    result = evaluate(matchedaction->after, &flag);
	}
    }

    free_parsy(matchedparsy);
    matchedparsy = NULL;

    return (result);
}


/************************************************************************
*
* bind_search() - Begin search to find objects matching the special
*	wildcards.  If all used wildcards are bound to objects, then
*	test the <before> condition of the action.  If the precondition
*	is satisfied, then add this combination of objects to the list
*	of matched objects in <matchedparsy>.
*
************************************************************************/

#ifdef _ANSI_
void bind_search(int offset)
#else
bind_search(offset)
int offset;
#endif
{
    int   i, flag;
    PARSY *ptr;

    if (offset >= MAXBIND) {
/*
print_tree(parserbindlist[1]->value.ptr.cdr, stdout); printf(" ");
print_tree(parserbindlist[4]->value.ptr.cdr, stdout); printf(" ");
print_tree(parserbindlist[7]->value.ptr.cdr, stdout); printf(" ");
print_tree(parserbindlist[10]->value.ptr.cdr, stdout); printf(" ");
*/
	if (evaluate(matchedaction->before, &flag)) {
/*
printf("succeeded\n");
*/
	    ptr = new_parsy();
	    ptr->next = matchedparsy;
	    matchedparsy = ptr;
	    for (i = 0; i < MAXBINDOBJ; ++i)
		ptr->obarray[i] = parserbindlist[(i*3)+1]->value.ptr.cdr;
	}
/*
else
printf("failed\n");
*/
    }
    else
	object_search(offset, objectroot);
}



/************************************************************************
*
* object_search() - Recursively search for an obejct matching the string
*	sequence given by the player for the indexed wildcard.  If a
*	match is found, continue on the search for the next wildcard for
*	all matches found.
*
************************************************************************/

#ifdef _ANSI_
void object_search(int offset, OBJECT *objptr)
#else
object_search(offset, objptr)
int    offset;
OBJECT *objptr;
#endif
{
    char  *wordptr;
    int   foo, nouncount;
    NODEZ *namebind, *ptrbind, *quantbind, *listptr, *quantptr;
    NODEZ *ptr;

    namebind = parserbindlist[offset];
    ptrbind = parserbindlist[offset+1];
    quantbind = parserbindlist[offset+2];

	/* if no wildcard string, skip this wildcard and proceed to the next */
    if (namebind->value.ptr.cdr == NULL) {
	bind_search(offset + 3);
	return;
    }

	/* otherwise, loop through all words and try to match against the
	   current object, iteratively searching all siblings, and recursively
	   searching all children */
    while (objptr != NULL) {
	listptr = namebind->value.ptr.cdr;
	nouncount = 0;
	while (listptr != NULL) {
	    ptr = listptr->value.ptr.car;
	    wordptr = ptr->value.idptr->name;
	    foo = 0;

		/* handle the occurance of a number */
	    if (is_integer(wordptr) || same_string(wordptr, "all")) {
		if ((quantptr == NULL) || (quantptr->type == TYPEstring)) {
		    if (same_string(wordptr, "all"))
			quantptr = ptr;
		    else {
			quantptr = new_node();
			quantptr->type = TYPEnumber;
			quantptr->value.number = str2int(wordptr);
		    }
		}
		else {
		    print_string("There are too many numbers in \\\"");
		    print_sentence(namebind->value.ptr.cdr);
		    print_string("\\\".\\n");
		}
	    }

		/* halt if reach a bad word */
	    else if (!dummy_word(wordptr)
			&& ((foo = match_word(wordptr, objptr)) == 0))
		break;

	    if (foo == 1)
		++nouncount;

	    listptr = listptr->value.ptr.cdr;
	}

	    /* if the list is null, then all of the words from the list
		matched those describing the object, so if at least one
		noun was matched, then an object match has been found */
	if ((listptr == NULL) && (nouncount > 0)) {
	    ptrbind->value.ptr.cdr = obj_to_node(objptr);
	    quantbind->value.ptr.cdr = quantptr;
	    
	    bind_search(offset + 3);

	    ptrbind->value.ptr.cdr = NULL;
	    quantbind->value.ptr.cdr = NULL;
	}

	object_search(offset, objptr->child);

	objptr = objptr->sibling;
    }
}



/************************************************************************
*
* which_object() - This function is used to resolve ambiguity about
*	which object is being referenced.  It takes a sentence typed
*	in by the used (as a list of strings) and attempts to match that
*	sentence against all of the indexd ambiguous objects in the
*	list produced by the initial parsing (found in <matchedparsy>).
*	If a match is found, all other objects are removed (except for
*	any other references to this object), and a true value is returned.
*	If no matches are found, this function returns false.
*
************************************************************************/

#ifdef _ANSI_
int which_object(int index, NODEZ *sentence)
#else
int which_object(index, sentence)
int   index;
NODEZ *sentence;
#endif
{
    OBJECT *objptr;
    PARSY  *pptr;

    pptr = matchedparsy;
    while (pptr != NULL) {
	objptr = pptr->obarray[index]->value.objptr;
	if (object_vs_sentence(objptr, sentence)) {
/*
printf(" matched %d nouns: ");
print_sentence(sentence);print_string("\n");
*/
	    matchedparsy = eliminate_parsy(matchedparsy, objptr, index);
	    return 1;
	}

	pptr = pptr->next;
    }

    return 0;
}



/************************************************************************
*
* eliminate_parsy() - Given a pointer to an object to keep, eliminate all
*	other objects from the list of ambiguous objects according to the
*	given index.
*
************************************************************************/

#ifdef _ANSI_
PARSY *eliminate_parsy(PARSY *pptr, OBJECT *objptr, int index)
#else
PARSY *eliminate_parsy(pptr, objptr, index)
PARSY  *pptr;
OBJECT *objptr;
int    index;
#endif
{
    PARSY *parptr;

    if (pptr == NULL)
	return (NULL);

    parptr = eliminate_parsy(pptr->next, objptr, index);
    if (pptr->obarray[index]->value.objptr == objptr) {
	pptr->next = parptr;
	return (pptr);
    }

    pptr->next = NULL;
    free_parsy(pptr);

    return (parptr);
}



/************************************************************************
*
* object_vs_sentence() - Returns true if the given list of words
*	describes the given object.  (The given list must include at
*	least one noun associated with the object.)
*
************************************************************************/

#ifdef _ANSI_
int object_vs_sentence(OBJECT *objptr, NODEZ *listptr)
#else
int object_vs_sentence(objptr, listptr)
OBJECT *objptr;
NODEZ  *listptr;
#endif
{
    char  *wordptr;
    int   foo, nouncount;
    NODEZ *ptr;

    nouncount = 0;
    while (listptr != NULL) {
	ptr = listptr->value.ptr.car;
	wordptr = ptr->value.idptr->name;
	foo = 0;

		/* fail if reach a word not matching against this object */
	if (!dummy_word(wordptr) && ((foo = match_word(wordptr, objptr)) == 0)
		     && !is_integer(wordptr) && !same_string(wordptr, "all"))
	    return 0;

	if (foo == 1)
	    ++nouncount;

	listptr = listptr->value.ptr.cdr;
    }

	/* return true if at least one of the nouns describing the object
	   were used in the given string (thus failing if only match against
	   adjectives or instances of articles */
    return (nouncount > 0);
}



/************************************************************************
*
* count_parsy() - Returns a count of the number of nodes in the list of
*	ambiguous object combinations resulting from initial parsing.
*
************************************************************************/

#ifdef _ANSI_
int count_parsy(PARSY *ptr)
#else
int count_parsy(ptr)
PARSY *ptr;
#endif
{
    int count;

    count = 0;
    while (ptr != NULL) {
	++count;
	ptr = ptr->next;
    }

    return (count);
}



/************************************************************************
*
* count_parsy_objs() - Returns a count of the number of different objects
*	found in the list of ambiguous objects, according to the given
*	index.  This works by checking for each object whether any more
*	pointers to that object are in the list, if so, it is ignored, if
*	not, it is added to the count of different objects.
*
************************************************************************/

#ifdef _ANSI_
int count_parsy_objs(PARSY *pptr, int index)
#else
int count_parsy_objs(pptr, index)
PARSY *pptr;
int   index;
#endif
{
    int   count;

    if (pptr->obarray[index] == NULL)
	return 0;

    count = 0;
    while (pptr != NULL) {
	if (!is_parsy_seen(pptr->obarray[index], pptr->next, index))
	    ++count;
	pptr = pptr->next;
    }

    return (count);
}



/************************************************************************
*
* is_parsy_seen() - Given a pointer to an object and a list of ambiguous
*	nodes, return true if the given object is found in the list.
*
************************************************************************/

#ifdef _ANSI_
int is_parsy_seen(NODEZ *nptr, PARSY *pptr, int index)
#else
int is_parsy_seen(nptr, pptr, index)
NODEZ *nptr;
PARSY *pptr;
int   index;
#endif
{
    while (pptr != NULL) {
	if ((nptr != NULL) && (pptr->obarray[index] != NULL)
		&& (nptr->value.objptr == pptr->obarray[index]->value.objptr))
	    return 1;

	pptr = pptr->next;
    }

    return (0);
}



/************************************************************************
*
* print_sentence() - given a list of words, print them out with spaces
*	between the words.  (This should be changed to not print spaces
*	before most punctuation marks (what about quotes, which may or
*	may not have spaces before them?).)
*
************************************************************************/

#ifdef _ANSI_
void print_sentence(NODEZ *ptr)
#else
print_sentence(ptr)
NODEZ *ptr;
#endif
{
    NODEZ *pt2;
    int   flag;

    flag = 0;
    while (ptr != NULL) {
	if (flag)
	    print_string(" ");

	flag = 1;
	pt2 = ptr->value.ptr.car;
	print_string(pt2->value.idptr->name);

	ptr = ptr->value.ptr.cdr;
    }
}



/************************************************************************
*
* is_integer() - Given a string, return true if it is an integer.
*
************************************************************************/

#ifdef _ANSI_
int is_integer(char *word)
#else
int is_integer(word)
char *word;
#endif
{
    char *next;

    next = word;
    ++next;

	/* abort if first digit is not a minus sign or digit, or if it
	   is a minus sign without a digit after it */
    if (((*word == '-') && (*next == '\0'))
	|| ((*word != '-') && ((*word < '0') || (*word > '9'))))
	return 0;

	/* skip first character, since have already check it */
    ++word;

	/* loop through remaining characters in string */
    while (*word != '\0') {
	if ((*word < '0') || (*word > '9'))
	    return 0;
	++word;
    }

    return 1;
}



/************************************************************************
*
* str2int() - Given a string, convert it into an integer;
*
************************************************************************/

#ifdef _ANSI_
int str2int(char *strg)
#else
int str2int(strg)
char *strg;
#endif
{
    int number, negative;

    number = 0;

	/* skip white space */
    while ((*strg == ' ') || (*strg == '\t'))
	++strg;

	/* if it is a negative number, set flag and strip off minus sign */
    negative = (*strg == '-');
    if (negative)
	++strg;

	/* loop through, converting the individual chars to digits */
    while ((*strg >= '0') && (*strg <= '9')) {
	number = (*strg - '0') + (number * 10);
	++strg;
    }

	/* negate if necessary */
    if (negative)
	number = -number;

    return (number);
}



/************************************************************************
*
* int2str() - Given an integer, convert it into a string.
*
************************************************************************/

#ifdef _ANSI_
void int2str(int number, char *final)
#else
int2str(number, final)
int  number;
char *final;
#endif
{
    int digit, negative;
    char strstack[64], *sptr;
    char *fptr;

    sptr = strstack;

	/* set a flag if negative, and convert to positive number */
    negative = (number < 0);
    if (negative)
	number = -number;

	/* ignore loop if is zero */
    if (number == 0) {
	*sptr = '0';
	++sptr;
    }

	/* otherwise loop, striping off low digit and adding to stack */
    else {
	while (number) {
	    digit = number % 10;
	    number = number / 10;
	    *sptr = '0' + digit;
	    ++sptr;
	}
    }

	/* add on the negative symbol if necessary */
    if (negative) {
	*sptr = '-';
	++sptr;
    }

	/* pop off stack, converting digits to correct order */
    fptr = final;
    do {
	--sptr;
	*fptr = *sptr;
	++fptr;
    } while (sptr != strstack);

    *fptr = '\0';
}



/************************************************************************
*
* return true of the given word is one to be ignored on command line
*
************************************************************************/

#ifdef _ANSI_
int dummy_word(char *word)
#else
int dummy_word(word)
char *word;
#endif
{
    return (same_string(word, "a") || same_string(word, "an")
	 || same_string(word, "the") || same_string(word, "of"));
}



/************************************************************************
*
* match_word() - Return true if the given word is a noun or adjective
*	describing the object.  Returns 0 if neither, 1 if noun, or 2 if
*	adjective.
*
************************************************************************/

#ifdef _ANSI_
int match_word(char *word, OBJECT *objptr)
#else
int match_word(word, objptr)
char   *word;
OBJECT *objptr;
#endif
{
    NODEZ  *ptr, *carptr;

	/* check to see if this is a noun associated with the object */
    ptr = find_property(objptr, nounid);
    if (ptr != NULL)
	ptr = ptr->value.ptr.cdr;

	/* compare against nouns, allowing comparison for different
	   possible plural forms of the noun */
    while ((ptr != NULL) && (ptr->type == TYPElisthead)) {
	carptr = ptr->value.ptr.car;
	if (((carptr->type == TYPEidname) || (carptr->type == TYPEstring))
		&& match_plurality(carptr->value.idptr->name, word))
	    return 1;

        ptr = ptr->value.ptr.cdr;
    }

	/* check to see if this is an adjective associated with the object */
    ptr = find_property(objptr, adjid);
    if (ptr != NULL)
	ptr = ptr->value.ptr.cdr;

	/* compare against adjectives, doing a straight string comparison
	   since adjectives are not normally pluralized */
    while ((ptr != NULL) && (ptr->type == TYPElisthead)) {
	carptr = ptr->value.ptr.car;
	if (((carptr->type == TYPEidname) || (carptr->type == TYPEstring))
		&& same_string(carptr->value.idptr->name, word))
	    return 2;

        ptr = ptr->value.ptr.cdr;
    }

    return 0;
}



/************************************************************************
*
* evaluate_daemons() - Sequentially evaluate all daemons which have been
*	spawned.  Halts when either all daemons have been evaluated, or
*	when the <maindone> flag is set by the (terminate) function.
*
************************************************************************/

#ifdef _ANSI_
void evaluate_daemons(void)
#else
evaluate_daemons()
#endif
{
    int    flag;
    DAEMON *prevptr, *curptr;
    NODEZ  *scopeptr, *ptr, *fptr;

    prevptr = NULL;
    curptr = daemonlist;
    while ((curptr != NULL) && !maindone) {

	    /* insert the daemon's variables as new local scope */
	scopeptr = new_node();
	scopeptr->type = TYPElisthead;
	scopeptr->value.ptr.cdr = curptr->variables;
	scopeptr->value.ptr.car = locals;
	locals = scopeptr;

	    /* set the self binding to the object the daemon was spawned on */
	selfbind->value.ptr.cdr = obj_to_node(curptr->objectptr);

	   /* evaluate the daemon */
	killthisdaemon = 0;
	fptr = find_function(curptr->idptr);
	fptr = fptr->value.ptr.cdr->value.ptr.cdr;
	while (fptr != NULL) {
	    if (fptr->type != TYPElisthead)
		break;

	    flag = 0;
	    ptr = evaluate(fptr->value.ptr.car, &flag);
	    if (flag || killthisdaemon)
		break;

	    fptr = fptr->value.ptr.cdr;
	}

	    /* get rid of the self binding */
	selfbind->value.ptr.cdr = NULL;

	    /* record new values for variables and pop local scope */
	curptr->variables = scopeptr->value.ptr.cdr;
	locals = locals->value.ptr.car;

	    /* if the daemon is to be killed, remove it from daemonlist */
	if (killthisdaemon) {
	    if (prevptr == NULL) {
		daemonlist = curptr->next;
		free_daemon(curptr);
		curptr = daemonlist;
	    }
	    else {
		prevptr->next = curptr->next;
		free_daemon(curptr);
		curptr = prevptr->next;
	    }
	}
	else {
	    prevptr = curptr;
	    curptr = curptr->next;
	}
    }
}



/************************************************************************
*
* set_seed() - Seed the random number generator and create a new random
*	seed for the next time the program is invoked.  The seed is
*	stored in the file "ginaseed".
*
************************************************************************/

#ifdef _ANSI_
void set_seed(void)
#else
set_seed()
#endif
{
    int   seed;
    FILE *fopen(), *fileptr;

    if ((fileptr = fopen("ginaseed", "r")) != NULL) {
	fscanf(fileptr, "%d", &seed);
	fclose(fileptr);
    }
    else
	seed = rand();

    srand(seed);
    seed = rand();

    if ((fileptr = fopen("ginaseed", "w")) != NULL) {
	fprintf(fileptr, "%d", seed);
	fclose(fileptr);
    }
}



/************************************************************************
*
* match_plurality() - Match an unknow to a given base word, allowing for
*	the various ways in which the base word can be made plural.
*
************************************************************************/

#ifdef _ANSI_
int match_plurality(char *baseword, char *unknown)
#else
int match_plurality(baseword, unknown)
char *baseword, *unknown;
#endif
{
    char newword[100];

    if (same_string(unknown, baseword)
	|| same_string(unknown, new_suffix(baseword, newword, "s", 0))
	|| same_string(unknown, new_suffix(baseword, newword, "es", 0))
	|| (has_suffix(baseword, "fe")
	    && same_string(unknown, new_suffix(baseword, newword, "ves", 2)))
	|| (has_suffix(baseword, "us")
	    && same_string(unknown, new_suffix(baseword, newword, "i", 2)))
	|| (has_suffix(baseword, "ex")
	    && same_string(unknown, new_suffix(baseword, newword, "ices", 2))))
	return 1;

    return 0;
}



/************************************************************************
*
* compare two strings, ignoring case, returning true if equivalent
*
************************************************************************/

#ifdef _ANSI_
int same_string(char *s1, char *s2)
#else
int same_string(s1, s2)
char *s1, *s2;
#endif
{
    for (;;) {
	if (*s1 == *s2) {
	    if (*s1 == '\0')
		return 1;
	}
	else if ((islower(*s1) ? *s1 : tolower(*s1)) !=
				(islower(*s2) ? *s2 : tolower(*s2)))
	    return 0;

	++s1;
	++s2;
    }
}



/************************************************************************
*
* has_suffix() - returns true if sufx is a proper suffix of word (i.e., a
*	     word is not a suffix of itself), thus "ob" is a proper suffix
*	     of "fob", but "fob" is not a proper suffix of "fob"
*
************************************************************************/

#ifdef _ANSI_
int has_suffix(char *word, char *sufx)
#else
int has_suffix(word, sufx)
char word[], sufx[];
#endif
{
    int i, j;

	/* get the offsets of last character in each word */
    i = strlen(word) - 1;
    j = strlen(sufx) - 1;

	/* if the suffix's length is >= that of the word, return false */
    if (j >= i)
	return 0;

	/* compare the last characters of the suffix with the last ones
	   of the word, until all of the characters of the suffix have
	   been checked, or until a mismatch is found */
    while (j >= 0) {
	if (word[i] != sufx[j])
	    return 0;
	--i;
	--j;
    }

    return 1;
}



/************************************************************************
*
* new_suffix() - given a word and a suffix, replace the last <len>
*		 characters of the word with the new suffix, storing
*		 the new word in <newword>, and returning the address
*		 of <newword>
*
************************************************************************/

#ifdef _ANSI_
char *new_suffix(char *word, char *newword, char *sufx, int len)
#else
char *new_suffix(word, newword, sufx, len)
char *word, *newword, *sufx;
int  len;
#endif
{
    char *nw;
    int  i;

    nw = newword;

    i = strlen(word) - len;

    while (i > 0) {
	*nw = *word;
	++nw;
	++word;
	--i;
    }

    while (*sufx != '\0') {
	*nw = *sufx;
	++nw;
	++sufx;
    }

    *nw = '\0';

    return (newword);
}
