/************************************************************************
*
* Program: GINA System LISP
* Module ginafunc.c
* Version 0.1, January 1994.
* Version 0.2, October 1994.
* Version 0.3, February 1995.
* Version 0.4, August 1995.
*
* Copyright 1994, 1995, Jeff Standish.  (jestandi@cs.indiana.edu)
* All rights reserved.
* Permission is hereby granted for unrestricted non-commercial use,
* provided full attribution of source code origin is included.
*
* This module contains all built-in functions for GINALISP.
*
* Version 0.2
* Revised into standard C, removing dependencies upon special functions
* for accessing memory via 8088.
*
* Version 0.3
* Corrected a bug with the (method...) function in which <self> was
* being reset to its new value before any of the arguments could be
* evaluated, which would screw things up if <self> were passed as an
* argument, or used in a function to produce an argument for (method...).
*
* 4-30-95: Bug fix with let_function(), which arose when (let...) was
* called with no local parameters, which caused <lastnew> to not be set,
* causing a segment fault.  Possibly this is the cause of that infrequent
* case of inexplicable segment faults which have occasionally occurred...
* but probably not.  This seems a predictable cause of a crash, whereas
* the ones which have thus far been occurring have appeared infrequently
* and with no apparent predictable cause.
*
* Version 0.4
* Another bug fix in (let...), in which variables declared without an
* initial value were not being inserted into the local scope, resulting
* in all assignments to them creating global variables.  Also, the <exitflag>
* is also now trapped by (let ...), which was not doing it previously for no
* reason I could figure (unless I just plain forgot to code that one line).
*
* The addition of the func_warning() function to display warning messages
* only when the *warning-messages* global variable is bound to a non-nil
* value during interpretation, so that minor warnings would not clutter
* up with screen during use of the system for normal game play.
*
* Addition of the following built-in lisp functions: append, str2sym,
* sym2str, readsentence, parser, ginalisp, nth, and dumpstate.
*
*************************************************************************
*
* Notes:
*
* Each of the functions in this module take two arguments: head and
* exitflag.  Head contains a pointer to the list which contains the
* invocation of the function.  Exitflag is set by those functions which
* break out of an evaluation loop, such as (exit), (return ...), and
* (terminate).  Flow control functions will monitor this flag after each
* function they evaluate, and will exit if this flag has been set (i.e.,
* (loop ...) will continue to evaluate its list of expressions until one
* of them sets exitflag to a non-zero value).
*
* Currently, (cons...), (let...), (begin...), (loop...), and (function...)
* will break if the exitflag is set.  However, only (begin...), (let ...),
* (loop ...), and (function...) currently reset exitflag to zero before
* exiting.  (cons...) should not reset exitflag to zero, so that (exit)'s
* can be propagated upwards.
*
* When a function has finished evaluating its arguments, it will return
* a pointer the list structure it has created.  Some functions always
* return an empty list (such as (exit), which sets exitflag and returns
* a nil pointer).
*
* NOTE: an assumption for tree structure passed to these functions
*	(which is held in the pointer <head>) is that the minimum number
*	of arguments/elements are in the tree, and that all elements in
*	the main list are sure to be type TYPElisthead, which simplifies
*	the functions so that they do not need to check that this list
*	is corrupt before they begin evaluating their functionality on
*	the given list of arguments
*
************************************************************************/


#include <stdio.h>
#include "ginas.h"


	/* externally defined global variables */
extern NODEZ	*functions, *locals;
extern OBJECT   *objectroot, *classlist, *lastclass;
extern int	nodetally, gcsize;
extern int	maindone;
extern int	debuglevel;
extern NODEZ	*truebind, *warningbind, *debugbind;
extern int	killthisdaemon;
extern OBJECT	*obspecroot;
extern char	specialchar[];


	/* each entry is of the form:
	 *	string identifier of operation,
	 *	number of arguments (if negative, then additional optional
	 *		arguments may be included, so a 2 means that function
	 *		requires exactly 2 arguments, while a -2 means that
	 *		function takes 2 or more arguments),
	 *	name of built-in function to call
	 *
	 * This makes adding new built-in functions simple, since all that
	 * is required is to add a new entry into the table (in alphabetical
	 * order), and to change the value of MAXFUNC, as defined in ginas.h
	 */
FUNCDATA funclist[MAXFUNC] = {
{ "!=",		 2, nequalq_function },
{ "%",		-2, mod_function },
{ "*",		-2, mul_function },
{ "+",		-2, add_function },
{ "-",		-2, sub_function },
{ "/",		-2, div_function },
{ "1+",		 1, add1_function },
{ "1-",		 1, sub1_function },
{ "<",		 2, less_function },
{ "<=",		 2, lesseq_function },
{ "=",		 2, equalq_function },
{ ">",		 2, great_function },
{ ">=",		 2, greateq_function },
{ "action",	 4, action_function },
{ "add",	-2, add_function },
{ "add1",	 1, add1_function },
{ "and",	-1, and_function },
{ "append",	-1, append_function },
{ "atom?",	 1, atomq_function },
{ "begin",	-1, begin_function },
{ "car",	 1, car_function },
{ "cdr",	 1, cdr_function },
{ "child",	 1, child_function },
{ "class",	 4, class_function },
{ "class?",	 1, classq_function },
{ "collect",	 0, collect_function },
{ "cond",	-1, cond_function },
{ "cons",	 2, cons_function },
{ "daemon",	-3, function_function },
{ "delprop",	 2, delprop_function },
{ "destroy",	 1, destroy_function },
{ "div",	-2, div_function },
{ "dumpids",	 0, dumpids_function },
{ "dumpstate",	 1, dumpstate_function },
{ "equal?",	 2, equalq_function },
{ "eval",	 1, eval_function },
{ "exit",	 0, exit_function },
{ "findobject",	-1, findobject_function },
{ "function",	-3, function_function },
{ "getprop",	 2, getprop_function },
{ "ginalisp",	 0, ginalisp_function },
{ "hasprop",	 2, hasprop_function },
{ "if",		-2, if_function },
{ "killdaemon",	 0, killdaemon_function },
{ "length",	 1, length_function },
{ "let",	-2, let_function },
{ "list",	-1, list_function },
{ "list?",	 1, listq_function },
{ "load",	 1, load_function },
{ "loadobjects", 2, loadobjects_function },
{ "loop",	-1, loop_function },
{ "member",	 2, member_function },
{ "method",	-2, method_function },
{ "mod",	-2, mod_function },
{ "move",	 2, move_function },
{ "mul",	-2, mul_function },
{ "nequal?",	 2, nequalq_function },
{ "not",	 1, nullq_function },
{ "nth",	 2, nth_function },
{ "null?",	 1, nullq_function },
{ "number?",	 1, numberq_function },
{ "object",	 5, object_function },
{ "object?",	 1, objectq_function },
{ "objectname",	 1, objectname_function },
{ "or",		-1, or_function },
{ "parent",	 1, parent_function },
{ "parser",	 2, parser_function },
{ "pluralof",	 2, pluralof_function },
{ "pprinttree",	 1, pprinttree_function },
{ "print",	-1, print_function },
{ "printtree",	 1, printtree_function },
{ "quote",	 1, quote_function },
{ "random",	 1, random_function },
{ "read",	 0, read_function },
{ "readsentence",0, readsentence_function },
{ "return",	 1, return_function },
{ "samestring?", 2, samestringq_function },
{ "saveobjects", 2, saveobjects_function },
{ "set",	 2, set_function },
{ "setprop",	 3, setprop_function },
{ "showaction",	 1, showaction_function },
{ "showclass",	 1, showclass_function },
{ "showfunc",	 1, showfunc_function },
{ "showobject",	 1, showobject_function },
{ "sibling",	 1, sibling_function },
{ "spawn",	-2, spawn_function },
{ "str2sym",	 1, str2sym_function },
{ "string?",	 1, stringq_function },
{ "sub",	-2, sub_function },
{ "sub1",	 1, sub1_function },
{ "subclassof?", 2, subclassofq_function },
{ "sym2str",	 1, sym2str_function },
{ "terminate",	 0, terminate_function }
};



/************************************************************************
*
* action_function() - This function is used to define actions.  It takes
*	four arguments: the action name, a list of command strings
*	which can invoke the action, a precondition, and action command
*	which is invoked if the precondition evaluates to true.
*
* (action hello (("hello") ("good" "day")) t (print "Hello there!\n"))
*	==> hello
*
************************************************************************/

#ifdef _ANSI_
NODEZ *action_function(NODEZ *head, int *exitflag)
#else
NODEZ *action_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    ACTION *actptr;
    NODEZ  *hptr, *nameptr;

	/* get the name of the action */
    hptr = head->value.ptr.cdr;
    nameptr = hptr->value.ptr.car;
    if ((nameptr == NULL) || (nameptr->type != TYPEidname)) {
	printf("Invalid action name: ");
	print_tree(nameptr, stdout);
	putchar('\n');
	return (NULL);
    }

	/* create a new action entry in the list of actions */
    actptr = new_action();
    actptr->idptr = nameptr->value.idptr;

	/* insert the list of words, preconditions, and affects
	   into the action entry */
    hptr = hptr->value.ptr.cdr;
    actptr->words = hptr->value.ptr.car;
    hptr = hptr->value.ptr.cdr;
    actptr->before = hptr->value.ptr.car;
    hptr = hptr->value.ptr.cdr;
    actptr->after = hptr->value.ptr.car;

	/* return the name of the action */
    return (nameptr);
}



/************************************************************************
*
* add_function() - Given two or more arguments, evaluate each in turn,
*	returning the sum of all their results.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *add_function(NODEZ *head, int *exitflag)
#else
NODEZ *add_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    int   val;
    NODEZ *hptr, *ptr;

	/* initialize the sum to zero */
    val = 0;
    hptr = head->value.ptr.cdr;

	/* loop through all of the expressions, summing up their results */
    while (hptr != NULL) {
	ptr = evaluate(hptr->value.ptr.car, exitflag);

	if ((ptr != NULL) && (ptr->type == TYPEnumber))
	    val += ptr->value.number;
	else
	    func_warning("(add ...) requires numbers as arguments", NULL,head);

	hptr = hptr->value.ptr.cdr;
    }

	/* create a new node for the result of the summation */
    ptr = new_node();
    ptr->type = TYPEnumber;
    ptr->value.number = val;

	/* return the sum */
    return (ptr);
}



/************************************************************************
*
* add1_function() - Given a numerical expression, return a value one
*	greater than the result of the expression.
*
* (+1 5)  ==>  6
*
************************************************************************/

#ifdef _ANSI_
NODEZ *add1_function(NODEZ *head, int *exitflag)
#else
NODEZ *add1_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    int   val;
    NODEZ *hptr, *ptr;

	/* evaluate the expression */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* if it is a number, create a value one greater than the result,
	   if not a number, then just return nil */
    if ((ptr != NULL) && (ptr->type == TYPEnumber)) {
	val = ptr->value.number;
	ptr = new_node();
	ptr->type = TYPEnumber;
	ptr->value.number = val + 1;
    }
    else
	ptr = NULL;

    return (ptr);
}



/************************************************************************
*
* and_function() - Given a sequence of one or more expressions, evaluate
*	each of them in turn, and return true if all evaluate to true.  If
*	any one of them evaluates to nil, it immediately halts, returning
*	nil and will not evaluate any of the remaining expressions.
*	NOTE: If the exitflag is set, it will immediately halt
*	evaluation and return the result of the expression.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *and_function(NODEZ *head, int *exitflag)
#else
NODEZ *and_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* initialize the result of the expression to true */
    ptr = truebind->value.ptr.cdr;
    hptr = head->value.ptr.cdr;

	/* loop through all expressions */
    while (hptr != NULL) {
	ptr = evaluate(hptr->value.ptr.car, exitflag);

	    /* the result is nil, or the exitflag is set, then return the
		result of the expression */
	if ((ptr == NULL) || *exitflag)
	    return (ptr);

	hptr = hptr->value.ptr.cdr;
    }

    return (ptr);
}



/************************************************************************
*
* append_function() - Given one or more lists, return a single list
*	comprised of all elements of the individual lists.
*
* (append '(a b c) '(d e f) '(g h i)) ==> (a b c d e f g h i)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *append_function(NODEZ *head, int *exitflag)
#else
NODEZ *append_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    return (append_func_rec1(head->value.ptr.cdr, head, exitflag));
}

#ifdef _ANSI_
NODEZ *append_func_rec1(NODEZ *hptr, NODEZ *head, int *exitflag)
#else
NODEZ *append_func_rec1(hptr, head, exitflag)
NODEZ *hptr, *head;
int   *exitflag;
#endif
{
    NODEZ *ptr, *pt2;

	/* handle hitting the end of the list of arguments */
    if (hptr == NULL)
	return (NULL);

	/* evaluate the current argument */
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if (ptr == NULL)
	return (append_func_rec1(hptr->value.ptr.cdr, head, exitflag));
    else if (ptr->type != TYPElisthead) {
	func_warning("(append ...): argument is not a proper list", NULL,head);
	return (ptr);
    }
    else if (hptr->value.ptr.cdr == NULL)
	return (ptr);

	/* get the result of remaining arguments */
    pt2 = append_func_rec1(hptr->value.ptr.cdr, head, exitflag);
    if (pt2 == NULL)
	return (ptr);

	/* add the current list to the results of recursion */
    return (append_func_rec2(ptr, pt2, head));
}

/* recursively cons up a new list from the given one, such that the tail
 * is composed of previous lists */
#ifdef _ANSI_
NODEZ *append_func_rec2(NODEZ *curptr, NODEZ *baseptr, NODEZ *head)
#else
NODEZ *append_func_rec2(curptr, baseptr)
NODEZ *curptr, *baseptr;
#endif
{
    NODEZ *ptr;

    if (curptr == NULL)
	return (baseptr);
    else if (curptr->type != TYPElisthead) {
	func_warning("(append ...): argument is not a proper list", NULL,head);
	return (baseptr);
    }

    return (constructor(curptr->value.ptr.car,
			append_func_rec2(curptr->value.ptr.cdr,baseptr,head)));
}



/************************************************************************
*
* atomq_function() - Takes one argument, which is evaluated.  If this
*	results in a list, the value nil is returned, otherwise it
*	returns true (nil is considered to be an atom by this function).
*
************************************************************************/

#ifdef _ANSI_
NODEZ *atomq_function(NODEZ *head, int *exitflag)
#else
NODEZ *atomq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* evaluate the given expression */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* if it is a list, return nil */
    if ((ptr != NULL) && (ptr->type == TYPElisthead))
	return (NULL);

	/* anything else is an atom, so return true */
    return (truebind->value.ptr.cdr);
}



/************************************************************************
*
* begin_function() - Given a sequence of one or more expressions,
*	evaluate each of them in turn until either done or until
*	the evaluation exitflag is set.  Returns the value of the
*	last expression evaluated.
*
*	The <exitflag> is trapped by this function.
*
* (begin <expression 1> ... <expression n>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *begin_function(NODEZ *head, int *exitflag)
#else
NODEZ *begin_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* the pointer to the first expression */
    hptr = head->value.ptr.cdr;
    ptr = NULL;

	/* loop through all of the expressions */
    while (hptr != NULL) {

	    /* halt if the expression list is corrupt */
	if (hptr->type != TYPElisthead)
	    break;

	    /* evaluate the expression, and if the exitflag is set,
		do not evaluate any more of the expressions */
	ptr = evaluate(hptr->value.ptr.car, exitflag);
	if (*exitflag) {
	    *exitflag = 0;
	    break;
	}

	    /* proceed to the next expression */
	hptr = hptr->value.ptr.cdr;
    }

	/* return the value of the last expression evaluated */
    return (ptr);
}



/************************************************************************
*
* car_function() - Given a list, return the first element of the list.
*	Returns nil if not given a good list.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *car_function(NODEZ *head, int *exitflag)
#else
NODEZ *car_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* evaluate the given expression */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* if is a good list, return the first element of it */
    if ((ptr != NULL) && (ptr->type == TYPElisthead))
	return (ptr->value.ptr.car);

    func_warning("(car ...) requires a non-nil argument", NULL, head);

	/* otherwise return nil */
    return (NULL);
}



/************************************************************************
*
* cdr_function() - Given a list, strip off the first element of the
*	list and return the remaining portion of the list.  Returns
*	nil if not given a good list.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *cdr_function(NODEZ *head, int *exitflag)
#else
NODEZ *cdr_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* evaluate the given expression */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* if a good list pointer, return the cdr of it */
    if ((ptr != NULL) && (ptr->type == TYPElisthead))
	return (ptr->value.ptr.cdr);

    func_warning("(cdr ...) requires a non-nil argument", NULL, head);

	/* otherwise return nil */
    return (NULL);
}



/************************************************************************
*
* child_function() - Given a pointer to an existing object, return the
*	pointer to its first child.  Returns nil if the object pointer
*	is bad, or if the object has no children.
*
* (child <object>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *child_function(NODEZ *head, int *exitflag)
#else
NODEZ *child_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    OBJECT *obptr;
    NODEZ  *hptr, *ptr;

	/* get the pointer to the object in question */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(child ...) requires an object pointer as argument",
		NULL, head);
	return (NULL);
    }

	/* return a pointer to its first child */
    return (obj_to_node(obptr->child));
}



/************************************************************************
*
* class_function() - Define a new class.  Returns the name of the
*	class if successful.
*
* (class <classname> <class list> <property list> <method list>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *class_function(NODEZ *head, int *exitflag)
#else
NODEZ *class_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *nameptr;
    OBJECT *clptr;

	/* get the name of the new class */
    hptr = head->value.ptr.cdr;
    nameptr = hptr->value.ptr.car;
    if ((nameptr == NULL) || (nameptr->type != TYPEidname)) {
	printf("Error: (class ...): class name must be idname\n");
	return (NULL);
    }

	/* create the new class, inserting it into the list of classes,
	   or redefine an existing class */
    if ((clptr = find_class(nameptr->value.idptr)) == NULL) {
	clptr = new_object();
	clptr->idptr = nameptr->value.idptr;
	if (lastclass == NULL)
	    classlist = lastclass = clptr;
	else {
	    lastclass->sibling = clptr;
	    lastclass = clptr;
	}
    }
    else
	func_warning("Redefinition of class %s", clptr->idptr->name, head);

	/* parse the lists of classes, properties, and methods */
    hptr = hptr->value.ptr.cdr;
    clptr->classes = make_class_list(hptr->value.ptr.car);
    hptr = hptr->value.ptr.cdr;
    clptr->properties = hptr->value.ptr.car;
    hptr = hptr->value.ptr.cdr;
    clptr->methods = hptr->value.ptr.car;

	/* return the name of the class */
    return (nameptr);
}



/************************************************************************
*
* classq_function() - Returns true if the given expression evaluates to
*	a class pointer.
*
* (class? <expression>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *classq_function(NODEZ *head, int *exitflag)
#else
NODEZ *classq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* evaluate the expression */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* return true if the result is a class pointer */
    if ((ptr != NULL) && (ptr->type == TYPEclass))
	return (truebind->value.ptr.cdr);

    return (NULL);
}



/************************************************************************
*
* collect_function() - forces garbage collection to occur on the next
*	cycle of execution
*
************************************************************************/

#ifdef _ANSI_
NODEZ *collect_function(NODEZ *head, int *exitflag)
#else
NODEZ *collect_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    nodetally = gcsize;

    return (NULL);
}



/************************************************************************
*
* cond_function() - Given a sequence of condition expressions, look at
*	each condition in turn until one of them evaluates to true,
*	then evaluate all of the expressions.  Note that (cond ...)
*	will not trap (exit) or (return ...), but it will stop when it
*	encounters such a function and pass the exitflag back up to
*	the calling function.
*
* (cond (<condition 1> <expression 1> ... <expression x>)
*       ...
*       (<condition y> <expression 1> ... <expression z>))
*
************************************************************************/

#ifdef _ANSI_
NODEZ *cond_function(NODEZ *head, int *exitflag)
#else
NODEZ *cond_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr, *rslt;

	/* move the pointer to the first condition list */
    rslt = NULL;
    hptr = head->value.ptr.cdr;

	/* loop through all conditions */
    while (hptr != NULL) {
	ptr = hptr->value.ptr.car;

	    /* if this is a good sublist, evaluate the test condition */
	if ((ptr != NULL) && (ptr->type == TYPElisthead)) {
	    rslt = evaluate(ptr->value.ptr.car, exitflag);

		/* abort if hit a return exitflag */
	    if (*exitflag)
		return (rslt);

		/* if the result is true, evaluate the subcondition */
	    if (rslt != NULL) {
		ptr = ptr->value.ptr.cdr;

		    /* evaluate each expression in the subcondition */
		while (ptr != NULL) {
		    if (ptr->type == TYPElisthead)
			rslt = evaluate(ptr->value.ptr.car, exitflag);
		    if (*exitflag)	
			return (rslt);
		    ptr = ptr->value.ptr.cdr;
		}
		return (rslt);
	    }
	}
	else
	    func_warning("(cond ...): badly-formed conditional expression",
		NULL, head);

	hptr = hptr->value.ptr.cdr;
    }

	/* return the result of the condition expression */
    return (rslt);
}



/************************************************************************
*
* cons_function() - Given two expressions, evaluate them and cons
*	together the result, returning the new list.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *cons_function(NODEZ *head, int *exitflag)
#else
NODEZ *cons_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *first, *rest;

	/* evaluate the two arguments */
    hptr = head->value.ptr.cdr;
    first = evaluate(hptr->value.ptr.car, exitflag);
    hptr = hptr->value.ptr.cdr;
    rest = evaluate(hptr->value.ptr.car, exitflag);

	/* cons them together and return the result */
    return (constructor(first, rest));
}



/************************************************************************
*
* delprop_function() - Given a pointer to an object and the name of a
*	property, remove than property from the object's list of
*	properties.  Will not affect any inherited properties, only those
*	those directly associated with this object.  Returns the name of
*	the property is successful, or nil if no such property can be
*	found in the object's definition.
*
* (delprop <object> <propname>)
* returns property name, or nil if no such property
*
************************************************************************/

#ifdef _ANSI_
NODEZ *delprop_function(NODEZ *head, int *exitflag)
#else
NODEZ *delprop_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    OBJECT *obptr;
    NODEZ  *hptr, *ptr, *nameptr, *prevptr, *proptr, *bindptr;

	/* get the pointer to the object */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car, exitflag))) ==NULL){
	func_warning("(delprop ...): first argument must be object pointer",
		NULL, head);
	return (NULL);
    }

	/* get the name of the property to delete */
    hptr = hptr->value.ptr.cdr;
    nameptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((nameptr == NULL) || (nameptr->type != TYPEidname)) {
	func_warning("(delprop ...) second argument is not a valid identifier",
		NULL, head);
	return (NULL);
    }

	/* loop through all of the properties of this object, and if the
	   desired property is found, remove it from the list and return
	   the name of the property to indicate success */
    prevptr = NULL;
    proptr = obptr->properties;
    while (proptr != NULL) {

	    /* loop at the current property binding, if it is a bad
		property definition, abort now */
	bindptr = proptr->value.ptr.car;
	if ((bindptr == NULL) || (bindptr->type != TYPElisthead)) {
	    func_warning("(delprop ...): invalid property list for \"%s\"",
		obptr->idptr->name, head);
	    return (NULL);
	}

	    /* if the property does not have a valid name, abort */
	ptr = bindptr->value.ptr.car;
	if ((ptr == NULL) || (ptr->type != TYPEidname)) {
	    func_warning("(delprop ...): invalid property name for \"%s\"",
		obptr->idptr->name, head);
	    return (NULL);
	}

	    /* if this is the desired property, remove it from the
		list and return its name */
	if (ptr->value.number == nameptr->value.number) {
	    if (prevptr == NULL)
		obptr->properties = proptr->value.ptr.cdr;
	    else
		prevptr->value.ptr.cdr = proptr->value.ptr.cdr;

	    return (ptr);
	}

	    /* else, move to the next property in the list */
	prevptr = proptr;
	proptr = proptr->value.ptr.cdr;
    }

	/* otherwise the property was not found, so return nil */
    return (NULL);
}



/************************************************************************
*
* destroy_function() - Given a pointer to an object, remove that object
*	from the object tree.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *destroy_function(NODEZ *head, int *exitflag)
#else
NODEZ *destroy_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr;
    OBJECT *obptr;

	/* get the pointer to the object to detroy */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car, exitflag))) ==NULL){
	func_warning("(destroy ...) not given valid identifier", NULL, head);
	return (NULL);
    }

	/* destroy the object and return true */
    destroy_object(obptr);
    return (truebind->value.ptr.cdr);
}



/************************************************************************
*
* div_function() - Given two or more arguments, evaluate each in turn,
*	taking the first and returning its value divided by all
*	successive values.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *div_function(NODEZ *head, int *exitflag)
#else
NODEZ *div_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    int   val;
    NODEZ *hptr, *ptr;

	/* evaluate the first argument, and use its value as the base which
	   will be divided by the values of the successive arguments */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    val = 0;
    if ((ptr != NULL) && (ptr->type == TYPEnumber))
	val = ptr->value.number;
    else
	func_warning("(div ...) requires numbers as arguments", NULL, head);

	/* evaluate all remaining arguments, each time making the new
	   result equal to the previous result divided by the current
	   argument's value */
    hptr = hptr->value.ptr.cdr;
    while (hptr != NULL) {
	ptr = evaluate(hptr->value.ptr.car, exitflag);
	if ((ptr != NULL) && (ptr->type == TYPEnumber)) {
	    if (ptr->value.number != 0)
		val /= ptr->value.number;
	    else
		puts("Error: division by zero, division ignored\n");
	}
	else
	    func_warning("(div ...) requires numbers as arguments", NULL,head);

	hptr = hptr->value.ptr.cdr;
    }

	/* insert the results into a new node, and return the pointer to it */
    ptr = new_node();
    ptr->type = TYPEnumber;
    ptr->value.number = val;

    return (ptr);
}



/************************************************************************
*
* dumpids_function() - print out the contents of the internal symbol
*	table
*
************************************************************************/

#ifdef _ANSI_
NODEZ *dumpids_function(NODEZ *head, int *exitflag)
#else
NODEZ *dumpids_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    dump_idnodes();

    return (NULL);
}



/************************************************************************
*
* dumpstate_function() - Print out state of the system in such a way
*	that it can be read back in as normal GINALISP instructions.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *dumpstate_function(NODEZ *head, int *exitflag)
#else
NODEZ *dumpstate_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr;

	/* get name of file to save it into */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) || (ptr->type != TYPEstring)) {
	func_warning("(dumpstate ...) requires a string as an argument",
		NULL, head);
	return (NULL);
    }

	/* save the object */
    dump_state(ptr->value.idptr->name);

	/* return true */
    return (truebind->value.ptr.cdr);
}



/************************************************************************
*
* equalq_function() - Given two expressions, evaluate them and return
*	true if they are equivalent.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *equalq_function(NODEZ *head, int *exitflag)
#else
NODEZ *equalq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr1, *ptr2;

	/* evaluate the two expressions */
    hptr = head->value.ptr.cdr;
    ptr1 = evaluate(hptr->value.ptr.car, exitflag);
    hptr = hptr->value.ptr.cdr;
    ptr2 = evaluate(hptr->value.ptr.car, exitflag);

	/* return true if they are equal */
    if (is_equal_tree(ptr1, ptr2))
	return (truebind->value.ptr.cdr);

	/* default to nil if not equal */
    return (NULL);
}



/************************************************************************
*
* eval_function() - Evaluate the given expression, and then evaluate the
*	result of that.
*
* (eval <expression>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *eval_function(NODEZ *head, int *exitflag)
#else
NODEZ *eval_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr;

    hptr = head->value.ptr.cdr;
    return (evaluate(evaluate(hptr->value.ptr.car, exitflag), exitflag));
}



/************************************************************************
*
* exit_function() - Set the flow control exitflag for breaking a layer of
*	flow control (begin, let, loop, etc).
*
************************************************************************/

#ifdef _ANSI_
NODEZ *exit_function(NODEZ *head, int *exitflag)
#else
NODEZ *exit_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    *exitflag = 1;
    return (NULL);
}



/************************************************************************
*
* findobject_function() - Given the name of an object, search for that
*	object in the object tree, and return a pointer to it if found,
*	or nil if no such object exists.  Optionally, a starting point
*	in the object tree may be given, so that the search will proceed
*	from there.
*
* (findobject 'objectname <optional starting pointer>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *findobject_function(NODEZ *head, int *exitflag)
#else
NODEZ *findobject_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr;
    OBJECT *startptr;

	/* get the object's name */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) || (ptr->type != TYPEidname)) {
	func_warning("(findobject ...) not given a proper identifier",
		NULL, head);
	return (NULL);
    }

	/* if no starting point is defined, search from the root of the
	   object tree, and return the result */
    hptr = hptr->value.ptr.cdr;
    if (hptr == NULL)
	return (obj_to_node(find_object(ptr->value.idptr, objectroot)));

	/* otherwise a starting point is given, so check to see if it is
	   a valid pointer, if not, complain and default to searching
	   from the root of the object tree */
    if ((startptr=good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag)))==NULL){
	func_warning("(findobject ...) not given an object pointer",
		NULL, head);
	startptr = objectroot;
    }

    if (startptr->idptr == ptr->value.idptr)
	return (obj_to_node(startptr));

	/* perform the search from the first child of the specified starting
	   point, and return the result */
    return (obj_to_node(find_object(ptr->value.idptr, startptr->child)));
}



/************************************************************************
*
* function_function() - Define a new function.  Inserts the function
*	definition into the list of functions, with very little error
*	checking.  Bugs in the function will not be discovered until
*	it is actually run.  Returns the name of the function if
*	successful, or nil otherwise.
*
* (function <function-name> <parameter-list> <expr 1> ... <expr n>)
* (function foo (x y) (add x y))
*
************************************************************************/

#ifdef _ANSI_
NODEZ *function_function(NODEZ *head, int *exitflag)
#else
NODEZ *function_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *nameptr, *ptr;

	/* get the name of the function being defined */
    hptr = head->value.ptr.cdr;
    nameptr = hptr->value.ptr.car;
    if ((nameptr == NULL) || (nameptr->type != TYPEidname)) {
	func_warning("(function ...): first argument must be symbol",
		NULL, head);
	return (NULL);
    }

	/* move head to the second argument, and check that the list of
	   arguments is actually a list so that this point can be ignored
	   when a function is executed */
    hptr = hptr->value.ptr.cdr;
    if ((hptr->value.ptr.car != NULL)
		&& (hptr->value.ptr.car->type != TYPElisthead)) {
	printf("(function ...): second argument must be list", NULL);
	return (NULL);
    }

	/* add this function to list of functions, either by replacing
	   any existing function with the given name, or creating a
	   new function listing */
    if ((ptr = find_function(nameptr->value.idptr)) != NULL)
	ptr->value.ptr.cdr = hptr;
    else
	functions = assign_binding(nameptr->value.idptr, hptr, functions);

	/* return the name of the function defined */
    return (nameptr);
}



/************************************************************************
*
* getprop_function() - Given a pointer to an object and the name of a
*	property, check to see if the object has or inherits this
*	property from one of its superclasses.  If the property is found,
*	its value is returned.  If not, the value nil is returned.
*
* (getprop <object> <propname>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *getprop_function(NODEZ *head, int *exitflag)
#else
NODEZ *getprop_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr, *proptr;
    OBJECT *obptr;

	/* get the pointer to the object, return nil if bad pointer */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(getprop ...) not given good object pointer",
		NULL, head);
	return (NULL);
    }

	/* get the name of the property, return nil if bad prop name */
    hptr = hptr->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) || (ptr->type != TYPEidname)) {
	func_warning("(getprop ...) given invalid property name",
		NULL, head);
	return (NULL);
    }

	/* search for the property, return nil if cannot be found */
    proptr = find_property(obptr, ptr->value.idptr);
    if ((proptr == NULL) || (proptr->type != TYPElisthead))
	return (NULL);

	/* otherwise the property was found, so return its value */
    return (proptr->value.ptr.cdr);
}



/************************************************************************
*
* ginalisp_function() - Starts up an interactive lisp session.
*	Returns nil.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *ginalisp_function(NODEZ *head, int *exitflag)
#else
NODEZ *ginalisp_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    gina_lisp(NULL, 0);
    return(NULL);
}



/************************************************************************
*
* great_function() - Given two numerical expressions, evaluate them and
*	return true if the first is greater than the second.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *great_function(NODEZ *head, int *exitflag)
#else
NODEZ *great_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr1, *ptr2;

	/* evaluate the two expressions */
    hptr = head->value.ptr.cdr;
    ptr1 = evaluate(hptr->value.ptr.car, exitflag);
    hptr = hptr->value.ptr.cdr;
    ptr2 = evaluate(hptr->value.ptr.car, exitflag);

	/* if either is nil, return nil */
    if ((ptr1 == NULL) || (ptr2 == NULL)) {
	func_warning("(> ...) requires two number arguments", NULL, head);
	return (NULL);
    }

	/* if both are numbers and the first is > the second, return true */
    if ((ptr1->type == TYPEnumber) && (ptr2->type == TYPEnumber)) {
	if (ptr1->value.number > ptr2->value.number)
	    return (truebind->value.ptr.cdr);
    }
    else
	func_warning("(> ...) requires two number arguments", NULL, head);

	/* default to returning nil */
    return (NULL);
}



/************************************************************************
*
* greateq_function() - Given two numerical expressions, evaluate them and
*	return true if the first is greater than or equal to the second.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *greateq_function(NODEZ *head, int *exitflag)
#else
NODEZ *greateq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr1, *ptr2;

	/* evaluate the two expressions */
    hptr = head->value.ptr.cdr;
    ptr1 = evaluate(hptr->value.ptr.car, exitflag);
    hptr = hptr->value.ptr.cdr;
    ptr2 = evaluate(hptr->value.ptr.car, exitflag);

	/* if either is nil, return nil */
    if ((ptr1 == NULL) || (ptr2 == NULL)) {
	func_warning("(>= ...) requires two number arguments", NULL, head);
	return (NULL);
    }

	/* if both are numbers and the first is >= the second, return true */
    if ((ptr1->type == TYPEnumber) && (ptr2->type == TYPEnumber)) {
	if (ptr1->value.number >= ptr2->value.number)
	    return (truebind->value.ptr.cdr);
    }
    else
	func_warning("(>= ...) requires two number arguments", NULL, head);

	/* default to returning nil */
    return (NULL);
}



/************************************************************************
*
* hasprop_function() - Given a pointer to an object and the name of a
*	property, return the property name if the object has this property,
*	or inherits it from one of its superclasses.  Otherwise return nil.
*
* (hasprop <object> <propname>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *hasprop_function(NODEZ *head, int *exitflag)
#else
NODEZ *hasprop_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr, *proptr;
    OBJECT *obptr;

	/* get the pointer to the object, return nil if bad pointer */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(hasprop ...): first argument must be an object pointer",
		NULL, head);
	return (NULL);
    }

	/* get the name of the property, return nil if bad prop name */
    hptr = hptr->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) || (ptr->type != TYPEidname)) {
	func_warning("(hasprop ...): not given a valid property name",
		NULL, head);
	return (NULL);
    }

	/* check to see if the property can be found,
	 * if not return nil */
    proptr = find_property(obptr, ptr->value.idptr);
    if ((proptr == NULL) || (proptr->type != TYPElisthead))
	return (NULL);

	/* otherwise property was found, so return the property's name */
    return (proptr->value.ptr.car);
}



/************************************************************************
*
* if_function() - Given a condition-expression, a then-expression, and
*	an optional else-expression, evaluate the condition-expression.
*	If this is non-nil, evaluate the then-expression.
*	If this is nil, evaluate the else-condition if given, otherwise
*	return just nil.
*
* (if <condition-expression> <then-expression> <optional else-expression>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *if_function(NODEZ *head, int *exitflag)
#else
NODEZ *if_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *thenptr, *elseptr, *rslt;

	/* evaluate the condition */
    hptr = head->value.ptr.cdr;
    rslt = evaluate(hptr->value.ptr.car, exitflag);

	/* get the <then> expression */
    hptr = hptr->value.ptr.cdr;
    thenptr = hptr->value.ptr.car;

	/* find the <else> expression, if one is given */
    if (hptr->value.ptr.cdr != NULL) {
	hptr = hptr->value.ptr.cdr;
	elseptr = hptr->value.ptr.car;
    }
    else
	elseptr = NULL;

	/* if the condition is non-nil, evaluate the <then> expression */
    if (rslt != NULL)
	return (evaluate(thenptr, exitflag));

	/* otherwise, evaluate the <else> expression if one was given */
    else if (elseptr != NULL)
	return (evaluate(elseptr, exitflag));

    return (rslt);
}



/************************************************************************
*
* killdaemon_function() - Kill the currently running daemon.  Only
*	effective when a daemon is running.  Otherwise it acts the
*	same as (exit).
*
************************************************************************/

#ifdef _ANSI_
NODEZ *killdaemon_function(NODEZ *head, int *exitflag)
#else
NODEZ *killdaemon_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
	/* set the exitflag to kill the currently running daemon */
    killthisdaemon = 1;
    *exitflag = 1;

    return (NULL);
}



/************************************************************************
*
* less_function() - Given two numerical expressions, evaluate them and
*	return true if the first is less than the second.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *less_function(NODEZ *head, int *exitflag)
#else
NODEZ *less_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr1, *ptr2;

	/* evaluate the two expressions */
    hptr = head->value.ptr.cdr;
    ptr1 = evaluate(hptr->value.ptr.car, exitflag);
    hptr = hptr->value.ptr.cdr;
    ptr2 = evaluate(hptr->value.ptr.car, exitflag);

	/* if either is nil, return nil */
    if ((ptr1 == NULL) || (ptr2 == NULL)) {
	func_warning("(< ...) requires two numerical arguments", NULL, head);
	return (NULL);
    }

	/* if both are numbers and the first is < the second, return true */
    if ((ptr1->type == TYPEnumber) && (ptr2->type == TYPEnumber)) {
	if (ptr1->value.number < ptr2->value.number)
	    return (truebind->value.ptr.cdr);
    }
    else
	func_warning("(< ...) requires two numerical arguments", NULL, head);

	/* default to returning nil */
    return (NULL);
}



/************************************************************************
*
* lesseq_function() - Given two numerical expressions, evaluate them and
*	return true if the first is less than or equal to the second.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *lesseq_function(NODEZ *head, int *exitflag)
#else
NODEZ *lesseq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr1, *ptr2;

	/* evaluate the two expressions */
    hptr = head->value.ptr.cdr;
    ptr1 = evaluate(hptr->value.ptr.car, exitflag);
    hptr = hptr->value.ptr.cdr;
    ptr2 = evaluate(hptr->value.ptr.car, exitflag);

	/* if either is nil, return nil */
    if ((ptr1 == NULL) || (ptr2 == NULL)) {
	func_warning("(< ...) requires two numerical arguments", NULL, head);
	return (NULL);
    }

	/* if both are numbers and the first is <= the second, return true */
    if ((ptr1->type == TYPEnumber) && (ptr2->type == TYPEnumber)) {
	if (ptr1->value.number <= ptr2->value.number)
	    return (truebind->value.ptr.cdr);
    }
    else
	func_warning("(< ...) requires two numerical arguments", NULL, head);

	/* default to returning nil */
    return (NULL);
}



/************************************************************************
*
* length_function() - Given a list, return the number of elements in the
*	list.  If the argument does not evaluate to a list, it returns a
*	value of 1.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *length_function(NODEZ *head, int *exitflag)
#else
NODEZ *length_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    int   count;
    NODEZ *hptr, *ptr;

	/* evaluate the expression */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    count = 0;

	/* loop through all of the elements in the list */
    while (ptr != NULL) {
	++count;
	if (ptr->type != TYPElisthead)
	    break;
	ptr = ptr->value.ptr.cdr;
    }

	/* create a new node with the count of elements found */
    ptr = new_node();
    ptr->type = TYPEnumber;
    ptr->value.number = count;

	/* return the count */
    return (ptr);
}



/************************************************************************
*
* let_function() - Add more variables to the current scope, and evaluate
*	one or more expressions which use these variables.  Returns the
*	value of the last expression evaluated within the (let ...).
*	The effects of an (exit) function are trapped.
*
* (let (<var-dec 1> ... <var-dec m>) <expression 1> ... <expression n>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *let_function(NODEZ *head, int *exitflag)
#else
NODEZ *let_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *scopeptr, *lastnew, *bindlist, *bindptr, *varptr, *ptr;

	/* clear the exitflag, and set the bindlist pointer to the
	   list of new variable bindings */
    *exitflag = 0;
    hptr = head->value.ptr.cdr;
    bindlist = hptr->value.ptr.car;
    scopeptr = NULL;
    lastnew = NULL;

	/* loop through all of the new variable declarations, adding them
	   to the local scope */
    while (bindlist != NULL) {

	    /* abort if the list structure is invalid */
	if (bindlist->type != TYPElisthead) {
	    func_warning("(let ...): invalid list of variables", NULL, head);
	    break;
	}

	    /* set pointers to next entry in list */
	bindptr = bindlist->value.ptr.car;
	bindlist = bindlist->value.ptr.cdr;

	    /* if the current binding is a good entry format,
		create a new local binding */
	if ((bindptr != NULL) && (bindptr->type == TYPElisthead)) {

		/* get pointer to name of variable, exit if not a symbol */
	    varptr = bindptr->value.ptr.car;
	    if ((varptr == NULL) || (varptr->type != TYPEidname)) {
		func_warning("(let ...): invalid variable name", NULL, head);
		continue;
	    }

		/* look at expression with value for the variable, halt if
		   not a good expression list (i.e. in case a dotted pair) */
	    bindptr = bindptr->value.ptr.cdr;
	    if ((bindptr != NULL) && (bindptr->type != TYPElisthead)) {
		func_warning("(let ...): invalid variable binding", NULL,head);
		continue;
	    }

		/* evaluate the value of the expression, and assign its
		   result as the initial value of the binding */
	    if (bindptr == NULL)
		ptr = NULL;
	    else
		ptr = evaluate(bindptr->value.ptr.car, exitflag);
	    scopeptr = assign_binding(varptr->value.idptr, ptr, scopeptr);

		/* if this is the first binding being added to the local
		   scope, then record the pointer to it, since it will be
		   the last one to remove when these variables are removed
		   from the local scope */
	    if (lastnew == NULL)
		lastnew = scopeptr;
	}
	else if (bindptr != NULL)
	    func_warning("(let ...): invalid variable declaration",
			NULL, head);
    }

	/* insert the list of new bindings into the local binding scope */
    if (lastnew) {
	lastnew->value.ptr.cdr = locals->value.ptr.cdr;
	locals->value.ptr.cdr = scopeptr;
    }

	/* evaluate all of the expressions, until either all have been
	   evaluated, or until one of them set the exitflag */
    ptr = NULL;
    hptr = hptr->value.ptr.cdr;
    while ((hptr != NULL) && (*exitflag == 0)) {
	if (hptr->type != TYPElisthead) {
	    func_warning("(let ...): invalid list of expressions", NULL, head);
	    break;
	}
	ptr = evaluate(hptr->value.ptr.car, exitflag);
	hptr = hptr->value.ptr.cdr;
    }
    *exitflag = 0;

	/* remove the new bindings from the local scope */
    if (lastnew)
	locals->value.ptr.cdr = lastnew->value.ptr.cdr;

	/* return the result of the last expression evaluated */
    return (ptr);
}



/************************************************************************
*
* list_function() - Given one or more expressions, evaluate each of them
*	and stick their results into one list.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *list_function(NODEZ *head, int *exitflag)
#else
NODEZ *list_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *rslt, *ptr;

	/* define the initial list to be nil,
	   and set head to the first expression given */
    rslt = NULL;
    hptr = head->value.ptr.cdr;

	/* loop through all of the expressions */
    while (hptr != NULL) {

	    /* if this is the first expression, make it the root of the list,
		otherwise append the new node to the end of the list */
	if (rslt == NULL)
	    ptr = rslt = new_node();
	else {
	    ptr->value.ptr.cdr = new_node();
	    ptr = ptr->value.ptr.cdr;
	}

	    /* evaluate the expression, and make its result the child of
		the newly created node */
	ptr->type = TYPElisthead;
	ptr->value.ptr.car = evaluate(hptr->value.ptr.car, exitflag);
	hptr = hptr->value.ptr.cdr;
    }

	/* return the pointer to the new list */
    return (rslt);
}



/************************************************************************
*
* listq_function() - Given an expression, return true if it is a list
*	(nils are considered lists), nil if not a list.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *listq_function(NODEZ *head, int *exitflag)
#else
NODEZ *listq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* evaluate the list */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* if the result is nil or a list, return true */
    if ((ptr == NULL) || (ptr->type == TYPElisthead))
	return (truebind->value.ptr.cdr);

	/* otherwise return nil */
    return (NULL);
}



/************************************************************************
*
* load_function() - Given the name of a file, open that file and parse
*	its contents as regular LISP code.  Returns true if it was at
*	least able to open the file and start parsing.  Returns nil if
*	not given a filename.
*
* (load "filename")
*
************************************************************************/

#ifdef _ANSI_
NODEZ *load_function(NODEZ *head, int *exitflag)
#else
NODEZ *load_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* get the name of the file from which to load */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* if not a string, abort */
    if ((ptr == NULL) || (ptr->type != TYPEstring)) {
	func_warning("(load ...) requires a string as first argument",
		NULL, head);
	return (NULL);
    }

	/* load the file */
    gina_lisp(ptr->value.idptr->name, 0);

	/* return true */
    return (truebind->value.ptr.cdr);
}



/************************************************************************
*
* loadobjects_function() - Given an object and a filename, open the file
*	and load all of the objects contained with in as children of the
*	specified object.  Returns true it was at least able to try and
*	load the file.  If the filename is bad, it returns nil.  If
*	the specified object pointer is nil, it dumps objects without
*	parents into the top level of the object tree.
*
* (loadobjects <objectptr> "filename")
*
************************************************************************/

#ifdef _ANSI_
NODEZ *loadobjects_function(NODEZ *head, int *exitflag)
#else
NODEZ *loadobjects_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr;
    OBJECT *obptr, *oldspec;

	/* get pointer to object for scope of loading */
    hptr = head->value.ptr.cdr;
    obptr = good_ob_ptr(evaluate(hptr->value.ptr.car, exitflag));

	/* get name of file from which to load */
    hptr = hptr->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* abort if the second argument is not a string */
    if ((ptr == NULL) || (ptr->type != TYPEstring)) {
	func_warning("(loadobjects ...) requires string as second argument",
		NULL, head);
	return (NULL);
    }

	/* set scope for (object...) to given object pointer */
    oldspec = obspecroot;
    if (obptr != NULL)
	obspecroot = obptr;

	/* load the file, then reset the scope of obspecroot */
    gina_lisp(ptr->value.idptr->name, 0);
    obspecroot = oldspec;

    return (truebind->value.ptr.cdr);
}



/************************************************************************
*
* loop_function() - Given one or more expressions, repeatedly evaluate
*	them until stopped by an (exit) or similar function.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *loop_function(NODEZ *head, int *exitflag)
#else
NODEZ *loop_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *rslt, *funcptr;

	/* set head as the first expression, and default rslt to nil */
    hptr = head->value.ptr.cdr;
    rslt = NULL;
    *exitflag = 0;

	/* loop until exitflag is set by (exit) or a similar function */
    while (*exitflag == 0) {

	    /* loop through all of the expressions in the loop */
	funcptr = hptr;
	while (funcptr != NULL) {
	    rslt = evaluate(funcptr->value.ptr.car, exitflag);

		/* if exitflag is set, then return the current result */
	    if (*exitflag) {
		*exitflag = 0;
		return (rslt);
	    }

	    funcptr = funcptr->value.ptr.cdr;
	}
    }

	/* this point should never be reached, but just to be safe... */
    return (rslt);
}



/************************************************************************
*
* member_function() - Given two expressions, evaluate both of them, and
*	returns the remaining portion of the second result if the first
*	result appears within the second.  This function only checks the
*	first layer of the second argument.
************************************************************************/

#ifdef _ANSI_
NODEZ *member_function(NODEZ *head, int *exitflag)
#else
NODEZ *member_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *valptr, *listptr;

	/* evaluate the two arguments */
    hptr = head->value.ptr.cdr;
    valptr = evaluate(hptr->value.ptr.car, exitflag);
    hptr = hptr->value.ptr.cdr;
    listptr = evaluate(hptr->value.ptr.car, exitflag);

	/* if the second is nil, don't bother to continue */
    if (listptr == NULL)
	return (NULL);

	/* if the second is not a list, complain and stop */
    if (listptr->type != TYPElisthead) {
	func_warning("(member ...): second argument must be a list",
		NULL, head);
	return (NULL);
    }

	/* search through each element in the list, and if the value
	   appears within it, return the remaining portion of the
	   second list */
    while (listptr != NULL) {
	if (listptr->type != TYPElisthead)
	    return (NULL);

	if (is_equal_tree(valptr, listptr->value.ptr.car))
	    return (listptr);

	listptr = listptr->value.ptr.cdr;
    }

    return (NULL);
}



/************************************************************************
*
* method_function() - Invoke a method on a specified object.  Returns the
*	value of the invoked method, or nil if unable to invoke the method.
*
* (method <object> <methodname> ...args...)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *method_function(NODEZ *head, int *exitflag)
#else
NODEZ *method_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr, *newself, *result, *methodptr;
    OBJECT *obptr;

	/* obtain the pointer to the object upon which to apply the method */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(method ...): first argument must be an object pointer",
		NULL, head);
	return (NULL);
    }

	/* get the name of the method to apply */
    hptr = hptr->value.ptr.cdr;
    ptr = hptr->value.ptr.car;
    if ((ptr == NULL) || (ptr->type != TYPEidname)) {
	func_warning("(method ...): second argument must be a method name",
		NULL, head);
	return (NULL);
    }

	/* find the method for this object; abort if no such method */
    methodptr = find_method(obptr, ptr->value.idptr);
    if (methodptr == NULL) {
	func_warning("(method ...): given method not found", NULL, head);
	return (NULL);
    }

	/* create binding with pointer to new object */
    newself = obj_to_node(obptr);

	/* evaluate the method */
    result = eval_user_func(hptr, methodptr, newself);

	/* return the result of the method */
    return (result);
}



/************************************************************************
*
* mod_function() - Given two or more arguments, evaluate each in turn,
*	taking the first and returning its value modulo all successive
*	values.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *mod_function(NODEZ *head, int *exitflag)
#else
NODEZ *mod_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    int   val;
    NODEZ *hptr, *ptr;


	/* evaluate the first argument, and use its value as the base which
	   will be modulus the values of the successive arguments */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    val = 0;
    if ((ptr != NULL) && (ptr->type == TYPEnumber))
	val = ptr->value.number;
    else
	func_warning("(mod ...): requires numbers as arguments", NULL, head);

	/* evaluate all remaining arguments, each time making the new
	   result equal to the previous result mod the current argument's
	   value */
    hptr = hptr->value.ptr.cdr;
    while (hptr != NULL) {
	ptr = evaluate(hptr->value.ptr.car, exitflag);
	if ((ptr != NULL) && (ptr->type == TYPEnumber)) {
	    if (ptr->value.number != 0)
		val %= ptr->value.number;
	    else
		func_warning("(mod ...): modulo by zero is not valid",
				NULL, head);
	}
	else
	    func_warning("(mod ...): requires numbers as arguments",
				NULL, head);

	hptr = hptr->value.ptr.cdr;
    }

	/* create a new node, into which can place the result to return */
    ptr = new_node();
    ptr->type = TYPEnumber;
    ptr->value.number = val;

	/* return the result */
    return (ptr);
}



/************************************************************************
*
* move_function() - Given two objects, make the second one the parent of
*	the first one.  Returns a pointer to the new parent, or nil if
*	if failed.
*
* (move <object> <newparentobject>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *move_function(NODEZ *head, int *exitflag)
#else
NODEZ *move_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr;
    OBJECT *obptr, *newparptr, *sibptr, *parptr;

	/* get the pointer to the first object, abort if invalid pointer */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(move ...) requires object pointer as first argument",
		NULL, head);
	return (NULL);
    }

	/* get the pointer to the second object, abort if invalid */
    hptr = hptr->value.ptr.cdr;
    if ((newparptr =good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag)))==NULL){
	func_warning("(move ...) requires object pointer as second argument",
		NULL, head);
	return (NULL);
    }

	/* if object a parent, remove it from it's parent's list of children */
    parptr = obptr->parent;
    if (parptr != NULL) {

	    /* remove object from its parent's list of children */
	if (parptr->child == obptr)
	    parptr->child = obptr->sibling;
	else {
	    sibptr = parptr->child;
	    while ((sibptr != NULL) && (sibptr->sibling != obptr))
		sibptr = sibptr->sibling;

	    if (sibptr == NULL) {
		printf("Fatal Error: object not child of parent\n");
		exit(1);
	    }

	    sibptr->sibling = obptr->sibling;
	}
    }

	/* make object into first child of new parent */
    obptr->parent = newparptr;
    obptr->sibling = newparptr->child;
    newparptr->child = obptr;

	/* return pointer to new parent */
    return (obj_to_node(newparptr));
}



/************************************************************************
*
* mul_function() - Given two or more arguments, evaluate them and
*	multiply their results together, ignoring any that are not
*	numbers, and returning the resulting product.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *mul_function(NODEZ *head, int *exitflag)
#else
NODEZ *mul_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    int   val;
    NODEZ *hptr, *ptr;

	/* look at the first argument and set the base product to one */
    hptr = head->value.ptr.cdr;
    val = 1;

	/* loop through all arguments, evaluate them, and if they are
	   numbers multiply them to the current product in val */
    while (hptr != NULL) {
	ptr = evaluate(hptr->value.ptr.car, exitflag);
	if ((ptr != NULL) && (ptr->type == TYPEnumber))
	    val *= ptr->value.number;
	else
	    func_warning("(mul ...) requires numbers as arguments", NULL,head);

	hptr = hptr->value.ptr.cdr;
    }

	/* create a new node to stick the result into, and return it */
    ptr = new_node();
    ptr->type = TYPEnumber;
    ptr->value.number = val;

    return (ptr);
}



/************************************************************************
*
* nequalq_function() - Given two expressions, evaluate them and return
*	true if they are not equal
*
************************************************************************/

#ifdef _ANSI_
NODEZ *nequalq_function(NODEZ *head, int *exitflag)
#else
NODEZ *nequalq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr1, *ptr2;

	/* evaluate the two expressions */
    hptr = head->value.ptr.cdr;
    ptr1 = evaluate(hptr->value.ptr.car, exitflag);
    hptr = hptr->value.ptr.cdr;
    ptr2 = evaluate(hptr->value.ptr.car, exitflag);

	/* return true if they are not equal */
    if (!is_equal_tree(ptr1, ptr2))
	return (truebind->value.ptr.cdr);

    return (NULL);
}



/************************************************************************
*
* nth_function() - Given a list and an index into that list, return the
*	nth element of the list.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *nth_function(NODEZ *head, int *exitflag)
#else
NODEZ *nth_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    int   number;
    NODEZ *hptr, *ptr;

	/* get the index of the element to look up */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) || (ptr->type != TYPEnumber)) {
	func_warning("(nth ...) takes number as first argument", NULL, head);
	return (NULL);
    }

	/* verify that it is a valid number */
    number = ptr->value.number;
    if (number < 0) {
	func_warning("(nth ...) given a negative index", NULL, head);
	return (NULL);
    }

	/* get the list in which to search */
    hptr = hptr->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) || (ptr->type != TYPElisthead)) {
	func_warning("(nth ...) takes list as second argument", NULL, head);
	return (NULL);
    }

	/* iteratively search the list for the desired element */
    for (;;) {
	if (ptr == NULL)
	    break;
	else if (number < 1) {
	    if (ptr->type == TYPElisthead)
		return (ptr->value.ptr.car);
	    else
		return (ptr);
	}
	else if (ptr->type != TYPElisthead)
	    break;
	else {
	    ptr = ptr->value.ptr.cdr;
	    --number;
	}
    }

    return (NULL);
}



/************************************************************************
*
* nullq_function() - Given an expression, evaluate it and return true
*	if it evaluates to nil.  Otherwise, return nil.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *nullq_function(NODEZ *head, int *exitflag)
#else
NODEZ *nullq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr;

	/* evaluate the expression, and return true if it is nil */
    hptr = head->value.ptr.cdr;
    if (evaluate(hptr->value.ptr.car, exitflag) == NULL)
	return (truebind->value.ptr.cdr);

	/* otherwise, return nil */
    return (NULL);
}



/************************************************************************
*
* numberq_function() - Given an expression, return true if it evaluates
*	to a number.  Otherwise, return false.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *numberq_function(NODEZ *head, int *exitflag)
#else
NODEZ *numberq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* evaluate the function */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* return true if it is a number */
    if ((ptr != NULL) && (ptr->type == TYPEnumber))
	return (truebind->value.ptr.cdr);

	/* otherwise it is not a number, so return nil */
    return (NULL);
}



/************************************************************************
*
* object_function() - Create a new object.
*
* (object <objectname> <parentname> <class list> <property list> <method list>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *object_function(NODEZ *head, int *exitflag)
#else
NODEZ *object_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *nptr, *nameptr, *root;
    OBJECT *obptr, *parptr;

    root = head;

	/* get name of object to create */
    hptr = head->value.ptr.cdr;
    nameptr = hptr->value.ptr.car;
    if ((nameptr == NULL) || (nameptr->type != TYPEidname)) {
	func_warning("(object ...): first argument must be an identifier",
		NULL, head);
	return (NULL);
    }

	/* get pointer to parent */
    hptr = hptr->value.ptr.cdr;
    nptr = hptr->value.ptr.car;
    if (nptr != NULL) {
	if (nptr->type != TYPEidname) {
	    func_warning("(object ...): second argument must be name of parent",
			NULL, head);
	    return (NULL);
	}

	if ((parptr = find_object(nptr->value.idptr, obspecroot)) == NULL) {
	    func_warning("(object ...): parent object %s not found",
			nptr->value.idptr->name, head);
	    return (NULL);
	}
    }

	/* create the object and insert it into the object tree */
    obptr = new_object();
    obptr->idptr = nameptr->value.idptr;
    if (nptr != NULL) {
	obptr->sibling = parptr->child;
	parptr->child = obptr;
	obptr->parent = parptr;
    }
    else {
	obptr->sibling = obspecroot->child;
	obspecroot->child = obptr;
	obptr->parent = obspecroot;
    }

	/* parse through the list of classes, properties, and methods */
    hptr = hptr->value.ptr.cdr;
    obptr->classes = make_class_list(hptr->value.ptr.car);
    hptr = hptr->value.ptr.cdr;
    obptr->properties = hptr->value.ptr.car;
    hptr = hptr->value.ptr.cdr;
    obptr->methods = hptr->value.ptr.car;

	/* return the name of the new object */
    return (nameptr);
}



/************************************************************************
*
* (objectname <objptr>)
* returns the name of the object, or nil if not an object
*
************************************************************************/

#ifdef _ANSI_
NODEZ *objectname_function(NODEZ *head, int *exitflag)
#else
NODEZ *objectname_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr;
    OBJECT *obptr;

	/* evaluate the expression, and return nil if it is not a good
	   object pointer */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(objectname ...): argument must be an object pointer",
		NULL, head);
	return (NULL);
    }

    ptr = new_node();
    ptr->type = TYPEidname;
    ptr->value.idptr = obptr->idptr;

	/* otherwise it is a good object pointer, so return the name */
    return (ptr);
}



/************************************************************************
*
* (object? <expr>)
* returns true of the expression evaluates to an object pointer
*
************************************************************************/

#ifdef _ANSI_
NODEZ *objectq_function(NODEZ *head, int *exitflag)
#else
NODEZ *objectq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr;
    OBJECT *obptr;

	/* evaluate the expression, and return nil if it is not a good
	   object pointer */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car, exitflag))) == NULL)
	return (NULL);

	/* otherwise it is a good object pointer, so return true */
    return (truebind->value.ptr.cdr);
}



/************************************************************************
*
* or_function() - Perform a logical-or on a sequence of one or more
*	expressions.  When reach an expression that evaluates to true
*	(non-nil), halt and return the value of the expression.  If none
*	of the expression evaluate to true, return nil.
*	If the exitflag is set, then no more expressions will be
*	evaluated, and the result of the current expression is returned
*	and exitflag propagates upwards.
*
* (or <expression 1> ... <expression n>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *or_function(NODEZ *head, int *exitflag)
#else
NODEZ *or_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* loop through all of the expressions;
	   when reach one that is evaluates to non-nil, halt and return
	   its value */
    hptr = head->value.ptr.cdr;
    while (hptr != NULL) {
	ptr = evaluate(hptr->value.ptr.car, exitflag);

	    /* if the expression is non-nil, or exitflag is set,
		then return the result of the expression */
	if ((ptr != NULL) || *exitflag)
	    return (ptr);

	hptr = hptr->value.ptr.cdr;
    }

	/* if reach here, then all of the expression evaluate to nil,
	   so return nil as the result */
    return (NULL);
}



/************************************************************************
*
* parent_function() - This function return a pointer to the parent object
*	of the given object pointer.
*
* (parent (findobject 'gadget))  ==>  widget
* provided widget is the parent of gadget
*
************************************************************************/

#ifdef _ANSI_
NODEZ *parent_function(NODEZ *head, int *exitflag)
#else
NODEZ *parent_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr;
    OBJECT *obptr;

    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(parent ...): requires an object pointer as argument",
		NULL, head);
	return (NULL);
    }

    return (obj_to_node(obptr->parent));
}



/************************************************************************
*
* parser_function() - Calls the built-in parser.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *parser_function(NODEZ *head, int *exitflag)
#else
NODEZ *parser_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr;
    OBJECT *obptr;

	/* get the object pointer for the given actor object */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* make certain it is a valid object pointer */
    if (ptr->type != TYPEobject) {
	func_warning("(parser...) takes object pointer as first argument",
		NULL, head);
	return (NULL);
    }
    else
	obptr = ptr->value.objptr;

	/* get the list of strings */
    hptr = hptr->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* make certain it is a valid list */
    if ((ptr != NULL) && (ptr->type != TYPElisthead)) {
	func_warning("(parser...) takes a list as second argument", NULL,head);
	return (NULL);
    }

    return (parser(obptr, ptr));
}



/************************************************************************
*
* pluralof_function() - Given two strings, return true if the first is
*	a plural of the second.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *pluralof_function(NODEZ *head, int *exitflag)
#else
NODEZ *pluralof_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    char  *s1ptr, *s2ptr;
    NODEZ *hptr, *ptr;

	/* get the first string */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) || (ptr->type != TYPEstring)) {
	func_warning("(pluralof ...): first argument is not a string",
		NULL, head);
	return (NULL);
    }

	/* get the second string */
    s1ptr = ptr->value.idptr->name;
    hptr = hptr->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) || (ptr->type != TYPEstring)) {
	func_warning("(pluralof ...): second argument is not a string",
		NULL, head);
	return (NULL);
    }

    s2ptr = ptr->value.idptr->name;

	/* return true if first string is a plural of the second */
    if (match_plurality(s2ptr, s1ptr))
	return(truebind->value.ptr.cdr);

	/* otherwise it is not, so return null */
    return (NULL);
}



/************************************************************************
*
* pprinttree_function() - This is supposed to be a pretty-print function,
*	but has not yet been implemented.  Currently it behaves the same
*	as (printtree ...)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *pprinttree_function(NODEZ *head, int *exitflag)
#else
NODEZ *pprinttree_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* evaluate the expression, print out its value, and then return
	   the value as the result */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    pretty_print_tree(ptr, 0, 0);

    return (ptr);
}



/************************************************************************
*
* print_function() - Given a list of arguments, evaluate each in turn
*	and print them to the screen, stripping the quotes off strings.
*	Return the value of the last expression in the list.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *print_function(NODEZ *head, int *exitflag)
#else
NODEZ *print_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* set to the first argument, and initialize the return value to nil */
    hptr = head->value.ptr.cdr;
    ptr = NULL;

	/* loop through all arguments, evaluating and printing each in turn */
    while (hptr != NULL) {

	    /* if the node is a list node, evaluate its car;
		otherwise there is a dotted pair here, so evaluate it and
		set the head to NULL to halt the loop */
	if (hptr->type == TYPElisthead) {
	    ptr = evaluate(hptr->value.ptr.car, exitflag);
	    hptr = hptr->value.ptr.cdr;
	}
	else {
	    ptr = evaluate(hptr, exitflag);
	    hptr = NULL;
	}

	    /* print out the result */
	screen_print(ptr);
    }

	/* return the last result found */
    return (ptr);
}



/************************************************************************
*
* printtree_function() - Given an expression, evaluate it, print it out,
*	and then return the result.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *printtree_function(NODEZ *head, int *exitflag)
#else
NODEZ *printtree_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* evaluate the expression, print out its value, and then return
	   the value as the result */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    print_tree(ptr, stdout);

    return (ptr);
}



/************************************************************************
*
* quote_function() - Return the given list without evaluating it.
*
* '(a b c)  ==>  (a b c)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *quote_function(NODEZ *head, int *exitflag)
#else
NODEZ *quote_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr;

    hptr = head->value.ptr.cdr;

    return (hptr->value.ptr.car);
}



/************************************************************************
*
* random_function() - Given an expression which evaluates to a number n,
*	return a random number between zero and n-1 inclusive.
*
* (random 10)  ==>  <pseudo-random between zero and 9 inclusive>
*
************************************************************************/

#ifdef _ANSI_
NODEZ *random_function(NODEZ *head, int *exitflag)
#else
NODEZ *random_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr, *newptr;

    newptr = new_node();
    newptr->type = TYPEnumber;
    newptr->value.number = 0;

    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

    if ((ptr == NULL) || (ptr->type != TYPEnumber)) {
	func_warning("(random ...): requires a numerical argument", NULL,head);
	return (newptr);
    }

    newptr->value.number = rand() % ptr->value.number;

    return (newptr);
}



/************************************************************************
*
* read_function() - prompt the user to input a LISP expression, which is
*	evaluated and returned
*
************************************************************************/

#ifdef _ANSI_
NODEZ *read_function(NODEZ *head, int *exitflag)
#else
NODEZ *read_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    get_token();

    return (parse_lisp());
}



/************************************************************************
*
* readsentence_function() - Prompts the user to type in a normal
*	sentence, and converts it into a list expression.
*
* "hello, fred"  ==>  ("hello" "," "fred")
*
************************************************************************/

#ifdef _ANSI_
NODEZ *readsentence_function(NODEZ *head, int *exitflag)
#else
NODEZ *readsentence_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    return(sentence_to_list());
}



/************************************************************************
*
* return() - the return command tells the interpreter to break out of
*	     evaluation of a list of expressions, so it is trapped by
*	     loop, prog, and all user-defined functions
*		a return will be propogated backwards through flow-control
*		expressions (if, cond, and, & or) and predicates (atom,
*		null, listq, numberq, & stringq), for any other built-in
*		function the propogation of a return is problematic
*
************************************************************************/

#ifdef _ANSI_
NODEZ *return_function(NODEZ *head, int *exitflag)
#else
NODEZ *return_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* evaluate the expression */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    *exitflag = 1;

    return (ptr);
}



/************************************************************************
*
* samestringq_function() - Returns true if the two given strings are
*	the same (ignoring case).
*
* (samestring <string1> <string2>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *samestringq_function(NODEZ *head, int *exitflag)
#else
NODEZ *samestringq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *s1, *s2;

    hptr = head->value.ptr.cdr;
    s1 = evaluate(hptr->value.ptr.car, exitflag);

    hptr = hptr->value.ptr.cdr;
    s2 = evaluate(hptr->value.ptr.car, exitflag);

    if ((s1 == NULL) || (s1->type != TYPEstring)
		|| (s2 == NULL) || (s2->type != TYPEstring)) {
	func_warning("(samestring? ...): both arguments must be strings",
		NULL, head);
	return (NULL);
    }

    if (same_string(s1->value.idptr->name, s2->value.idptr->name))
	return (truebind->value.ptr.cdr);

    return (NULL);
}



/************************************************************************
*
* saveobjects_function() - Save the named object and all of its children
*	into the named output file, returning true if successful.
*
* (saveobjects <objectptr> "filename")
*
************************************************************************/

#ifdef _ANSI_
NODEZ *saveobjects_function(NODEZ *head, int *exitflag)
#else
NODEZ *saveobjects_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr;
    OBJECT *obptr;

	/* get pointer to object to save */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(saveobjects ...): invalid object pointer", NULL, head);
	return (NULL);
    }

	/* get name of file to save it into */
    hptr = hptr->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) || (ptr->type != TYPEstring)) {
	func_warning("(saveobjects ...): second argument must be a string",
		NULL, head);
	return (NULL);
    }

	/* save the object */
    save_objects(obptr, ptr->value.idptr->name);

	/* return true */
    return (truebind->value.ptr.cdr);
}



/************************************************************************
*
* set_function() - Given the name of a variable and an expression,
*	evaluate the expression and assign its value to the variable,
*	returning the name of the variable if successful, or nil if not.
*
* (set count 13)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *set_function(NODEZ *head, int *exitflag)
#else
NODEZ *set_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *nameptr, *valueptr, *newptr;

	/* get the name of the variable to change */
    hptr = head->value.ptr.cdr;
    nameptr = hptr->value.ptr.car;
    if ((nameptr == NULL) || (nameptr->type != TYPEidname)) {
	func_warning("(set ...): first argument must be an identifier",
		NULL, head);
	return (NULL);
    }

	/* get the new value for the variable binding */
    hptr = hptr->value.ptr.cdr;
    valueptr = evaluate(hptr->value.ptr.car, exitflag);

	/* add this binding to list of bindings: if the correct scoped
	   binding cannot be found, create a new global binding */
    if ((newptr = find_binding(nameptr->value.idptr)) != NULL)
	newptr->value.ptr.cdr = valueptr;
    else
	assign_global(nameptr->value.idptr, valueptr);

	/* return the name of the variable */
    return (nameptr);
}



/************************************************************************
*
* setprop_function() - Given an object pointer and the name of a property,
*	assign the new property value to this property.  Returns the name
*	of the property if it is successful.
*
* (setprop <object> <propname> <propvalue>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *setprop_function(NODEZ *head, int *exitflag)
#else
NODEZ *setprop_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr, *nameptr, *valptr, *bindptr;
    OBJECT *obptr;

	/* get the pointer to the object, return nil if no such object */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(setprop ...): first argument must be an object pointer",
		NULL, head);
	return (NULL);
    }

	/* get the name of the property, return nil if invalid name */
    hptr = hptr->value.ptr.cdr;
    nameptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((nameptr == NULL) || (nameptr->type != TYPEidname)) {
	func_warning("(setprop ...): invalid property name", NULL, head);
	return (NULL);
    }

	/* get the value to assign to the property */
    hptr = hptr->value.ptr.cdr;
    valptr = evaluate(hptr->value.ptr.car, exitflag);

	/* if the object does not have this property, then create a new
	   property binding on the object and assign it the value,
	   otherwise just reset the value of the property to the new value */
    bindptr = find_prop_rec(obptr, nameptr->value.idptr);
    if (bindptr == NULL) {
	bindptr = constructor(nameptr, valptr);
	obptr->properties = constructor(bindptr, obptr->properties);
    }
    else
	bindptr->value.ptr.cdr = valptr;

	/* return the name of the property */
    return (nameptr);
}



/************************************************************************
*
* showaction_function() - Given the name of an action, return a list
*	structure showing the definition of that action.  If no such
*	action exists, return nil
*
* (showaction 'actionname)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *showaction_function(NODEZ *head, int *exitflag)
#else
NODEZ *showaction_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    ACTION *actptr;
    NODEZ  *hptr, *ptr;

	/* get the name of the action to show */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) | (ptr->type != TYPEidname)) {
	func_warning("(showaction ...): invalid action name", NULL, head);
	return (NULL);
    }

	/* if no such action exists, return nil */
    actptr = find_action(ptr->value.idptr);
    if (actptr == NULL)
	return (NULL);

	/* the action exists, so assemble a list structure and return that */
    return (assemble_action(actptr));
}



/************************************************************************
*
* showclass_function() - Given the name of a class, return a list
*	struction showing the definition of the class.  Returns nil
*	if no such class exists.
*
* (showclass 'classname)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *showclass_function(NODEZ *head, int *exitflag)
#else
NODEZ *showclass_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    OBJECT *clptr;
    NODEZ  *hptr, *ptr;

	/* get the name of the class to show */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) | (ptr->type != TYPEidname)) {
	func_warning("(showclass ...): invalid class name", NULL, head);
	return (NULL);
    }

	/* if no such class, return a nil pointer */
    clptr = find_class(ptr->value.idptr);
    if (clptr == NULL)
	return (NULL);

	/* otherwise class exists, so assemble a list structure and
	   return that */
    return (assemble_object(clptr, 0));
}



/************************************************************************
*
* showfunc_function() - Given the name of a function, returns a list
*	structure showing the definition of the function.  Returns nil
*	if no such function exists.
*
* (showfunc 'function-name)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *showfunc_function(NODEZ *head, int *exitflag)
#else
NODEZ *showfunc_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr, *rslt;

	/* get the name of the function to show */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* if the function exists, get a pointer to its definition,
	   otherwise complain and set up a nil pointer */
    if ((ptr != NULL) && (ptr->type == TYPEidname))
	rslt = find_function(ptr->value.idptr);
    else {
	func_warning("(showfunc ...): invalid function name", NULL, head);
	rslt = NULL;
    }

	/* return the pointer that was found */
    return (rslt);
}



/************************************************************************
*
* showobject_function() - Given a pointer to an object, return a list
*	structure that is similar to the one used to define the object
*	with the (object ...) function.  Returns nil if no such object
*	exists.
*
* (showobject <object>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *showobject_function(NODEZ *head, int *exitflag)
#else
NODEZ *showobject_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr;
    OBJECT *obptr;

	/* get the pointer to the object; return nil of no such object
	   can be found */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(showobject ...): invalid object pointer", NULL, head);
	return (NULL);
    }

	/* successful: return the object list struction */
    return (assemble_object(obptr, 1));
}



/************************************************************************
*
* sibling_function() - Given a pointer to an object, return the sibling
*	of that object.
*
* (sibling <object>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *sibling_function(NODEZ *head, int *exitflag)
#else
NODEZ *sibling_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr;
    OBJECT *obptr;

    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(sibling ...): requires an object pointer as argument",
		NULL, head);
	return (NULL);
    }

    return (obj_to_node(obptr->sibling));
}



/************************************************************************
*
* spawn_function() - Spawn off a daemon to run in the background.  The
*	daemon may be attached to an existing object, or may be independent
*	of all objects.
*
* (spawn <object pointer> <daemon-name> <arg 1> ... <arg n>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *spawn_function(NODEZ *head, int *exitflag)
#else
NODEZ *spawn_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    DAEMON *daeptr;
    NODEZ  *hptr, *nameptr, *fptr, *vptr;
    OBJECT *objptr;

	/* get pointer to object daemon is spawned on, if any */
    hptr = head->value.ptr.cdr;
    objptr = good_ob_ptr(evaluate(hptr->value.ptr.car, exitflag));

	/* get name of daemon being spawned */
    hptr = hptr->value.ptr.cdr;
    nameptr = hptr->value.ptr.car;
    if ((nameptr == NULL) || (nameptr->type != TYPEidname)) {
	func_warning("(spawn ...): invalid daemon/function name", NULL, head);
	return (NULL);
    }

	/* find the pointer to the daemon being spawned */
    if ((fptr = find_function(nameptr->value.idptr)) == NULL) {
	func_warning("(spawn ...): daemon %s is not defined",
		nameptr->value.idptr->name, head);
	return (NULL);
    }

	/* set pointer to list of variables for function */
    vptr = fptr->value.ptr.cdr->value.ptr.car;

	/* move hptr to point to the list of arguments for the daemon,
	   if any were given */
    hptr = hptr->value.ptr.cdr;

	/* create the new daemon */
    daeptr = new_daemon();
    daeptr->idptr = nameptr->value.idptr;
    daeptr->variables = eval_binding_list(hptr, vptr, head);
    daeptr->objectptr = objptr;

    return (truebind->value.ptr.cdr);
}



/************************************************************************
************************************************************************/

#ifdef _ANSI_
NODEZ *str2sym_function(NODEZ *head, int *exitflag)
#else
NODEZ *str2sym_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    char  *sptr;
    int   flag;
    NODEZ *hptr, *ptr, *symptr;

    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

    if ((ptr == NULL) || (ptr->type != TYPEstring)) {
	func_warning("(str2sym ...) takes a string as argument", NULL, head);
	return (NULL);
    }

    flag = 0;
    sptr = ptr->value.idptr->name;

    if (*sptr != '\0') {
	flag = 1;
	while (*sptr != '\0') {
	    if (specialchar[*sptr]) {
		flag = 0;
		break;
	    }
	    ++sptr;
	}
    }

    if (!flag) {
	func_warning("(str2sym ...): string cannot be converted to symbol",
		NULL, head);
	return (NULL);
    }

    symptr = new_node();
    symptr->type = TYPEidname;
    symptr->value.idptr = ptr->value.idptr;

    return (symptr);
}



/************************************************************************
*
* stringq_function() - Given one argument, evaluate it and return true
*	if it is a string.
*
* (string? "foobar")  ==>  t
*
************************************************************************/

#ifdef _ANSI_
NODEZ *stringq_function(NODEZ *head, int *exitflag)
#else
NODEZ *stringq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *hptr, *ptr;

	/* evaluate the first argument */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* if it is a string, return true */
    if ((ptr != NULL) && (ptr->type == TYPEstring))
	return (truebind->value.ptr.cdr);

    return (NULL);
}



/************************************************************************
*
* sub_function() - Given two or more arguments that are numbers, take
*	the first number and subtract the values of all successive
*	arguments.
*
* (sub 15 3 1)  ==>  11
*
************************************************************************/

#ifdef _ANSI_
NODEZ *sub_function(NODEZ *head, int *exitflag)
#else
NODEZ *sub_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    int   val;
    NODEZ *hptr, *ptr;

    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    val = 0;
    if ((ptr != NULL) && (ptr->type == TYPEnumber))
	val = ptr->value.number;
    else
	func_warning("(sub ...) requires numerical arguments", NULL, head);

    hptr = hptr->value.ptr.cdr;
    while (hptr != NULL) {
	ptr = evaluate(hptr->value.ptr.car, exitflag);

	if ((ptr != NULL) && (ptr->type == TYPEnumber))
	    val -= ptr->value.number;
	else
	    func_warning("(sub ...) requires numerical arguments", NULL, head);

	hptr = hptr->value.ptr.cdr;
    }

    ptr = new_node();
    ptr->type = TYPEnumber;
    ptr->value.number = val;

    return (ptr);
}



/************************************************************************
*
* sub1_function() - given an expression that evaluates to a number,
*	return one less than that number
*
* (sub1 15)  ==>  14
*
************************************************************************/

#ifdef _ANSI_
NODEZ *sub1_function(NODEZ *head, int *exitflag)
#else
NODEZ *sub1_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    int   val;
    NODEZ *hptr, *ptr;

	/* evaluate the argument of the expression */
    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

	/* if it is a number, return one less than that number,
	   otherwise return a nil value */
    if ((ptr != NULL) && (ptr->type == TYPEnumber)) {
	val = ptr->value.number;
	ptr = new_node();
	ptr->type = TYPEnumber;
	ptr->value.number = val - 1;
    }
    else {
	func_warning("(sub1 ...) requires a numerical argument", NULL, head);
	ptr = NULL;
    }

	/* return the final value */
    return (ptr);
}



/************************************************************************
*
* subclassofq_function() - Given a pointer to an object and the name
*	of a class, return true if the object inherits properties from
*	the named class.
*
* (subclassof? <objectptr> <classname>)
*
************************************************************************/

#ifdef _ANSI_
NODEZ *subclassofq_function(NODEZ *head, int *exitflag)
#else
NODEZ *subclassofq_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ  *hptr, *ptr;
    OBJECT *obptr;

	/* look at the first argument, and check that it is a good
	   object pointer; if not, then return nil */
    hptr = head->value.ptr.cdr;
    if ((obptr = good_ob_ptr(evaluate(hptr->value.ptr.car,exitflag))) == NULL){
	func_warning("(subclassof? ...): first argument must be object pointer",
		NULL, head);
	return (NULL);
    }

	/* evaluate the second argument, and check that it is symbol */
    hptr = hptr->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);
    if ((ptr == NULL) || (ptr->type != TYPEidname)) {
	func_warning("(subclassof? ...): class number must be an identifier",
		NULL, head);
	return (NULL);
    }

	/* call is_subclass() with the object pointer and the given
	   class name, and return a true value if it is a subclass */
    if (is_subclass(obptr->classes, ptr->value.idptr))
	return (truebind->value.ptr.cdr);

	/* if not a subclass, then return nil */
    return (NULL);
}



/************************************************************************
*
* sym2str_function() - Given a lisp symbol, convert it into a string.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *sym2str_function(NODEZ *head, int *exitflag)
#else
NODEZ *sym2str_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    char  *cptr;
    NODEZ *hptr, *ptr, *strptr;

    hptr = head->value.ptr.cdr;
    ptr = evaluate(hptr->value.ptr.car, exitflag);

    if ((ptr == NULL) || (ptr->type != TYPEidname)) {
	func_warning("(sym2str ...) takes symbol as argument", NULL, head);
	return (NULL);
    }

    cptr = ptr->value.idptr->name;
    while (*cptr != '\0') {
	if (*cptr == '"') {
	    func_warning("(sym2str ...): symbols cannot contain quotes",
		    NULL, head);
	    return (NULL);
	}
	++cptr;
    }

    strptr = new_node();
    strptr->type = TYPEstring;
    strptr->value.idptr = ptr->value.idptr;

    return (strptr);
}



/************************************************************************
*
* terminate_function() - behaves the same as (exit), but also sets
*	the maindone flag, which causes the parser to halt before it
*	prompts the player for another command
*
************************************************************************/

#ifdef _ANSI_
NODEZ *terminate_function(NODEZ *head, int *exitflag)
#else
NODEZ *terminate_function(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    maindone = 1;
    *exitflag = 1;

    return (NULL);
}



/************************************************************************
*
* func_warning() - If the *warning-messages* binding is non-nil, then
*	display the given warning messages, along with the lisp
*	structure which generated the warning.
*
************************************************************************/

#ifdef _ANSI_
void func_warning(char *mesg, char *strg, NODEZ *head)
#else
func_warning(mesg, strg, head)
char  *mesg, *strg;
NODEZ *head;
#endif
{
    if (warningbind->value.ptr.cdr != NULL) {
	new_line();
	printf("Warning: ");
	printf(mesg, strg);
	printf("\n");
	print_tree(head, stdout);
	printf("\n");
	if (debugbind->value.ptr.cdr != NULL) {
	    ++debuglevel;
	    gina_lisp(NULL, 0);
	    --debuglevel;
	}
    }
}
