#define PRGNAME "D64ToInf"
#define PRGVERSION "3.02"
#define PRGDATE "22.5.2001"

/*
 *  d64toinf.c
 *
 *  - extracts z-code datafile from D64 image(s)
 *
 *  infile:   D64 disk image of an Infocom game
 *  infile2:  D64 disk image of second disk (only with V4 and V5)
 *  outfile:  extracted datafile
 *
 *  Author: Paul David Doherty <42@pdd.de>
 *
 *  v1.0:    26 Apr 1993
 *  v1.0a:   24 May 1993
 *  v1.10:    4 Mar 1996  completely rewritten, renamed, and updated
 *  v1.20:   15 Jul 1996  support for V4 and V5 added
 *  v2.0:    29 Jul 1996  support for 5J added, C128 version numbers fixed
 *  v2.0a:   24 Nov 1996  fixed lint warnings
 *  v2.1:    27 Aug 1999  switched 3_1/3_2 around
 *                        changed check2 for C128-5E
 *  v3.0:     8 May 2001  detects Plus/4 versions, some cleanups
 *                        typo removed thanks to Fredrik Ramsberg
 *  v3.01:   14 May 2001  FreeBSD patch supplied by Mcclain Looney
 *                        added DJGPP patches
 *  v3.02:   22 May 2001  small fix suggested by Miron Schmidt
 *
 *  ToDo:   hack interpreter to support longer games (???)
 */

#if defined(AZTEC_C) || defined(LATTICE)
#define AMIGA
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>

#ifdef __TURBOC__
#include <io.h>
#include <sys\stat.h>
#define S_IRUSR S_IREAD
#define S_IWUSR S_IWRITE
#endif

#ifdef __GNUC__
#include <sys/stat.h>
#endif

#ifdef FREEBSD
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#endif

#ifdef __linux__
#include <unistd.h>
#endif

#ifdef __DJGPP__
#include <sys/types.h>
#include <io.h>
#include <unistd.h>
#endif

#ifndef S_IRUSR
#define S_IRUSR 0400
#define S_IWUSR 0200
#endif

#if defined(__TURBOC__) || defined(__DJGPP__) || (defined(__GNUC__) && !defined(__unix__))
#define O_SCRIVERE O_WRONLY|O_BINARY
#define O_LEGGERE O_RDONLY|O_BINARY
#else
#define O_SCRIVERE O_WRONLY
#define O_LEGGERE O_RDONLY
#endif

#define TRUE 1
#define FALSE 0

typedef unsigned char type8;
typedef unsigned short type16;
typedef unsigned long int type32;

/* Amiga version string */
char *amiverstag = "\0$VER: " PRGNAME " " PRGVERSION " (" PRGDATE ")";

#define MAXLENGTH 131072L
/*
   longest possible datafile:
   C-64 (01-A), using tracks 4-35         ->  32 * 16 * 256 = 131072
   C-64 (B), using tracks 5-16 / 19-35    ->  29 * 16 * 256 = 118784
   C-64 (C-H), using tracks 5-16 / 18-35  ->  30 * 17 * 256 = 130560
   Apple II, using tracks 4-35            ->  32 * 16 * 256 = 131072

   (So the maximum length of a V3 datafile is 130560 bytes; only the
   earliest versions of the C64 interpreter allowed for datafiles with
   length 131072.)
 */

int fdi1, fdi2, fdo;

static type8 inputbuf[256];
type32 length;
type8 version, alternate_version;
type8 endfile_reached = FALSE;
type16 chksum;
type16 ckdsum = 0;
type32 bytecount = 0;
type32 file_length;

void
ex (char *error)
{
  fprintf (stderr, PRGNAME ": %s\n", error);
  exit (1);
}

void
usage (void)
{
  fprintf (stderr,
	   PRGNAME " version " PRGVERSION " (released " PRGDATE "):\n");
  fprintf (stderr,
	   "Extracts z-code datafile from D64 image(s)\n");
  fprintf (stderr,
	   "(c) 1993,96,99,2001 by Paul David Doherty <42@pdd.de>\n\n");
  fprintf (stderr,
	   "Usage: " PRGNAME " infile.d64 [infile2.d64] outfile.dat\n");
  exit (1);
}

/*
   V3 interpreters use tracks 5-16 and 19-35 for the data. Some also
   use tracks 4, 17 or 18. The early ones used the first 16 sectors of
   each track, the later ones use 17 sectors.

   V4 interpreters use full tracks 3-10 and sectors 11:0-6 on disk 1,
   or the full tracks 3-18 (except 18:0 and 18:1) and sectors 19:0-10;
   plus the whole second disk (except 18:0, and sometimes 20:0).

   V5 interpreters use full tracks 5-12 and sectors 13:0-6 on disk 1,
   or full tracks 5-20 (except 18:0 and 18:1) and sectors 21:0-14;
   plus the whole second disk (except 18:0, and sometimes 20:0).
 */

struct params
{
  type8 check_number;		/* we've got three checks... */
  type8 internal_version;	/* for reference purposes */
  char *version_name;
  type8 version_number;		/* V3...V5 or V6 for special check */
  type8 computer_type;		/* 0: C64
				   1: C128
				   2: C64/128
				   3: Plus/4 */
  type8 tr4;			/* V3: track 4 used? 
				   V4: tracks 12-19 used? 
				   V5: tracks 13-21 used? */
  type8 tr17;			/* V3: track 17 used?
				   V5: peculiarities for C128-5E */
  type8 tr18;			/* V3: track 18 used? [starts at sec 2] */
  type8 sec_used;		/* V3: 16 = 0..15, 17 = 0..16 */
  type8 id[6];
}
version_params[] =
{
/*
 * check 1: start of sector 2:11 (offset 0x2000)
 */
  1, 0, "01", 3, 0, TRUE, TRUE, TRUE, 16, 0x00, 0xd0, 0xf5, 0x60, 0x20, 0xf9,
    1, 1, "02", 3, 0, TRUE, TRUE, TRUE, 16, 0xf5, 0x60, 0x20, 0xf7, 0x1f, 0x20,
    1, 2, "A", 3, 0, TRUE, TRUE, TRUE, 16, 0xa0, 0x03, 0x20, 0x7b, 0x27, 0xa9,
    1, 3, "B", 3, 0, FALSE, FALSE, FALSE, 16, 0x0f, 0xa8, 0xa6, 0x78, 0x20, 0xba,
    1, 4, "C", 3, 0, FALSE, FALSE, TRUE, 17, 0xff, 0xa2, 0x0f, 0x20, 0xc9, 0xff,
    1, 5, "D", 3, 0, FALSE, FALSE, TRUE, 17, 0xdd, 0x27, 0x09, 0x30, 0x99, 0xf0,
    1, 6, "E", 3, 0, FALSE, FALSE, TRUE, 17, 0xff, 0xa2, 0x00, 0xc9, 0x0a, 0x90,
    1, 7, "F", 3, 0, FALSE, FALSE, TRUE, 17, 0xa2, 0xef, 0xa0, 0x27, 0xa9, 0x01,
    1, 8, "G", 3, 0, FALSE, FALSE, TRUE, 17, 0x69, 0x6e, 0x75, 0x65, 0x2e, 0x0d,
    1, 9, "H", 3, 0, FALSE, FALSE, TRUE, 17, 0x45, 0x54, 0x55, 0x52, 0x4e, 0x5d,
    1, 10, "A", 4, 1, TRUE, 0, 0, 0, 0xdd, 0x30, 0x2d, 0xf0, 0x05, 0xca,
    1, 11, "B", 4, 1, TRUE, 0, 0, 0, 0x2a, 0x29, 0x7f, 0xc9, 0x0d, 0xf0,
    1, 12, "L", 4, 2, FALSE, 0, 0, 0, 0x02, 0x69, 0x20, 0xa2, 0x06, 0xdd,
    1, 13, "A", 5, 1, TRUE, 0, 0, 0, 0x2e, 0x0d, 0xa9, 0xfb, 0x60, 0x85,
    1, 14, "J", 5, 2, FALSE, 0, 0, 0, 0xa5, 0x76, 0x85, 0xe5, 0xa2, 0x2d,
/* next two get version 6; check again at 3:5 */
    1, 15, "E/C", 6, 2, FALSE, 0, 0, 0, 0x2e, 0xa5, 0x76, 0x85, 0xe5, 0xa2,
    1, 16, "E/H", 6, 2, FALSE, 0, 0, 0, 0x36, 0x4c, 0x01, 0x2d, 0x20, 0x0c,
/*
 * check 2: start of sector 3:5 (offset 0x2f00) for V5 disambiguation only
 */
    2, 17, "E", 5, 1, TRUE, TRUE, 0, 0, 0x06, 0x20, 0x55, 0x05, 0x4c, 0xee,
    2, 18, "C", 5, 2, FALSE, 0, 0, 0, 0xf0, 0x06, 0x20, 0x55, 0x05, 0x4c,
    2, 19, "E", 5, 2, FALSE, 0, 0, 0, 0x20, 0x27, 0x35, 0xc9, 0x59, 0xf0,
    2, 20, "H", 5, 2, FALSE, 0, 0, 0, 0x60, 0x20, 0x28, 0x35, 0xc9, 0x59,
/*
 * check 3: start of sector 3:15 (offset 0x3900) for Plus/4 only
 */
    3, 21, "B", 3, 3, FALSE, FALSE, FALSE, 16, 0x18, 0x69, 0x02, 0x85, 0x11, 0x90,
    4, 127, "", 0, 127, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

static char *computer_type[] =
{
  "Commodore 64", "Commodore 128", "Commodore 64/128", "Commodore Plus/4"
};

void
docheck (type8 check_number)
{
  type8 i, counter, loop;
  type8 act_id[6];

  for (i = 0; i < 6; i++)
    act_id[i] = inputbuf[i];
  counter = loop = 0;
  do
    {
      if (version_params[loop].check_number == check_number)
	{
	  counter = 1;
	  for (i = 0; i < 6; i++)
	    if (act_id[i] != version_params[loop].id[i])
	      counter = 0;
	}
      loop++;
    }
  while ((version_params[loop].check_number != 4)
	 && (counter == 0));

  version = loop;
  if (counter == 1)
    version--;
}

void
print_version (void)
{
  if (version_params[version].internal_version == 127)
    ex ("unknown C-64 interpreter");

  printf ("%s V%d Interpreter %s",
	  computer_type[version_params[version].computer_type],
	  version_params[version].version_number,
	  version_params[version].version_name);
  if ((alternate_version != version)
      && (version_params[alternate_version].internal_version != 127))
    printf (" / %s V%d Interpreter %s",
	    computer_type[version_params[alternate_version].computer_type],
	    version_params[alternate_version].version_number,
	    version_params[alternate_version].version_name);
  printf ("\n");
}

void
skip256 (type16 secs)
{
  type16 i;
  for (i = 0; i < secs; i++)
    length = read (fdi1, inputbuf, 256);
  if (length != 256)
    ex ("input file too short");
}

void
trans256 (type16 secs)
{
  type16 i;
  type32 j, bytes_used;

  for (i = 0; i < secs; i++)
    {
      bytes_used = 256;
      skip256 (1);

      if (endfile_reached == FALSE)
	{
	  if (bytecount == 0)
	    {
	      if (inputbuf[0x00] != version_params[version].version_number)
		ex ("data and interpreter do not match");

	      printf ("Datafile %d/%c%c%c%c%c%c, ",
		      inputbuf[0x02] * 256 + inputbuf[0x03],
		      inputbuf[0x12], inputbuf[0x13], inputbuf[0x14],
		      inputbuf[0x15], inputbuf[0x16], inputbuf[0x17]);

	      file_length = ((256L * (type32) inputbuf[0x1a]) +
			     (type32) inputbuf[0x1b]) * 2L;
	      if (file_length == 0L)
		file_length = MAXLENGTH;
	      if ((version_params[version].version_number == 4) ||
		  (version_params[version].version_number == 5))
		file_length = file_length * 2L;
	      printf ("length: %lu\n", file_length);

	      chksum = inputbuf[0x1c] * 256 + inputbuf[0x1d];
	      for (j = 0x40; j < 256; j++)
		ckdsum = ckdsum + inputbuf[j];
	    }

	  bytecount = bytecount + 256;
	  if (bytecount >= file_length)
	    {
	      bytes_used = (type32) (bytecount - file_length);
	      bytes_used = 256L - bytes_used;
	      endfile_reached = TRUE;
	    }
	  if (bytecount != 256)
	    for (j = 0; j < bytes_used; j++)
	      ckdsum = ckdsum + inputbuf[j];
	  write (fdo, inputbuf, bytes_used);
	}
    }
}

int
main (int argc, char **argv)
{
  type16 i;

  if (argc == 1)
    usage ();
  else if ((argc == 2) || (argc > 4))
    ex ("needs 2 or 3 filenames");

#ifdef __TURBOC__
  _fmode = O_BINARY;
#endif

  if ((fdi1 = open (argv[1], O_LEGGERE, 0)) == -1)
    ex ("could not open input file");
  if (argc == 4)
    if ((fdi2 = open (argv[2], O_LEGGERE, 0)) == -1)
      ex ("could not open second input file");
  if ((fdo = open (((argc == 4) ? argv[3] : argv[2]), O_SCRIVERE | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR)) == -1)
    ex ("could not create output file");

  skip256 (33);			/* get 2:11 for check 1 */
  docheck (1);
  alternate_version = version;
  skip256 (9);			/* reach track 3 */

  if (version_params[version].version_number == 4)
    {
      print_version ();
      trans256 (21);
    }
  else
    {
      skip256 (6);		/* get 3:5 for check 2 */
      if (version_params[version].version_number == 6)
	{
	  docheck (2);
	  alternate_version = version;
	}
      skip256 (10);		/* get 3:15 for check 3 */

/* do Plus/4 check if C-64 or nothing found */
      if ((version_params[version].version_number == 3) ||
	  (version_params[version].version_number == 0))
	{
	  docheck (3);
	  if (version_params[alternate_version].internal_version != 127)
	    {
	      i = version;
	      version = alternate_version;
	      alternate_version = i;
	    }
	}
      print_version ();
      skip256 (5);
    }

/* track 4 -- end of all checks */
  if (version_params[version].version_number == 4)
    {
      trans256 (154);		/* 4:0 - 11:6 */
      if (version_params[version].tr4)	/* C128 version */
	{
	  trans256 (140);	/* 11:7 - 17:20 */
	  skip256 (1);
	  trans256 (29);	/* 18:1 - 19:10 */
	}
    }
  else if (version_params[version].version_number == 5)
    {
      skip256 (21);
      trans256 (175);		/* 5:0 - 13:6 */
      if (version_params[version].tr4)	/* C128 version */
	{
	  trans256 (98);	/* 13:7 - 17:20 */
	  skip256 (1);
	  trans256 (37);	/* 18:1 - 19:18 */
	  if (version_params[version].tr17)	/* C128-5E skips 20:0 */
	    skip256 (1);
	  else
	    trans256 (1);
	  trans256 (33);	/* 20:1 - 21:14 */
	  if (version_params[version].tr17)	/* C128-5E uses 21:15 */
	    trans256 (1);
	}
    }
  else
/* V3 */
    {
/* track 4 */
      if (version_params[version].tr4)
	trans256 (version_params[version].sec_used);
      else
	skip256 (version_params[version].sec_used);
      skip256 (21 - version_params[version].sec_used);

      for (i = 5; i < 17; i++)
	{
	  trans256 (version_params[version].sec_used);
	  skip256 (21 - version_params[version].sec_used);
	}

/* track 17 */
      if (version_params[version].tr17)
	trans256 (version_params[version].sec_used);
      else
	skip256 (version_params[version].sec_used);
      skip256 (21 - version_params[version].sec_used);

/* track 18 */
      skip256 (2);
      if (version_params[version].tr18)
	trans256 (version_params[version].sec_used);
      else
	skip256 (version_params[version].sec_used);
      if (version_params[version].sec_used == 16)
	skip256 (1);

      for (i = 19; i < 36; i++)
	{
	  trans256 (version_params[version].sec_used);
	  if (version_params[version].sec_used == 16)
	    skip256 (1);
	  if (i < 31)
	    skip256 (1);
	  if (i < 25)
	    skip256 (1);
	}
    }

/* done with disk 1 */
  if (argc == 3)
    if (((version_params[version].version_number == 4)
	 || (version_params[version].version_number == 5))
	&&
	((file_length > 88064L) ||
	 ((file_length > 44800L) && (!version_params[version].tr4))))
      ex ("second disk missing; output file too short");

  if ((argc == 4)
      && ((version_params[version].version_number == 4) ||
	  (version_params[version].version_number == 5)))
    {
      close (fdi1);
      fdi1 = fdi2;
/* skip sector 18:0 */
      trans256 (357);
      skip256 (1);
/*
   skip sector 20:0, unless it is C128 V4 or V5 game -- except
   if it is C128 V5 interpreter E (what a mess!)
 */
      trans256 (37);
      if (((version_params[version].version_number == 4)
	   || (version_params[version].version_number == 5))
	  && (version_params[version].tr4)
	  && (!(version_params[version].tr17)))
	trans256 (1);
      else
	skip256 (1);
      trans256 (287);
    }

  printf ("Checksum %s\n", (chksum == ckdsum) ? "okay" : "incorrect!");
  if (chksum != ckdsum)
    ex ("output file corrupt");

  close (fdi1);
  close (fdo);
  return 0;
}
