/*
 * table.c
 *
 * Table handling opcodes
 *
 */

#include <math.h>
#include "frotz.h"

/*
 * z_copy_table
 *
 * Copy a table or fill it with zeroes. There are three cases:
 *
 * 1) Destination is 0: Source is filled with zeroes.
 * 2) Size is positive: Source is copied to destination in either
 * forward or backward fashion. Both tables may overlap ("memmove").
 * 3) Size is negative: Source is copied forwards to destination.
 *
 */

void z_copy_table (zword first, zword second, zword size)
{
    zword addr;
    zbyte value;
    int i;

    if (second == 0)

	/* Fill source with zeros */

	for (i = 0; i < size; i++)
	    z_storeb (first, i, 0);

    else if ((short) size < 0 || first > second)

	/* Copy forwards */

	for (i = 0; i < abs (size); i++) {
	    addr = first + i;
	    LOW_BYTE (addr, value)
	    z_storeb (second, i, value);
	}

    else

	/* Copy backwards */

	for (i = size - 1; i >= 0; i--) {
	    addr = first + i;
	    LOW_BYTE (addr, value)
	    z_storeb (second, i, value);
	}

}/* z_copy_table */

/*
 * z_loadb
 *
 * Load a byte from a table of bytes.
 *
 */

void z_loadb (zword addr, zword offset)
{
    zbyte value;

    addr += offset;
    LOW_BYTE (addr, value)
    store (value);

}/* z_loadb */

/*
 * z_loadw
 *
 * Load a word from a table of words.
 *
 */

void z_loadw (zword addr, zword offset)
{
    zword value;

    addr += (offset << 1);
    LOW_WORD (addr, value)
    store (value);

}/* z_loadw */

/*
 * z_scan_table
 *
 * Scan a table of bytes or words looking for a target byte or word.
 * The optional 4th parameter gives the address step (bottom 7 bits)
 * and the type of the table (top bit: 1 for word, 0 for byte table).
 * (The default is word table and address step 2.) If the search is
 * successful, the address of the first match in the table is stored
 * and a branch is made. Otherwise, 0 is stored.
 *
 */

void z_scan_table (int argc, zword *argv)
{
    zword addr;
    zword len;
    zword wvalue;
    zbyte bvalue;
    zbyte type;
    zbyte step;
    int i;

    /* Supply default parameters */

    if (argc < 4)
	argv[3] = 0x82;

    addr = argv[1];
    len = argv[2];
    type = argv[3] & 0x80;
    step = argv[3] & 0x7f;

    if (type != 0) /* Scan word array */

	for (i = 0; i < len; i++) {

	    LOW_WORD (addr, wvalue)

	    if (wvalue == argv[0]) {
		store (addr);
		branch (1);
		return;
	    }

	    addr += step;
	}

    else /* Scan byte array */

	for (i = 0; i < len; i++) {

	    LOW_BYTE (addr, bvalue)

	    if (bvalue == argv[0]) {
		store (addr);
		branch (1);
		return;
	    }

	    addr += step;
	}

    /* Store zero if the search fails */

    store (0);
    branch (0);

}/* z_scan_table */

/*
 * z_storeb
 *
 * Store a byte into a table of bytes.
 *
 */

void z_storeb (zword addr, zword offset, zword value)
{

    addr += offset;

    /* Check if this toggles the scripting flag */

    if (addr == H_FLAGS + 1)
	if (value & SCRIPTING_FLAG)
	    z_output_stream (2, 0);
	else
	    z_output_stream (-2, 0);

    /* Are we writing outside of the dynamic memory? */

    if (addr >= h_dynamic_size)
	os_fatal ("Store out of dynamic memory");

    /* Store the byte */

    SET_BYTE (addr, value)

}/* z_storeb */

/*
 * z_storew
 *
 * Store a word into a table of words.
 *
 */

void z_storew (zword addr, zword offset, zword value)
{

    addr += (offset << 1);

    z_storeb (addr, 0, hi (value));
    z_storeb (addr, 1, lo (value));

}/* z_storew */
