/*
 * stream.c
 *
 * IO stream implementation
 *
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "frotz.h"

static zword memory_table;

static FILE *sfp = NULL;
static FILE *rfp = NULL;
static FILE *pfp = NULL;

static char script_name[MAX_FILE_NAME + 1] = DEFAULT_SCRIPT_NAME;
static char command_name[MAX_FILE_NAME + 1] = DEFAULT_COMMAND_NAME;

static char *euro_substitute =
"\
aeoeueAeOeUess>><<e \
i y E I a e i o u y \
A E I O U Y a e i o \
u A E I O U a e i o \
u A E I O U a A o O \
a n o A N O aeAEc C \
ththThThL oeOE! ? \
";

static void close_playback (void);
static void close_record (void);
static void close_script (void);

static void open_playback (void);
static void open_record (void);
static void open_script (void);

static int playback_char (int);
static void record_char (int, int);

/*
 * memory_char
 *
 * Redirect a character to a table within the dynamic memory. The
 * first word of the table holds the number of characters printed.
 *
 */

void memory_char (int c)
{
    zword size;
    zword addr;

    LOW_WORD (memory_table, size)
    addr = memory_table + size + 2;
    size++;
    SET_BYTE (addr, c)
    SET_WORD (memory_table, size)

}/* memory_char */

/*
 * open_playback
 *
 * Open a command file for playback.
 *
 */

static void open_playback (void)
{
    char new_command_name[MAX_FILE_NAME + 1];

    /* Get command file name */

    if (os_get_file_name (new_command_name, command_name, FILE_PLAYBACK) != 0)

	if ((pfp = fopen (new_command_name, "rt")) != 0) {
	    strcpy (command_name, new_command_name);
	    replaying = 1;
	} else {
	    display_string ("Cannot open command file");
	    display_new_line ();
	}

}/* open_playback */

/*
 * playback_char
 *
 * Get a single character (a plain character or a function key) from
 * the command file.
 *
 */

static int playback_char (int force_encoding)
{
    int code;
    int c;

    /* Literal characters are all characters except '[' */

    c = fgetc (pfp);
    if (c != '[' && force_encoding == 0)
	return (c);

    /* Read an encoded character */

    if (c != '[')
	return (-1);

    code = 0;
    do {
	c = fgetc (pfp);
	if (isdigit (c))
	    code = 10 * code + c - '0';
    } while (isdigit (c));

    if (c != ']')
	return (-1);

    /* Return character */

    return (code);

}/* playback_char */

/*
 * playback_key
 *
 * Get a key from the command file.
 *
 */

int playback_key (void)
{
    int key;

    /* Read key */

    key = playback_char (0);
    if (key == 253 || key == 254) {
	mouse_x = playback_char (1);
	mouse_y = playback_char (1);
    }

    /* Next character should be newline */

    if (fgetc (pfp) != '\n')
	key = -1;

    /* Stop playback if problems occur */

    if (key == -1)
	close_playback ();

    /* Return key value */

    return (key);

}/* playback_key */

/*
 * playback_line
 *
 * Get a line of input from the command file.
 *
 */

int playback_line (int max_size, char *buffer)
{
    int row, col;
    int read_size;
    int length;
    int terminator;
    int c;

    /* Remove initial input from screen */

    length = os_text_length (buffer);

    if (length != 0) {
	os_get_cursor (&row, &col);
	os_erase_area (row, col - length, row, col - 1);
	os_set_cursor (row, col - length);
    }

    /* Read input buffer */

    read_size = 0;

    for (;;) {

	c = playback_char (0);
	if (c == -1) {
	    terminator = -1;
	    break;
	}

	/* Insert key 13 if the current character is newline */

	if (c == '\n') {
	    ungetc ('\n', pfp);
	    c = 13;
	}

	/* Check for terminating key */

	if (is_terminator (c) != 0) {
	    terminator = c;
	    if (terminator == 253 || terminator == 254) {
		mouse_x = playback_char (1);
		mouse_y = playback_char (1);
	    }
	    break;
	}

	/* Add character to the buffer */

	if (read_size < max_size)
	    buffer[read_size++] = c;
    }

    buffer[read_size] = 0;

    /* Next character should be newline */

    if (fgetc (pfp) != '\n')
	terminator = -1;

    /* Stop playback if problems occur */

    if (terminator == -1)
	close_playback ();

    /* Display input line on screen */

    display_string (buffer);

    /* Return terminating key value */

    return (terminator);

}/* playback_line */

/*
 * close_playback
 *
 * Stop playback from a command file.
 *
 */

static void close_playback (void)
{

    fclose (pfp);
    replaying = 0;

}/* close_playback */

/*
 * open_record
 *
 * Turn on recording of player's input to a command file.
 *
 */

static void open_record (void)
{
    char new_command_name[MAX_FILE_NAME + 1];

    /* Get recording file name */

    if (os_get_file_name (new_command_name, command_name, FILE_RECORD) != 0)

	if ((rfp = fopen (new_command_name, "wt")) != NULL) {
	    strcpy (command_name, new_command_name);
	    recording = 1;
	} else {
	    display_string ("Cannot open command file");
	    display_new_line ();
	}

}/* open_record */

/*
 * record_char
 *
 * Write a single character (a plain character or a function key) to
 * the command file.
 *
 */

static void record_char (int c, int force_encoding)
{
    int i;

    /* Store all characters from 32 to 126 except '[' as literals */

    if (c >= 32 && c <= 126 && c != '[' && force_encoding == 0) {
	fputc (c, rfp);
	return;
    }

    /* Store an encoded character */

    fputc ('[', rfp);

    for (i = 10000; i != 0; i /= 10)
	if (c >= i || i == 1)
	    fputc ('0' + (c / i) % 10, rfp);

    fputc (']', rfp);

}/* record_char */

/*
 * record_key
 *
 * Write a key followed by a new line to the command file.
 *
 */

void record_key (int key)
{

    /* Write key */

    record_char (key, 0);
    if (key == 253 || key == 254) {
	record_char (mouse_x, 1);
	record_char (mouse_y, 1);
    }

    /* Write a newline */

    fputc ('\n', rfp);

    /* Stop recording if problems occur */

    if (ferror (rfp) != 0)
	close_record ();

}/* record_key */

/*
 * record_line
 *
 * Write a line of input to the command file.
 *
 */

void record_line (const char *buffer, int terminator)
{
    int i;

    /* Copy input line to command file */

    for (i = 0; buffer[i] != 0; i++)
	record_char ((unsigned char) buffer[i], 0);

    if (terminator != 13) {
	record_char (terminator, 0);
	if (terminator == 253 || terminator == 254) {
	    record_char (mouse_x, 1);
	    record_char (mouse_y, 1);
	}
    }

    /* Write a newline */

    fputc ('\n', rfp);

    /* Stop recording if problems occur */

    if (ferror (rfp) != 0)
	close_record ();

}/* record_line */

/*
 * close_record
 *
 * Turn off recording of all input to a command file.
 *
 */

static void close_record (void)
{

    fclose (rfp);
    recording = 0;

}/* close_record */

/*
 * open_script
 *
 * Open the transscript file.
 *
 */

static void open_script (void)
{
    char new_script_name[MAX_FILE_NAME + 1];

    /* Get transscript file name */

    if (os_get_file_name (new_script_name, script_name, FILE_SCRIPT) != 0)

	if ((sfp = fopen (new_script_name, "wt")) != NULL) {
	    strcpy (script_name, new_script_name);
	    scripting = 1;
	} else {
	    display_string ("Cannot open transscript file");
	    display_new_line ();
	}

    /* Set or clear the scripting flag */

    if (scripting)
	h_flags |= SCRIPTING_FLAG;
    else
	h_flags &= ~SCRIPTING_FLAG;

    SET_WORD (H_FLAGS, h_flags)

}/* open_script */

/*
 * script_backup
 *
 * Remove the last n characters from the transscript file.
 *
 */

void script_backup (int n)
{

    fseek (sfp, -n, SEEK_CUR);

}/* script_backup */

/*
 * script_char
 *
 * Write a character to the transscript file.
 *
 */

void script_char (int c)
{

    if (c >= 155 && c <= 223) {

	c = (c - 155) << 1;

	fputc (euro_substitute[c], sfp);

	if (euro_substitute[c + 1] != ' ')
	    fputc (euro_substitute[c + 1], sfp);

    } else fputc (c, sfp);

}/* script_char */

/*
 * script_string
 *
 * Write a string of characters to the transscript file.
 *
 */

void script_string (const char *s)
{
    int i;

    for (i = 0; s[i] != 0; i++)
	script_char ((unsigned char) s[i]);

}/* script_string */

/*
 * close_script
 *
 * Close the transscript file.
 *
 */

static void close_script (void)
{

    fclose (sfp);
    scripting = 0;

    /* Clear the scripting flag */

    h_flags &= ~SCRIPTING_FLAG;
    SET_WORD (H_FLAGS, h_flags)

}/* close_script */

/*
 * z_input_stream
 *
 * Choose the input stream which can be either 0 (keyboard) or 1 (playback).
 *
 */

void z_input_stream (zword stream)
{

    if (stream == 0) {

	/* Turn on keyboard input */

	if (replaying)
	    close_playback ();

    } else {

	/* Turn on playback from a file */

	if (!replaying)
	    open_playback ();
    }

}/* z_input_stream */

/*
 * z_output_stream
 *
 * Open and close various output streams. These can be:
 *
 *    1 = the screen
 *    2 = transscript file
 *    3 = redirection to memory
 *    4 = command line recording
 *
 * In addition, this interpreter uses stream 5 if it wants to
 * print a debugging message (which is passed without buffering
 * to the interface).
 *
 */

void z_output_stream (zword stream, zword table)
{

    switch ((short) stream) {

	case 1: /* Turn on text output */
	    outputting = 1;
	    break;

	case -1: /* Turn off text output */
	    outputting = 0;
	    break;

	case 2: /* Turn on scripting */
	    if (!scripting)
		open_script ();
	    break;

	case -2: /* Turn off scripting */
	    if (scripting)
		close_script ();
	    break;

	case 3: /* Turn on output redirection */
	    redirecting = 1;
	    memory_table = table;
	    SET_WORD (memory_table, 0)
	    break;

	case -3: /* Turn off output redirection */
	    redirecting = 0;
	    break;

	case 4: /* Turn on input recording */
	    if (!recording)
		open_record ();
	    break;

	case -4: /* Turn off input recording */
	    if (recording)
		close_record ();
	    break;

	case 5: /* Turn on message printing */
	    os_message_start ();
	    message = 1;
	    break;

	case -5: /* Turn off message printing */
	    os_message_end ();
	    message = 0;
	    break;
    }

}/* z_output_stream */
