/*
 * sound.c
 *
 * High level sound routines
 *
 */

#include "frotz.h"

static zword sound_routine = 0;

/*
 * end_of_sound
 *
 * Call the Z-code routine which was given as the last parameter of
 * a sound_effect call. Note that this may happen during a hardware
 * interrupt (which requires very careful programming).
 *
 */

void end_of_sound (void)
{

    call_interrupt (sound_routine);

}/* end_of_sound */

/*
 * z_sound_effect
 *
 * Handle sound effects. These can be either plain beep sounds
 * (argv[0] == 0 high-pitched and argv[0] == 1 low-pitched) or
 * samples (argv[0] >= 3). If it is a sample then the second
 * argument is what to do:
 *
 *    1 = prepare sample
 *    2 = start sample (and load it, if necessary)
 *    3 = stop sample (if it is running)
 *    4 = finish with sample (ie. remove it from memory)
 *
 * When a sample must be played, the third argument holds the volume
 * in its lower byte (ranging from 1 to 8, and reportedly 255 being a
 * default volume) and (in V5 only) the #repeats to play in its upper
 * byte (255 meaning forever). The last argument points to a Z-code
 * function which is called after the sound is finished.
 *
 */

void z_sound_effect (int argc, zword *argv)
{

    /* Play beep sound if there is only one argument */

    if (argc <= 1) {
	os_beep (argv[0]);
	return;
    }

    /* Don't continue if the interface does not support sound */

    if (h_version >= V4 && !(h_flags & SOUND_FLAG))
	return;
    if (h_version == V3 && !(h_flags & OLD_SOUND_FLAG))
	return;

    /* Prevent any end-of-sound interrupts */

    sound_routine = 0;

    /* The following is a workaround for The Lurking Horror. Close to
       the end of the game, a sequence of samples is played during one
       turn. On slow Amigas, this isn't a major problem, but on a fast
       machine each sample gets almost immediately stopped by the next
       one. As a result, a click is all that can be heard. Therefore,
       if no input happened since the current sample was started then
       wait until the current sample finishes. */

    if (h_version == V3 && input_flag == 0)
	os_wait_sample ();

    /* Perform the effect */

    switch (argv[1]) {

	case 1:	os_prepare_sample (argv[0]);
		break;
	case 2: os_start_sample (argv[0], lo (argv[2]), hi (argv[2]));
		sound_routine = (argc < 4) ? 0 : argv[3];
		input_flag = 0;
		break;
	case 3: os_stop_sample ();
		break;
	case 4: os_finish_with_sample ();
		break;
    }

}/* z_sound_effect */
