/*
 * screen.c
 *
 * Generic screen manipulation routines
 *
 */

#include "frotz.h"

static int saved_row = 0;
static int saved_col = 0;

/*
 * display_new_line
 *
 * Print a newline to the screen.
 *
 */

void display_new_line (void)
{
    int y, x;

    /* If the cursor has not already reached the bottom line then
       it is simply moved to the start of the next line. Otherwise
       the lower window is scrolled up one line. */

    os_get_cursor (&y, &x);

    if (y == h_screen_rows) {

	os_scroll_area (upper_size + 1, 1, h_screen_rows, h_screen_cols);
	os_set_cursor (y, 1);

    } else os_set_cursor (y + 1, 1);

    /* See if we have filled the screen. The spare line is for the
       [MORE] prompt. After the [MORE] has been displayed the line
       counter must be reset. In the upper window, there is no such
       concept as [MORE] prompts. */

    if (cwin == LOWER_WINDOW)

	if (++line_count >= h_screen_rows - upper_size - 1) {
	    if (!replaying)
		os_more_prompt ();
	    line_count = option_context_lines;
	}

}/* display_new_line */

/*
 * display_string
 *
 * Print a string of characters to the screen.
 *
 */

void display_string (const char *s)
{
    int i;

    for (i = 0; s[i] != 0; i++)
	os_display_char ((unsigned char) s[i]);

}/* display_string */

/*
 * z_erase_line
 *
 * Clear a line. The cursor stays put.
 *
 */

void z_erase_line (zword value)
{
    int y, x;

    os_get_cursor (&y, &x);

    /* Erase from the cursor position to the end of the line */

    os_erase_area (y, x, y, h_screen_cols);

}/* z_erase_line */

/*
 * z_erase_window
 *
 * Clear one or all windows on the screen.
 *
 */

void z_erase_window (zword window)
{

    if ((short) window == -2) {

	/* Erase the screen */

	os_erase_area (1, 1, h_screen_rows, h_screen_cols);
	line_count = 0;

    } else if ((short) window == -1) {

	/* Erase and unsplit the screen */

	os_erase_area (1, 1, h_screen_rows, h_screen_cols);
	line_count = 0;

	if (h_version <= V4)
	    os_set_cursor (h_screen_rows, 1);
	else
	    os_set_cursor (1, 1);

	upper_size = (h_version <= V3) ? 1 : 0;

    } else if (window == LOWER_WINDOW) {

	/* Erase the lower window */

	os_erase_area (upper_size + 1, 1, h_screen_rows, h_screen_cols);
	line_count = 0;

	saved_row = (h_version <= V4) ? h_screen_rows : upper_size + 1;
	saved_col = 1;

	if (cwin == LOWER_WINDOW)
	    os_set_cursor (saved_row, saved_col);

    } else if (window == UPPER_WINDOW) {

	/* Erase the upper window */

	os_erase_area (1, 1, upper_size, h_screen_cols);

	if (cwin == UPPER_WINDOW)
	    os_set_cursor (1, 1);
    }

}/* z_erase_window */

/*
 * z_get_cursor
 *
 * Write the cursor coordinates into a table.
 *
 */

void z_get_cursor (zword table)
{
    int y, x;

    /* Can only read cursor position in upper window */

    if (cwin == UPPER_WINDOW) {
	os_get_cursor (&y, &x);
	z_storew (table, 1, y);
	z_storew (table, 2, x);
    }

}/* z_get_cursor */

/*
 * z_set_colour
 *
 * Set the colour of the screen which can be set to 1 of 10 values:
 *
 *    0 = current colour
 *    1 = machine default
 *    2 = black
 *    3 = red
 *    4 = green
 *    5 = brown
 *    6 = blue
 *    7 = magenta
 *    8 = cyan
 *    9 = white
 *
 */

void z_set_colour (zword foreground, zword background)
{
    static int current_foreground = 1;
    static int current_background = 1;

    flush_buffer ();

    if (foreground == 0)
	foreground = current_foreground;
    if (background == 0)
	background = current_background;
    os_set_colour (foreground, background);

    current_foreground = foreground;
    current_background = background;

}/* z_set_colour */

/*
 * z_set_cursor
 *
 * Set the cursor position in the upper window only.
 *
 */

void z_set_cursor (zword y, zword x)
{

    /* Can only move cursor in upper window */

    if (cwin == UPPER_WINDOW)
	os_set_cursor (y, x);

}/* z_set_cursor */

/*
 * z_set_window
 *
 * Put the cursor in the upper or lower window. The cursor is free to move
 * in the upper window, but is fixed to the input line in the lower window.
 *
 */

void z_set_window (zword window)
{

    /* Flush all buffered output first */

    flush_buffer ();

    if (window == UPPER_WINDOW) {

	/* Select upper window */

	if (cwin == LOWER_WINDOW)
	    os_get_cursor (&saved_row, &saved_col);

	os_set_cursor ((h_version <= V3) ? 2 : 1, 1);

	cwin = UPPER_WINDOW;

    } else if (window == LOWER_WINDOW) {

	/* Select lower window */

	if (cwin == UPPER_WINDOW)
	    os_set_cursor (saved_row, saved_col);

	cwin = LOWER_WINDOW;
    }

}/* z_set_window */

/*
 * pad_line
 *
 * Pad the status line with spaces up to the given position.
 *
 */

static void pad_line (int column)
{
    int row, col;

    os_get_cursor (&row, &col);

    while (col++ <= column)
	os_display_char (' ');

}/* pad_line */

/*
 * z_show_status
 *
 * Output the status line for V1 to V3 games.
 *
 */

void z_show_status (void)
{
    zword global0;
    zword global1;
    zword global2;
    zword addr;
    int hours;

    /* One V5 game (Wishbringer Solid Gold) contains this opcode by
       accident, so just return if the version number does not fit */

    if (h_version >= V4)
	return;

    /* Read all relevant global variables from the memory of the
       Z-machine into local variables */

    addr = h_globals;
    LOW_WORD (addr, global0)
    addr += 2;
    LOW_WORD (addr, global1)
    addr += 2;
    LOW_WORD (addr, global2)

    /* Move the cursor to the top line of the upper window, set the
       reverse rendition and print the status line */

    z_set_window (UPPER_WINDOW);
    z_buffer_mode (0);
    z_set_text_style (REVERSE_STYLE);
    z_set_cursor (1, 1);

    /* Print the object description for the global variable 0 */

    pad_line (1);
    z_print_obj (global0);
    pad_line (h_screen_cols - 9);

    /* A header flag tells us whether we have to display the current
       time or the score and moves information */

    if (h_config & CONFIG_TIME) {

	/* Print hours and minutes from global variables 1 and 2 */

	hours = (global1 + 11) % 12 + 1;
	os_display_char ((hours < 10) ? ' ' : '1');
	os_display_char ('0' + hours % 10);
	os_display_char (':');

	os_display_char ((global2 < 10) ? '0' : (global2 / 10));
	os_display_char ('0' + global2 % 10);
	os_display_char (' ');

	os_display_char ((hours >= 12) ? 'p' : 'a');
	os_display_char ('m');

    } else {

	/* Print score and moves from global variables 1 and 2 */

	z_print_num (global1);
	z_print_char ('/');
	z_print_num (global2);
    }

    /* Pad the end of status line with spaces */

    pad_line (h_screen_cols);

    /* Return to the lower window */

    z_set_window (LOWER_WINDOW);
    z_buffer_mode (1);
    z_set_text_style (ROMAN_STYLE);

}/* z_show_status */

/*
 * z_split_window
 *
 * Set the size of the upper window. The upper window appears below
 * the status line in V3 games. In this implementation, however, the
 * status line is treated as a part of the upper window.
 *
 */

void z_split_window (zword lines)
{
    int y, x;

    /* Account for the status line in V3 */

    if (h_version == V3)
	lines++;

    /* Erase the upper window in V3 only */

    if (h_version == V3 && lines != 1)
	os_erase_area (2, 1, lines, h_screen_cols);

    /* Don't let the cursor drop out of the upper window */

    if (cwin == UPPER_WINDOW) {
	os_get_cursor (&y, &x);
	if (y > lines)
	    os_set_cursor ((h_version <= V3) ? 2 : 1, 1);
    }

    /* Don't let the cursor fall into the upper window */

    if (cwin == LOWER_WINDOW) {
	os_get_cursor (&y, &x);
	if (y <= lines)
	    os_set_cursor (lines + 1, 1);
    }

    /* Set the new window size */

    upper_size = lines;

}/* z_split_window */
