/*
 * process.c
 *
 * Main interpreter loop and some basic opcodes
 *
 */

#include "frotz.h"

#define FUNC 0
#define PROC 1
#define INTR 2

static int interrupt_finished = 0;

static zword load_operand (int);

/*
 * branch
 *
 * Take a jump after an instruction based on the flag, either true or
 * false. The jump can be modified by the change logic flag. Normally
 * jumps are taken when the flag is true. When the change logic flag
 * is set then the jump is taken when flag is false. The jump can either
 * be a byte or a positive or negative word range jump. An additional
 * feature is the return option. If the jump offset is zero or one then
 * that literal value is passed to the Z-machine, instead of a jump
 * being taken.
 *
 */

void branch (int flag)
{
    long current_pc;
    zword offset;
    zbyte specifier;
    zbyte off1;
    zbyte off2;

    /* Read specifier */

    CODE_BYTE (specifier)

    /* Handle reverse logic flag */

    if (specifier & 0x80)
	flag = !flag;

    /* Offset is in bottom 6 bits */

    off1 = specifier & 0x3f;

    /* We may need another byte to assemble the offset */

    if (!(specifier & 0x40)) {

	/* Propagate sign bit */

	if (off1 & 0x20)
	    off1 |= 0xc0;

	/* Fetch second byte */

	CODE_BYTE (off2)

	hi (offset) = off1;
	lo (offset) = off2;

    } else offset = off1;

    /* Branch if the flag is clear */

    if (flag == 0)

	if (offset > 1) {

	    GET_PC (current_pc)
	    current_pc += (short) offset - 2;
	    SET_PC (current_pc)

	} else z_ret (offset);

}/* branch */

/*
 * call_interrupt
 *
 * Call interrupt routine for z_read or z_sound_effect.
 *
 */

int call_interrupt (zword addr)
{

    /* Calls to address 0 return false */

    if (addr == 0)
	return (0);

    /* Call the interpreter directly */

    z_call (1, &addr, INTR);

    interpret ();
    interrupt_finished--;

    /* Return result of interrupt routine */

    return (*(sp++));

}/* call_interrupt */

/*
 * load_operand
 *
 * Load an operand, either a variable or a constant.
 *
 */

static zword load_operand (int type)
{
    zword addr;
    zword woperand;
    zbyte variable;
    zbyte boperand;

    if (type == 2) {

	/* Variable */

	CODE_BYTE (variable)

	if (variable == 0)
	    return (*(sp++));
	if (variable < 16)
	    return (*(fp - variable));

	addr = h_globals + ((variable - 16) << 1);
	LOW_WORD (addr, woperand)
	return (woperand);

    } else if (type == 1) {

	/* Small constant */

	CODE_BYTE (boperand)
	return (boperand);

    } else {

	/* Large constant */

	CODE_WORD (woperand)
	return (woperand);
    }

}/* load_operand */

/*
 * interpret
 *
 * Interpret Z-code
 *
 */

void interpret ()
{
    zword operand[8];
    zbyte opcode;
    zbyte specifier1;
    zbyte specifier2;
    zbyte optype;
    int extended;
    int count;
    int i;

    extended = 0;

    /* Loop until interrupt finished or quit instruction executed */

    while (interrupt_finished == 0) {

	/* Load opcode */

	CODE_BYTE (opcode)
	if (opcode == 0xbe) {
	    CODE_BYTE (opcode)
	    extended = 1;
	}

	if (opcode < 0x80 || opcode >= 0xc0 || extended) {

	    if (opcode < 0x80 && !extended) {

		/* Load two operands */

		operand[0] = load_operand ((opcode & 0x40) ? 2 : 1);
		operand[1] = load_operand ((opcode & 0x20) ? 2 : 1);
		count = 2;

		/* Opcode number is in bottom 5 bits */

		opcode &= 0x1f;

	    } else {

		/* Load operand specifier(s) */

		CODE_BYTE (specifier1)
		if (opcode == 0xec || opcode == 0xfa)
		    CODE_BYTE (specifier2)

		/* Load variable number of operands */

		count = 0;
		for (i = 6; i >= 0; i -= 2) {
		    optype = (specifier1 >> i) & 0x03;
		    if (optype == 3)
			break;
		    operand[count++] = load_operand (optype);
		}

		/* Load extra operands for extended CALL instructions */

		if (opcode == 0xec || opcode == 0xfa)
		    for (i = 6; i >= 0; i -= 2) {
			optype = (specifier2 >> i) & 0x03;
			if (optype == 3)
			    break;
			operand[count++] = load_operand (optype);
		    }

		/* Opcode number is in bottom 6 bits */

		opcode &= 0x3f;
	    }

	    if (extended) {

		/* Reset "extended" flag */

		extended = 0;

		/* These opcodes are reserved for future specification */

		if (opcode >= 0x1d)
		    continue;

		switch (opcode) {

		    /* Extended instructions */

		    case 0x00: z_save (count, operand); break;
		    case 0x01: z_restore (count, operand); break;
		    case 0x02: z_log_shift (operand[0], operand[1]); break;
		    case 0x03: z_arith_shift (operand[0], operand[1]); break;
		    case 0x04: z_set_font (operand[0]); break;

		    case 0x09: z_save_undo (); break;
		    case 0x0a: z_restore_undo (); break;

		    default: os_fatal ("Unknown opcode");
		}

	    } else

		switch (opcode) {

		    /* Two operand instructions */

		    case 0x01: z_je (count, operand); break;
		    case 0x02: z_jl (operand[0], operand[1]); break;
		    case 0x03: z_jg (operand[0], operand[1]); break;
		    case 0x04: z_dec_chk (operand[0], operand[1]); break;
		    case 0x05: z_inc_chk (operand[0], operand[1]); break;
		    case 0x06: z_jin (operand[0], operand[1]); break;
		    case 0x07: z_test (operand[0], operand[1]); break;
		    case 0x08: z_or (operand[0], operand[1]); break;
		    case 0x09: z_and (operand[0], operand[1]); break;
		    case 0x0a: z_test_attr (operand[0], operand[1]); break;
		    case 0x0b: z_set_attr (operand[0], operand[1]); break;
		    case 0x0c: z_clear_attr (operand[0], operand[1]); break;
		    case 0x0d: z_store (operand[0], operand[1]); break;
		    case 0x0e: z_insert_obj (operand[0], operand[1]); break;
		    case 0x0f: z_loadw (operand[0], operand[1]); break;
		    case 0x10: z_loadb (operand[0], operand[1]); break;
		    case 0x11: z_get_prop (operand[0], operand[1]); break;
		    case 0x12: z_get_prop_addr (operand[0], operand[1]); break;
		    case 0x13: z_get_next_prop (operand[0], operand[1]); break;
		    case 0x14: z_add (operand[0], operand[1]); break;
		    case 0x15: z_sub (operand[0], operand[1]); break;
		    case 0x16: z_mul (operand[0], operand[1]); break;
		    case 0x17: z_div (operand[0], operand[1]); break;
		    case 0x18: z_mod (operand[0], operand[1]); break;
		    case 0x19: z_call (count, operand, FUNC); break;
		    case 0x1a: z_call (count, operand, PROC); break;
		    case 0x1b: z_set_colour (operand[0], operand[1]); break;
		    case 0x1c: z_throw (operand[0], operand[1]); break;

		    /* Variable operand instructions */

		    case 0x20: z_call (count, operand, FUNC); break;
		    case 0x21: z_storew (operand[0], operand[1], operand[2]); break;
		    case 0x22: z_storeb (operand[0], operand[1], operand[2]); break;
		    case 0x23: z_put_prop (operand[0], operand[1], operand[2]); break;
		    case 0x24: z_read (count, operand); break;
		    case 0x25: z_print_char (operand[0]); break;
		    case 0x26: z_print_num (operand[0]); break;
		    case 0x27: z_random (operand[0]); break;
		    case 0x28: z_push (operand[0]); break;
		    case 0x29: z_pull (operand[0]); break;
		    case 0x2a: z_split_window (operand[0]); break;
		    case 0x2b: z_set_window (operand[0]); break;
		    case 0x2c: z_call (count, operand, FUNC); break;
		    case 0x2d: z_erase_window (operand[0]); break;
		    case 0x2e: z_erase_line (operand[0]); break;
		    case 0x2f: z_set_cursor (operand[0], operand[1]); break;
		    case 0x30: z_get_cursor (operand[0]); break;
		    case 0x31: z_set_text_style (operand[0]); break;
		    case 0x32: z_buffer_mode (operand[0]); break;
		    case 0x33: z_output_stream (operand[0], operand[1]); break;
		    case 0x34: z_input_stream (operand[0]); break;
		    case 0x35: z_sound_effect (count, operand); break;
		    case 0x36: z_read_char (count, operand); break;
		    case 0x37: z_scan_table (count, operand); break;
		    case 0x38: z_not (operand[0]); break;
		    case 0x39: z_call (count, operand, PROC); break;
		    case 0x3a: z_call (count, operand, PROC); break;
		    case 0x3b: z_tokenise (count, operand); break;
		    case 0x3c: z_encode_text (operand[0], operand[1], operand[2], operand[3]); break;
		    case 0x3d: z_copy_table (operand[0], operand[1], operand[2]); break;
		    case 0x3e: z_print_table (count, operand); break;
		    case 0x3f: z_check_arg_count (operand[0]); break;

		    default: os_fatal ("Unknown opcode");
		}

	} else if (opcode < 0xb0) {

	    /* Load operand */

	    operand[0] = load_operand ((opcode >> 4) & 0x03);

	    switch (opcode & 0x0f) {

		/* Single operand instructions */

		case 0x00: z_jz (operand[0]); break;
		case 0x01: z_get_sibling (operand[0]); break;
		case 0x02: z_get_child (operand[0]); break;
		case 0x03: z_get_parent (operand[0]); break;
		case 0x04: z_get_prop_len (operand[0]); break;
		case 0x05: z_inc (operand[0]); break;
		case 0x06: z_dec (operand[0]); break;
		case 0x07: z_print_addr (operand[0]); break;
		case 0x08: z_call (1, operand, FUNC); break;
		case 0x09: z_remove_obj (operand[0]); break;
		case 0x0a: z_print_obj (operand[0]); break;
		case 0x0b: z_ret (operand[0]); break;
		case 0x0c: z_jump (operand[0]); break;
		case 0x0d: z_print_paddr (operand[0]); break;
		case 0x0e: z_load (operand[0]); break;
		case 0x0f: if (h_version <= V4)
			       z_not (operand[0]);
			   else
			       z_call (1, operand, PROC);
			   break;

		default: os_fatal ("Unknown opcode");
	    }

	} else

	    switch (opcode & 0x0f) {

		/* Zero operand instructions */

		case 0x00: z_ret (1); break;
		case 0x01: z_ret (0); break;
		case 0x02: z_print (); break;
		case 0x03: z_print_ret (); break;
		case 0x04: /* nop */ break;
		case 0x05: z_save (0, 0); break;
		case 0x06: z_restore (0, 0); break;
		case 0x07: z_restart (); break;
		case 0x08: z_ret (*(sp++)); break;
		case 0x09: if (h_version <= V4)
			       z_pop ();
			   else
			       z_catch ();
			   break;
		case 0x0a: return;
		case 0x0b: z_new_line (); break;
		case 0x0c: z_show_status (); break;
		case 0x0d: z_verify (); break;

		case 0x0f: /* piracy */ branch (1); break;

		default: os_fatal ("Unknown opcode");
	    }
    }

}/* interpret */

/*
 * store
 *
 * Store an operand, either as a variable or pushed on the stack.
 *
 */

void store (zword value)
{
    zword addr;
    zbyte variable;

    /* Read operand specifier byte */

    CODE_BYTE (variable)

    /* Store value */

    if (variable == 0)
	*(--sp) = value;
    else if (variable <= 15)
	*(fp - variable) = value;
    else {
	addr = h_globals + ((variable - 16) << 1);
	SET_WORD (addr, value)
    }

}/* store */

/*
 * z_call
 *
 * Call a subroutine. Save PC and FP then load new PC and initialise
 * new stack frame.
 *
 */

void z_call (int argc, zword *argv, int type)
{
    long current_pc;
    zword arg;
    zbyte count;
    int i;

    /* Convert calls to 0 as returning FALSE */

    if (argv[0] == 0) {
	if (type == FUNC)
	    store (0);
	return;
    }

    /* Save current PC, FP and argument count on stack */

    GET_PC (current_pc)

    *(--sp) = current_pc >> 9;
    *(--sp) = current_pc & 0x1ff;
    *(--sp) = fp - (stack + 1);
    *(--sp) = argc - 1;

    hi (*sp) = type;

    /* Create FP for new subroutine and load new PC */

    fp = sp;

    current_pc = ((long) h_functions_offset << 3) + ((long) argv[0] << story_shift);
    if (current_pc >= story_size)
	os_fatal ("Call to illegal address");

    SET_PC (current_pc)

    /* Read argument count and initialise local variables */

    CODE_BYTE (count)
    if (count > 15)
	os_fatal ("Call to non-routine");

    for (i = 1; i <= count; i++) {
	if (h_version <= V4)
	    CODE_WORD (arg)
	else
	    arg = 0;
	*(--sp) = (--argc > 0) ? argv[i] : arg;
    }

}/* z_call */

/*
 * z_catch
 *
 * Return the value of the frame pointer for later use with z_throw.
 *
 */

void z_catch (void)
{

    store (fp - stack);

}/* z_catch */

/*
 * z_check_arg_count
 *
 * Branch if the given argument is present.
 *
 */

void z_check_arg_count (zword argc)
{

    branch (argc <= (lo (*fp)));

}/* z_check_arg_count */

/*
 * z_dec
 *
 * Decrement a variable.
 *
 */

void z_dec (zword variable)
{
    zword value;
    zword addr;

    if (variable == 0)
	(*fp)--;
    else if (variable < 16)
	(*(fp - variable))--;
    else {
	addr = h_globals + ((variable - 16) << 1);
	LOW_WORD (addr, value)
	value--;
	SET_WORD (addr, value)
    }

}/* z_dec */

/*
 * z_dec_chk
 *
 * Decrement a variable and then check its value against a target.
 *
 */

void z_dec_chk (zword variable, zword target)
{
    zword value;
    zword addr;

    if (variable == 0)
	value = --(*fp);
    else if (variable < 16)
	value = --(*(fp - variable));
    else {
	addr = h_globals + ((variable - 16) << 1);
	LOW_WORD (addr, value)
	value--;
	SET_WORD (addr, value)
    }

    branch ((short) value < (short) target);

}/* z_dec_chk */

/*
 * z_inc
 *
 * Increment a variable.
 *
 */

void z_inc (zword variable)
{
    zword value;
    zword addr;

    if (variable == 0)
	(*fp)++;
    else if (variable < 16)
	(*(fp - variable))++;
    else {
	addr = h_globals + ((variable - 16) << 1);
	LOW_WORD (addr, value)
	value++;
	SET_WORD (addr, value)
    }

}/* z_inc */

/*
 * z_inc_chk
 *
 * Increment a variable and then check its value against a target.
 *
 */

void z_inc_chk (zword variable, zword target)
{
    zword value;
    zword addr;

    if (variable == 0)
	value = ++(*fp);
    else if (variable < 16)
	value = ++(*(fp - variable));
    else {
	addr = h_globals + ((variable - 16) << 1);
	LOW_WORD (addr, value)
	value++;
	SET_WORD (addr, value)
    }

    branch ((short) value > (short) target);

}/* z_inc_chk */

/*
 * z_jump
 *
 * Unconditional jump. Jump is PC relative.
 *
 */

void z_jump (zword offset)
{
    long current_pc;

    GET_PC (current_pc)

    current_pc += (short) offset - 2;
    if (current_pc >= story_size)
	os_fatal ("Jump to illegal address");

    SET_PC (current_pc)

}/* z_jump */

/*
 * z_load
 *
 * Set the value of a variable.
 *
 */

void z_load (zword variable)
{
    zword value;
    zword addr;

    if (variable == 0)
	value = *sp;
    else if (variable < 16)
	value = *(fp - variable);
    else {
	addr = h_globals + ((variable - 16) << 1);
	LOW_WORD (addr, value)
    }

    store (value);

}/* z_load */

/*
 * z_pop
 *
 * Pop a value off the system stack and throw it away.
 *
 */

void z_pop (void)
{

    sp++;

}/* z_pop */

/*
 * z_pull
 *
 * Pop a variable off the system stack and store it in a variable.
 *
 */

void z_pull (zword variable)
{

    z_store (variable, *(sp++));

}/* z_pull */

/*
 * z_push
 *
 * Push a value onto the system stack.
 *
 */

void z_push (zword value)
{

    *(--sp) = value;

}/* z_push */

/*
 * z_ret
 *
 * Return from subroutine. Restore the old stack frame.
 *
 */

void z_ret (zword value)
{
    long current_pc;
    zbyte type;

    /* Clean stack */

    sp = fp;

    /* Remove argument count, FP and PC from the stack */

    type = hi (*sp);

    sp++;
    fp = *(sp++) + (stack + 1);
    current_pc = *(sp++);
    current_pc |= (long) *(sp++) << 9;

    SET_PC (current_pc)

    /* Return the value (type == FUNC), push it onto the system stack
       (type == INTR) or throw it away (type == PROC) */

    if (type == FUNC)
	store (value);

    if (type == INTR) {
	interrupt_finished++;
	*(--sp) = value;
    }

}/* z_ret */

/*
 * z_store
 *
 * Store a variable, either: a local variable, a global variable
 * or the top of the stack.
 *
 */

void z_store (zword variable, zword value)
{
    zword addr;

    if (variable == 0)
	*sp = value;
    else if (variable <= 15)
	*(fp - variable) = value;
    else {
	addr = h_globals + ((variable - 16) << 1);
	SET_WORD (addr, value)
    }

}/* z_store */

/*
 * z_throw
 *
 * Remove one or more stack frames and return (see catch).
 *
 */

void z_throw (zword value, zword new_fp)
{

    fp = stack + new_fp;

    /* Return value to the Z-machine */

    z_ret (value);

}/* z_throw */
