/*
 * object.c
 *
 * Object manipulation routines
 *
 */

#include "frotz.h"

#define O1_PARENT 4
#define O1_SIBLING 5
#define O1_CHILD 6
#define O1_PROPERTY_OFFSET 7
#define O1_SIZE 9

#define P1_MAX_PROPERTIES 0x20

#define O4_PARENT 6
#define O4_SIBLING 8
#define O4_CHILD 10
#define O4_PROPERTY_OFFSET 12
#define O4_SIZE 14

#define P4_MAX_PROPERTIES 0x40

#define MAX_OBJECT 2000

static zword object_address (int obj);
static zword first_property (int obj);
static zword next_property (zword addr);

/*
 * object_address
 *
 * Calculate the address of an object in the data area.
 *
 */

static zword object_address (int obj)
{

    /* Check object number */

    if (obj > MAX_OBJECT || h_version <= V3 && obj > 255)
	os_fatal ("Illegal object number");

    /* Return object address */

    if (h_version <= V3)
	return (h_objects + (2 * (P1_MAX_PROPERTIES - 1)) + ((obj - 1) * O1_SIZE));
    else
	return (h_objects + (2 * (P4_MAX_PROPERTIES - 1)) + ((obj - 1) * O4_SIZE));

}/* object_address */

/*
 * object_name
 *
 * Return the address of the given object's name.
 *
 */

zword object_name (int obj)
{
    zword obj_addr;
    zword name_addr;

    obj_addr = object_address (obj);

    /* The object name is found at the start of the property list */

    if (h_version <= V3)
	obj_addr += O1_PROPERTY_OFFSET;
    else
	obj_addr += O4_PROPERTY_OFFSET;

    LOW_WORD (obj_addr, name_addr)

    return (name_addr);

}/* object_name */

/*
 * first_property
 *
 * Calculate the start address of the property list associated with
 * an object.
 *
 */

static zword first_property (int obj)
{
    zword prop_addr;
    zbyte size;

    /* Fetch address of object name */

    prop_addr = object_name (obj);

    /* Get length of object name */

    LOW_BYTE (prop_addr, size)

    /* Add name length to pointer */

    return (prop_addr + (size << 1) + 1);

}/* first_property */

/*
 * next_property
 *
 * Calculate the address of the next property in a property list.
 *
 */

static zword next_property (zword prop_addr)
{
    zbyte value;

    /* Load the current property id */

    LOW_BYTE (prop_addr, value)
    prop_addr++;

    /* Calculate the length of this property */

    if (h_version <= V3)
	value >>= 5;
    else if (!(value & 0x80))
	value >>= 6;
    else {
	LOW_BYTE (prop_addr, value)
	value &= 0x3f;
    }

    /* Add property length to current property pointer */

    return (prop_addr + value + 1);

}/* next_property */

/*
 * z_clear_attr
 *
 * Clear an attribute bit.
 *
 */

void z_clear_attr (zword obj, zword bit)
{
    zword obj_addr;
    zbyte value;

    if (bit >= 48 || h_version <= V3 && bit >= 32)
	os_fatal ("Illegal attribute number");

    /* If we are monitoring attribute assignment display a short note */

    if (option_attribute_assignment != 0) {
	z_output_stream (5, 0);
	display_string ("clear ");
	z_print_obj (obj);
	display_string ("(");
	z_print_num (bit);
	display_string (")");
	z_output_stream (-5, 0);
    }

    /* Get attribute address */

    obj_addr = object_address (obj) + (bit >> 3);

    /* Clear attribute bit */

    LOW_BYTE (obj_addr, value)
    value &= ~(0x80 >> (bit & 7));
    SET_BYTE (obj_addr, value)

}/* z_clear_attr */

/*
 * z_jin
 *
 * Branch if object 2 is the parent of object 1.
 *
 */

void z_jin (zword obj1, zword obj2)
{
    zword obj_addr;
    zword wparent;
    zbyte bparent;

    /* If we are monitoring object location display a short note */

    if (option_object_locating != 0) {
	z_output_stream (5, 0);
	display_string ("check ");
	z_print_obj (obj1);
	display_string (" in ");
	z_print_obj (obj2);
	z_output_stream (-5, 0);
    }

    obj_addr = object_address (obj1);

    if (h_version <= V3) {

	/* Get parent id from object */

	obj_addr += O1_PARENT;
	LOW_BYTE (obj_addr, bparent)

	/* Branch if the parent is obj2 */

	branch (bparent == obj2);

    } else {

	/* Get parent id from object */

	obj_addr += O4_PARENT;
	LOW_WORD (obj_addr, wparent)

	/* Branch if the parent is obj2 */

	branch (wparent == obj2);
    }

}/* z_jin */

/*
 * z_get_child
 *
 * Load the child id of an object and branch if the child id is not 0.
 *
 */

void z_get_child (zword obj)
{
    zword obj_addr;
    zword wchild;
    zbyte bchild;

    /* If we are monitoring object locating display a short note */

    if (option_object_locating != 0) {
	z_output_stream (5, 0);
	display_string ("examine contents of ");
	z_print_obj (obj);
	z_output_stream (-5, 0);
    }

    obj_addr = object_address (obj);

    if (h_version <= V3) {

	/* Get child id from object */

	obj_addr += O1_CHILD;
	LOW_BYTE (obj_addr, bchild)

	/* Store child id and branch */

	store (bchild);
	branch (bchild);

    } else {

	/* Get child id from object */

	obj_addr += O4_CHILD;
	LOW_WORD (obj_addr, wchild)

	/* Store child id and branch */

	store (wchild);
	branch (wchild);
    }

}/* z_get_child */

/*
 * z_get_next_prop
 *
 * Store the number of the next property. If the current property is
 * zero then store the number of the first property instead.
 *
 */

void z_get_next_prop (zword obj, zword prop)
{
    zword prop_addr;
    zbyte value;

    /* Load address of first property */

    prop_addr = first_property (obj);

    if (prop != 0) {

	/* Scan down the property list */

	do {
	    LOW_BYTE (prop_addr, value)
	    prop_addr = next_property (prop_addr);
	} while ((value & property_mask) > prop);

	/* Exit if the property does not exist */

	if ((value & property_mask) != prop)
	    os_fatal ("No such property");
    }

    /* Return the property id */

    LOW_BYTE (prop_addr, value)
    store (value & property_mask);

}/* z_get_next_prop */

/*
 * z_get_parent
 *
 * Load the parent id of an object.
 *
 */

void z_get_parent (zword obj)
{
    zword obj_addr;
    zword wparent;
    zbyte bparent;

    /* If we are monitoring object locating display a short note */

    if (option_object_locating != 0) {
	z_output_stream (5, 0);
	display_string ("locate ");
	z_print_obj (obj);
	z_output_stream (-5, 0);
    }

    obj_addr = object_address (obj);

    if (h_version <= V3) {

	/* Get parent id from object */

	obj_addr += O1_PARENT;
	LOW_BYTE (obj_addr, bparent)

	/* Store parent */

	store (bparent);

    } else {

	/* Get parent id from object */

	obj_addr += O4_PARENT;
	LOW_WORD (obj_addr, wparent)

	/* Store parent */

	store (wparent);
    }

}/* z_get_parent */

/*
 * z_get_prop
 *
 * Load a property from a property list. Properties are held in list
 * sorted by property id in descending order. There is also a concept
 * of a default property for reading only. The default properties are
 * held in a table right at the start of the object table.
 *
 */

void z_get_prop (zword obj, zword prop)
{
    zword prop_addr;
    zword wprop_val;
    zbyte bprop_val;
    zbyte value;

    /* Load address of first property */

    prop_addr = first_property (obj);

    /* Scan down the property list */

    for (;;) {
	LOW_BYTE (prop_addr, value)
	if ((value & property_mask) <= prop)
	    break;
	prop_addr = next_property (prop_addr);
    }

    if ((value & property_mask) == prop) {

	/* Load property (byte or word sized) */

	prop_addr++;

	if (h_version <= V3 && !(value & 0xe0) || h_version >= V4 && !(value & 0xc0)) {

	    LOW_BYTE (prop_addr, bprop_val)
	    wprop_val = bprop_val;

	} else LOW_WORD (prop_addr, wprop_val)

    } else {

	/* Load default value */

	prop_addr = h_objects + ((prop - 1) << 1);
	LOW_WORD (prop_addr, wprop_val)
    }

    /* Store the property value */

    store (wprop_val);

}/* z_get_prop */

/*
 * z_get_prop_addr
 *
 * Get the address of the given property.
 *
 */

void z_get_prop_addr (zword obj, zword prop)
{
    zword prop_addr;
    zbyte value;

    /* "Beyond Zork" accidently uses the address of the vocabulary
       entry circlet instead of its object number... */

    if (beyond_zork_flag != 0 && obj >= 1000) {
	store (0);
	return;
    }

    /* Load address of first property */

    prop_addr = first_property (obj);

    /* Scan down the property list */

    for (;;) {
	LOW_BYTE (prop_addr, value)
	if ((value & property_mask) <= prop)
	    break;
	prop_addr = next_property (prop_addr);
    }

    /* Calculate the property address or return zero */

    if ((value & property_mask) == prop) {

	if (h_version >= V4 && (value & 0x80))
	    prop_addr++;
	store (prop_addr + 1);

    } else store (0);

}/* z_get_prop_addr */

/*
 * z_get_prop_len
 *
 * Calculate the length of a given property.
 *
 */

void z_get_prop_len (zword prop_addr)
{
    zbyte value;

    /* Back up the property pointer to the property id */

    prop_addr--;

    /* Calculate length of property */

    LOW_BYTE (prop_addr, value)

    if (h_version <= V3)
	value = (value >> 5) + 1;
    else if (!(value & 0x80))
	value = (value >> 6) + 1;
    else
	value &= 0x3f;

    /* Store length of property */

    store (value);

}/* z_get_prop_len */

/*
 * z_get_sibling
 *
 * Load the sibling id of an object and branch if the sibling is not 0.
 *
 */

void z_get_sibling (zword obj)
{
    zword obj_addr;
    zword wsibling;
    zbyte bsibling;

    obj_addr = object_address (obj);

    if (h_version <= V3) {

	/* Get sibling id from object */

	obj_addr += O1_SIBLING;
	LOW_BYTE (obj_addr, bsibling)

	/* Store sibling and branch */

	store (bsibling);
	branch (bsibling);

    } else {

	/* Get sibling id from object */

	obj_addr += O4_SIBLING;
	LOW_WORD (obj_addr, wsibling)

	/* Store sibling and branch */

	store (wsibling);
	branch (wsibling);
    }

}/* z_get_sibling */

/*
 * z_insert_obj
 *
 * Insert object 1 as the child of object 2 after first removing it from
 * its previous parent. The object is inserted at the front of the child
 * object chain.
 *
 */

void z_insert_obj (zword obj1, zword obj2)
{
    zword obj1_addr;
    zword obj2_addr;
    zword wchild;
    zbyte bchild;
    int saved;

    /* If we are monitoring object movements display a short note */

    if (option_object_movement != 0) {
	z_output_stream (5, 0);
	display_string ("move ");
	z_print_obj (obj1);
	display_string (" to ");
	z_print_obj (obj2);
	z_output_stream (-5, 0);
    }

    /* Get addresses of both objects */

    obj1_addr = object_address (obj1);
    obj2_addr = object_address (obj2);

    /* Remove object 1 from current parent */

    saved = option_object_movement;
    option_object_movement = 0;

    z_remove_obj (obj1);

    option_object_movement = saved;

    /* Make object 1 first child of object 2 */

    if (h_version <= V3) {
	obj1_addr += O1_PARENT;
	SET_BYTE (obj1_addr, obj2)
	obj2_addr += O1_CHILD;
	LOW_BYTE (obj2_addr, bchild)
	SET_BYTE (obj2_addr, obj1)
	obj1_addr += O1_SIBLING - O1_PARENT;
	SET_BYTE (obj1_addr, bchild)
    } else {
	obj1_addr += O4_PARENT;
	SET_WORD (obj1_addr, obj2)
	obj2_addr += O4_CHILD;
	LOW_WORD (obj2_addr, wchild)
	SET_WORD (obj2_addr, obj1)
	obj1_addr += O4_SIBLING - O4_PARENT;
	SET_WORD (obj1_addr, wchild)
    }

}/* z_insert_obj */

/*
 * z_put_prop
 *
 * Set a property to the given value.
 *
 */

void z_put_prop (zword obj, zword prop, zword prop_val)
{
    zword prop_addr;
    zbyte value;

    /* Load address of first property */

    prop_addr = first_property (obj);

    /* Scan down the property list */

    for (;;) {
	LOW_BYTE (prop_addr, value)
	if ((value & property_mask) <= prop)
	    break;
	prop_addr = next_property (prop_addr);
    }

    /* Exit if the property does not exist */

    if ((value & property_mask) != prop)
	os_fatal ("No such property");

    /* Store the new property value (byte or word sized) */

    prop_addr++;

    if (h_version <= V3 && !(value & 0xe0) || h_version >= V4 && !(value & 0xc0))
	SET_BYTE (prop_addr, prop_val)
    else
	SET_WORD (prop_addr, prop_val)

}/* z_put_prop */

/*
 * z_remove_obj
 *
 * Remove an object by unlinking from its parent object and from its
 * siblings.
 *
 */

void z_remove_obj (zword obj)
{
    zword obj_addr;
    zword parent_addr;
    zword sibling_addr;
    zword wparent;
    zword wyounger_sibling;
    zword wolder_sibling;
    zbyte bparent;
    zbyte byounger_sibling;
    zbyte bolder_sibling;

    /* If we are monitoring object movements display a short note */

    if (option_object_movement != 0) {
	z_output_stream (5, 0);
	display_string ("remove ");
	z_print_obj (obj);
	z_output_stream (-5, 0);
    }

    obj_addr = object_address (obj);

    if (h_version <= V3) {

	/* Get parent of object, and return if no parent */

	obj_addr += O1_PARENT;
	LOW_BYTE (obj_addr, bparent)
	if (bparent == 0)
	    return;

	/* Get (older) sibling of object and set both parent and sibling
	   pointers to 0 */

	SET_BYTE (obj_addr, 0)
	obj_addr += O1_SIBLING - O1_PARENT;
	LOW_BYTE (obj_addr, bolder_sibling)
	SET_BYTE (obj_addr, 0)

	/* Get first child of parent (the youngest sibling of the object) */

	parent_addr = object_address (bparent) + O1_CHILD;
	LOW_BYTE (parent_addr, byounger_sibling)

	/* Remove object from the list of siblings */

	if (byounger_sibling == obj)
	    SET_BYTE (parent_addr, bolder_sibling)
	else {
	    do {
		sibling_addr = object_address (byounger_sibling) + O1_SIBLING;
		LOW_BYTE (sibling_addr, byounger_sibling)
	    } while (byounger_sibling != obj);
	    SET_BYTE (sibling_addr, bolder_sibling)
	}

    } else {

	/* Get parent of object, and return if no parent */

	obj_addr += O4_PARENT;
	LOW_WORD (obj_addr, wparent)
	if (wparent == 0)
	    return;

	/* Get (older) sibling of object and set both parent and sibling
	   pointers to 0 */

	SET_WORD (obj_addr, 0)
	obj_addr += O4_SIBLING - O4_PARENT;
	LOW_WORD (obj_addr, wolder_sibling)
	SET_WORD (obj_addr, 0)

	/* Get first child of parent (the youngest sibling of the object) */

	parent_addr = object_address (wparent) + O4_CHILD;
	LOW_WORD (parent_addr, wyounger_sibling)

	/* Remove object from the list of siblings */

	if (wyounger_sibling == obj)
	    SET_WORD (parent_addr, wolder_sibling)
	else {
	    do {
		sibling_addr = object_address (wyounger_sibling) + O4_SIBLING;
		LOW_WORD (sibling_addr, wyounger_sibling)
	    } while (wyounger_sibling != obj);
	    SET_WORD (sibling_addr, wolder_sibling)
	}
    }

}/* z_remove_obj */

/*
 * z_set_attr
 *
 * Set an attribute bit.
 *
 */

void z_set_attr (zword obj, zword bit)
{
    zword obj_addr;
    zbyte value;

    if (bit >= 48 || h_version <= V3 && bit >= 32)
	os_fatal ("Illegal attribute number");

    /* If we are monitoring attribute assignment display a short note */

    if (option_attribute_assignment != 0) {
	z_output_stream (5, 0);
	display_string ("set ");
	z_print_obj (obj);
	display_string ("(");
	z_print_num (bit);
	display_string (")");
	z_output_stream (-5, 0);
    }

    /* Get attribute address */

    obj_addr = object_address (obj) + (bit >> 3);

    /* Load attribute byte */

    LOW_BYTE (obj_addr, value)

    /* Set attribute bit */

    value |= 0x80 >> (bit & 7);

    /* Store attribute byte */

    SET_BYTE (obj_addr, value)

}/* z_set_attr */

/*
 * z_test_attr
 *
 * Test if an attribute bit is set.
 *
 */

void z_test_attr (zword obj, zword bit)
{
    zword obj_addr;
    zbyte value;

    if (bit >= 48 || h_version <= V3 && bit >= 32)
	os_fatal ("Illegal attribute number");

    /* If we are monitoring attribute testing display a short note */

    if (option_attribute_testing != 0) {
	z_output_stream (5, 0);
	display_string ("test ");
	z_print_obj (obj);
	display_string ("(");
	z_print_num (bit);
	display_string (")");
	z_output_stream (-5, 0);
    }

    /* Get attribute address */

    obj_addr = object_address (obj) + (bit >> 3);

    /* Load attribute byte */

    LOW_BYTE (obj_addr, value)

    /* Test attribute */

    branch (value & (0x80 >> (bit & 7)));

}/* z_test_attr */
