/*
 * math.c
 *
 * Arithmetic, compare and logical opcodes
 *
 */

#include <stdlib.h>
#include <time.h>
#include "frotz.h"

/*
 * random_number
 *
 * Calculate a random number between 1 and range (if range > 0),
 * randomize (if range = 0) or make the random number generator
 * predictable (if range < 0). Following G. Nelson's suggestion,
 * a seed value less than 1000 generates a sequence running from
 * 1 to |range|, and a seed value greater or equal 1000 is used
 * as a standard random number seed.
 *
 */

int random_number (int range)
{
    static int interval = 0;
    static int counter = 0;
    int result;

    if (range == 0) {

	/* Randomize by using the current time */

	srand (time (NULL));
	interval = 0;
	result = 0;

    } else if (range > -1000 && range < 0) {

	/* Start the sequence 1,...,range,1,...,range... */

	interval = -range;
	counter = 0;
	result = 0;

    } else if (range <= -1000) {

	/* Seed the random number generator with -range */

	srand (-range);
	interval = 0;
	result = 0;

    } else if (interval != 0) {

	/* Generate a random number in "sequence" mode */

	result = counter % range + 1;
	if (++counter == interval)
	    counter = 0;

    } else result = rand () % range + 1;

    return (result);

}/* random_number */

/*
 * z_add
 *
 * Add two operands.
 *
 */

void z_add (zword a, zword b)
{

    store (a + b);

}/* z_add */

/*
 * z_and
 *
 * Bitwise AND.
 *
 */

void z_and (zword a, zword b)
{

    store (a & b);

}/* z_and */

/*
 * z_arith_shift
 *
 * Aritmetic shift +/- b bits.
 *
 */

void z_arith_shift (zword a, zword b)
{

    if ((short) b > 0)
	store ((short) a << b);
    else
	store ((short) a >> -b);

}/* z_arith_shift */

/*
 * z_div
 *
 * Unsigned division.
 *
 */

void z_div (zword a, zword b)
{

    if (b == 0)
	os_fatal ("Division by zero");

    store (a / b);

}/* z_div */

/*
 * z_je
 *
 * Jump if operand 1 is equal to any other operand.
 *
 */

void z_je (int argc, zword *argv)
{

    branch (argc >= 2 && (argv[0] == argv[1] ||
	    argc >= 3 && (argv[0] == argv[2] ||
	    argc == 4 && (argv[0] == argv[3]))));

}/* z_je */

/*
 * z_jg
 *
 * Branch if operand 1 is greater than operand 2.
 *
 */

void z_jg (zword a, zword b)
{

    branch ((short) a > (short) b);

}/* z_jg */

/*
 * z_jl
 *
 * Branch if operand 1 is less than operand 2.
 *
 */

void z_jl (zword a, zword b)
{

    branch ((short) a < (short) b);

}/* z_jl */

/*
 * z_jz
 *
 * Branch if operand is zero.
 *
 */

void z_jz (zword a)
{

    branch (a == 0);

}/* z_jz */

/*
 * z_log_shift
 *
 * Logical shift +/- b bits.
 *
 */

void z_log_shift (zword a, zword b)
{

    if ((short) b > 0)
	store (a << b);
    else
	store (a >> -b);

}/* z_log_shift */

/*
 * z_mod
 *
 * Remainder after unsigned division.
 *
 */

void z_mod (zword a, zword b)
{

    if (b == 0)
	os_fatal ("Division by zero");

    store (a % b);

}/* z_mod */

/*
 * z_mul
 *
 * Multiply two operands.
 *
 */

void z_mul (zword a, zword b)
{

    store (a * b);

}/* z_mul */

/*
 * z_not
 *
 * Bitwise NOT.
 *
 */

void z_not (zword a)
{

    store (~a);

}/* z_not */

/*
 * z_or
 *
 * Bitwise OR.
 *
 */

void z_or (zword a, zword b)
{

    store (a | b);

}/* z_or */

/*
 * z_random
 *
 * Random number between 1 and operand.
 *
 */

void z_random (zword range)
{

    store (random_number ((short) range));

}/* z_random */

/*
 * z_sub
 *
 * Subtract two operands.
 *
 */

void z_sub (zword a, zword b)
{

    store (a - b);

}/* z_sub */

/*
 * z_test
 *
 * Branch if all bits of "flags" are set in "bitmap".
 *
 */

void z_test (zword bitmap, zword flags)
{

    branch ((bitmap & flags) == flags);

}/* z_test */
