/*
 * input.c
 *
 * High level input routines
 *
 */

#include <ctype.h>
#include <string.h>
#include "frotz.h"

static char euro_tolower[] = {
    155, 156, 157, 155, 156, 157, 161, 162, 163, 164,
    165, 166, 164, 165, 169, 170, 171, 172, 173, 174,
    169, 170, 171, 172, 173, 174, 181, 182, 183, 184,
    185, 181, 182, 183, 184, 185, 191, 192, 193, 194,
    195, 191, 192, 193, 194, 195, 201, 201, 203, 203,
    205, 206, 207, 205, 206, 207, 211, 211, 213, 213,
    215, 216, 215, 216, 219, 220, 220, 222, 223
};

/*
 * is_terminator
 *
 * Check if the given key is an input terminator.
 *
 */

int is_terminator (int key)
{
    zword addr;
    zbyte c;

    if (key == 0 || key == 13 || key >= HOT_KEY_MIN && key <= HOT_KEY_MAX)
	return (1);

    addr = h_terminating_keys;

    if (addr != 0)

	do {
	    LOW_BYTE (addr, c)
	    if (c == key || c == 255)
		return (1);
	    addr++;
	} while (c != 0);

    return (0);

}/* is_terminator */

/*
 * z_read
 *
 * Read a line of input with optional timeout.
 *
 *    argv[0] = text buffer address
 *    argv[1] = token buffer address
 *    argv[2] = timeout value in 1/10 seconds (optional)
 *    argv[3] = timeout action routine (optional)
 *
 */

void z_read (int argc, zword *argv)
{
    char buffer[120];
    zbyte max_size;
    zbyte c;
    int read_size;
    int terminator;
    int success;
    int i;

    /* Supply default parameters */

    if (argc < 3)
	argv[2] = 0;
    if (argc < 4)
	argv[3] = 0;

    /* Set the "input_flag" (for a workaround in z_sound_effect) */

    input_flag = 1;

    /* Refresh status line (V1 to V3 only) */

    if (h_version <= V3)
	z_show_status ();

    /* Flush any buffered output before read */

    flush_buffer ();

    /* Calculate maximum input size */

    LOW_BYTE (argv[0], max_size)
    if (h_version <= V4)
	max_size--;

    /* Calculate initial input size */

    if (h_version >= V5)
	LOW_BYTE (argv[0] + 1, read_size)
    else
	read_size = 0;

    /* Copy initial input to local buffer */

    for (i = 0; i < read_size; i++) {
	LOW_BYTE (argv[0] + 2 + i, c)
	buffer[i] = c;
    }
    buffer[i] = 0;

    /* Remove initial input from transscript file */

    if (scripting && cwin == LOWER_WINDOW)
	script_backup (read_size);

continue_input:

    /* Read input from command file or keyboard */

    if (replaying)
	terminator = playback_line (max_size, buffer);
    else
	terminator = os_read (max_size, buffer, argv[2]);

    /* Retry if input failed */

    if (terminator == -1)
	goto continue_input;

    /* Copy input line to command file */

    if (recording && !replaying)
	record_line (buffer, terminator);

    /* Handle timeouts */

    if (terminator == 0) {

	newline_flag = 0;

	/* Continue input if the timeout routine returns 0 */

	if (call_interrupt (argv[3]) == 0) {

	    /* The input line may have to be redrawn */

	    if (newline_flag != 0 && cwin == LOWER_WINDOW)
		display_string (buffer);

	    goto continue_input;
	}
    }

    /* Reset line counter (but account for the following newline) */

    line_count = option_context_lines - 1;

    /* Handle hot keys */

    if (terminator >= HOT_KEY_MIN && terminator <= HOT_KEY_MAX) {

	if (cwin == LOWER_WINDOW) {

	    display_new_line ();

	    switch (terminator) {
		case HOT_KEY_RECORDING: hot_key_recording (); break;
		case HOT_KEY_PLAYBACK: hot_key_playback (); break;
		case HOT_KEY_SEED: hot_key_seed (); break;
		case HOT_KEY_UNDO: hot_key_undo (&success); break;
	    }

	    if (terminator == HOT_KEY_UNDO && success == 2)
		return;

	    display_string ("Input continues...");
	    display_new_line ();
	    display_new_line ();
	    display_string (">");
	    display_string (buffer);
	}

	goto continue_input;
    }

    /* Handle mouse clicks */

    if (terminator == 253 || terminator == 254) {
	z_storew (h_mouse_table, 1, mouse_x);
	z_storew (h_mouse_table, 2, mouse_y);
    }

    /* Copy input line to transscript file */

    if (scripting && cwin == LOWER_WINDOW)
	script_string (buffer);

    /* Print a newline if the input terminated normally */

    if (terminator == 13)
	z_new_line ();

    /* Lowercase local buffer */

    read_size = strlen (buffer);

    for (i = 0; i < read_size; i++) {
	c = buffer[i];
	if (c >= 155 && c <= 223)
	   c = euro_tolower[c - 155];
	else
	   c = tolower (c);
	buffer[i] = c;
    }

    /* Copy local buffer back to dynamic memory */

    for (i = 0; i < read_size; i++)
	z_storeb (argv[0], ((h_version <= V4) ? 1 : 2) + i, buffer[i]);

    /* In V1 to V4 add a null character */

    if (h_version <= V4)
	z_storeb (argv[0], read_size + 1, 0);

    /* In V5+ write length of input into text buffer */

    if (h_version >= V5)
	z_storeb (argv[0], 1, read_size);

    /* Tokenise line if a token buffer is present */

    if (argv[1] != 0)
	z_tokenise (2, argv);

    /* Perform save_undo for V1 to V4 games */

    if (h_version <= V4)
	save_undo ();

    /* Return line terminator to Z-machine (V5+ only) */

    if (h_version >= V5)
	store (terminator);

}/* z_read */

/*
 * z_read_char
 *
 * Read one character with optional timeout
 *
 *    argv[0] = input device (should be 1)
 *    argv[1] = timeout value in 1/10 seconds (optional)
 *    argv[2] = timeout action routine (optional)
 *
 */

void z_read_char (int argc, zword *argv)
{
    int key;

    /* Supply default parameters */

    if (argc < 3)
	argv[2] = 0;
    if (argc < 2)
	argv[1] = 0;

    /* Set the "input_flag" (for a workaround in z_sound_effect) */

    input_flag = 1;

    /* Flush any buffered output before read */

    flush_buffer ();

    /* Fail the call if the first argument is not 1 */

    if (argv[0] != 1) {
	store (0);
	return;
    }

continue_input:

    /* Read input from command file or keyboard */

    if (replaying)
	key = playback_key ();
    else
	key = os_read_char (argv[1]);

    /* Retry if input failed */

    if (key == -1)
	goto continue_input;

    /* Copy key to command file */

    if (recording && !replaying)
	record_key (key);

    /* Handle timeouts */

    if (key == 0)
	if (call_interrupt (argv[2]) == 0)
	    goto continue_input;

    /* Reset line counter */

    line_count = option_context_lines;

    /* Handle mouse clicks */

    if (key == 253 || key == 254) {
	z_storew (h_mouse_table, 1, mouse_x);
	z_storew (h_mouse_table, 2, mouse_y);
    }

    /* Return key value */

    store (key);

}/* z_read_char */
