/*
 * fmemory.c
 *
 * Memory related routines (fast version without virtual memory)
 *
 */

#include <alloc.h>
#include <ctype.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "frotz.h"

#define MAX_UNDO 20

#define RESERVE_MEM 4096

static char save_name[MAX_FILE_NAME + 1] = DEFAULT_SAVE_NAME;

zbyte *zmp = 0;
zbyte *pcp = 0;

static FILE *story_fp;

static zbyte *undo_zmp[MAX_UNDO];
static zbyte *undo_stack[MAX_UNDO];
static int undo_slots = 0;
static int undo_count = 0;
static int undo_valid = 0;

static void restart_header (void);

static void load_header (void);
static void load_story (void);

static void get_default_name (char *, zword);

static int old_restore (FILE *);

/*
 * restart_header
 *
 * Set all header fields which hold information about the interpreter.
 *
 */

static void restart_header (void)
{

    if (h_version == V3 && option_tandy_bit != 0)
	h_config |= CONFIG_TANDY;

    SET_BYTE (H_CONFIG, h_config)
    SET_WORD (H_FLAGS, h_flags)

    if (h_version >= V4) {
	SET_BYTE (H_INTERPRETER_NUMBER, h_interpreter_number)
	SET_BYTE (H_INTERPRETER_VERSION, h_interpreter_version)
	SET_BYTE (H_SCREEN_ROWS, h_screen_rows)
	SET_BYTE (H_SCREEN_COLS, h_screen_cols)
    }

    if (h_version >= V5) {
	SET_WORD (H_SCREEN_WIDTH, h_screen_width)
	SET_WORD (H_SCREEN_HEIGHT, h_screen_height)
	SET_BYTE (H_FONT_WIDTH, h_font_width)
	SET_BYTE (H_FONT_HEIGHT, h_font_height)
    }

    if (h_version >= V5 && (h_config & CONFIG_COLOUR)) {
	SET_BYTE (H_DEFAULT_BACKGROUND, h_default_background)
	SET_BYTE (H_DEFAULT_FOREGROUND, h_default_foreground)
    }

    SET_BYTE (H_SPECIFICATION_HIGH, 0)
    SET_BYTE (H_SPECIFICATION_LOW, 2)

}/* restart_header */

/*
 * load_header
 *
 * Load header and copy header entries to global variables.
 *
 */

static void load_header (void)
{
    zword addr;
    int i;

    /* Move to the start of the story file */

    rewind (story_fp);

    /* Allocate memory for header data */

    if ((zmp = (zbyte *) farmalloc (64)) == NULL)
	os_fatal ("Out of memory");

    /* Load header into memory */

    if (fread (zmp, 1, 64, story_fp) != 64)
	os_fatal ("Story file read error");

    /* Copy header fields to global variables */

    LOW_BYTE (H_VERSION, h_version)

    if (h_version == V1 || h_version == V2 || h_version == V3) {
	story_shift = 1;
	property_mask = 0x1f;
    } else if (h_version == V4 || h_version == V5 || h_version == V7) {
	story_shift = 2;
	property_mask = 0x3f;
    } else if (h_version == V8) {
	story_shift = 3;
	property_mask = 0x3f;
    } else os_fatal ("Unsupported Z-code version");

    LOW_BYTE (H_CONFIG, h_config)

    if (h_version == V3 && (h_config & CONFIG_BYTE_SWAPPED))
	os_fatal ("Byte swapped story file");

    LOW_WORD (H_RELEASE, h_release);
    LOW_WORD (H_START_PC, h_start_pc);
    LOW_WORD (H_DICTIONARY, h_dictionary);
    LOW_WORD (H_OBJECTS, h_objects);
    LOW_WORD (H_GLOBALS, h_globals);
    LOW_WORD (H_DYNAMIC_SIZE, h_dynamic_size);
    LOW_WORD (H_FLAGS, h_flags);

    for (i = 0, addr = H_SERIAL; i < 6; i++, addr++)
	LOW_BYTE (addr, h_serial[i])

    if (h_release == 47 || h_release == 49 || h_release == 51 || h_release == 57)
	beyond_zork_flag = (h_serial[0] == '8' && h_serial[1] == '7');

    LOW_WORD (H_ABBREVIATIONS, h_abbreviations);
    LOW_WORD (H_FILE_SIZE, h_file_size);

    if (h_file_size != 0) {

	/* Calculate story file size in bytes */

	story_size = (long) h_file_size << 1;

	if (h_version >= V4)
	    story_size <<= 1;
	if (h_version >= V6)
	    story_size <<= 1;

    } else {

	/* Some old games lack the file size entry */

	fseek (story_fp, 0, SEEK_END);
	story_size = ftell (story_fp);
	fseek (story_fp, 64, SEEK_SET);
    }

    LOW_WORD (H_CHECKSUM, h_checksum);
    LOW_WORD (H_ALPHABET, h_alphabet);
    LOW_WORD (H_FUNCTIONS_OFFSET, h_functions_offset);
    LOW_WORD (H_STRINGS_OFFSET, h_strings_offset);
    LOW_WORD (H_TERMINATING_KEYS, h_terminating_keys);
    LOW_WORD (H_MOUSE_TABLE, h_mouse_table);

}/* load_header */

/*
 * load_story
 *
 * Load the entire story file into memory.
 *
 */

static void load_story (void)
{
    zbyte *ptr;
    long len;
    unsigned count;

    /* Move to the start of the story file */

    rewind (story_fp);

    /* Allocate memory for story data */

    if ((zmp = (zbyte *) farrealloc (zmp, story_size)) == NULL)
	os_fatal ("Out of memory");

    /* Load story file into memory */

    ptr = zmp;

    for (len = story_size; len > 0; len -= 0xf000) {

	/* Load chunk of up to 60KB (DOS only manages chunks < 64KB) */

	count = (len >= 0xf000) ? 0xf000 : len;

	if (fread (ptr, 1, count, story_fp) != count)
	    os_fatal ("Story file read error");

	/* Increase pointer by 60 KB (this is quite DOS specific) */

	FP_SEG (ptr) += 0xf00;
    }

}/* load_story */

/*
 * init_memory
 *
 * Allocate memory and load the story file.
 *
 */

void init_memory (void)
{
    void *reserved;

    /* Open story file */

    if ((story_fp = fopen (story_name, "rb")) == NULL)
	os_fatal ("Cannot open story file");

    /* Initialise Z-machine memory */

    load_header ();
    load_story ();

    /* Initialise undo slots */

    reserved = malloc (RESERVE_MEM);

    if (option_undo_slots > MAX_UNDO)
	option_undo_slots = MAX_UNDO;

    for (undo_slots = 0; undo_slots < option_undo_slots; undo_slots++) {

	undo_zmp[undo_slots] = (zbyte *) malloc (h_dynamic_size);
	undo_stack[undo_slots] = (zbyte *) malloc (sizeof (stack));

	if (undo_stack[undo_slots] != NULL && undo_zmp[undo_slots] == NULL)
	    free (undo_stack[undo_slots]);
	if (undo_stack[undo_slots] == NULL || undo_zmp[undo_slots] == NULL)
	    break;
    }

    free (reserved);

}/* init_memory */

/*
 * reset_memory
 *
 * Close the story file and deallocate memory.
 *
 */

void reset_memory (void)
{

    /* Close story file */

    fclose (story_fp);

    /* Free Z-machine memory */

    farfree (zmp);

    /* Free undo slots */

    while (undo_slots-- != 0) {
	free (undo_zmp[undo_slots]);
	free (undo_stack[undo_slots]);
    }

}/* reset_memory */

/*
 * restore_undo
 *
 * Restore the current Z-machine state from memory. If the flag is
 * set, this state will be removed from the stack such that later
 * restore_undo calls can refer to earlier states. If this flag is
 * cleared, the state will not be removed such that later calls can
 * refer to the same state. The latter policy is used to implement
 * the restore_undo opcode which doesn't have a concept of multiple
 * undo. The return value is defined as in z_restore_undo.
 *
 */

int restore_undo (int flag)
{
    long current_pc;

    /* Check if undo is available first */

    if (undo_slots == 0)
	return (-1);
    if (undo_valid == 0)
	return (0);

    /* Restore state from undo memory */

    if (undo_count == 0)
	undo_count = undo_slots;

    memcpy (zmp, undo_zmp[undo_count - 1], h_dynamic_size);
    memcpy (stack, undo_stack[undo_count - 1], sizeof (stack));

    hi_word (current_pc) = stack[0];
    lo_word (current_pc) = stack[1];
    sp = stack + stack[2];
    fp = stack + stack[3];

    SET_PC (current_pc)

    /* Adjust undo counters */

    if (flag != 0) {
	undo_count--;
	undo_valid--;
    }

    return (2);

}/* restore_undo */

/*
 * save_undo
 *
 * Save the current Z-machine state to memory. The return value is
 * defined as in z_save_undo.
 *
 */

int save_undo (void)
{
    long current_pc;

    /* Check if undo is available first */

    if (undo_slots == 0)
	return (-1);

    /* Copy current state to undo memory */

    if (undo_count == undo_slots)
	undo_count = 0;

    GET_PC (current_pc)

    stack[0] = hi_word (current_pc);
    stack[1] = lo_word (current_pc);
    stack[2] = sp - stack;
    stack[3] = fp - stack;

    memcpy (undo_zmp[undo_count], zmp, h_dynamic_size);
    memcpy (undo_stack[undo_count], stack, sizeof (stack));

    /* Adjust undo counters */

    if (++undo_count == undo_slots)
	undo_count = 0;
    if (++undo_valid > undo_slots)
	undo_valid = undo_slots;

    return (1);

}/* save_undo */

/*
 * z_restart
 *
 * Restart game by initialising environment and setting PC to the
 * start address of the program.
 *
 */

void z_restart (void)
{
    static first_time = 1;
    long current_pc;

    /* Restart the screen */

    os_set_font (1);
    os_set_colour (1, 1);
    os_set_text_style (ROMAN_STYLE);

    z_erase_window (-1);
    z_buffer_mode (1);

    /* Randomize */

    random_number (0);

    /* Reload dynamic area */

    if (first_time == 0) {

	rewind (story_fp);
	fread (zmp, 1, h_dynamic_size, story_fp);

    } else first_time = 0;

    /* Initialise story header */

    restart_header ();

    /* Initialise PC, SP and FP */

    current_pc = h_start_pc;
    sp = stack + STACK_SIZE;
    fp = stack + STACK_SIZE;

    SET_PC (current_pc)

}/* z_restart */

/*
 * get_default_name
 *
 * Read a default file name from the memory of the Z-machine and
 * copy it to an array.
 *
 */

static void get_default_name (char *default_name, zword table)
{
    zbyte len;
    zbyte c;
    int i;

    if (table != 0) {

	LOW_BYTE (table, len)

	for (i = 0; i < len; i++) {
	    table++;
	    LOW_BYTE (table, c)
	    default_name[i] = c;
	}

	default_name[i] = 0;

	if (strchr (default_name, '.') == 0)
	    strcpy (default_name + i, ".aux");

    } else strcpy (default_name, DEFAULT_AUXILARY_NAME);

}/* get_default_name */

/*
 * old_restore
 *
 * This routines tries to load old save files (ie. uncompressed
 * files from Zip 2.00 to 2.04). It returns true if successful.
 *
 */

static int old_restore (FILE *gfp)
{
    long current_pc;
    long len;

    fseek (gfp, 0, SEEK_END);
    len = ftell (gfp);
    fseek (gfp, 0, SEEK_SET);

    if (len != sizeof (stack) + h_dynamic_size)
	return (0);

    fread (stack, 1, sizeof (stack), gfp);
    fread (zmp, 1, h_dynamic_size, gfp);

    sp = stack + stack[0] + 1;
    fp = stack + *(sp++) + 1;

    current_pc = *(sp++);
    current_pc |= (long) *(sp++) << 9;
    SET_PC (current_pc)

    return (1);

}/* old_restore */

/*
 * z_restore
 *
 * Restore game state from disk. Returns:
 *
 *    0 = restore failed
 *    2 = restore succeeded
 *
 * All arguments are optional. If these are given then only a part of
 * the dynamic memory is restored. The area starts at address argv[0]
 * and is argv[1] bytes long. Finally, argv[2] points to a suggested
 * file name if present. When restore is used in this long form, the
 * return value is the number of bytes read.
 *
 */

void z_restore (int argc, zword *argv)
{
    char new_save_name[MAX_FILE_NAME + 1];
    char default_name[MAX_FILE_NAME + 1];
    FILE *gfp;
    zword release;
    zword checksum;
    zword addr;
    long current_pc;
    int current_sp;
    int current_fp;
    int success;
    int skip;
    int i;

    success = 0;

    if (argc != 0) {

	/* Get the file name */

	get_default_name (default_name, (argc >= 3) ? argv[2] : 0);

	if (os_get_file_name (new_save_name, default_name, FILE_LOAD_AUX) == 0)
	    goto finished;

	/* Open auxilary file */

	if ((gfp = fopen (new_save_name, "rb")) == NULL)
	    goto finished;

	/* Load auxilary file */

	success = fread (zmp + argv[0], 1, argv[1], gfp);

	/* Close auxilary file */

	fclose (gfp);

    } else {

	/* Get the file name */

	if (os_get_file_name (new_save_name, save_name, FILE_RESTORE) == 0)
	    goto finished;

	/* Open game file */

	if ((gfp = fopen (new_save_name, "rb")) == NULL)
	    goto finished;

	/* Load game file */

	if (old_restore (gfp) == 0) {

	    hi (release) = fgetc (gfp);
	    lo (release) = fgetc (gfp);
	    hi (checksum) = fgetc (gfp);
	    lo (checksum) = fgetc (gfp);

	    if (release != h_release || checksum != h_checksum) {
		display_string ("Invalid save file");
		display_new_line ();
		fclose (gfp);
		goto finished;
	    }

	    hi (hi_word (current_pc)) = 0;
	    lo (hi_word (current_pc)) = fgetc (gfp);
	    hi (lo_word (current_pc)) = fgetc (gfp);
	    lo (lo_word (current_pc)) = fgetc (gfp);
	    hi (current_sp) = fgetc (gfp);
	    lo (current_sp) = fgetc (gfp);
	    hi (current_fp) = fgetc (gfp);
	    lo (current_fp) = fgetc (gfp);

	    SET_PC (current_pc)
	    sp = stack + current_sp;
	    fp = stack + current_fp;

	    for (i = current_sp; i < STACK_SIZE; i++) {
		hi (stack[i]) = fgetc (gfp);
		lo (stack[i]) = fgetc (gfp);
	    }

	    rewind (story_fp);

	    for (addr = 0; addr < h_dynamic_size; addr++) {
		skip = fgetc (gfp);
		for (i = 0; i < skip; i++)
		    zmp[addr++] = fgetc (story_fp);
		zmp[addr] = fgetc (gfp);
		fgetc (story_fp);
	    }

	    if (ferror (gfp) != 0 || ferror (story_fp) != 0)
		os_fatal ("Read from save file failed");
	}

	/* Close game file */

	fclose (gfp);

	/* Reset upper window (V3 only) */

	if (h_version == V3)
	    z_split_window (0);

	/* Initialise story header */

	restart_header ();

	/* Success */

	strcpy (save_name, new_save_name);
	success = 2;
    }

finished:

    if (h_version <= V3)
	branch (success);
    else
	store (success);

}/* z_restore */

/*
 * z_restore_undo
 *
 * Restore the current Z machine state from memory. Returns:
 *
 *   -1 = feature unavailable
 *    0 = restore failed
 *    2 = restore succeeded
 *
 */

void z_restore_undo (void)
{

    store (restore_undo (0));

}/* z_restore_undo */

/*
 * z_save
 *
 * Save game state to disk. Returns:
 *
 *    0 = save failed
 *    1 = save succeeded
 *
 * All arguments are optional. If these are given then only a part
 * of the dynamic memory is saved. The area starts at address argv[0]
 * and is argv[1] bytes long. Finally, argv[2] points to a suggested
 * file name if present.
 *
 */

void z_save (int argc, zword *argv)
{
    char new_save_name[MAX_FILE_NAME + 1];
    char default_name[MAX_FILE_NAME + 1];
    FILE *gfp;
    zword addr;
    long current_pc;
    int current_sp;
    int current_fp;
    int success;
    int skip;
    int i;

    success = 0;

    if (argc != 0) {

	/* Get the file name */

	get_default_name (default_name, (argc >= 3) ? argv[2] : 0);

	if (os_get_file_name (new_save_name, default_name, FILE_SAVE_AUX) == 0)
	    goto finished;

	/* Open auxilary file */

	if ((gfp = fopen (new_save_name, "wb")) == NULL)
	    goto finished;

	/* Write auxilary file */

	success = fwrite (zmp + argv[0], 1, argv[1], gfp);

	/* Close auxilary file */

	fclose (gfp);

    } else {

	/* Get the file name */

	if (os_get_file_name (new_save_name, save_name, FILE_SAVE) == 0)
	    goto finished;

	/* Open game file */

	if ((gfp = fopen (new_save_name, "wb")) == NULL)
	    goto finished;

	/* Write game file */

	fputc (hi (h_release), gfp);
	fputc (lo (h_release), gfp);
	fputc (hi (h_checksum), gfp);
	fputc (lo (h_checksum), gfp);

	GET_PC (current_pc)
	current_sp = sp - stack;
	current_fp = fp - stack;
	fputc (lo (hi_word (current_pc)), gfp);
	fputc (hi (lo_word (current_pc)), gfp);
	fputc (lo (lo_word (current_pc)), gfp);
	fputc (hi (current_sp), gfp);
	fputc (lo (current_sp), gfp);
	fputc (hi (current_fp), gfp);
	fputc (lo (current_fp), gfp);

	for (i = current_sp; i < STACK_SIZE; i++) {
	    fputc (hi (stack[i]), gfp);
	    fputc (lo (stack[i]), gfp);
	}

	rewind (story_fp);

	for (addr = 0, skip = 0; addr < h_dynamic_size; addr++)
	    if (zmp[addr] != fgetc (story_fp) || skip == 255 || addr == h_dynamic_size - 1) {
		fputc (skip, gfp);
		fputc (zmp[addr], gfp);
		skip = 0;
	    } else skip++;

	if (ferror (gfp) != 0 || ferror (story_fp) != 0) {
	    display_string ("Write to save file failed");
	    display_new_line ();
	    fclose (gfp);
	    goto finished;
	}

	/* Close game file */

	fclose (gfp);

	/* Success */

	strcpy (save_name, new_save_name);
	success = 1;
    }

finished:

    if (h_version <= V3)
	branch (success);
    else
	store (success);

}/* z_save */

/*
 * z_save_undo
 *
 * Save the current Z machine state in memory for a future undo. Returns:
 *
 *    -1 = feature unavailable
 *     0 = save failed
 *     1 = save succeeded
 *
 */

void z_save_undo (void)
{

    store (save_undo ());

}/* z_save_undo */

/*
 * z_verify
 *
 * Verify game ($verify verb). Add all bytes in story file except for
 * bytes in the story file header.
 *
 */

void z_verify (void)
{
    zword checksum;
    long len;

    checksum = 0;

    /* Sum all bytes in game file except header bytes */

    fseek (story_fp, 64, SEEK_SET);

    for (len = story_size; len > 64; len--)
	checksum += fgetc (story_fp);

    /* Jump if the checksum is equal */

    branch (checksum == h_checksum);

}/* z_verify */
