/*
 * buffer.c
 *
 * Text buffering and word wrapping
 *
 */

#include "frotz.h"

#define TEXT_BUFFER_SIZE 100

static int buffering = 1;

static char buffer[TEXT_BUFFER_SIZE];
static int buffer_pos = 0;

static int line_len = 0;

/*
 * flush_buffer
 *
 * Send output buffer to the screen.
 *
 */

void flush_buffer (void)
{
    int word_len;
    int code;
    int i;

    buffer[buffer_pos] = 0;

    /* Get the length of the word in screen units */

    word_len = os_text_length (buffer);

    /* Print a newline if the word does not fit */

    if (line_len + word_len > h_screen_cols - option_right_margin) {

	if (outputting)
	    display_new_line ();
	if (scripting && cwin == LOWER_WINDOW)
	    script_char ('\n');

	line_len = 0;
    }

    /* Print the buffer */

    for (i = 0; buffer[i] != 0; i++) {

	code = (unsigned char) buffer[i];

	switch (code) {

	    /* Style changes */

	    case CODE_ROMAN_STYLE:
		os_set_text_style (ROMAN_STYLE); break;
	    case CODE_REVERSE_STYLE:
		os_set_text_style (REVERSE_STYLE); break;
	    case CODE_BOLDFACE_STYLE:
		os_set_text_style (BOLDFACE_STYLE); break;
	    case CODE_EMPHASIS_STYLE:
		os_set_text_style (EMPHASIS_STYLE); break;
	    case CODE_FIXED_WIDTH_STYLE:
		os_set_text_style (FIXED_WIDTH_STYLE); break;

	    /* Font changes */

	    case CODE_TEXT_FONT:
		os_set_font (TEXT_FONT); break;
	    case CODE_PICTURE_FONT:
		os_set_font (PICTURE_FONT); break;
	    case CODE_GRAPHICS_FONT:
		os_set_font (GRAPHICS_FONT); break;
	    case CODE_FIXED_WIDTH_FONT:
		os_set_font (FIXED_WIDTH_FONT); break;

	    /* Plain characters */

	    default:
		if (outputting)
		    os_display_char (code);
		if (scripting && cwin == LOWER_WINDOW)
		    script_char (code);
	}
    }

    line_len += word_len;

    /* Reset the buffer state */

    buffer_pos = 0;

}/* flush_buffer */

/*
 * z_buffer_mode
 *
 * Set the format mode flag. If buffering is on then characters are
 * written into an output buffer to make line wrapping possible.
 *
 */

void z_buffer_mode (zword flag)
{

    buffering = flag;

}/* z_buffer_mode */

/*
 * z_new_line
 *
 * Just flush the current contents of the output buffer followed
 * by a newline.
 *
 */

void z_new_line (void)
{

    /* Only flush buffer if output redirection is off */

    if (!redirecting) {

	flush_buffer ();

	if (outputting)
	    display_new_line ();
	if (scripting && cwin == LOWER_WINDOW)
	    script_char ('\n');

	line_len = 0;

	/* If this is the lower window set the "newline_flag". This flag
	   is actually a workaround for a problem with the read opcode.
	   This flag helps to find out whether the input line has to be
	   redrawn after a timeout or not. */

	if (outputting && cwin == LOWER_WINDOW)
	    newline_flag = 1;

    } else memory_char (13);

}/* z_new_line */

/*
 * z_print_char
 *
 * High level character output routine.
 *
 */

void z_print_char (zword c)
{

    /* ASCII value 0 prints nothing and should be ignored */

    if (c == 0)
	return;

    /* ASCII value 13 means newline */

    if (c == 13) {
	z_new_line ();
	return;
    }

    if (message)

	/* We are currently printing a debugging message, so just
	   pass the output to the interface. */

	os_display_char (c);

    else if (redirecting)

	/* Redirection to memory turns all other output streams off.
	   Output is never buffered in this stream. */

	memory_char (c);

    else if (!buffering || h_version >= V5 && cwin == UPPER_WINDOW) {

	/* No buffering or output redirection, just output the character
	   to the screen or to the transscript file. */

	if (outputting)
	    os_display_char (c);
	if (scripting && cwin == LOWER_WINDOW)
	    script_char (c);

    } else {

	/* Buffering is on */

	if (c == ' ') {

	    /* Next character is space so flush the buffer */

	    flush_buffer ();

	    if (line_len < h_screen_cols - option_right_margin) {

		/* Print the space if it fits into the line... */

		if (outputting)
		    os_display_char (' ');
		if (scripting && cwin == LOWER_WINDOW)
		    script_char (c);

		line_len++;

	    } else {

		/* ...or replace it by newline otherwise. */

		if (outputting)
		    display_new_line ();
		if (scripting && cwin == LOWER_WINDOW)
		    script_char ('\n');

		line_len = 0;
	    }

	} else {

	    /* Plain character, add it to the buffer */

	    buffer[buffer_pos++] = c;

	    if (buffer_pos == TEXT_BUFFER_SIZE)
		os_fatal ("Text buffer overflow");
	}

	/* Paul D. Doherty suggests that hyphenated words should be
	   wrapped, too. This avoids large empty gaps now and then. */

	if (c == '-')
	    flush_buffer ();
    }

}/* z_print_char */

/*
 * z_set_font
 *
 * Set font for text output which can be:
 *
 *    0 = current font
 *    1 = text font
 *    2 = picture font (never used)
 *    3 = graphics font
 *    4 = fixed width font
 *
 * If the font isn't available return 0, otherwise return the number
 * of the current font. The specification demands that fonts may be
 * changed in the middle of a word, so font changes must be buffered
 * as well.
 *
 */

void z_set_font (zword font)
{
    static int current_font = TEXT_FONT;
    int code;

    /* See if the font is available */

    if (os_font_available (font) == 0) {
	store (0);
	return;
    }

    /* Return the current font to the Z-machine */

    store (current_font);
    current_font = font;

    /* Activate the font if buffering is off */

    if (!buffering || h_version >= V5 && cwin == UPPER_WINDOW) {
	os_set_font (font);
	return;
    }

    /* Otherwise store the new font in the output buffer */

    switch (font) {
	case TEXT_FONT: code = CODE_TEXT_FONT; break;
	case PICTURE_FONT: code = CODE_PICTURE_FONT; break;
	case GRAPHICS_FONT: code = CODE_GRAPHICS_FONT; break;
	case FIXED_WIDTH_FONT: code = CODE_FIXED_WIDTH_FONT; break;
    }

    buffer[buffer_pos++] = code;

    if (buffer_pos == TEXT_BUFFER_SIZE)
	os_fatal ("Text buffer overflow");

}/* z_set_font */

/*
 * z_set_text_style
 *
 * Set the text style which can be
 *
 *    0 = reset to roman style (plain style)
 *    1 = reverse mode
 *    2 = boldface
 *    4 = emphasis (italics or underlining)
 *    8 = fixed width
 *
 * Since the style may change in the middle of a word, it must be
 * buffered like text.
 *
 */

void z_set_text_style (zword style)
{
    int code;

    /* Things are easy if buffering is off */

    if (!buffering || h_version >= V5 && cwin == UPPER_WINDOW) {
	os_set_text_style (style);
	return;
    }

    /* Otherwise store the new style in the output buffer */

    switch (style) {
       case ROMAN_STYLE: code = CODE_ROMAN_STYLE; break;
       case REVERSE_STYLE: code = CODE_REVERSE_STYLE; break;
       case BOLDFACE_STYLE: code = CODE_BOLDFACE_STYLE; break;
       case EMPHASIS_STYLE: code = CODE_EMPHASIS_STYLE; break;
       case FIXED_WIDTH_STYLE: code = CODE_FIXED_WIDTH_STYLE; break;
    }

    buffer[buffer_pos++] = code;

    if (buffer_pos == TEXT_BUFFER_SIZE)
	os_fatal ("Text buffer overflow");

}/* z_set_text_style */
