/*
 * borland.c
 *
 * IO interface for MS-DOS and Borland C
 *
 */

#include <alloc.h>
#include <ctype.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\timeb.h>
#include "frotz.h"

#define CLICK_DELAY	250
#define DEFAULT_MODE	4
#define DEFAULT_VOLUME  15
#define LOW_BEEP_FREQ	440
#define LOW_BEEP_TIME	150
#define HIGH_BEEP_FREQ	1190
#define HIGH_BEEP_TIME	75
#define HISTORY_LINES	15

#define INFORMATION	"\
FROTZ V1.01 - plays all Infocom games except version 6 (graphic) games.\n\
Written by Stefan Jokisch in 1995. This is a standard 0.2 interpreter.\n\
\n\
\t-d #\tdisplay mode (0 mono, 1 text, 2 CGA, 3 MCGA, 4 EGA)\n\
\n\
\t-f #\tforeground colour\n\
\t-b #\tbackground colour\n\
\t-F #\treverse mode foreground colour\n\
\t-B #\treverse mode background colour\n\
\t-e #\temphasis colour\n\
\n\
\t-w #\tscreen width\n\
\t-h #\tscreen height\n\
\t-r #\tright margin\n\
\t-c #\tcontext lines\n\
\n\
\t-u #\tundo slots (0 turns undo off)\n\
\n\
\t-o\tmonitor object movement\n\
\t-O\tmonitor object locating\n\
\t-a\tmonitor attribute assignment\n\
\t-A\tmonitor attribute testing\n\
\t-t\tset the Tandy bit"

#define BLACK 0
#define BLUE 1
#define GREEN 2
#define CYAN 3
#define RED 4
#define MAGENTA 5
#define BROWN 6
#define LIGHTGRAY 7
#define DARKGRAY 8
#define LIGHTBLUE 9
#define LIGHTGREEN 10
#define LIGHTCYAN 11
#define LIGHTRED 12
#define LIGHTMAGENTA 13
#define YELLOW 14
#define WHITE 15

#define write_dsp(v)	while(inportb(sound_adr+12)&0x80);outportb(sound_adr+12,(v))

int getopt (int, char *[], const char *);

extern const char *optarg;
extern int optind;

static int display_mode = -1;

static int user_foreground = -1;
static int user_background = -1;
static int user_reverse_fg = -1;
static int user_reverse_bg = -1;
static int user_screen_width = -1;
static int user_screen_height = -1;
static int user_emphasis = -1;

static int current_style = 0;
static int current_font = 0;
static int current_foreground = 0;
static int current_background = 0;

static int message_dumping = 0;

static int saved_style = 0;
static int saved_font = 0;

static int screen_fg = 0;
static int screen_bg = 0;
static int text_fg = 0;
static int text_bg = 0;

static char history[HISTORY_LINES][80];
static int history_count = 0;

static int play_part = 0;
static int play_repeats = 0;
static int play_default = 0;

static void interrupt (*saved_interrupt)() = 0;

static int sound_adr = 0;
static int sound_irq = 0;
static int sound_dma = 0;

static void *sample_data = 0;
static long sample_adr1 = 0;
static long sample_adr2 = 0;
static unsigned sample_len1 = 0;
static unsigned sample_len2 = 0;

static int current_sample = 0;

static int overwrite_mode = 0;

static char graphic_font[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x30, 0x70, 0xFF, 0x70, 0x30, 0x00, 0x00,
    0x00, 0x0C, 0x0E, 0xFF, 0x0E, 0x0C, 0x00, 0x00,
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80,
    0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00,
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
    0x08, 0x08, 0x08, 0xFF, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xFF, 0x08, 0x08, 0x08,
    0x08, 0x08, 0x08, 0x08, 0x0F, 0x08, 0x08, 0x08,
    0x10, 0x10, 0x10, 0x10, 0xF0, 0x10, 0x10, 0x10,
    0x10, 0x10, 0x10, 0x10, 0x1F, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x1F, 0x10, 0x10, 0x10, 0x10,
    0x00, 0x00, 0x00, 0xF8, 0x08, 0x08, 0x08, 0x08,
    0x08, 0x08, 0x08, 0x08, 0xF8, 0x00, 0x00, 0x00,
    0x10, 0x10, 0x10, 0x10, 0x1F, 0x20, 0x40, 0x80,
    0x80, 0x40, 0x20, 0x1F, 0x10, 0x10, 0x10, 0x10,
    0x01, 0x02, 0x04, 0xF8, 0x08, 0x08, 0x08, 0x08,
    0x08, 0x08, 0x08, 0x08, 0xF8, 0x04, 0x02, 0x01,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
    0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F,
    0x08, 0x08, 0x08, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x08, 0x08, 0x08,
    0xF8, 0xF8, 0xF8, 0xF8, 0xFF, 0xF8, 0xF8, 0xF8,
    0x1F, 0x1F, 0x1F, 0x1F, 0xFF, 0x1F, 0x1F, 0x1F,
    0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F,
    0x00, 0x00, 0x00, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
    0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0x00, 0x00, 0x00,
    0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x20, 0x40, 0x80,
    0x80, 0x40, 0x20, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F,
    0x01, 0x02, 0x04, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
    0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0x04, 0x02, 0x01,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
    0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00,
    0x00, 0xFF, 0x80, 0x80, 0x80, 0x80, 0xFF, 0x00,
    0x00, 0xFF, 0xC0, 0xC0, 0xC0, 0xC0, 0xFF, 0x00,
    0x00, 0xFF, 0xE0, 0xE0, 0xE0, 0xE0, 0xFF, 0x00,
    0x00, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0x00,
    0x00, 0xFF, 0xF8, 0xF8, 0xF8, 0xF8, 0xFF, 0x00,
    0x00, 0xFF, 0xFC, 0xFC, 0xFC, 0xFC, 0xFF, 0x00,
    0x00, 0xFF, 0xFE, 0xFE, 0xFE, 0xFE, 0xFF, 0x00,
    0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00,
    0x00, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00,
    0x81, 0x42, 0x24, 0x18, 0x18, 0x24, 0x42, 0x81,
    0x08, 0x08, 0x08, 0x08, 0xFF, 0x08, 0x08, 0x08,
    0x18, 0x3C, 0x7E, 0x18, 0x18, 0x18, 0x18, 0x00,
    0x00, 0x18, 0x18, 0x18, 0x18, 0x7E, 0x3C, 0x18,
    0x18, 0x3C, 0x7E, 0x18, 0x18, 0x7E, 0x3C, 0x18,
    0xFF, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0xFF,
    0x00, 0x3C, 0x66, 0x06, 0x18, 0x00, 0x18, 0x00,
    0x63, 0x76, 0x5C, 0x60, 0x78, 0x66, 0x60, 0x00,
    0x78, 0x6E, 0x63, 0x7C, 0x63, 0x6E, 0x78, 0x00,
    0x00, 0x1E, 0x1B, 0x18, 0x18, 0xD8, 0x78, 0x00,
    0x63, 0x63, 0x77, 0x49, 0x77, 0x63, 0x63, 0x00,
    0x63, 0x77, 0x6B, 0x6B, 0x63, 0x63, 0x63, 0x00,
    0x49, 0x52, 0x64, 0x68, 0x70, 0x60, 0x60, 0x00,
    0x00, 0x63, 0x36, 0x1C, 0x1C, 0x36, 0x63, 0x00,
    0x63, 0x53, 0x4F, 0x63, 0x79, 0x65, 0x63, 0x00,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x00,
    0x18, 0x18, 0x7E, 0x99, 0x7E, 0x18, 0x18, 0x00,
    0x60, 0x60, 0x78, 0x6C, 0x67, 0x63, 0x63, 0x00,
    0x38, 0x3C, 0x36, 0x30, 0x30, 0x30, 0x30, 0x00,
    0x77, 0x6B, 0x77, 0x63, 0x63, 0x63, 0x63, 0x00,
    0x18, 0xD8, 0x78, 0x3C, 0x1E, 0x1B, 0x18, 0x00,
    0x71, 0x6A, 0x64, 0x71, 0x6A, 0x64, 0x60, 0x00,
    0x60, 0x60, 0x63, 0x67, 0x6F, 0x7B, 0x73, 0x00,
    0x18, 0x18, 0x18, 0x18, 0x3C, 0x5A, 0xDB, 0x00,
    0x7C, 0x63, 0x63, 0x7C, 0x66, 0x63, 0x63, 0x00,
    0x60, 0x63, 0x67, 0x6F, 0x7B, 0x73, 0x03, 0x00,
    0x18, 0x3C, 0x5A, 0x99, 0x18, 0x18, 0x18, 0x00,
    0x70, 0x78, 0x6C, 0x66, 0x63, 0x63, 0x63, 0x00,
    0x7E, 0x03, 0x7E, 0x03, 0x03, 0x03, 0x7C, 0x00,
    0x70, 0x6C, 0x63, 0x6C, 0x78, 0x60, 0x60, 0x00,
    0xDB, 0x5A, 0x3C, 0x18, 0x18, 0x18, 0x18, 0x00,
    0x78, 0x64, 0x52, 0x49, 0x4F, 0x49, 0x49, 0x00,
    0x60, 0x63, 0x66, 0x6C, 0x78, 0x60, 0x60, 0x00,
    0xE7, 0xC3, 0x81, 0xE7, 0xE7, 0xE7, 0xE7, 0xFF,
    0xFF, 0xE7, 0xE7, 0xE7, 0xE7, 0x81, 0xC3, 0xE7,
    0xE7, 0xC3, 0x81, 0xE7, 0xE7, 0x81, 0xC3, 0xE7,
    0xFF, 0xC3, 0x99, 0xF9, 0xE7, 0xFF, 0xE7, 0xFF
};

static char euro_map[] = {
    0x84, 0x94, 0x81, 0x8e, 0x99, 0x9a, 0xe1, 0xaf, 0xae, 0x89,
    0x8b, 0x98, 0xd3, 0xd8, 0xa0, 0x82, 0xa1, 0xa2, 0xa3, 0xec,
    0xb5, 0x90, 0xd6, 0xe0, 0xe9, 0xed, 0x85, 0x8a, 0x8d, 0x95,
    0x97, 0xb7, 0xd4, 0xde, 0xe3, 0xeb, 0x83, 0x88, 0x8c, 0x93,
    0x96, 0xb6, 0xd2, 0xd7, 0xe2, 0xea, 0x86, 0x8f, 0x9b, 0x9d,
    0xc6, 0xa4, 0xe4, 0xc7, 0xa5, 0xe5, 0x91, 0x92, 0x87, 0x80,
    0xd0, 0xe7, 0xd1, 0xe8, 0x9c, 0xff, 0xff, 0xad, 0xa8
};

static void adjust_style (void);

static int htoi (const char*);

static int init_mouse (void);
static int init_sound (void);

static void reset_sound (void);

static void start_of_dma (long, unsigned);
static void interrupt end_of_dma (void);
static void get_sample_name (char *, int);

static void graphic_cursor (void);

static int get_mouse (void);
static int get_key (long, short);

static void shift_cursor (int);
static void cursor_left (int *);
static void cursor_right (int *, int);
static void delete_char (int *, int *, char *);
static void delete_left (int *, int *, char *);
static void erase_input (int *, int *, char *);
static void first_char (int *);
static void insert_char (int, int *, int *, char *, int);
static void last_char (int *, int);
static void prev_word (int *, const char *);
static void next_word (int *, int, const char *);
static void store_input (const char *);
static void get_this_entry (int *, int *, char *, int *, int);
static void get_prev_entry (int *, int *, char *, int *, int);
static void get_next_entry (int *, int *, char *, int *, int);

/*
 * os_beep
 *
 * Play a beep sound. Ideally, the sound should be high- (number == 1)
 * or low-pitched (number == 2).
 *
 */

void os_beep (int number)
{

    /* Turn on speaker and wait a moment */

    if (number == 1) {
	sound (HIGH_BEEP_FREQ);
	delay (HIGH_BEEP_TIME);
    } else if (number == 2) {
	sound (LOW_BEEP_FREQ);
	delay (LOW_BEEP_TIME);
    }

    /* Turn off speaker */

    nosound ();

}/* os_beep */

/*
 * os_display_char
 *
 * Display a character from the current font using the current colours
 * and text style. The cursor moves to the next position or stays put
 * if it reaches the right margin. Printable codes are ASCII values from
 * 32 to 126 and European characters from 155 to 223 (as defined in the
 * specification of the Z-machine). Note that the screen doesn't scroll
 * after writing to the bottom right position.
 *
 */

void os_display_char (int c)
{
    const char scaler[] = { 0, 0, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 7, 7 };
    const char *table;
    char *screen1;
    char *screen2;
    int x, y;
    int i;

    /* Return if we are discarding the current message */

    if (message_dumping != 0)
	return;

    /* Get cursor position */

    asm mov ah,3
    asm mov bh,0
    asm int 0x10
    asm mov al,dh
    asm xor ah,ah
    asm mov y,ax
    asm xor dh,dh
    asm mov x,dx

    /* Print character */

    if (current_font == GRAPHICS_FONT) {

	/* Graphic font */

	if (display_mode == 2) {

	    /* Graphic font in CGA mode */

	    screen1 = MK_FP (0xb800, 320 * y + x);
	    screen2 = MK_FP (0xba00, 320 * y + x);

	    table = &graphic_font[(c - 32) << 3];

	    for (i = 0; i < 4; i++) {
		*screen1 = (text_fg == 1) ? *table++ : (*table++) & 0xff;
		*screen2 = (text_fg == 2) ? *table++ : (*table++) & 0xff;
		screen1 += 80;
		screen2 += 80;
	    }

	} if (display_mode == 3 || display_mode == 4) {

	    /* Graphic font in MCGA/EGA mode */

	    if (display_mode == 3)
		screen1 = MK_FP (0xa000, 640 * y + x);
	    else
		screen1 = MK_FP (0xa000, 1120 * y + x);

	    table = &graphic_font[(c - 32) << 3];

	    outport (0x03ce, 0x0205);
	    outport (0x03ce, 0xff08);

	    asm les bx,screen1
	    asm mov al,byte ptr text_bg
	    asm mov es:[bx],al
	    asm mov al,es:[bx]

	    if (display_mode == 3)

		for (i = 0; i < 8; i++) {
		    outportb (0x03cf, table[i]);
		    *screen1 = text_fg;
		    screen1 += 80;
		}

	    else

		for (i = 0; i < 14; i++) {
		    outportb (0x03cf, table[scaler[i]]);
		    *screen1 = text_fg;
		    screen1 += 80;
		}

	    outport (0x03ce, 0x0005);
	    outport (0x03ce, 0xff08);
	}

    } else {

	/* Text font */

	if (beyond_zork_flag == 0) {

	    /* Print "oe" ligature */

	    if (c == 220) {
		os_display_char ('o');
		os_display_char ('e');
		return;
	    }

	    /* Print "Oe" ligature */

	    if (c == 221) {
		os_display_char ('O');
		os_display_char ('e');
		return;
	    }

	    /* Print other accented characters */

	    if (c >= 155 && c <= 223)
		c = (unsigned char) euro_map[c - 155];
	}

	if (display_mode <= 1) {

	    /* Text font in mono/text mode */

	    asm mov ah,9
	    asm mov bh,0
	    asm mov bl,byte ptr text_bg
	    asm mov cl,4
	    asm shl bl,cl
	    asm or bl,byte ptr text_fg
	    asm mov cx,1
	    asm mov al,byte ptr c
	    asm int 0x10

	} else if (display_mode == 2) {

	    /* Text font in CGA mode */

	    if (text_fg == 0) {

		/* Text font in CGA mode (reverse style) */

		screen1 = MK_FP (0xb800, 320 * y + x);
		screen2 = MK_FP (0xba00, 320 * y + x);

		for (i = 0; i < 4; i++) {
		    *screen1 = 0xff;
		    *screen2 = 0xff;
		    screen1 += 80;
		    screen2 += 80;
		}

		asm mov ah,9
		asm mov al,byte ptr c
		asm mov bh,0
		asm mov bl,0x81
		asm mov cx,1
		asm int 0x10

	    } else {

		/* Text font in CGA mode (roman style) */

		asm mov ah,9
		asm mov al,byte ptr c
		asm mov bh,0
		asm mov bl,1
		asm mov cx,1
		asm int 0x10
	    }

	} else if (display_mode == 3 || display_mode == 4) {

	    /* Text font in MCGA/EGA mode */

	    asm mov ah,9
	    asm mov bh,0
	    asm mov cx,1
	    asm mov bl,byte ptr text_bg
	    asm mov al,219
	    asm int 0x10
	    asm mov bl,byte ptr text_fg
	    asm xor bl,byte ptr text_bg
	    asm or bl,0x80
	    asm mov al,byte ptr c
	    asm int 0x10
	}
    }

    /* Underline character */

    if (current_style & EMPHASIS_STYLE) {

	if (display_mode == 2) {

	    /* Underlining in CGA mode */

	    screen1 = MK_FP (0xba00, 320 * y + 240 + x);

	    if (text_fg == 1)
		*screen1 = 0xff;
	    else
		*screen1 = 0;

	} else if (display_mode == 3 || display_mode == 4) {

	    /* Underlining in MCGA/EGA mode */

	    if (display_mode == 3)
		screen1 = MK_FP (0xa000, 640 * y + 560 + x);
	    else
		screen1 = MK_FP (0xa000, 1120 * y + 1040 + x);

	    outport (0x03ce, 0x0205);
	    outport (0x03ce, 0xff08);

	    *screen1 = text_fg;

	    outport (0x03ce, 0x0005);
	}
    }

    /* Move cursor to the next position */

    if (++x < h_screen_cols) {
	asm mov ah,2
	asm mov bh,0
	asm mov dh,byte ptr y
	asm mov dl,byte ptr x
	asm int 0x10
    }

}/* os_display_char */

/*
 * os_erase_area
 *
 * Fill a rectangular area of the screen with the current background
 * colour. Top left coordinates are (1,1). The cursor does not move.
 *
 */

void os_erase_area (int top, int left, int bottom, int right)
{
    int attribute;

    if (display_mode <= 1)
	attribute = (screen_bg << 4) | screen_fg;
    else if (display_mode == 2)
	attribute = 0;
    else
	attribute = screen_bg;

    asm mov ah,6
    asm mov al,0
    asm mov ch,byte ptr top
    asm dec ch
    asm mov cl,byte ptr left
    asm dec cl
    asm mov dh,byte ptr bottom
    asm dec dh
    asm mov dl,byte ptr right
    asm dec dl
    asm mov bh,byte ptr attribute
    asm int 0x10

}/* os_erase_area */

/*
 * os_fatal
 *
 * Display error message and stop interpreter.
 *
 */

void os_fatal (const char *s)
{

    /* Stop current sample */

    os_stop_sample ();

    /* Reset the screen (if it has been initialised) */

    if (h_interpreter_number == INTERP_MSDOS)
	os_reset_screen ();

    /* Display error message */

    fputs ("\nFatal error: ", stderr);
    fputs (s, stderr);
    fputs ("\n", stderr);

    /* Exit indicating an error */

    exit (1);

}/* os_fatal */

/*
 * os_finish_with_sample
 *
 * Remove the current sample from memory (if any).
 *
 */

void os_finish_with_sample (void)
{

    /* Stop current sample */

    os_stop_sample ();

    /* Free memory */

    if (current_sample != 0) {
	free (sample_data);
	current_sample = 0;
    }

}/* os_finish_with_sample */

/*
 * os_font_available
 *
 * Return true if the given font is supported by this interface. The
 * font can be any of
 *
 *    TEXT_FONT
 *    PICTURE_FONT
 *    GRAPHICS_FONT
 *    FIXED_WIDTH_FONT
 *
 * The fonts are defined in the specification of the Z-machine.
 *
 */

int os_font_available (int font)
{

    if (font == TEXT_FONT)
	return (1);
    if (font == GRAPHICS_FONT && display_mode >= 2)
	return (1);
    if (font == FIXED_WIDTH_FONT)
	return (1);

    return (0);

}/* os_font_available */

/*
 * os_get_cursor
 *
 * Return the current cursor position. Top left coordinates are (1,1).
 *
 */

void os_get_cursor (int *row, int *col)
{
    int x, y;

    asm mov ah,3
    asm mov bh,0
    asm int 0x10
    asm mov al,dh
    asm xor ah,ah
    asm inc al
    asm mov y,ax
    asm xor dh,dh
    asm inc dl
    asm mov x,dx

    *row = y;
    *col = x;

}/* os_get_cursor */

/*
 * os_get_file_name
 *
 * Return the name of a file. Flag can be one of:
 *
 *    FILE_SAVE     - Save file (write only)
 *    FILE_RESTORE  - Save file (read only)
 *    FILE_SCRIPT   - Transscript file
 *    FILE_RECORD   - Command file for recording (write only)
 *    FILE_PLAYBACK - Command file for playback (read only)
 *    FILE_SAVE_AUX - Save auxilary ("preferred settings") file
 *    FILE_LOAD_AUX - Load auxilary ("preferred settings") file
 *
 * The length of the file name is limited by MAX_FILE_NAME. Ideally
 * an interpreter should open a file requester to ask for the file
 * name. If it is unable to do that then it should use diplay_string
 * and display_new_line to prompt for a file name.
 *
 */

int os_get_file_name (char *file_name, char *default_name, int flag)
{
    const char *extension;
    FILE *fp;
    int c;

    /* Select appropriate extension */

    if (flag == FILE_SAVE || flag == FILE_RESTORE)
	extension = ".sav";
    else if (flag == FILE_SCRIPT)
	extension = ".scr";
    else if (flag == FILE_RECORD || flag == FILE_PLAYBACK)
	extension = ".rec";
    else
	extension = ".aux";

    /* Prompt for the file name */

    display_string ("Enter a file name (extension \"");
    display_string (extension);
    display_string ("\" will be added).");
    display_new_line ();
    display_string ("Default is \"");
    display_string (default_name);
    display_string ("\": ");

    /* Read the file name */

    file_name[0] = 0;

    do {
	c = os_read (MAX_FILE_NAME, file_name, 0);
    } while (c != 13);

    display_new_line ();

    /* If nothing typed then use the default name */

    if (file_name[0] == 0)
	strcpy (file_name, default_name);

    /* Add the extension */

    if (strchr (file_name, '.') == 0)
	if (strlen (file_name) + strlen (extension) <= MAX_FILE_NAME)
	    strcat (file_name, extension);

    /* Check if we are going to overwrite the file */

    if (flag == FILE_SAVE || flag == FILE_SCRIPT || flag == FILE_RECORD || flag == FILE_SAVE_AUX) {

	/* Try to open the file */

	if ((fp = fopen (file_name, "rb")) != NULL) {

	    fclose (fp);

	    /* If it succeeded then prompt to overwrite */

	    display_string ("You are about to write over an existing file.");
	    display_new_line ();
	    display_string ("Proceed? (Y/N) ");

	    do {
		c = toupper (get_key (0, 0));
	    } while (c != 'Y' && c != 'N');

	    os_display_char (c);
	    display_new_line ();

	    /* If no overwrite then fail the routine */

	    if (c == 'N')
		return (0);
	}
    }

    /* Success */

    return (1);

}/* os_get_file_name */

/*
 * init_mouse
 *
 * Reset the mouse driver. Returns true if a mouse driver was found.
 *
 */

static int init_mouse (void)
{
    int success;

    asm mov ax,0
    asm int 0x33
    asm mov success,ax

    return (success);

}/* init_mouse */

/*
 * htoi
 *
 * Convert a string containing a hex number to integer (this is cheaper
 * than using scanf which makes a DOS program almost 2KB longer).
 *
 */

static int htoi (const char *s)
{
    int number;
    char c;

    number = 0;

loop:

    c = toupper (*(s++));

    if (c >= '0' && c <= '9') {
	number = 16 * number + c - '0';
	goto loop;
    }

    if (c >= 'A' && c <= 'F') {
	number = 16 * number + c - 'A' + 10;
	goto loop;
    }

    return (number);

}/* htoi */

/*
 * init_sound
 *
 * Initialise the sound board and various sound related variables.
 *
 */

static int init_sound (void)
{
    const char *blaster;

    /* Fetch the BLASTER environment variable */

    if ((blaster = getenv ("BLASTER")) == NULL)
	return (0);

    /* Read the IRQ, port address and DMA channel */

    sound_irq = atoi (strchr (blaster, 'I') + 1);
    sound_adr = htoi (strchr (blaster, 'A') + 1);
    sound_dma = atoi (strchr (blaster, 'D') + 1);

    /* Reset the DSP */

    outportb (sound_adr + 6, 1);
    inportb (sound_adr + 6);
    inportb (sound_adr + 6);
    inportb (sound_adr + 6);
    outportb (sound_adr + 6, 0);

    /* Reset the mixer (requires Soundblaster Pro or better) */

    outportb (sound_adr + 4, 0);
    outportb (sound_adr + 5, 0);

    /* Turn on speakers */

    write_dsp (0xd1);

    /* Install and enable the end_of_dma interrupt */

    if (sound_irq < 8) {

	saved_interrupt = getvect (0x08 + sound_irq);
	setvect (0x08 + sound_irq, end_of_dma);
	outportb (0x21, inportb (0x21) & ~(1 << sound_irq));

    } else {

	sound_irq &= 0x07;

	saved_interrupt = getvect (0x70 + sound_irq);
	setvect (0x70 + sound_irq, end_of_dma);
	outportb (0xa1, inportb (0xa1) & ~(1 << sound_irq));
	outportb (0x21, inportb (0x21) & ~0x04);

	sound_irq |= 0x08;
    }

    /* Indicate success */

    return (1);

}/* init_sound */

/*
 * os_init_screen
 *
 * Initialise the IO interface. Prepare screen and other devices
 * (mouse, sound board). Set various OS depending story file header
 * entries:
 *
 *     h_config (aka flags 1)
 *     h_flags (aka flags 2)
 *     h_screen_cols (aka screen width in characters)
 *     h_screen_rows (aka screen height in lines)
 *     h_screen_width
 *     h_screen_height
 *     h_font_width (should be 1)
 *     h_font_height (should be 1)
 *     h_default_foreground
 *     h_default_background
 *     h_interpreter_number
 *     h_interpreter_version
 *
 * See the specification of the Z-machine for more information.
 *
 */

void os_init_screen (void)
{
    const colour_map[] = {
	BLACK_COLOUR,
	BLUE_COLOUR,
	GREEN_COLOUR,
	CYAN_COLOUR,
	RED_COLOUR,
	MAGENTA_COLOUR,
	YELLOW_COLOUR,
	WHITE_COLOUR
    };

    /* If the display mode has not already been set by the user then
       see if this is a monochrome board. If so, set the display mode
       to 0. Otherwise check the graphics flag of the story. Select a
       graphic mode if it is set, or text mode if it is not. */

    if (display_mode == -1) {

	asm mov ah,15
	asm int 0x10
	asm mov display_mode,0
	asm cmp al,7
	asm je set_video_mode

	display_mode = (h_flags & GRAPHICS_FLAG) ? DEFAULT_MODE : 1;
    }

    /* Activate the desired display mode. Turn off the cursor and
       enable bright background colours (EGA/VGA boards only). */

set_video_mode:
    asm mov ax,display_mode
    asm cmp ax,0
    asm jne no_mono
    asm mov al,7
    asm jmp call_bios
no_mono:
    asm cmp ax,1
    asm jne no_text
    asm mov al,3
    asm jmp call_bios
no_text:
    asm cmp ax,2
    asm jne no_cga
    asm mov al,6
    asm jmp call_bios
no_cga:
    asm cmp ax,3
    asm jne no_mcga
    asm mov al,14
    asm jmp call_bios
no_mcga:
    asm mov al,16
call_bios:
    asm mov ah,0
    asm int 0x10
    asm mov ah,16
    asm mov al,3
    asm mov bl,0
    asm int 0x10
    asm mov ah,1
    asm mov cx,-1
    asm int 0x10

    /* Handle various game flags. These flags are set if the game wants
       to use certain features. The flags must be cleared if the feature
       is not available. */

    if (h_flags & GRAPHICS_FLAG)
	if (display_mode <= 1)
	    h_flags &= ~GRAPHICS_FLAG;

    if (h_version == V3 && (h_flags & OLD_SOUND_FLAG))
	if (init_sound () == 0)
	    h_flags &= ~OLD_SOUND_FLAG;

    if (h_version >= V5 && (h_flags & UNDO_FLAG))
	if (option_undo_slots == 0)
	    h_flags &= ~UNDO_FLAG;

    if (h_flags & SOUND_FLAG)
	if (init_sound () == 0)
	    h_flags &= ~SOUND_FLAG;

    if (h_flags & MOUSE_FLAG)
	if (init_mouse () == 0)
	    h_flags &= ~MOUSE_FLAG;

    if (h_flags & COLOUR_FLAG)
	if (display_mode == 0 || display_mode == 2)
	    h_flags &= ~COLOUR_FLAG;

    /* Set various bits in the configuration byte. These bits tell
       the game which features are supported by the interpreter. */

    if (h_version == V3)
	h_config |= CONFIG_SPLITSCREEN;

    if (h_version >= V4 && display_mode != 2)
	h_config |= CONFIG_BOLDFACE;

    if (h_version >= V4)
	h_config |= CONFIG_EMPHASIS;

    if (h_version >= V4)
	h_config |= CONFIG_FIXED;

    if (h_version == V5 && display_mode != 0 && display_mode != 2)
	h_config |= CONFIG_COLOUR;

    if (h_version >= V4)
	h_config |= CONFIG_TIMEDINPUT;

    /* Store the default colours. This is slightly complicated
       because PCs and Z-machines use different colour schemes. */

    if (user_foreground == -1)
	h_default_foreground = WHITE_COLOUR;
    else
	h_default_foreground = colour_map[user_foreground & 7];

    if (user_background == -1)
	h_default_background = BLUE_COLOUR;
    else
	h_default_background = colour_map[user_background & 7];

    /* Set the screen dimensions in character grid positions. */

    if (user_screen_height == -1)
	h_screen_rows = 25;
    else
	h_screen_rows = user_screen_height;

    if (user_screen_width == -1)
	h_screen_cols = 80;
    else
	h_screen_cols = user_screen_width;

    /* At the moment, the following settings are mandatory. */

    h_screen_width = h_screen_cols;
    h_screen_height = h_screen_rows;

    h_font_width = 1;
    h_font_height = 1;

    /* Set the interpreter number (a constant telling the game which
       operating system it runs on) and the interpreter version. */

    h_interpreter_number = INTERP_MSDOS;
    h_interpreter_version = 'A';

}/* os_init_screen */

/*
 * os_message_start
 *
 * Prepare for printing a "debugging" message. The text of the message
 * will be passed to os_display_char. Possibly, the interface may want
 * to direct the message to an additional "messages" window. Otherwise,
 * it should display the message on the game screen -- or discard it if
 * the upper window is selected (ie. if cwin == UPPER_WINDOW).
 *
 */

void os_message_start (void)
{
    int row, col;

    if (cwin == LOWER_WINDOW) {

	/* Select text font and roman style */

	saved_style = current_style;
	saved_font = current_font;
	os_set_font (TEXT_FONT);
	os_set_text_style (ROMAN_STYLE);

	/* Print newline if necessary */

	os_get_cursor (&row, &col);
	if (col != 1)
	    display_new_line ();

	/* Introduce message by a left bracket */

	display_string ("  [");

    } else message_dumping = 1;

}/* os_message_start */

/*
 * os_message_end
 *
 * Stop printing a debugging message.
 *
 */

void os_message_end (void)
{

    if (cwin == LOWER_WINDOW) {

	/* Print right bracket to finish message */

	display_string ("]");
	display_new_line ();

	/* Restore font and text style */

	os_set_font (saved_font);
	os_set_text_style (saved_style);

    } else message_dumping = 0;

}/* os_message_end */

/*
 * os_more_prompt
 *
 * Display a MORE prompt, wait for a keypress and remove the MORE
 * prompt from the screen.
 *
 */

void os_more_prompt (void)
{
    int saved_font;
    int saved_style;
    int row, col;

    /* Prepare the MORE prompt by selecting the text font and roman style */

    saved_font = current_font;
    saved_style = current_style;
    os_set_font (TEXT_FONT);
    os_set_text_style (ROMAN_STYLE);

    /* Print MORE message and wait for a key */

    os_get_cursor (&row, &col);
    display_string ("[MORE]");
    get_key (0, 0);

    /* Remove MORE prompt from the screen and restore cursor position */

    os_set_cursor (row, col);
    os_erase_area (row, col, row, col + 5);
    os_set_font (saved_font);
    os_set_text_style (saved_style);

}/* os_more_prompt */

/*
 * get_sample_name
 *
 * Build the file name for a given sample number.
 *
 */

static void get_sample_name (char *sample_name, int number)
{
    const char *ptr1;
    const char *ptr2;
    int i;

    sample_name = stpcpy (sample_name, "SOUND\\");

    for (ptr1 = ptr2 = story_name; *ptr1 != 0; ptr1++)
	if (*ptr1 == '\\' || *ptr1 == ':')
	    ptr2 = ptr1 + 1;

    for (i = 0; i < 6 && *ptr2 != '.' && *ptr2 != 0; i++)
	*sample_name++ = *(ptr2++);

    *sample_name++ = '0' + number / 10;
    *sample_name++ = '0' + number % 10;

    stpcpy (sample_name, ".SND");

}/* get_sample_name */

/*
 * os_prepare_sample
 *
 * Load the given sample from the disk.
 *
 */

void os_prepare_sample (int number)
{
    char sample_name[80];
    FILE *fp;
    unsigned frequency;
    unsigned length;
    unsigned char repeats;

    /* Return if the sample is already loaded */

    if (current_sample == number)
	return;

    /* Throw away previous sample */

    os_finish_with_sample ();

    /* Build the name of the sample file */

    get_sample_name (sample_name, number);

    /* Open the sample file */

    if ((fp = fopen (sample_name, "rb")) == NULL)
	return;

    /* Read repetition byte */

    fseek (fp, 2, SEEK_SET);
    repeats = fgetc (fp);

    if (repeats == 0)
	play_default = 255;
    else
	play_default = repeats;

    /* Read frequency */

    fseek (fp, 4, SEEK_SET);
    hi (frequency) = fgetc (fp);
    lo (frequency) = fgetc (fp);

    /* Read length */

    fseek (fp, 8, SEEK_SET);
    hi (length) = fgetc (fp);
    lo (length) = fgetc (fp);

    /* Allocate memory */

    if ((sample_data = malloc (length)) == NULL) {
	fclose (fp);
	return;
    }

    /* Read sample data */

    if (fread (sample_data, 1, length, fp) != length) {
	free (sample_data);
	fclose (fp);
	return;
    }

    /* Set various sound related variables */

    sample_adr1 = ((long) FP_SEG (sample_data) << 4) | FP_OFF (sample_data);
    sample_adr2 = (sample_adr1 + 0xffff) & ~0xffffL;

    sample_len1 = sample_adr2 - sample_adr1;
    sample_len2 = length - sample_len1;

    if (sample_len1 > length) {
	sample_len1 = length;
	sample_len2 = 0;
    }

    /* Set time constant */

    write_dsp (0x40);
    write_dsp (256 - 1000000L / frequency);

    /* Close sample file */

    fclose (fp);

    /* Success */

    current_sample = number;

}/* os_prepare_sample */

/*
 * os_process_arguments
 *
 * Handle command line switches. Some variables may be set to activate
 * certain features of Frotz:
 *
 *     option_attribute_assignment
 *     option_attribute_testing
 *     option_context_lines
 *     option_object_locating
 *     option_object_movement
 *     option_right_margin
 *     option_tandy_bit
 *     option_undo_slots
 *
 * The name of the story file is stored in "story_name".
 *
 */

void os_process_arguments (int argc, char *argv[])
{
    int errflg = 0;
    int c;

    /* Parse the options */

    while ((c = getopt (argc, argv, "aAb:B:c:d:e:f:F:h:oOr:tu:w:")) != EOF)
	switch (c) {
	    case 'a': option_attribute_assignment = 1;
		      break;
	    case 'A': option_attribute_testing = 1;
		      break;
	    case 'b': user_background = atoi (optarg);
		      break;
	    case 'B': user_reverse_bg = atoi (optarg);
		      break;
	    case 'c': option_context_lines = atoi (optarg);
		      break;
	    case 'd': display_mode = atoi (optarg);
		      break;
	    case 'e': user_emphasis = atoi (optarg);
		      break;
	    case 'f': user_foreground = atoi (optarg);
		      break;
	    case 'F': user_reverse_fg = atoi (optarg);
		      break;
	    case 'h': user_screen_height = atoi (optarg);
		      break;
	    case 'o': option_object_movement = 1;
		      break;
	    case 'O': option_object_locating = 1;
		      break;
	    case 'r': option_right_margin = atoi (optarg);
		      break;
	    case 't': option_tandy_bit = 1;
		      break;
	    case 'u': option_undo_slots = atoi (optarg);
		      break;
	    case 'w': user_screen_width = atoi (optarg);
		      break;
	    default : errflg = 1;
		      break;
	}

    /* Display usage */

    if (errflg != 0 || optind >= argc) {
	puts (INFORMATION);
	exit (1);
    }

    /* Store the story file name */

    story_name = argv[optind];

}/* os_process_arguments */

/*
 * graphic_cursor
 *
 * Display a cursor in graphic mode using XOR operation.
 *
 */

static void graphic_cursor (void)
{
    char *screen1;
    char *screen2;
    int x, y;
    int i;

    asm mov ah,3
    asm mov bh,0
    asm int 0x10
    asm mov al,dh
    asm xor ah,ah
    asm mov y,ax
    asm xor dh,dh
    asm mov x,dx

    if (display_mode == 2) {

	screen1 = MK_FP (0xb800, 320 * y + x);
	screen2 = MK_FP (0xba00, 320 * y + x);

	for (i = 0; i < 4; i++) {
	    *screen1 = *screen1 ^ 0xff;
	    *screen2 = *screen2 ^ 0xff;
	    screen1 += 80;
	    screen2 += 80;
	}

    } else if (display_mode == 3 || display_mode == 4) {

	if (display_mode == 3)
	    screen1 = MK_FP (0xa000, 640 * y + x);
	else
	    screen1 = MK_FP (0xa000, 1120 * y + x);

	outport (0x03ce, 0x1803);
	outport (0x03ce, 0x0205);
	outport (0x03ce, 0xff08);

	for (i = 0; i < ((display_mode == 3) ? 8 : 14); i++) {
	    asm les bx,screen1
	    asm mov al,es:[bx]
	    asm mov al,byte ptr text_fg
	    asm xor al,byte ptr text_bg
	    asm mov es:[bx],al
	    asm add word ptr screen1,80
	}

	outport (0x03ce, 0x0003);
	outport (0x03ce, 0x0005);
	outport (0x03ce, 0xff08);
    }

}/* graphic_cursor */

/*
 * get_mouse
 *
 * Report any mouse clicks.
 *
 */

static int get_mouse ()
{
    int value;

    /* Read the current mouse status */

    asm mov ax,6
    asm mov bx,0
    asm int 0x33
    asm mov mouse_x,cx
    asm mov mouse_y,dx
    asm cmp bx,0
    asm jne single_click
    asm mov value,0
    asm jmp done

    /* Wait for a second mouse click */

single_click:

    delay (CLICK_DELAY);

    asm mov ax,6
    asm mov bx,0
    asm int 0x33
    asm cmp bx,0
    asm jne double_click
    asm mov value,254
    asm jmp done

    /* The mouse button has been pressed twice */

double_click:

    asm mov mouse_x,cx
    asm mov mouse_y,dx
    asm mov value,253

    /* Store the mouse coordinates in character grid positions */

done:

    mouse_x = mouse_x / 8 + 1;
    mouse_y = mouse_y / ((display_mode <= 3) ? 8 : 14) + 1;

    /* Return key (253 for double click, 254 for single click) */

    return (value);

}/* get_mouse */

/*
 * get_key
 *
 * Read a key or a mouse click or abort when the time limit is exceeded.
 *
 */

static int get_key (long target_time, short target_millitm)
{
    struct timeb now;
    int key;
    int c;
    int i;

    /* Turn on cursor */

    if (display_mode <= 1) {

	asm mov ah,1
	asm cmp overwrite_mode,0
	asm jne overwrite
	asm mov cx,0x0a0b
	asm cmp display_mode,0
	asm je call_bios
	asm mov cx,0x0506
	asm jmp call_bios
    overwrite:
	asm mov cx,0x080f
	asm cmp display_mode,0
	asm jne call_bios
	asm mov cx,0x0408
    call_bios:
	asm int 0x10

    } else graphic_cursor ();

    /* Turn mouse pointer on */

    if (h_flags & MOUSE_FLAG) {
	asm mov ax,1
	asm int 0x33
    }

    for (;;) {

	/* Check for timeout */

	if (target_time != 0) {
	    ftime (&now);
	    if (now.time > target_time || now.time == target_time && now.millitm >= target_millitm) {
		key = 0;
		goto finished;
	    }
	}

	/* Check for mouse click */

	if (h_flags & MOUSE_FLAG) {
	    key = get_mouse ();
	    if (key != 0)
		goto finished;
	}

	/* Check for keypress */

	asm mov ah,1
	asm int 0x16
	asm jnz key_available
	asm xor ax,ax
	asm jmp key_done
    key_available:
	asm mov ah,0
	asm int 0x16
    key_done:
	asm mov key,ax

	/* Special keys */

	switch (key) {
	    case 0x4800: key = 129; goto finished; /* Up arrow    */
	    case 0x5000: key = 130; goto finished; /* Down arrow  */
	    case 0x4b00: key = 131; goto finished; /* Left arrow  */
	    case 0x4d00: key = 132; goto finished; /* Right arrow */
	    case 0x3b00: key = 133; goto finished; /* F1          */
	    case 0x3c00: key = 134; goto finished; /* F2          */
	    case 0x3d00: key = 135; goto finished; /* F3          */
	    case 0x3e00: key = 136; goto finished; /* F4          */
	    case 0x3f00: key = 137; goto finished; /* F5          */
	    case 0x4000: key = 138; goto finished; /* F6          */
	    case 0x4100: key = 139; goto finished; /* F7          */
	    case 0x4200: key = 140; goto finished; /* F8          */
	    case 0x4300: key = 141; goto finished; /* F9          */
	    case 0x4400: key = 142; goto finished; /* F10         */
	    case 0x5230: key = 145; goto finished; /* NumberPad 0 */
	    case 0x4f31: key = 146; goto finished; /* NumberPad 1 */
	    case 0x5032: key = 147; goto finished; /* NumberPad 2 */
	    case 0x5133: key = 148; goto finished; /* NumberPad 3 */
	    case 0x4b34: key = 149; goto finished; /* NumberPad 4 */
	    case 0x4c35: key = 150; goto finished; /* NumberPad 5 */
	    case 0x4d36: key = 151; goto finished; /* NumberPad 6 */
	    case 0x4737: key = 152; goto finished; /* NumberPad 7 */
	    case 0x4838: key = 153; goto finished; /* NumberPad 8 */
	    case 0x4939: key = 154; goto finished; /* NumberPad 9 */
	    case 0x4700: key = 256; goto finished; /* Home        */
	    case 0x4f00: key = 257; goto finished; /* End         */
	    case 0x7300: key = 258; goto finished; /* Word left   */
	    case 0x7400: key = 259; goto finished; /* Word right  */
	    case 0x5300: key = 260; goto finished; /* Delete      */
	    case 0x4900: key = 261; goto finished; /* Page up     */
	    case 0x5100: key = 262; goto finished; /* Page down   */
	    case 0x5200: key = 263; goto finished; /* Insert      */
	    case 0x1300: key = HOT_KEY_RECORDING; goto finished;
	    case 0x1900: key = HOT_KEY_PLAYBACK; goto finished;
	    case 0x1f00: key = HOT_KEY_SEED; goto finished;
	    case 0x1600: key = HOT_KEY_UNDO; goto finished;
	}

	/* Standard keys */

	key &= 0xff;
	if (key >= 32 && key <= 126 || key == 8 || key == 13 || key == 27)
	    goto finished;

	/* European characters */

	for (i = 155; i <= 223; i++) {
	    c = (unsigned char) euro_map[i - 155];
	    if (key == c) {
		key = i;
		goto finished;
	    }
	}
    }

finished:

    /* Turn off mouse pointer */

    if (h_flags & MOUSE_FLAG) {
	asm mov ax,2
	asm int 0x33
    }

    /* Turn off cursor */

    if (display_mode <= 1) {

	asm mov ah,1
	asm mov cx,-1
	asm int 0x10

    } else graphic_cursor ();

    /* Return key value */

    return (key);

}/* get_key */

/*
 * shift_cursor
 *
 * Move the hardware cursor to the left or right.
 *
 */

static void shift_cursor (int shift)
{
    int row, col;

    os_get_cursor (&row, &col);
    os_set_cursor (row, col + shift);

}/* shift_cursor */

/*
 * cursor_left
 *
 * Move the cursor one character to the left.
 *
 */

static void cursor_left (int *pos)
{

    if (*pos == 0)
	return;

    shift_cursor (-1);
    (*pos)--;

}/* cursor_left */

/*
 * cursor_right
 *
 * Move the cursor one character to the right.
 *
 */

static void cursor_right (int *pos, int read_size)
{

    if (*pos == read_size)
	return;

    shift_cursor (+1);
    (*pos)++;

}/* cursor_right */

/*
 * delete_char
 *
 * Delete the character below the cursor.
 *
 */

static void delete_char (int *pos, int *read_size, char *buffer)
{

    if (*pos == *read_size)
	return;

    buffer += *pos;
    (*read_size)--;

    memmove (buffer, buffer + 1, *read_size + 1 - *pos);

    display_string (buffer);
    os_display_char (' ');
    shift_cursor (*pos - *read_size - 1);

}/* delete_char */

/*
 * delete_left
 *
 * Delete the character to the left of the cursor.
 *
 */

static void delete_left (int *pos, int *read_size, char *buffer)
{

    if (pos == 0)
	return;

    cursor_left (pos);
    delete_char (pos, read_size, buffer);

}/* delete_left */

/*
 * erase_input
 *
 * Clear the input line.
 *
 */

static void erase_input (int *pos, int *read_size, char *buffer)
{
    int i;

    shift_cursor (-(*pos));

    for (i = 0; i < *read_size; i++)
	os_display_char (' ');

    shift_cursor (-(*read_size));

    *read_size = 0;
    *buffer = 0;
    *pos = 0;

}/* erase_input */

/*
 * first_char
 *
 * Move the cursor to the beginning of the input line.
 *
 */

static void first_char (int *pos)
{

    shift_cursor (-(*pos));
    *pos = 0;

}/* first_char */

/*
 * insert_char
 *
 * Insert a character into the input buffer.
 *
 */

static void insert_char (int max_size, int *pos, int *read_size, char *buffer, int c)
{

    if (overwrite_mode != 0 && *pos < *read_size) {

	buffer[*pos] = c;

	os_display_char (c);
	shift_cursor (-1);
	cursor_right (pos, *read_size);

    } else {

	if (*read_size == max_size)
	    return;

	buffer += *pos;
	(*read_size)++;

	memmove (buffer + 1, buffer, *read_size - *pos);
	*buffer = c;

	display_string (buffer);
	shift_cursor (*pos - *read_size);
	cursor_right (pos, *read_size);
    }

}/* insert_char */

/*
 * last_char
 *
 * Move the cursor to the end of the input line.
 *
 */

static void last_char (int *pos, int read_size)
{

    shift_cursor (read_size - *pos);
    *pos = read_size;

}/* last_char */

/*
 * prev_word
 *
 * Move the cursor to the start of the previous word.
 *
 */

static void prev_word (int *pos, const char *buffer)
{

    while (*pos > 0) {
	cursor_left (pos);
	if (buffer[*pos] != ' ' && buffer[*pos - 1] == ' ')
	    break;
    }

}/* prev_word */

/*
 * next_word
 *
 * Move the cursor to the start of the next word.
 *
 */

static void next_word (int *pos, int read_size, const char *buffer)
{

    while (*pos < read_size) {
	cursor_right (pos, read_size);
	if (buffer[*pos] != ' ' && buffer[*pos - 1] == ' ')
	    break;
    }

}/* next_word */

/*
 * store_input
 *
 * Copy the current input line to the history buffer.
 *
 */

static void store_input (const char *buffer)
{

    if (buffer[0] == 0)
	return;

    memmove (history[history_count], buffer, 80);

    if (++history_count == HISTORY_LINES)
	history_count = 0;

}/* store_input */

/*
 * get_this_entry
 *
 * Get the current entry from the history buffer and copy it to the
 * input line.
 *
 */

static void get_this_entry (int *pos, int *read_size, char *buffer, int *entry, int max_size)
{

    erase_input (pos, read_size, buffer);

    memmove (buffer, history[*entry], max_size + 1);
    buffer[max_size] = 0;

    display_string (buffer);

    *read_size = *pos = strlen(buffer);

}/* get_this_entry */

/*
 * get_prev_entry
 *
 * Fetch the previous history entry and display it on the input line.
 *
 */

static void get_prev_entry (int *pos, int *read_size, char *buffer, int *entry, int max_size)
{

    if (--(*entry) == -1)
	*entry = HISTORY_LINES - 1;

    get_this_entry (pos, read_size, buffer, entry, max_size);

}/* get_prev_entry */

/*
 * get_next_entry
 *
 * Fetch the next history entry and display it on the input line.
 *
 */

static void get_next_entry (int *pos, int *read_size, char *buffer, int *entry, int max_size)
{

    if (++(*entry) == HISTORY_LINES)
	*entry = 0;

    get_this_entry (pos, read_size, buffer, entry, max_size);

}/* get_next_entry */

/*
 * os_read
 *
 * Read a line of input from the keyboard into a buffer. The buffer
 * may already be primed with some text. In this case, the "initial"
 * text is already displayed on the screen. After the input action
 * is complete, the function returns with the terminating key value.
 * The buffer should not exceed max_size characters plus an extra
 * 0 character.
 *
 * Terminating keys are the return key (13) and all function keys
 * (see the specification of the Z-machine) which are accepted by
 * the is_terminator function. Mouse clicks behave like function
 * keys except that the mouse position is stored in mouse_x and
 * mouse_y (top left coordinates are (1,1)).
 *
 * Furthermore, Frotz introduces some special terminating keys:
 *
 *     HOT_KEY_PLAYBACK (Alt-P)
 *     HOT_KEY_RECORD (Alt-R)
 *     HOT_KEY_SEED (Alt-S)
 *     HOT_KEY_UNDO (Alt-U)
 *
 * If the timeout argument is not zero, the input gets interrupted
 * after timeout/10 seconds (and the return value is 0).
 *
 * Note that the screen is not scrolled after the return key was
 * pressed. Also note that the cursor should be placed at the end
 * of the input line when the function returns.
 *
 */

int os_read (int max_size, char *buffer, int timeout)
{
    struct timeb now;
    int row, col;
    int read_size;
    int pos;
    int entry;
    int c;

    overwrite_mode = 0;

    read_size = pos = strlen (buffer);
    entry = history_count;

    /* Calculate characters left on input line */

    os_get_cursor (&row, &col);
    if (max_size > h_screen_cols + read_size - col)
	max_size = h_screen_cols + read_size - col;

    /* Calculate time limit */

    if (timeout != 0) {

	ftime (&now);
	now.millitm += (timeout % 10) * 100;
	if (now.millitm >= 1000) {
	    now.millitm -= 1000;
	    now.time++;
	}
	now.time += timeout / 10;

    } else now.time = 0;

    /* Loop until the input is finished */

next_key:

    c = get_key (now.time, now.millitm);

    /* Backspace, erase the character to the left of the cursor */

    if (c == 8) {
	delete_left (&pos, &read_size, buffer);
	goto next_key;
    }

    /* Return key, store the input line in the history buffer */

    if (c == 13 && cwin == LOWER_WINDOW) {
	last_char (&pos, read_size);
	store_input (buffer);
    }

    /* Escape key, erase the current input line */

    if (c == 27) {
	erase_input (&pos, &read_size, buffer);
	goto next_key;
    }

    /* Plain character, add it to the input line */

    if (c >= 32 && c <= 127) {
	insert_char (max_size, &pos, &read_size, buffer, c);
	goto next_key;
    }

    /* Cursor up, fetch previous history entry */

    if (c == 129 && cwin == LOWER_WINDOW) {
	get_prev_entry (&pos, &read_size, buffer, &entry, max_size);
	goto next_key;
    }

    /* Cursor down, fetch next history entry */

    if (c == 130 && cwin == LOWER_WINDOW) {
	get_next_entry (&pos, &read_size, buffer, &entry, max_size);
	goto next_key;
    }

    /* Cursor left, move the character one character to the left */

    if (c == 131 && cwin == LOWER_WINDOW) {
	cursor_left (&pos);
	goto next_key;
    }

    /* Cursor right, move the cursor one character to the right */

    if (c == 132 && cwin == LOWER_WINDOW) {
	cursor_right (&pos, read_size);
	goto next_key;
    }

    /* European character, add it to the input line */

    if (c >= 155 && c <= 223) {
	insert_char (max_size, &pos, &read_size, buffer, c);
	goto next_key;
    }

    /* Home key, move cursor to the beginning of the input */

    if (c == 256 && cwin == LOWER_WINDOW) {
	first_char (&pos);
	goto next_key;
    }

    /* End key, move cursor to the end of the input */

    if (c == 257 && cwin == LOWER_WINDOW) {
	last_char (&pos, read_size);
	goto next_key;
    }

    /* Ctrl cursor left, move to previous word */

    if (c == 258 && cwin == LOWER_WINDOW) {
	prev_word (&pos, buffer);
	goto next_key;
    }

    /* Ctrl cursor right, move to next word */

    if (c == 259 && cwin == LOWER_WINDOW) {
	next_word (&pos, read_size, buffer);
	goto next_key;
    }

    /* Delete key, erase the character under the cursor */

    if (c == 260 && cwin == LOWER_WINDOW) {
	delete_char (&pos, &read_size, buffer);
	goto next_key;
    }

    /* Page up, this is a replacement for "Cursor up" */

    if (c == 261)
	c = 129;

    /* Page down, this is a replacement for "Cursor down" */

    if (c == 262)
	c = 130;

    /* Insert, toggle overwrite mode */

    if (c == 263 && cwin == LOWER_WINDOW) {
	overwrite_mode = !overwrite_mode;
	goto next_key;
    }

    /* Check for terminating key */

    if (pos < read_size || is_terminator (c) == 0)
	goto next_key;

    /* Return terminating key */

    return (c);

}/* os_read */

/*
 * os_read_char
 *
 * Read a single character from the keyboard (or a mouse click) and
 * return it. Input aborts after timeout/10 seconds.
 *
 */

int os_read_char (int timeout)
{
    struct timeb now;
    int key;

    /* Calculate time limit */

    if (timeout != 0) {

	ftime (&now);
	now.millitm += (timeout % 10) * 100;
	if (now.millitm >= 1000) {
	    now.millitm -= 1000;
	    now.time++;
	}
	now.time += timeout / 10;

    } else now.time = 0;

    /* Read a key and ignore the special keys from 256 onwards */

    do {

	key = get_key (now.time, now.millitm);

	/* Convert cursor up [down] */

	if (key == 261)
	    key = 129;
	if (key == 262)
	    key = 130;

    } while (key >= 256);

    /* Return key value */

    return (key);

}/* os_read_char */

/*
 * reset_sound
 *
 * Reset the sound board before the program stops.
 *
 */

static void reset_sound (void)
{

    /* Turn off speakers */

    write_dsp (0xd0);

    /* Restore old interrupt */

    if (sound_irq < 8) {

	outportb (0x21, inportb (0x21) | (1 << sound_irq));
	setvect (0x08 + sound_irq, saved_interrupt);

    } else {

	sound_irq &= 0x07;
	outportb (0xa1, inportb (0xa1) | (1 << sound_irq));
	setvect (0x70 + sound_irq, saved_interrupt);
	sound_irq |= 0x08;
    }

}/* reset_sound */

/*
 * os_reset_screen
 *
 * Reset the screen before the program ends.
 *
 */

void os_reset_screen (void)
{

    /* Print a message and wait for a keypress */

    display_string ("[Hit any key to exit.]");
    get_key (0, 0);

    /* Activate a monochrome or colour text mode */

    asm mov al,7
    asm cmp display_mode,0
    asm je call_bios
    asm mov al,3
call_bios:
    asm mov ah,0
    asm int 0x10

    /* Reset the sound board */

    if (h_flags & SOUND_FLAG)
	reset_sound ();
    if ((h_version == V3) && (h_flags & OLD_SOUND_FLAG))
	reset_sound ();

}/* os_reset_screen */

/*
 * os_scroll_area
 *
 * Scroll a rectangular area of the screen and fill the empty lines
 * with the current background colour. Top left coordinates are (1,1).
 * The cursor stays put.
 *
 */

void os_scroll_area (int top, int left, int bottom, int right)
{
    int attribute;

    if (display_mode <= 1)
	attribute = (screen_bg << 4) | screen_fg;
    else if (display_mode == 2)
	attribute = 0;
    else
	attribute = screen_bg;

    asm mov ah,6
    asm mov al,1
    asm mov ch,byte ptr top
    asm dec ch
    asm mov cl,byte ptr left
    asm dec cl
    asm mov dh,byte ptr bottom
    asm dec dh
    asm mov dl,byte ptr right
    asm dec dl
    asm mov bh,byte ptr attribute
    asm int 0x10

}/* os_scroll_area */

/*
 * adjust_style
 *
 * Set the current colours. This combines the current colour selection
 * and the current text style.
 *
 */

static void adjust_style (void)
{
    const colour_map[] = {
	0, 0,
	BLACK,
	RED,
	GREEN,
	BROWN,
	BLUE,
	MAGENTA,
	CYAN,
	LIGHTGRAY
    };

    if (display_mode == 0) {

	/* Monochrome video boards. These cannot do much, but they can
	   display bright and underlined (!) characters. */

	screen_fg = 7;
	screen_bg = 0;

	if (current_style & REVERSE_STYLE) {
	    text_fg = 0;
	    text_bg = 7;
	} else {
	    text_fg = 7;
	    text_bg = 0;
	}

	if (current_style & EMPHASIS_STYLE)
	    text_fg = 1;

	if (current_style & BOLDFACE_STYLE)
	    text_fg ^= 8;

    } else if (display_mode == 2) {

	/* CGA boards. The screen is black and white and all we can do
	   is to display reverse or normal characters. */

	if (current_style & REVERSE_STYLE)
	    text_fg = 0;
	else
	    text_fg = 1;

    } else {

	/* Colour boards. Things are complicated because the player may
	   optionally set default colours for normal or reverse video. */

	if (current_foreground == DEFAULT_COLOUR)
	    screen_fg = (user_foreground == -1) ? LIGHTGRAY : user_foreground;
	else
	    screen_fg = colour_map[current_foreground];

	if (current_background == DEFAULT_COLOUR)
	    screen_bg = (user_background == -1) ? BLUE : user_background;
	else
	    screen_bg = colour_map[current_background];

	if (current_style & REVERSE_STYLE) {

	    if (current_background == DEFAULT_COLOUR && user_reverse_fg != -1)
		text_fg = user_reverse_fg;
	    else
		text_fg = screen_bg;

	    if (current_foreground == DEFAULT_COLOUR && user_reverse_bg != -1)
		text_bg = user_reverse_bg;
	    else
		text_bg = screen_fg;

	} else {

	    text_fg = screen_fg;
	    text_bg = screen_bg;
	}

	if (display_mode == 1 && (current_style & EMPHASIS_STYLE))
	    text_fg = (user_emphasis == -1) ? YELLOW : user_emphasis;

	if (current_style & BOLDFACE_STYLE)
	    text_fg ^= 8;
    }

}/* adjust_style */

/*
 * os_set_colour
 *
 * Set the foreground and background colours which can be:
 *
 *     DEFAULT_COLOUR
 *     BLACK_COLOUR
 *     RED_COLOUR
 *     GREEN_COLOUR
 *     YELLOW_COLOUR
 *     BLUE_COLOUR
 *     MAGENTA_COLOUR
 *     CYAN_COLOUR
 *     WHITE_COLOUR
 *
 */

void os_set_colour (int new_foreground, int new_background)
{

    /* Store values in global variables */

    current_foreground = new_foreground;
    current_background = new_background;

    /* Apply changes */

    adjust_style ();

}/* os_set_colour */

/*
 * os_set_cursor
 *
 * Place the text cursor at the given coordinates. Top left is (1,1).
 *
 */

void os_set_cursor (int row, int col)
{

    asm mov ah,2
    asm mov bh,0
    asm mov dh,byte ptr row
    asm mov dl,byte ptr col
    asm dec dh
    asm dec dl
    asm int 0x10

}/* os_set_cursor */

/*
 * os_set_font
 *
 * Set the font for text output. The interpreter takes care not to
 * choose fonts which aren't supported by the interface.
 *
 */

void os_set_font (int new_font)
{

    current_font = new_font;

}/* os_set_font */

/*
 * os_set_text_style
 *
 * Set a text style which can be:
 *
 *     ROMAN_STYLE
 *     REVERSE_STYLE
 *     BOLDFACE_STYLE
 *     EMPHASIS_STYLE (aka underline aka italics)
 *     FIXED_WIDTH_STYLE
 *
 * Note that ROMAN_STYLE resets the text style back to normal.
 *
 */

void os_set_text_style (int new_style)
{

    /* Roman resets the text style */

    if (new_style == ROMAN_STYLE)
	current_style = ROMAN_STYLE;
    else
	current_style |= new_style;

    /* Apply changes */

    adjust_style ();

}/* os_set_text_style */

/*
 * start_of_dma
 *
 * Start the DMA transfer to the sound board.
 *
 */

static void start_of_dma (long address, unsigned length)
{
    const dma_offset_port[] = { 0x00, 0x02, 0x04, 0x06 };
    const dma_page_port[] = { 0x87, 0x83, 0x81, 0x82 };
    const dma_size_port[] = { 0x01, 0x03, 0x05, 0x07 };
    const dma_mask_port = 0x0a;
    const dma_mode_port = 0x0b;
    const dma_reset_port = 0x0c;

    length--;

    /* Set up DMA chip */

    outportb (dma_mask_port, 0x04 | sound_dma);
    outportb (dma_reset_port, 0x00);
    outportb (dma_mode_port, 0x48 | sound_dma);
    outportb (dma_offset_port[sound_dma], lo (lo_word (address)));
    outportb (dma_offset_port[sound_dma], hi (lo_word (address)));
    outportb (dma_page_port[sound_dma], lo (hi_word (address)));
    outportb (dma_size_port[sound_dma], lo (length));
    outportb (dma_size_port[sound_dma], hi (length));
    outportb (dma_mask_port, 0x00 | sound_dma);

    /* Play 8-bit mono sample */

    write_dsp (0x14);
    write_dsp (lo (length));
    write_dsp (hi (length));

    length++;

}/* start_of_dma */

/*
 * end_of_dma
 *
 * This is an interrupt function which is called when the DMA transfer
 * is complete.
 *
 */

static void interrupt end_of_dma (void)
{

    /* This function is called when a hardware interrupt signals the
       end of the current sound. We may have to play the second half
       of the sound effect, or we may have to repeat the sound. But
       if we are finished, we must call the end_of_sound routine. */

    if (play_part == 1 && sample_len2 != 0) {

	/* Play second part */

	play_part = 2;
	start_of_dma (sample_adr2, sample_len2);

    } else if (play_repeats == 255 || --play_repeats != 0) {

	/* Play another cycle */

	play_part = 1;
	start_of_dma (sample_adr1, sample_len1);

    } else {

	/* Stop playing */

	play_part = 0;
	end_of_sound ();
    }

    /* Tell interrupt controller(s) we are done */

    if (sound_irq >= 8)
	outportb (0xa0, 0x20);
    outportb (0x20, 0x20);

    /* Tell sound board we are done */

    inportb (sound_adr + 14);

}/* end_of_dma */

/*
 * os_start_sample
 *
 * Play the current sample at the given volume (ranging from 1 to 8,
 * and 255 being the default volume). The sound is played once or
 * several times in the background (255 meaning forever). In Z-code 3,
 * the repeats value is always 0 and the number of the repeats is taken
 * from the sound file itself. The end_of_sound function is called as
 * soon as the sound finishes.
 *
 */

void os_start_sample (int number, int volume, int repeats)
{

    /* Prepare sample */

    os_stop_sample ();
    os_prepare_sample (number);

    if (current_sample == 0)
	return;

    /* Set repeats if these are not zero */

    if (repeats != 0)
	play_repeats = repeats;
    else
	play_repeats = play_default;

    /* Set volume (requires Soundblaster Pro or better) */

    if (volume == 255)
	volume = DEFAULT_VOLUME;
    else if (volume > 0)
	volume = (volume << 1) - 1;

    outportb (sound_adr + 4, 0x22);
    outportb (sound_adr + 5, (volume << 4) | volume);

    /* Play sample using DMA */

    play_part = 1;
    start_of_dma (sample_adr1, sample_len1);

}/* os_start_sample */

/*
 * os_stop_sample
 *
 * Turn off the current sample.
 *
 */

void os_stop_sample (void)
{

    if (play_part != 0) {
	write_dsp (0xd0);
	play_part = 0;
    }

}/* os_stop_sample */

/*
 * os_text_length
 *
 * Calculate the length of a word in screen units. Apart from letters,
 * the word may contain special codes indicating changing text styles
 * or fonts. Currently, following codes exist:
 *
 *    CODE_NORMAL_STYLE
 *    CODE_REVERSE_STYLE
 *    CODE_BOLDFACE_STYLE
 *    CODE_EMPHASIS_STYLE
 *    CODE_FIXED_WIDTH_STYLE
 *    CODE_TEXT_FONT
 *    CODE_PICTURE_FONT
 *    CODE_GRAPHICS_FONT
 *    CODE_FIXED_WIDTH_FONT
 *
 */

int os_text_length (const char *buffer)
{
    int length;
    int i;
    int c;

    length = 0;

    for (i = 0; buffer[i] != 0; i++) {

	c = (unsigned char) buffer[i];

	/* Since this interface doesn't support proportional width fonts
	   yet, every normal characters requires exactly one screen unit. */

	if (c >= 32 && c <= 126 || c >= 155 && c <= 223)
	    length++;

	/* There are two characters (namely oe and OE ligatures) which
	   are not part of the IBM font 850 and need two screen units. */

	if (c == 220 || c == 221)
	    length++;
    }

    return (length);

}/* os_text_length */

/*
 * os_wait_sample
 *
 * Stop repeating the current sample and wait until it finishes.
 *
 */

void os_wait_sample (void)
{

    play_repeats = 1;

    /* Wait for hardware interrupt */

    while (play_part != 0);

}/* os_wait_sample */
