/*
 * file "S5text.c"
 *
 *
 */

#include "frotz.h"
#include "S5frotz.h"
#include "S5api.h"


/*
 * os_font_data
 *
 * Return true if the given font is available. The font can be
 *
 *    TEXT_FONT
 *    PICTURE_FONT
 *    GRAPHICS_FONT
 *    FIXED_WIDTH_FONT
 *
 * The font size should be stored in "height" and "width". If the given
 * font is unavailable then these values must _not_ be changed.
 *
 */

short os_font_data (struct sg *g, short font, short *height, short *width)
{

    /* All fonts of this interface have the same size */

    *height = g->h_font_height;
    *width = g->h_font_width;

    /* Not every font is available in every mode */

    if (font == TEXT_FONT)
	return TRUE;
    if (font == FIXED_WIDTH_FONT)
	return TRUE;

    /* Unavailable font */

    return FALSE;

}/* os_font_data */

/*
 * switch_scrn_attr
 *
 * Parts of the screen are usually erased to background colour.  However,
 * for deleting text in the input routine it can be useful to erase to
 * the current text background colour.  The two colours can be different,
 * for example when the reverse text style is used.  This helper function
 * toggles between the two possible behaviours.
 *
 */

void switch_scrn_attr (struct sg *g, short flag)
{
    byte scrn_bg;
    byte scrn_fg;

    if (flag) {
	scrn_bg = g->text_bg;
	scrn_fg = g->text_fg;
    } else {
	scrn_bg = (unsigned char)g->bg;
	scrn_fg = (unsigned char)g->fg;
    }
	
	g->scrn_attr = scrn_bg;
}/* switch_scrn_attr */

/*
 * adjust_style
 *
 * Set the current colours. This combines the current colour selection
 * and the current text style.
 *
 */

void adjust_style (struct sg *g)
{
	short attr;

	attr =  (g->current_style & REVERSE_STYLE ? 1 : 0);
	attr |= (g->current_style & EMPHASIS_STYLE ? 2 : 0);
	attr |= (g->current_style & BOLDFACE_STYLE ? 4 : 0);
	SrvTextAttr(g,attr);
    
    /* Set the screen attribute for scrolling and erasing */

    switch_scrn_attr (g,FALSE);

}/* adjust_style */


/*
 * os_set_text_style
 *
 * Set the current text style. Following flags can be set:
 *
 *     REVERSE_STYLE
 *     BOLDFACE_STYLE
 *     EMPHASIS_STYLE (aka underline aka italics)
 *     FIXED_WIDTH_STYLE
 *
 */

void os_set_text_style (struct sg *g, short new_style)
{

    g->current_style = new_style;

    /* Apply changes */

    adjust_style (g);

}/* os_set_text_style */

/*
 * os_set_font
 *
 * Set the font for text output. The interpreter takes care not to
 * choose fonts which aren't supported by the interface.
 *
 */

void os_set_font (struct sg *g, short new_font)
{

    g->current_font = new_font;

}/* os_set_font */

/*
 * os_display_char
 *
 * Display a character of the current font using the current colours and
 * text style. The cursor moves to the next position. Printable codes are
 * all ASCII values from 32 to 126, ISO Latin-1 characters from 160 to
 * 255, ZC_GAP (gap between two sentences) and ZC_INDENT (paragraph
 * indentation). The screen should not be scrolled after printing to the
 * bottom right corner.
 *
 */

void os_display_char (struct sg *g, zchar c)
{
    /* Handle special indentations */

    if (c == ZC_INDENT)
	{ 
		SrvPutChar(g,g->cursor_x++, g->cursor_y, ' ');
		SrvPutChar(g,g->cursor_x++, g->cursor_y, ' ');
		SrvPutChar(g,g->cursor_x++, g->cursor_y, ' ');
		return;
	}
    if (c == ZC_GAP)
	{ 
		SrvPutChar(g,g->cursor_x++, g->cursor_y, ' ');
		SrvPutChar(g,g->cursor_x++, g->cursor_y, ' ');
		return;
	}

    /* Display character */
    SrvPutChar(g,g->cursor_x++, g->cursor_y, c);
    /* Move cursor to next position */

}/* os_display_char */

/*
 * os_display_string
 *
 * Display string including formatting.
 *
 */

void os_display_string (struct sg *g, const zchar *s)
{

    zchar c;
	int i=0;

	// Go through the string and check for special formatting. If any exists print out the 
	// current string and do the special formatting. Keep doing this until the string
	// has been processed. This creates as large a string as 
	// possible to send to the server for higher performance
	while ((c = *s++) != 0)
	{
		// Translate FROTZ bordering fonts from funny EPOC equivalents
		// to nearest looking.
		if (c > 170) {
			if (c == 196) c = 151;
			else if (c == 179) c = 124;
			else if (c == 191 || c == 192 || c == 217 || c == 218) c = 32;
		}

			
		if (c == ZC_NEW_FONT || c == ZC_NEW_STYLE || c == ZC_INDENT || c == ZC_GAP)
		{
	
			// String contains characters before these control codes are reached
			// so printout this string first
			if (i > 0)
			{
				g->tempstr[i] = 0;
				SrvPutString(g,g->cursor_x, g->cursor_y, g->tempstr);
				g->cursor_x += os_string_width(g,g->tempstr);
				i=0;
			}
				
			if (c == ZC_NEW_FONT || c == ZC_NEW_STYLE) 
			{
				short arg = *s++;

				if (c == ZC_NEW_FONT)
					os_set_font (g,arg);
				if (c == ZC_NEW_STYLE)
					os_set_text_style (g,arg);

			}
			else os_display_char (g,c);

		}
		else g->tempstr[i++] = c;

	}

	// String contains chars so print out what remains
	if (i > 0) 
	{
		g->tempstr[i] = 0;
		SrvPutString(g,g->cursor_x, g->cursor_y, g->tempstr);
		g->cursor_x += os_string_width(g,g->tempstr);
	}

 
}/* os_display_string */

/*
 * os_string_width
 *
 * Calculate the length of a word in screen units. Apart from letters,
 * the word may contain special codes:
 *
 *    ZC_NEW_STYLE - next character is a new text style
 *    ZC_NEW_FONT  - next character is a new font
 *
 */

short os_string_width (struct sg *g, const zchar *s)
{
    short width = 0;

    short saved_font = g->current_font;
    short saved_style = g->current_style;

    zchar c;

    while ((c = *s++) != 0)
	if (c == ZC_NEW_STYLE || c == ZC_NEW_FONT) {

	    short arg = *s++;

	    if (c == ZC_NEW_FONT)
		g->current_font = arg;
	    if (c == ZC_NEW_STYLE)
		g->current_style = arg;

	} else width += 1;

    g->current_font = saved_font;
    g->current_style = saved_style;

    return width;

}/* os_string_width */

/*
 * os_set_cursor
 *
 * Place the text cursor at the given coordinates. Top left is (1,1).
 *
 */

void os_set_cursor (struct sg *g, short y, short x)
{
    g->cursor_y = y - 1;
    g->cursor_x = x - 1;
	SrvSetCursor(g,g->cursor_x, g->cursor_y);
}/* os_set_cursor */

/*
 * os_more_prompt
 *
 * Display a MORE prompt, wait for a keypress and remove the MORE
 * prompt from the screen.
 *
 */

void os_more_prompt (struct sg *g)
{
    short saved_x;

    /* Save text font and style */

    short saved_font = g->current_font;
    short saved_style = g->current_style;

    /* Choose plain text style */

    g->current_font = TEXT_FONT;
    g->current_style = 0;

    adjust_style (g);

    /* Wait until the user presses a key */

    saved_x = g->cursor_x;

    os_display_string (g,(zchar *) "[MORE]");
    os_read_key (g,0, TRUE);

    os_erase_area (g,
		g->cursor_y + 1,
		saved_x + 1,
		g->cursor_y + g->h_font_height,
		g->cursor_x + 1);

    g->cursor_x = saved_x;

    /* Restore text font and style */

    g->current_font = saved_font;
    g->current_style = saved_style;

    adjust_style (g);

}/* os_more_prompt */
