module AR::Game::Perception

  def known_objects
    self.known_object_ids.map{|id| self.world.find id}
  end

  def known_objects=obj_arr
    self.known_object_ids = obj_arr.map(&:id)
  end



  def self.init ar_object
    ar_object.class.attr_serializeable :known_object_ids 

    ar_object.register_gauge "perception"
    ar_object.register_gauge "critical_hit"
    ar_object.after_gauges_init do |obj|
      perc = obj.gauge("perception")
      crit = obj.gauge("critical_hit")
      perc.parents! crit, with: AR::Game::GaugeBehavior::LinearGaugeRelation, data: {factor: 1}
      perc.change_current_and_max 1
      perc.manual_child_sync
    end
    

    ar_object.class.attr_serializeable :perception_account do |perc_acc_data, op, inst|
      case op
      when :load
        ret = {}
        perc_acc_data.each do |k, v|
          if k.to_s.to_i == 0
            ret[k.to_s] = v
          else
            ret[k.to_s.to_i] = v
          end
        end
        ret
      when :unload
        perc_acc_data
      end
    end
    ar_object.perception_account = {}
    ar_object.known_object_ids = []
  end

  def perception_lvl
    gauge("perception").modified_value
  end

  def self.emit_discovery_when_necessary
    AR::Events::Resolution.new event_class: AR::Events::PerceptionAccountChangedEvent, as: "perceiver" do |e|
      results=[]
      perceiver = e.perceiver
      discoverable = e.discoverable
      change       = e.change

      can_see = perceiver.can_see? discoverable
      perceiver.change_perception_for discoverable, change
      potential_to_see = perceiver.potential_to_see? discoverable

      if potential_to_see != can_see
        if potential_to_see
          results << (e.engine.create AR::Events::DiscoveryMadeEvent, ({perceiver: perceiver, discovered: discoverable}))
        else
          results << (e.engine.create AR::Events::DiscoveryLostEvent, ({perceiver: perceiver, undiscovered: discoverable}))
        end
      end 

      results
    end
  end

  def self.add_to_known_objects_on_discovery
    AR::Events::Resolution.new event_class: AR::Events::DiscoveryMadeEvent, as: "perceiver" do |e|
      perceiver = e.perceiver
      discovered = e.discovered
      perceiver.known_object_ids.push discovered.id
      nil
    end
  end

  def self.rem_from_known_objects_on_undiscovered
    AR::Events::Resolution.new as: "perceiver", event_class: AR::Events::DiscoveryLostEvent do |e|
      perceiver = e.perceiver
      undiscovered= e.undiscovered
      perceiver.known_object_ids.delete undiscovered.id
      perceiver.perception_account.delete undiscovered.id
      nil
    end
  end


  def self.discovery_lost_on_object_died ar_object
    AR::Events::Subscription.new event_class: AR::Events::ARObjectDied do |e|
      died = e.ar_object 
      if  ar_object.can_see? died and ar_object != died 
          AR::Events::DiscoveryLostEvent.new perceiver: ar_object, undiscovered: died
      else
        nil
      end
    end
  end
  def self.discovery_lost_on_item_added_to_inventory ar_object
    AR::Events::Subscription.new event_class: AR::Events::ItemAddedToInventoryEvent do |e|
      item = e.item
      if  ar_object.can_see? item and ar_object != e.inventory_owner
          (ar_object.create AR::Events::DiscoveryLostEvent, ({perceiver: ar_object, undiscovered: item}))
      else
        nil
      end
    end
  end


  def self.extended ar_object
    self.init ar_object

    ar_object.resolutions << emit_discovery_when_necessary
    ar_object.resolutions << add_to_known_objects_on_discovery
    ar_object.resolutions << rem_from_known_objects_on_undiscovered
    ar_object.subscriptions << discovery_lost_on_item_added_to_inventory(ar_object)
    ar_object.subscriptions << discovery_lost_on_object_died(ar_object)
  end

  #should really just be called internally by the extending class:w


  def perception_for object
    init_perception_to object if perception_account[object.id].nil?
    perception_account[object.id]
  end

  #should change as a resolution for the 
  #perception gained event
  def change_perception_for discoverable, amount
    p_v = perception_for discoverable
    perception_account[discoverable.id] = p_v + amount

    #do object lost sight/gained sight events if nessecary
    perception_account[discoverable.id]
  end

  def can_see? discoverable
    known_objects.include? discoverable
  end 

  def now_sees discoverable
    known_object_ids.push discoverable.id
  end

  def potential_to_see? discoverable
    discoverable.discovery_threshold <= perception_for(discoverable)
  end

  def init_perception_to object
    perception_account[object.id] = 0
  end

  def del_perception_to discoverable
    perception_account.delete object.id
  end

end

