#
# StoryFactory - Interactive Fiction compiler/interpreter.
# Copyright PW.Soft. 2003
# Email to pwsoft@syntiac.com or on the web http://www.syntiac.com
#
#

use strict;
use warnings;
my $version = "0.75";
$| = 1;



#
# Get commandline options
#

my $scriptFileName;
my $outputCFileName;
my $outputHTMLFileName;
my $outputMAPFileName;

if (exists $ARGV[0]) {
  $scriptFileName = $ARGV[0];
  # output filename is same as scriptname but with extension .cpp
  $outputCFileName = $scriptFileName;
  $outputCFileName =~ s/(.*)\.(.*)$/$1.cpp/;
  $outputHTMLFileName = $scriptFileName;
  $outputHTMLFileName =~ s/(.*)\.(.*)$/$1.html/;
  $outputMAPFileName = $scriptFileName;
  $outputMAPFileName =~ s/(.*)\.(.*)$/$1.map/;
} else {
  print "Usage: storyfactory <scriptname> [debug]\n";
  exit(1);
}


#
# open files
#
open(CCODE,    ">$outputCFileName" ) or die "Can't open $outputCFileName: $!";
open(HTMLCODE, ">$outputHTMLFileName" ) or die "Can't open $outputHTMLFileName: $!";


#
# VM configuration
#
my $maxCodeMem = 65536;
my $maxinputlen = 800;
my $dataStackSize = 128;
my $returnStackSize = 128;

#
# video output
#
my $screenWidth = 80;
my $scrollBack = 25;


#
# VM
#
my $firstOpcode    = 0xE000;
my $jumpRange      = 0x0200;

my $elseOpcode     = $firstOpcode;
my $eqifOpcode     = $elseOpcode + $jumpRange;
my $neqifOpcode    = $eqifOpcode + $jumpRange;
my $ifOpcode       = $neqifOpcode + $jumpRange;
my $nifOpcode      = $ifOpcode + $jumpRange;
my $jumpbackOpcode = $nifOpcode + $jumpRange;    # used for [loop]
my $forthOpcodes   = $jumpbackOpcode + $jumpRange;

my $litOpcode      = $forthOpcodes +  0;
my $getOpcode      = $forthOpcodes +  1;
my $putOpcode      = $forthOpcodes +  2;
my $retOpcode      = $forthOpcodes +  3;
my $fretOpcode     = $forthOpcodes +  4;
my $tretOpcode     = $forthOpcodes +  5;
my $dropfretOpcode = $forthOpcodes +  6;
my $droptretOpcode = $forthOpcodes +  7;
my $executeOpcode  = $forthOpcodes +  8;

my $dropOpcode     = $forthOpcodes +  9;
my $swapOpcode     = $forthOpcodes + 10;
my $dupOpcode      = $forthOpcodes + 11;
my $overOpcode     = $forthOpcodes + 12;
my $thirdOpcode    = $forthOpcodes + 13;

my $torOpcode      = $forthOpcodes + 14;
my $rfromOpcode    = $forthOpcodes + 15;
my $fetchOpcode    = $forthOpcodes + 16;
my $storeOpcode    = $forthOpcodes + 17;

my $eqOpcode       = $forthOpcodes + 18;
my $neqOpcode      = $forthOpcodes + 19;
my $bigOpcode      = $forthOpcodes + 20;
my $smallOpcode    = $forthOpcodes + 21;

my $orOpcode       = $forthOpcodes + 22;
my $andOpcode      = $forthOpcodes + 23;

my $addOpcode      = $forthOpcodes + 24;

my $zeroOpcode     = $forthOpcodes + 25;
my $oneOpcode      = $forthOpcodes + 26;
my $twoOpcode      = $forthOpcodes + 27;
my $threeOpcode    = $forthOpcodes + 28;

my $inputOpcode    = $forthOpcodes + 29;  # read and parse input, only returns when valid input is entered
my $dotOpcode      = $forthOpcodes + 30;  # print TOS as integer

my $addConstOpcode = $forthOpcodes + 256;  # add small constant to TOS
my $subConstOpcode = $forthOpcodes + 512;  # subtract small constant to TOS
my $wordOpcode     = $forthOpcodes + 768;

printf ("VM next opcode (must less or equal 0x10000): 0x%04X\n", ($wordOpcode + 256));

my $nospaceWords = 6;



#
# Compiler variables
#
my @compileStack=();
my @mapfileLines=();

my $pass      = 0;

my $codeCount = 0;
my $varCount  = 0;
my $charCount = 0;
my $wordCount = 0;

my $wordsOffset = 0;
my $varsOffset  = 0;

my %words=();
my $wordCount = 0;



# the generated code
my @code=();
#my @words=();
my %globalVars=();
my %subroutines=();
my %objects=();
my %defines=();

my $currentObject;
my $prevObject=0;
my @database;
my $gameTitle = "<no title>";


sub word {
  my $word;
  my $w = shift;
  my $f = shift;

# Check if word object
  if (exists $objects{$w}) {
    return $objects{$w};
  }
# Check if word is variable name
  if (exists $globalVars{$w}) {
    return $globalVars{$w};
  }
# Check if word is subroutine name
  if (exists $subroutines{$w}) {
    return $subroutines{$w};
  }
# Search if word already exists
  if (exists $words{$w}) {
    return $words{$w};
  }
# Create word if new

  my $len = int((length($w)+1)/2) + 1;

  if ($pass == 2) {
    if (!defined($f)) {
      $f=0;
    }
    push(@mapfileLines, sprintf("%04X %04X %04X  \"%s\"\n", $wordsOffset + $charCount, $len, $wordsOffset + $charCount + $len, $w));
    $code[$wordsOffset + $charCount] = $wordOpcode + length($w) + $f;
    for(my $i = 0; $i < length($w); $i++) {
      if (!defined($code[$wordsOffset + $charCount + int($i/2) + 1])) {
        $code[$wordsOffset + $charCount + int($i/2) + 1] = ord(substr $w, $i, 1)*256;
      } else {
        $code[$wordsOffset + $charCount + int($i/2) + 1] += ord(substr $w, $i, 1);
      }
    }

    $wordCount++;

    my $loww = lc($w);
    if (!($loww eq $w)) {
      print "warning '$w' has capitals.\n";
    }
  }

  $words{$w} = $wordsOffset + $charCount;
  $charCount = $charCount + $len;
  return $words{$w};
}


#
# Fill default word list
#
sub initWords {

  # initialise vars
  %words=();
  $charCount = 0;

  # no leading space words
#  word('\n');
  word('',     0xC0);
  word('.',    0xC0);
  word('?',    0xC0);
  word(',',    0x40);
  word(';',    0xC0);
  word(':',    0xC0);

  # leading space words
  word('``',   0x80);
  word('\'\'', 0xC0);
}

sub initCode {
  @code=(
    0,              # 0 first object
    0,              # 1 main code
    0,              # 2 parsed input position
    $wordsOffset,   # 3 words start addr
  );

  $codeCount=4;
#  %globalVars=();
#  %subroutines=();
#  %objects=();
  %defines=();
  @database=();
  $prevObject = 0;
}

sub findObject {
  my $label = shift;
  my $object;

  foreach $object (@database) {
#    print "findloc: " . $object->{'label'} . "\n";
    if ($object->{'label'} eq $label) {
      return $object->{'offset'};
    }
  }
  print "location '$label' not found in database.";
  die;
}

sub storeCode {
  my $data = shift;
  $code[$codeCount] = $data;
  $codeCount++;
}

#
# Peephole optimizer
#
my $lastOpcode = 0;
my $retSaved=0;
my $dropXretSaved=0;
sub storeOpcode {
  my $data = shift;

  if (($data == $retOpcode)
     && (($lastOpcode == $retOpcode)
     ||  ($lastOpcode == $fretOpcode)
     ||  ($lastOpcode == $tretOpcode)
     ||  ($lastOpcode == $dropfretOpcode)
     ||  ($lastOpcode == $droptretOpcode)
        )
     ) {
    $retSaved += 1;
  } elsif ($data == $fretOpcode && $lastOpcode == $dropOpcode) {
    $code[$codeCount-1] = $dropfretOpcode;
    $lastOpcode = $dropfretOpcode;
    $dropXretSaved += 1;
  } elsif ($data == $tretOpcode && $lastOpcode == $dropOpcode) {
    $code[$codeCount-1] = $droptretOpcode;
    $lastOpcode = $droptretOpcode;
    $dropXretSaved += 1;
  } else {
    storeCode($data);
    $lastOpcode = $data;
  }

# comment this out to disable peephole optimizer
#  $lastOpcode=0;
}

sub parseCode {
  my $script = shift;
  my $word;

  while (<$script>) {
    s/\n//;
    next if (/^\s*#.*/);
    if (/^\s*}/) {
      storeOpcode($retOpcode);
      $lastOpcode = 0;
      last;
    }

    split;
    #split(' ', @_);
    foreach $word (@_) {
      if ($word =~ /^&(.*)/) {
        storeCode($litOpcode);
        $word = $1;
      }
      # preprocessor, process defines
      if (exists $defines{$word}) {
        $word = $defines{$word};
      }

      if ($word =~ /\[obj:([a-zA-Z_]+)\]/) {
        storeCode(findObject($1));
      } elsif ($word eq '/self') {
        storeOpcode($litOpcode);
        storeCode($currentObject->{'offset'});
      } elsif ($word eq '/lit') {
        storeOpcode($litOpcode);
      } elsif ($word eq '/get') {
        storeOpcode($getOpcode);
      } elsif ($word eq '/to') {
        storeOpcode($putOpcode);
      } elsif ($word eq '/ret') {
        storeOpcode($retOpcode);
      } elsif ($word eq '/fret') {
        storeOpcode($fretOpcode);
      } elsif ($word eq '/tret') {
        storeOpcode($tretOpcode);
      } elsif ($word eq '/execute') {
        storeOpcode($executeOpcode);
      } elsif ($word eq '/fetch') {
        storeOpcode($fetchOpcode);
      } elsif ($word eq '/store') {
        storeOpcode($storeOpcode);
      } elsif ($word eq '/drop') {
        storeOpcode($dropOpcode);
      } elsif ($word eq '/swap') {
        storeOpcode($swapOpcode);
      } elsif ($word eq '/dup') {
        storeCode($dupOpcode);
      } elsif ($word eq '/over') {
        storeOpcode($overOpcode);
      } elsif ($word eq '/third') {
        storeOpcode($thirdOpcode);
      } elsif ($word eq '/eq') {
        storeOpcode($eqOpcode);
      } elsif ($word eq '[>]') {
        storeOpcode($bigOpcode);
      } elsif ($word eq '[<]') {
        storeOpcode($smallOpcode);
      } elsif ($word eq '/or') {
        storeOpcode($orOpcode);
      } elsif ($word eq '/and') {
        storeOpcode($andOpcode);

      } elsif ($word eq '[>r]') {
        storeOpcode($torOpcode);
      } elsif ($word eq '[r>]') {
        storeOpcode($rfromOpcode);
      } elsif ($word eq '/add') {
        storeOpcode($addOpcode);
      } elsif ($word eq '/0') {
        storeOpcode($zeroOpcode);
      } elsif ($word eq '/1') {
        storeOpcode($oneOpcode);
      } elsif ($word eq '/2') {
        storeOpcode($twoOpcode);
      } elsif ($word eq '/3') {
        storeOpcode($threeOpcode);
      } elsif ($word eq '/input') {
        storeOpcode($inputOpcode);
      } elsif ($word eq '/dot') {
        storeOpcode($dotOpcode);
      } elsif ($word eq '/do') {      
        push(@compileStack, 0);
        push(@compileStack, $codeCount);
      } elsif ($word eq '/loop') {      
        my $offset=pop(@compileStack);
        storeOpcode($jumpbackOpcode + $codeCount - $offset);

        $offset=pop(@compileStack);
        while ($offset > 0) {
          $code[$offset-1]=$code[$offset-1]+$codeCount-$offset;
          $offset=pop(@compileStack);
        }
      } elsif ($word eq '/eqwhile') {
        my $temp=pop(@compileStack);
        storeOpcode($eqifOpcode);
        push(@compileStack, $codeCount);
        push(@compileStack, $temp);
      } elsif ($word eq '/neqwhile') {
        my $temp=pop(@compileStack);
        storeOpcode($neqifOpcode);
        push(@compileStack, $codeCount);
        push(@compileStack, $temp);
      } elsif ($word eq '/if') {
        storeOpcode($ifOpcode);
        push(@compileStack, $codeCount);
      } elsif ($word eq '/nif') {
        storeOpcode($nifOpcode);
        push(@compileStack, $codeCount);
      } elsif ($word eq '/eqif') {
        storeOpcode($eqifOpcode);
        push(@compileStack, $codeCount);
      } elsif ($word eq '/neqif') {
        storeOpcode($neqifOpcode);
        push(@compileStack, $codeCount);
      } elsif ($word eq '/else') {
        my $offset=pop(@compileStack);
        storeOpcode($elseOpcode);
        $code[$offset-1]=$code[$offset-1]+$codeCount-$offset;
        push(@compileStack, $codeCount);
      } elsif ($word eq '/then') {
        my $offset=pop(@compileStack);
        $code[$offset-1]=$code[$offset-1]+$codeCount-$offset;
        $lastOpcode = 0;
      } elsif ($word =~ /\[\+([0-9]+)\]/) {
        storeOpcode($addConstOpcode + $1);
      } elsif ($word =~ /\[\-([0-9]+)\]/) {
        storeOpcode($subConstOpcode + $1);
      } elsif ($word =~ /^[0-9]+$/) {
        storeOpcode($word);
      } elsif ($word eq '^') {
        # newline (assumes crlf is first string in word array, which it is ;-)
        storeOpcode($wordsOffset);
      } else {
#                print "opcode: '$word'\n";
        storeOpcode(word($word));
      }
    }
  }
}

sub parseGlobalVars {
  my $script = shift;
  my $word;

  while (<$script>) {
    s/\n//;
    last if (/^\s*}/);

    split;
    foreach $word (@_) {
      if ($word eq '#') {
        last
      }
      $globalVars{$word} = $codeCount;
      if ($pass == 1) {
        push(@mapfileLines, sprintf("%04X 0001 %04X  variable: $word\n", $codeCount, $codeCount+1));
      }
      storeCode(0);
    }
  }
}


#
# Read script into memory and pre-process some info.
#
sub parseScript {
  my $filename = shift;
  my $script;

  open($script, $filename ) or die "Can't open $filename: $!";

  while (<$script>) {
    s/\n//;
    if (/#.*/) {
  #    print "comment: $_\n";
    } elsif (/^\s*title\s*(.*)/) {
      $gameTitle = $1;
    } elsif (/^\s*main\s*{/) {
      $code[1] = $codeCount;
      parseCode($script);
    } elsif (/^\s*include\s*([a-zA-Z0-9_.\/]*)/) {      
      parseScript($1);
    } elsif (/^\s*define\s+([a-zA-Z_][a-zA-Z0-9_]*)\s+([a-zA-Z0-9_]+)/) {
  #    printf "define $1 = $2\n" ;
      $defines{$1} = $2;
    } elsif (/^\s*vars\s*{/) {
      parseGlobalVars($script);
    } elsif (/^\s*sub\s*(.+?)\s*{/) {
      if (exists $subroutines{$1}) {
        if ($subroutines{$1} != $codeCount) {
          print "procedure redefinition of '$1'\n";
          die;
        }
      }
      if ($pass == 1) {
        push(@mapfileLines, sprintf("%04X ---- ----  routine: $1\n", $codeCount));
      }
      $subroutines{$1} = $codeCount;
      parseCode($script);
    } elsif (/^\s*object\s+([a-zA-Z_]+)\s*{/) {
      $currentObject = {};
      $currentObject->{'label'}=$1;
      $currentObject->{'offset'}=$codeCount;
      $objects{$1}=$codeCount;
      if ($pass == 1) {
        push(@mapfileLines, sprintf("\n%04X ---- ----  object: $currentObject->{'label'}\n", $codeCount));
      }
#      print "object '$1' offset '$currentObject->{'offset'}' {\n";

      # register object tree
      if ($code[0] eq 0) {
        $code[0]=$#code+1;
      }
      storeCode(0); # next object
      storeCode(0); # parent
      storeCode(0); # worn by parent
      storeCode(0); # noun runcode
      storeCode(0); # actor runcode

      my $has_name = 0;
      my $has_alias = 1; # set to 1 because aliases are disabled in the game


      while (<$script>) {
        s/\n//;

        if (/^\s*name:\s+(.*\w)/) {
          my $word;
          my @t;

          $currentObject->{'name'}=$1;
#          print "Object $currentObject->{'label'} has in-game name '$1'\n";
  
          @t = split(' ', $currentObject->{'name'});
          foreach $word (@t) {
            storeCode(word($word));
          }
          storeCode($retOpcode);
          $has_name = 1;
        } elsif (/^\s*alias:\s+(.*\w)/) {
          my $word;
          my @t;

          if ($has_name == 0) {
            print "Object $currentObject->{'label'} declares alias but no name.\n"
          }
  
          $currentObject->{'alias'}=$1;
          print "Object $currentObject->{'label'} has in-game alias '$1'\n";

          @t = split(' ', $currentObject->{'alias'});
          foreach $word (@t) {
#            print "worda: '$word'\n";
            storeCode(word($word));
          }
          storeCode($retOpcode);
          $has_alias = 1;
        } else {
          if ($has_name == 0) {
            storeCode($retOpcode);
            $has_name = 1;
          }
          if ($has_alias == 0) {
            storeCode($retOpcode);
            $has_alias = 1;
          }
        }
        if (/^\s*location:\s+(.*\w)/) {
          $currentObject->{'location'}=$1;
#          print "Object $currentObject->{'label'} has in-game location '$1'\n";
        }      
        if (/^\s*worn:\s+(.*\w)/) {
          $currentObject->{'worn'}=$1;
#          print "Object $currentObject->{'label'} has in-game location '$1'\n";
        }      
        if (/^\s*vars\s*{/) {
          parseGlobalVars($script);
          next;
        }
        if (/^\s*sub\s+([a-zA-Z_]+)\s*{/) {
#          print "define object subroutine '$1'\n";
          if ($pass == 1) {
            push(@mapfileLines, sprintf("%04X ---- ----  routine: $1\n", $codeCount));
          }
          if (exists $subroutines{$1}) {
            if ($subroutines{$1} != $codeCount) {
              print "procedure redefinition of '$1'\n";
              die;
            }
          }
          $subroutines{$1} = $codeCount;
#          print "routine at $subroutines{$1}\n";
          parseCode($script);
          next;
        }
        if (/^\s*run\s*{/) {
          $code[$currentObject->{'offset'}+3] = $codeCount;
          parseCode($script);
          next;
        }
        if (/^\s*noun\s*{/) {
          $code[$currentObject->{'offset'}+3] = $codeCount;
          parseCode($script);
          next;
        }
        if (/^\s*actor\s*{/) {
          $code[$currentObject->{'offset'}+4] = $codeCount;
          parseCode($script);
          next;
        }
#        if (/^\s*before\s*{/) {
#          $code[$currentObject->{'offset'}+2] = $codeCount;
  #        print "code at " . $codeCount . "\n";
  #            print $codeCount;
#          parseCode($script);
#          next;
#        }
        #if (/^\s*after\s*{/) {
#          $code[$currentObject->{'offset'}+3] = $codeCount;
  #        print "code at " . $codeCount . "\n";
  #            print $codeCount;
#          parseCode($script);
#          next;
#        }
        if (/^\s*}\s*/) {
          if (defined $currentObject->{'location'}) {
            my $parent = findObject($currentObject->{'location'});
#            print "found parent at offset $parent\n";
            $code[$currentObject->{'offset'}+1]=$parent;       # Parent
          } else {
            print "warning: no location given for object $currentObject->{'label'}\n";
          }
          if (defined $currentObject->{'worn'}) {
            my $parent = findObject($currentObject->{'worn'});
            $code[$currentObject->{'offset'}+2]=$parent;
          }
          # Linked list of objects
          $code[$prevObject]=$currentObject->{'offset'};
          $prevObject=$currentObject->{'offset'};

          # add object to database
          push(@database, $currentObject);

          # leave this nesting level
          last;
        }


  #         print "test: $currentObject['label']";

      }
      if ($pass == 1) {
        push(@mapfileLines, sprintf("---- %04X %04X  object: $currentObject->{'label'}\n\n", $codeCount - $prevObject, $codeCount));
      }
    }
  #   elsif (/^\s*$/) {
  #    print "empty line\n";
  #  }
  }
  close($script);
}

sub printOffsets {
  printf ("code : %04x + %04x => %04x\n", 0, $codeCount,  scalar(@code));
  printf ("words: %04x + %04x => %04x  count = $wordCount\n", $wordsOffset, $charCount, $varsOffset);
  printf ("vars : %04x + %04x => %04x\n", $varsOffset, 0, $varsOffset);
}

#
# PASS 1
# get length of everything and calculate offsets
#
print "*** PASS 1\n";
$pass = 1;
initCode();
initWords();
parseScript($scriptFileName);
$wordsOffset = $codeCount;
$varsOffset = $wordsOffset + $charCount;
printOffsets();

#
# PASS 2
# generate code
#
print "*** PASS 2\n";
$pass = 2;
initCode();
initWords();
parseScript($scriptFileName);
if (scalar(@code) != ($wordsOffset + $charCount)) {
  print "*** COMPILE ERROR DURING PASS 2 ***\n";
}
#$wordsOffset = scalar(@code);
#$varsOffset = $wordsOffset + $charCount;
printOffsets();

# store parsed input location
$code[2] = scalar(@code);

my $optTotal = $retSaved + $dropXretSaved;
print sprintf("peephole optimisation result\nret = %dx  drop+tret or drop+fret = %dx\ntotal = %d opcodes is %d bytes. Resulting game is %d bytes.", $retSaved, $dropXretSaved, $optTotal, $optTotal*2, $code[2]*2);

##########################################################################
#
#
# Export as ANSI-C program
#
#
##########################################################################

#
# write C-headers and system depended macros
#
print CCODE <<EOF
/*
 * Automatically generated from '$scriptFileName' by StoryFactory $version.
 * If changes are required change the script instead of this file.
 *
 * Copyright PW.Soft. 2003
 * Email to pwsoft\@syntiac.com or on the web http://www.syntiac.com
 *
 */ 
#if (!defined _DEBUG) && (defined _MSC_VER)
#if (defined SMALL_EXE)
#define STRICT
#define WIN32_LEAN_AND_MEAN
// Windows tiny exe
#include <windows.h>
#include <conio.h>
#include <string.h>
#pragma comment(linker,"/ENTRY:main");
#define vprintf wvprintf
#define vsprintf wvsprintf
#define printf(x, y) _cputs(y);
#else
// Windows standard
#pragma comment(linker,"/opt:nowin98 /OPT:REF /MAP /MAPINFO:FIXUPS")  /* VC6: turn off 4kbyte alignment. */
#include <stdarg.h>
#include <ctype.h>
#include <stdlib.h>
#include <conio.h>
#include <stdio.h>
#include <string.h>
#endif
#else
// Unix / linux / mac
#include <stdarg.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#ifndef _getch
#define _getch() getc(stdin)
#endif
#endif


EOF
;

print CCODE "#define INIT_LENGTH " . scalar(@code) . "\n";
print CCODE "static const unsigned short initCode[INIT_LENGTH] = {\n";
my $item;
my $itemcount=0;
foreach $item (@code) {
  print CCODE sprintf("0x%04X, ", $item);
  $itemcount++;
  if ($itemcount > 7) {
    $itemcount=0;
    print CCODE "\n";
  }
}
print CCODE "};\n";


print CCODE <<EOF

#define MAXCODEMEM   $maxCodeMem
static unsigned short code[MAXCODEMEM];


FILE *logfile;

static void ll(const char *formatstr, ...) {
  char tmp[2048];
  va_list(arglist);
  va_start(arglist, formatstr);     

  vsprintf(tmp, formatstr, arglist);
  if (logfile) {
    fprintf(logfile, "%s", tmp);
    fflush(logfile);
  }

  va_end(arglist);
}

static void lp(const char *formatstr, ...) {
  char tmp[2048];
  va_list(arglist);
  va_start(arglist, formatstr);     

  vsprintf(tmp, formatstr, arglist);
  if (logfile) {
    fprintf(logfile, "%s", tmp);
    fflush(logfile);
  }
  printf("%s", tmp);

  va_end(arglist);
}

static int curCol;
static int capitalLetter;
static void printWord(int index) {
  int lencode = code[index++];
  int len = lencode & 0x3F;
  int i;

  if (len==0) {
    curCol=0;
    lp("\\n");
    capitalLetter = 1;
  } else {
    if (len+curCol > ($screenWidth - 2) || (len+curCol > ($screenWidth - 4) && len > 1)) {
      curCol=0;
      lp("\\n");
    }
    if ((lencode & 0x40) == 0) {
      lp(" ");
      curCol++;
    }
    for(i=0;i<len;i++) {
      char c;
      if ((i&1)==0) {
        c=code[index + i/2]>>8;
      } else {
        c=code[index + i/2]&0xFF;
      }
      if (capitalLetter && c>='a' && c<='z') {
        lp("%c", c+('A'-'a'));
      } else {
        lp("%c", c);
      }
      capitalLetter = 0;
    }
    capitalLetter=(lencode & 0x80)?1:0;
    curCol+=len;
  }
}

static int findWord(const char *str, int len) {
  int index = code[3];
  
  while (index < code[2]) {
    int wordlen=code[index]&0x3F;
    if (wordlen == len) {
      bool found=true;

      for(int i=0; i<len; i++) {
        char c;
        if ((i&1)==0) {
          c=code[index + 1 + i/2]>>8;
        } else {
          c=code[index + 1 + i/2]&0xFF;
        }
        if (tolower(str[i]) != tolower(c)) {
          found=false;
          break;
        }        
      }
      if (found) return index;
    }
    index += ((code[index]&0x3F)+1)/2+1;
  }
  return 0;
}

static int parsedCount;
static int parseInput(const char *input) {
  int len, word;
  int err=0;
  int i=0;
//  int p=0;

  parsedCount=0;
  while(input[i]) {
    len=0;
    while (input[i]==' ') {
      i++;
    }
    if (input[i]=='\\0') break;
    while((input[i+len]>='a' && input[i+len]<='z') ||
          (input[i+len]>='A' && input[i+len]<='Z') ||
           input[i+len]=='\\'') {
      len++;
    }
    if (len==0) len=1;

    word=findWord(input+i, len);
    if (word) {
      code[code[2]+(parsedCount++)]=word;
//      lp("%4X ", word);
    } else {
      err++;
      lp("'");
      for(int q=0;q<len;q++) {
        lp("%c", input[i+q]);
      }
      lp("' is not recognised\\n");
    }
    i+=len;
  }
//  lp("\\n");
  code[code[2]+parsedCount]=$retOpcode;
  return err;
}

#define MAXINPUTLEN $maxinputlen
#define DATASTACKSIZE $dataStackSize
#define RETURNSTACKSIZE $returnStackSize
static char curInput[MAXINPUTLEN];
static int dataStack[DATASTACKSIZE];
static int returnStack[RETURNSTACKSIZE];
static int dsp;
static int rsp;
static int pc;
static void readInput(void) {
  int pos=0;
  int c;

  if ( dsp != 0 ) {
    lp("\\n(%d stack leak)> ", dsp);
  } else {
    lp("\\n> ");
  }

  c=_getch();
  while (pos==0 || (c!=10 && c!=13)) {
    if (c==8 && pos>0) {
      pos--;
      printf("%c %c", 8, 8);
    }
    if (c>=32 && c<127 && pos<MAXINPUTLEN) {
      curInput[pos++]=c;
      printf("%c", c);
    }
    c=_getch();
  }
  curInput[pos]='\\0';
//  lp("\\r> ");
  ll("%s", curInput);
  lp("\\n");
  
  curCol=0;
  capitalLetter=1;
//  printf("you typed '%s'\\n", curInput);
}

static int pop(void) {
  if (dsp>0) {
    return dataStack[--dsp];
  } else {
    lp("\\nError: stack underflow PC=%04X (%d)\\n", pc, pc);
    return 0;    
  }
}
static void push(int value) {
  dataStack[dsp++]=value;
}
static void pushr(int value) {
  returnStack[rsp++]=value;
}

static void run(void) {

  pc=code[1];
  curCol=0;
  capitalLetter=1;
  dsp=0;
  rsp=0;
  for(;;) {
    if (pc>=$firstOpcode) {
      lp("Error: PC out of range 0x%08X", pc);
      return;
    }
    switch(code[pc++]) {
    case $litOpcode:
      dataStack[dsp++]=code[pc++];
      break;
    case $getOpcode:
      dataStack[dsp++]=code[code[pc++]];
      break;
    case $putOpcode:
      code[code[pc++]]=pop();
      break;
    case $retOpcode:
    _retopcode:
      if (rsp > 0) {
        pc = returnStack[--rsp];
      } else {
        return;
      }
      break;
    case $fretOpcode:
      push(0);
      goto _retopcode;
    case $tretOpcode:
      push(1);
      goto _retopcode;
    case $dropfretOpcode:
      (void)pop();
      push(0);
      goto _retopcode;
    case $droptretOpcode:
      (void)pop();
      push(1);
      goto _retopcode;
    case $executeOpcode:
      pushr(pc);
      pc=pop();
      break;
    case $fetchOpcode: {
        int addr=pop();
        push(code[addr]);
      } break;
    case $storeOpcode: {
        int addr=pop();
        int v=pop();
        code[addr]=v;
      } break;
    case $bigOpcode: {
        int top=pop();
        int bottom=pop();
        push((bottom>top)?1:0);
      } break;
    case $smallOpcode: {
        int top=pop();
        int bottom=pop();
        push((bottom<top)?1:0);
      } break;
    case $eqOpcode: {
        int top=pop();
        int bottom=pop();
        push((bottom==top)?1:0);
      } break;
    case $orOpcode: {
        int top=pop();
        int bottom=pop();
        push((bottom|top)?1:0);
      } break;
    case $andOpcode: {
        int top=pop();
        int bottom=pop();
        push((bottom&top)?1:0);
      } break;

    case $dropOpcode:
      (void)pop();
      break;
    case $swapOpcode: {
        int v1=pop();
        int v2=pop();
        push(v1);
        push(v2);
      } break;
    case $dupOpcode: {
        int v=pop();
        push(v);
        push(v);
      } break;
    case $overOpcode: {
        int top=pop();
        int bottom=pop();
        push(bottom);
        push(top);
        push(bottom);
      } break;
    case $thirdOpcode: {
        int top=pop();
        int middle=pop();
        int bottom=pop();
        push(bottom);
        push(middle);
        push(top);
        push(bottom);
      } break;
    case $torOpcode: {
        int top=pop();
        pushr(top);
      } break;
    case $rfromOpcode: {
        // No error checking !
        push(returnStack[--rsp]);
      } break;

    case $addOpcode: {
        int top=pop();
        int bottom=pop();
        push(bottom+top);
      } break;

    case $zeroOpcode: {
        push(0);
      } break;
    case $oneOpcode: {
        push(1);
      } break;
    case $twoOpcode: {
        push(2);
      } break;
    case $threeOpcode: {
        push(3);
      } break;

    case $inputOpcode: {
//        printf("\\ndebug dsp=%d\\n", dsp);
        do {
          readInput();
        } while (parseInput(curInput)>0);
      } break;
    case $dotOpcode: {
        lp(" %d", pop());
      } break;
    default:
      if ((code[pc-1] & 0xFE00 ) == $elseOpcode) {
        pc+=code[pc-1] & 0x01FF;
        continue;
      }
      if ((code[pc-1] & 0xFE00 ) == $ifOpcode) {
        int a=pop();
        if (a == 0) {
          pc+=code[pc-1] & 0x01FF;
        }
        continue;
      }
      if ((code[pc-1] & 0xFE00 ) == $nifOpcode) {
        int a=pop();
        if (a != 0) {
          pc+=code[pc-1] & 0x01FF;
        }
        continue;
      }
      if ((code[pc-1] & 0xFE00 ) == $eqifOpcode) {
        int a=pop();
        int b=pop();
        if (a!=b) {
          pc+=code[pc-1] & 0x01FF;
        }
        continue;
      }
      if ((code[pc-1] & 0xFE00 ) == $neqifOpcode) {
        int a=pop();
        int b=pop();
        if (a==b) {
          pc+=code[pc-1] & 0x01FF;
        }
        continue;
      }
      if ((code[pc-1] & 0xFE00 ) == $jumpbackOpcode) {
        pc-=(code[pc-1] & 0x01FF)+1;
        continue;
      }
      if ((code[pc-1] & 0xFF00 ) == $addConstOpcode) {
        int d=pop() + (code[pc-1] & 0x00FF);
        push(d);
        continue;
      }
      if ((code[pc-1] & 0xFF00 ) == $subConstOpcode) {
        int d=pop() - (code[pc-1] & 0x00FF);
        push(d);
        continue;
      }
      if ((code[pc-1] & 0xFF00 ) == $wordOpcode) {
        printWord(pc-1);
        pc += code[pc-1] & 0x00FF;
        goto _retopcode;
      }
      if (code[pc-1] < $firstOpcode) {
        // Call routine
        pushr(pc);
        pc=code[pc-1];
        continue;
      }
    }
  }
}

int main(int argc, char **argv) {
  logfile=fopen("alice.log", "a+");
  lp("StoryFactory $version runtime. (c) 2003 Peter Wendrich\\nLoader... %d code cells  %d word cells  %d total cells  %d bytes\\n", $codeCount, $charCount, INIT_LENGTH, 2*INIT_LENGTH);
  memcpy(code, initCode, INIT_LENGTH*sizeof(unsigned short));
  run();
//  fgetc(stdin);
  if (logfile) {
    fclose(logfile);
  }
  return 0;
}
EOF
;

##########################################################################
#
#
# Export as HTML / JavaScript file
#
#
##########################################################################
print HTMLCODE <<EOF;
<html>
<head>
<title>$gameTitle</title>
</head>
<script language="JavaScript">
EOF

print HTMLCODE "var runCode;\nvar initCode=[";
$itemcount=0;
foreach $item (@code) {
  if ($itemcount > 0) {
    print HTMLCODE ",";
  }
  if (($itemcount%8) == 0) {
    print HTMLCODE "\n";
  }
  print HTMLCODE "$item";
  $itemcount++;
}
print HTMLCODE "];\n\n";

print HTMLCODE <<EOF;

var lines=1;
var virtScreen="";
function lp(str) {
  virtScreen = virtScreen + str;
  if (str == "\\n") {
    if (lines>=$scrollBack) {
      virtScreen = virtScreen.substring(virtScreen.indexOf("\\n")+1);
    } else {
      lines++;
    }
  }
}

var curCol=0;
var capitalLetter=1;
function printWord(index) {
  var lencode = runCode[index++];
  var len = lencode & 0x3F;
  var i;

  if (len==0) {
    curCol=0;
    lp("\\n");
    capitalLetter = 1;
  } else {
    if (len+curCol > ($screenWidth - 2) || (len+curCol > ($screenWidth - 4) && len > 1)) {
      curCol=0;
      lp("\\n");
    }
    if ((lencode & 0x40) == 0) {
      lp(" ");
      curCol++;
    }
    for(i=0;i<len;i++) {
      var c;
      if ((i&1)==0) {
        c=String.fromCharCode(runCode[index + (i>>1)]>>8);
      } else {
        c=String.fromCharCode(runCode[index + (i>>1)]&0xFF);
      }
      if (capitalLetter!=0) {
        c=c.toUpperCase();
      }
      lp(c);
      capitalLetter = 0;
    }
    capitalLetter=(lencode & 0x80)?1:0;
    curCol+=len;
  }
}

var pc, dsp, rsp;
var dStack=[0];
var rStack=[0];
function pop() {
  if (dsp>0) {
    return dStack[--dsp];
  } else {
    lp("\\n");
    lp("Error: stack underflow pc="+ pc);
    lp("\\n");
    return 0;
  }
}

function push(value) {
  dStack[dsp++]=value;
}

function pushr(value) {
  rStack[rsp++]=value;
}

function init() {
  document.f.input.focus();
  lp("StoryFactory $version runtime. (c) 2003 Peter Wendrich");
  lp("\\n");
  lp("Loader... " + initCode[3] + " code cells  " + (initCode.length-initCode[3]) + " word cells  " + initCode.length + " total cells  " + initCode.length * 2 + " bytes.");
  lp("\\n");
  runCode=initCode;
  pc=runCode[1];
  dsp=0;
  rsp=0;
}

function findWord(str) {
  var index = runCode[3];

  while (index < runCode[2]) {
    var wordlen = runCode[index]&0x3F;
    if (wordlen == str.length) {
      var found = 1;
      var i;

      for(i=0; i<wordlen; i++) {
        var c;

        if ((i&1)==0) {
          c=runCode[index + 1 + (i>>1)]>>8;
        } else {
          c=runCode[index + 1 + (i>>1)]&0xFF;
        }
        if (c>=65 && c<=90) c+=32;
        if (str.charCodeAt(i) != c) {
          found = 0;
          break;
        }        
      }
      if (found) return index;
    }
    index += (((runCode[index]&0x3F)+1)>>1)+1;
  }
  return 0;
}

function parse() {
  var input = document.f.input.value.toLowerCase();
  var inputlen = input.length;
  var i=0;
  var pp=runCode[2];

  if (inputlen == 0) {
    return -1;
  }

  lp(input);
  lp("\\n");

  while(i < inputlen) {
    var len;
    var word;

    len=0;
    while ((input.substring(i, i+1)) == " ") {
      i++;
    }
    if (i >= inputlen) break;
    while((i+len < inputlen) && ((input.substring(i+len, i+len+1)>="a" && input.substring(i+len, i+len+1)<="z") || input.substring(i+len, i+len+1)=="'")) {
      len++;
    }
    if (len == 0) {
      len=1;
    }

    word = findWord(input.substring(i, i+len));
    if (word == 0) {
      lp("'" + input.substring(i, i+len) + "' unrecognised.");
      lp("\\n");
      lp("> ");
      document.f.input.value = "";
      return -1;
    }
    runCode[pp] = word;
    i+=len;
    pp++;
  }

  document.f.input.value = "";
  runCode[pp]=$retOpcode;
  return 0;
}

var state=0;
function runVM() {
  curCol=0;
  capitalLetter=1;
  for(;;) {
    switch(runCode[pc++]) {
    case $litOpcode:
      push(runCode[pc++]);
      break;
    case $getOpcode:
      push(runCode[runCode[pc++]]);
      break;
    case $putOpcode:
      runCode[runCode[pc++]]=pop();
      break;
    case $retOpcode:
      if (rsp > 0) {
        pc = rStack[--rsp];
      } else {
        state=2;
        return;
      }
      break;
    case $fretOpcode:
      push(0);
      if (rsp > 0) {
        pc = rStack[--rsp];
      } else {
        state=2;
        return;
      }
      break;
    case $tretOpcode:
      push(1);
      if (rsp > 0) {
        pc = rStack[--rsp];
      } else {
        state=2;
        return;
      }
      break;
    case $dropfretOpcode:
      pop();
      push(0);
      if (rsp > 0) {
        pc = rStack[--rsp];
      } else {
        state=2;
        return;
      }
      break;
    case $droptretOpcode:
      pop();
      push(1);
      if (rsp > 0) {
        pc = rStack[--rsp];
      } else {
        state=2;
        return;
      }
      break;
    case $executeOpcode:
      pushr(pc);
      pc=pop();
      break;
    case $fetchOpcode: {
        var addr=pop();
        push(runCode[addr]);
      } break;
    case $storeOpcode: {
        var addr=pop();
        var v=pop();
        runCode[addr]=v;
      } break;
    case $dropOpcode:
      pop();
      break;
    case $swapOpcode: {
        var v1=pop();
        var v2=pop();
        push(v1);
        push(v2);
      } break;
    case $dupOpcode: {
        var v=pop();
        push(v);
        push(v);
      } break;
    case $overOpcode: {
        var top=pop();
        var bottom=pop();
        push(bottom);
        push(top);
        push(bottom);
      } break;
    case $thirdOpcode: {
        var top=pop();
        var middle=pop();
        var bottom=pop();
        push(bottom);
        push(middle);
        push(top);
        push(bottom);
      } break;
    case $eqOpcode: {
        var top=pop();
        var bottom=pop();
        push((bottom==top)?1:0);
      } break;
    case $bigOpcode: {
        var top=pop();
        var bottom=pop();
        push((bottom>top)?1:0);
      } break;
    case $smallOpcode: {
        var top=pop();
        var bottom=pop();
        push((bottom<top)?1:0);
      } break;
    case $orOpcode: {
        var top=pop();
        var bottom=pop();
        push((bottom|top)?1:0);
      } break;
    case $andOpcode: {
        var top=pop();
        var bottom=pop();
        push((bottom&top)?1:0);
      } break;

    case $torOpcode: {
        var top=pop();
        pushr(top);
      } break;
    case $rfromOpcode: {
        push(rStack[--rsp]);
      } break;

    case $addOpcode: {
        var top=pop();
        var bottom=pop();
        push(bottom+top);
      } break;

    case $zeroOpcode: {
        push(0);
      } break;
    case $oneOpcode: {
        push(1);
      } break;
    case $twoOpcode: {
        push(2);
      } break;
    case $threeOpcode: {
        push(3);
      } break;

    case $inputOpcode: {
        state=1;
        lp("\\n");
        lp("> ");
        return;
      } break;
    case $dotOpcode: {
        lp(" " + pop());
      } break;
    default:
      if ((runCode[pc-1] & 0xFE00 ) == $elseOpcode) {
        pc+=runCode[pc-1] & 0x01FF;
        continue;
      }
      if ((runCode[pc-1] & 0xFE00 ) == $ifOpcode) {
        var a=pop();
        if (a == 0) {
          pc+=runCode[pc-1] & 0x01FF;
        }
        continue;
      }
      if ((runCode[pc-1] & 0xFE00 ) == $nifOpcode) {
        var a=pop();
        if (a != 0) {
          pc+=runCode[pc-1] & 0x01FF;
        }
        continue;
      }
      if ((runCode[pc-1] & 0xFE00 ) == $eqifOpcode) {
        var a=pop();
        var b=pop();
        if (a!=b) {
          pc+=runCode[pc-1] & 0x01FF;
        }
        continue;
      }
      if ((runCode[pc-1] & 0xFE00 ) == $neqifOpcode) {
        var a=pop();
        var b=pop();
        if (a==b) {
          pc+=runCode[pc-1] & 0x01FF;
        }
        continue;
      }
      if ((runCode[pc-1] & 0xFE00 ) == $jumpbackOpcode) {
        pc-=(runCode[pc-1] & 0x01FF)+1;
        continue;
      }
      if ((runCode[pc-1] & 0xFF00 ) == $addConstOpcode) {
        var d=pop() + (runCode[pc-1] & 0x00FF);
        push(d);
        continue;
      }
      if ((runCode[pc-1] & 0xFF00 ) == $subConstOpcode) {
        var d=pop() - (runCode[pc-1] & 0x00FF);
        push(d);
        continue;
      }
      if ((runCode[pc-1] & 0xFF00 ) == $wordOpcode) {
        printWord(pc-1);
        if (rsp > 0) {
          pc = rStack[--rsp];
        } else {
          state=2;
          return;
        }
        continue;
      }
      if (runCode[pc-1] < $firstOpcode) {
        // Call routine
        pushr(pc);
        pc=runCode[pc-1];
        continue;
      }
    }
  }
}

function run() {
  switch(state) {
  case 0:
    init();
    runVM();
    break;
  case 1:
    if (parse()>=0) {
      runVM();
    }
    break;
  default:
    lp("Game is not running.");
    lp("\\n");
    break;
  }
  document.f.output.value = virtScreen;
}
</script>
<body onLoad="run();">
<table cellspacing="0" cellpadding="0" align="center" border="0"><tr><td align="center">
<form name="f" onSubmit="run(); return false;">
<textarea style="overflow:hidden" onkeypress="return false;" onfocus="this.blur();" name="output" cols="$screenWidth" rows="26" wrap="off"></textarea><BR>
<input type="text" name="input" size="$screenWidth">
</td></tr></table>
</form><HR><FONT SIZE="-1"><I>Automatically generated from '$scriptFileName' by StoryFactory $version.<BR>Copyright 2003 PW.Soft. Email to <A HREF="mailto:pwsoft\@syntiac.com">pwsoft\@syntiac.com</A> or on the web <A HREF="http://www.syntiac.com">http://www.syntiac.com</A></I></FONT>
</body>
</html>
EOF



##########################################################################
#
#
# Write MAP file
#
#
##########################################################################
my $line;
open(MAPFILE,  ">$outputMAPFileName" ) or die "Can't open $outputMAPFileName: $!";
print MAPFILE "start len  end\n";
print MAPFILE "--------------------------------------------\n\n";
foreach $line (@mapfileLines) {
  print MAPFILE $line;
}
close(MAPFILE);



##########################################################################
#
#
# Close open files
#
#
##########################################################################
close(CCODE);
close(HTMLCODE);

##########################################################################
#
#
# Game debugger
# Run game with this compiler for test purposes
#
#
##########################################################################
my @runcode;
my $curInput;
sub readInput {
  print "\n>";
  $curInput=<STDIN>;
  print $curInput;
}
sub findword {  
#  my $w = shift;
#  my $i = 0;
#  my $word;

#  foreach $word ( @words) {
#    if ($w eq $word) {
#      return $i;
#    }
#    $i++;
#  }
  
#  print "'$w' is not recognised.\n";
#  tr/[A-Z]/[a-z];
  my $w = shift;
  my $index = $words{"$w"};
#  print "$index\n";
  if (defined($index)) {
#    print "$index\n";
    return $index;
  } else {
    print "'$w' is not recognised.\n";
  }
  return -1;
}

sub parseInput {
  my @sinput;
  my $w;
  my $p = $runcode[2];
  
  @sinput = split(' ', $curInput);
  foreach $w (@sinput) {
    my $i = findword($w);
    if ($i == -1) {
      return -1;
    }
    $runcode[$p] = $i;
    $p++;
  }
  if ($p eq $runcode[2]) {
    return -1;
  } else {
    $runcode[$p] = $retOpcode;
    return 0;
  }
}

sub debugger {
  my $pc;
  my $opcode;
  my $curCol=0;
  my @dataStack=();
  my @returnStack=();
  print "*** Invoking debugging environment\n";

  @runcode=@code;
  print "StoryFactory $version debuging run.\nLoader... $#code code cells  ? bytes\n";

  $pc=$runcode[1];
  print "pc = '$pc'\n";

# test
#  $pc=0;

  while(1) {
    if ($pc > $firstOpcode) {
      print "\nInvalid value for program counter. PC=$pc\n";
      last
    }
    $opcode = $runcode[$pc];
    $pc++;
#    print $opcode;
    if ($opcode == $litOpcode) {
      push(@dataStack, $runcode[$pc]);
      $pc++;
      next;
    }
    if ($opcode == $getOpcode) {
      push(@dataStack, $runcode[$runcode[$pc]]);
      $pc++;
      next;
    }
    if ($opcode == $putOpcode) {
      $runcode[$runcode[$pc]]=pop(@dataStack);
      $pc++;
      next;
    }
    if ($opcode == $retOpcode) {
      if (scalar(@returnStack) > 0) {
        $pc = pop(@returnStack);
      } else {
        return;
      }
      next;
    }
    if ($opcode == $fretOpcode) {
      push(@dataStack, 0);
      if (scalar(@returnStack) > 0) {
        $pc = pop(@returnStack);
      } else {
        return;
      }
      next;
    }
    if ($opcode == $tretOpcode) {
      push(@dataStack, 1);
      if (scalar(@returnStack) > 0) {
        $pc = pop(@returnStack);
      } else {
        return;
      }
      next;
    }
    if ($opcode == $dropfretOpcode) {
      pop(@dataStack);
      push(@dataStack, 0);
      if (scalar(@returnStack) > 0) {
        $pc = pop(@returnStack);
      } else {
        return;
      }
      next;
    }
    if ($opcode == $droptretOpcode) {
      pop(@dataStack);
      push(@dataStack, 1);
      if (scalar(@returnStack) > 0) {
        $pc = pop(@returnStack);
      } else {
        return;
      }
      next;
    }
    if ($opcode == $executeOpcode) {
      push(@returnStack, $pc);
      $pc=pop(@dataStack);
      next;
    }
    if ($opcode == $fetchOpcode) {
      my $addr=pop(@dataStack);
      push(@dataStack, $runcode[$addr]);
      next;
    }
    if ($opcode == $storeOpcode) {      
      my $addr=pop(@dataStack);
      my $v=pop(@dataStack);
      $runcode[$addr] = $v;
      next;
    }
    if ($opcode == $dropOpcode) {
      pop(@dataStack);
      next;
    }
    if ($opcode == $swapOpcode) {
      my $v1=pop(@dataStack);
      my $v2=pop(@dataStack);
      push(@dataStack, $v1);
      push(@dataStack, $v2);
      next;
    }
    if ($opcode == $dupOpcode) {
      my $v=pop(@dataStack);
      push(@dataStack, $v);
      push(@dataStack, $v);
      next;
    }
    if ($opcode == $overOpcode) {
      my $top=pop(@dataStack);
      my $bottom=pop(@dataStack);
      push(@dataStack, $bottom);
      push(@dataStack, $top);
      push(@dataStack, $bottom);
      next;
    }
    if ($opcode == $thirdOpcode) {
      my $top=pop(@dataStack);
      my $middle=pop(@dataStack);
      my $bottom=pop(@dataStack);
      push(@dataStack, $bottom);
      push(@dataStack, $middle);
      push(@dataStack, $top);
      push(@dataStack, $bottom);
      next;
    }

    if ($opcode == $bigOpcode) {
      my $top=pop(@dataStack);
      my $bottom=pop(@dataStack);
      push(@dataStack, ($bottom>$top)?1:0);
      next;
    }
    if ($opcode == $smallOpcode) {
      my $top=pop(@dataStack);
      my $bottom=pop(@dataStack);
      push(@dataStack, ($bottom<$top)?1:0);
      next;
    }
    if ($opcode == $eqOpcode) {
      my $top=pop(@dataStack);
      my $bottom=pop(@dataStack);
      push(@dataStack, ($bottom==$top)?1:0);
      next;
    }
    if ($opcode == $orOpcode) {
      my $top=pop(@dataStack);
      my $bottom=pop(@dataStack);
      push(@dataStack, ($bottom or $top)?1:0);
      next;
    }
    if ($opcode == $andOpcode) {
      my $top=pop(@dataStack);
      my $bottom=pop(@dataStack);
      push(@dataStack, ($bottom and $top)?1:0);
      next;
    }

    if ($opcode ==  $torOpcode) {
      push(@returnStack, pop(@dataStack));
      next;
    }
    if ($opcode ==  $rfromOpcode) {
      push(@dataStack, pop(@returnStack));
      next;
    }

    if ($opcode ==  $addOpcode) {
      my $top=pop(@dataStack);
      my $bottom=pop(@dataStack);
      push(@dataStack, $bottom + $top);
      next;
    }

    if ($opcode ==  $zeroOpcode) {
      push(@dataStack, 0);
      next;
    }
    if ($opcode ==  $oneOpcode) {
      push(@dataStack, 1);
      next;
    }
    if ($opcode ==  $twoOpcode) {
      push(@dataStack, 2);
      next;
    }
    if ($opcode ==  $threeOpcode) {
      push(@dataStack, 3);
      next;
    }

    if ($opcode == $inputOpcode) {
#      printf("\\ndebug dsp=%d\\n", dsp);
      # dump dataStack
      my $v;
      foreach $v (@dataStack) {
        print $v . " ";
      }
      do {
        readInput();
      } while (parseInput($curInput)<0);
      $curCol = 0;
      next;
    }
    if ($opcode == $dotOpcode) {
      print sprintf(" %d\n", pop(@dataStack));
      next;
    }
    if (($opcode & 0xFE00 ) == $elseOpcode) {
      $pc+=$runcode[$pc-1] & 0x01FF;
      next;
    }
    if (($opcode & 0xFE00) == $ifOpcode) {
      my $a=pop(@dataStack);
      if ($a == 0) {
        $pc+=$runcode[$pc-1] & 0x01FF;
      }
      next;
    }
    if (($opcode & 0xFE00) == $nifOpcode) {
      my $a=pop(@dataStack);
      if ($a != 0) {
        $pc+=$runcode[$pc-1] & 0x01FF;
      }
      next;
    }
    if (($opcode & 0xFE00) == $eqifOpcode) {
      my $a=pop(@dataStack);
      my $b=pop(@dataStack);
      if ($a != $b) {
        $pc+=$runcode[$pc-1] & 0x01FF;
      }
      next;
    }
    if (($opcode & 0xFE00) == $neqifOpcode) {
      my $a=pop(@dataStack);
      my $b=pop(@dataStack);
      if ($a == $b) {
        $pc+=$runcode[$pc-1] & 0x01FF;
      }
      next;
    }
    if (($opcode & 0xFE00 ) == $jumpbackOpcode) {
      $pc-=($runcode[$pc-1] & 0x01FF)+1;
      next;
    }
    if (($opcode & 0xFF00 ) == $addConstOpcode) {
      my $v=pop(@dataStack) + ($opcode & 0x00FF);
      push(@dataStack, $v);
      next;
    }
    if (($opcode & 0xFF00 ) == $subConstOpcode) {
      my $v=pop(@dataStack) - ($opcode & 0x00FF);
      push(@dataStack, $v);
      next;
    }
    if (($opcode & 0xFF00 ) == $wordOpcode) {
      my $lencode = $opcode-$wordOpcode;
      my $len = $lencode & 0x3F;

      if ($len == 0) {
        print "\n";
        $curCol = 0;
      } else {
        if ($len+$curCol > ($screenWidth - 2) || ($len+$curCol > ($screenWidth - 4) && $len > 1)) {
          print "\n";
          $curCol = 0;
        }
        if (($lencode & 0x40) == 0) {
          print " ";
          $curCol++;
        }
        my $i;
        for($i=0;$i<$len;$i++) {
          my $c;
          if (($i&1)==0) {
            $c=$runcode[$pc + $i/2]>>8;
          } else {
            $c=$runcode[$pc + $i/2]&0xFF;
          }
#          if (capitalLetter && c>='a' && c<='z') {
#            lp("%c", c+('A'-'a'));
#          } else {
#            lp("%c", c);
#          }
#          capitalLetter = 0;
           print sprintf("%c", $c);
        }
#        capitalLetter=(lencode & 0x80)?1:0;
        $curCol+=$len;
      }

      if (scalar(@returnStack) > 0) {
        $pc = pop(@returnStack);
      } else {
        return;
      }
      next;
    }
    if ($opcode < $firstOpcode) {
      # Call
      push(@returnStack, $pc);
      $pc=$opcode;
      next;
    }
    print "\nUnknown opcode $opcode\n";
    die;
  }
}

if (exists $ARGV[1]) {
  debugger();
}

