{*********************************************************}
{                                                         }
{    Calmira System Library 3.1                           }
{    by Li-Hsin Huang & Erwin Dokter                      }
{    released into the public domain january 2001         }
{                                                         }
{*********************************************************}

unit MiscUtil;

{ Some useful Delphi and Windows routines }

interface

uses Classes, SysUtils, Forms, WinTypes, IniFiles, Menus, Buttons,
  StdCtrls, Dialogs, ExtCtrls, Graphics, TabNotBk, Controls;

const
  MsgDialogSounds: Boolean = False;
  MaxHistorySize: Integer = 24;

function Min(a, b: Integer): Integer;
function Max(a, b: Integer): Integer;
{ Returns the smaller and larger of two values respectively }

function Range(n, lower, upper: Integer): Integer;
{ Constrains n to a lower and upper limit }

function Sign(x: Integer): Integer;
{ Returns 1 if x > 0, -1 if x < 0 and 0 if x = 0 }

procedure Border3d(Canvas: TCanvas; Width, Height: Integer);
{ Draws a raised 3D border on a canvas, typically used in an
  OnPaint method of a TForm }

procedure RecessBevel(Canvas: TCanvas; R: TRect);
{ Draws a lowered 3D frame on a canvas, alternative to using
  bevels }

procedure ErrorMsg(const msg: string);
{ Displays a message dialog box indicating an error }

procedure ErrorMsgRes(Ident: Word);

procedure ErrorMsgResFmt(Ident: Word; const Args : array of const);

procedure PlaySound(const filename: TFilename);
{ Plays the specified WAV file as a sound effect.  If the filename
  is <None>, nothing is played }

function Intersects(const R, S: TRect): Boolean;
{ Returns True if the two rectangles intersect }

function NormalizeRect(p, q: TPoint): TRect;
{ Returns a rectangle defined by any two points.  When dragging a
  selection box with a mouse, the fixed corner and the moving
  corner may not always be top left and bottom right respectively.
  This function creates a valid TRect out of them }

function TimeStampToDate(FileDate: Longint): TDateTime;
{ Converts a DOS timestamp to TDateTime.  If the timestamp is invalid
  (some programs use invalid stamps as markers), the current date
  is returned instead of raising EConvertError }

function GetRadioIndex(const R: array of TRadioButton): Integer;
procedure SetRadioIndex(const R: array of TRadioButton; index: Integer);
function GetMenuCheck(const M: array of TMenuItem): Integer;
procedure SetMenuCheck(const M: array of TMenuItem; index: Integer);
function GetButtonState(const B: array of TSpeedButton): Integer; { 3.0 }
procedure SetButtonState(const B: array of TSpeedButton; index: Integer); { 3.0 }
{ These routines are useful for setting and querying the state of
  several controls.  Use them to simulate arrays and as an alternative
  to TRadioGroup. }

{ 3.1 }
function GetFlag(const Data: Pointer; Flag: Longint): Boolean;
function SetFlag(const Data: Pointer; Flag: Longint): Pointer;
function ClearFlag(const Data: Pointer; Flag: Longint): Pointer;
function ToggleFlag(const Data: Pointer; Flag: Longint): Pointer;
{ These routines let you directly manipulate bits in a data property,
  which many Delphi components, like grids and outlines have.  They
  usually hold a pointer to a data object in memory, but is often
  (ab)used to store flags instead.  Declare your flags using the
  represented bit values; 1, 2, 4, 8, 16, 32, etc... }

procedure RefreshCursor;
{ Updates the cursor image when you have changed the Cursor or DragCursor
  property of a control }

procedure UpdateScreen;
{ Forces all vivible windows to repaint }

procedure ShowHourGlass;
{ Displays the hourglass cursor immediately }

procedure ShowArrow;
{ Displays the standard arrow }

function AddHistory(Combo: TComboBox): Boolean;
{ Adds a combo box's Text string to its listbox, but only if the
  string is not empty and not already present in the list.  The item is
  inserted at the top of the list, and if there are more than 24 items,
  the bottom one is removed.  Returns true if the list is modified }

procedure AssignHistoryText(Combo: TCombobox; const NewText: string);

function MsgDialog(const Msg: string; AType: TMsgDlgType;
  AButtons: TMsgDlgButtons; HelpCtx: Longint): Word;
{ Calls the MessageDialog function, but also plays a suitable sound
  effect from the Control Panel settings.  The MsgDialogSounds variable
  enables the sounds }

function MsgDialogRes(Ident: Word; AType: TMsgDlgType;
  AButtons: TMsgDlgButtons; HelpCtx: Longint): Word;

function MsgDialogResFmt(Ident: Word; const Args: array of const;
  AType: TMsgDlgType; AButtons: TMsgDlgButtons; HelpCtx: Longint): Word;

function ShowModalDialog(FormClass: TFormClass): TModalResult;
{ A very simple way of displaying a dynamic modal form -- just pass the
  form's class name e.g. TForm1, and an instance will be created,
  shown as a modal dialog and then destroyed. }

function InitBitmap(ABitmap: TBitmap; AWidth, AHeight: Integer;
  Color: TColor): TBitmap;
{ Initialises the bitmap's dimensions and fills it with the chosen colour }

procedure ShrinkIcon(H: HIcon; Glyph: TBitmap; BGColor: TColor); { 3.0 }
{ Shrinks a 32 x 32 icon down to a 16 x 16 bitmap }

function CreateBrushPattern(bgColor, fgColor: TColor): TBitmap; { 3.1 }
{ Creates an 8 by 8 mesh pattern to use in brushes }

procedure Sleep(MSec: Longint); { 3.1 }
{ Suspend the application for a specified number of milliseconds.
  Also works when Application is nil. }

function GetTimerCount: Longint;
{ Acurate alternative to GetTickCount }

procedure CopyStringsToClipboard(strings: TStrings);

function ShortTimeToStr(Time: TDateTime): string;

procedure FreePageHandles(Notebook: TNotebook);

function GetMinPosition(Wnd: HWND): TPoint;

procedure MoveDesktopIcon(Wnd: HWND; pt: TPoint);

procedure GetHeaderDivisions(H: THeader; A: array of PInteger);

procedure EnableControlList(C: array of TControl; Enable: Boolean);

procedure Beep;

const
  RepaintBeforeHourglass: Integer = 1;
  DarkIconStretch: Boolean = True;
  HQIconStretch: Boolean = False; { 3.1 }

implementation

uses WinProcs, MMSystem, ShellAPI, Strings, FileCtrl, Clipbrd, ToolHelp,
  Messages;

function Min(a, b: Integer): Integer; assembler;
asm
  MOV	AX, a
  CMP	AX, b
  JLE	@@1
  MOV	AX, b
@@1:
end;

function Max(a, b: Integer): Integer; assembler;
asm
  MOV	AX, a
  CMP	AX, b
  JGE	@@1
  MOV	AX, b
@@1:
end;

function Range(n, lower, upper: Integer): Integer; assembler;
asm
  MOV  AX, n
  CMP  AX, lower
  JGE  @@1
  MOV  AX, lower
  JMP  @finish
@@1:
  CMP  AX, upper
  JLE  @finish
  MOV  AX, upper
  JMP  @finish
@@2:
  MOV  AX, lower
@finish:
end;

function Sign(x: Integer): Integer; assembler;
asm
  MOV  AX, X
  CMP  AX, 0
  JL   @@1
  JG   @@2
  XOR  AX, AX
  JMP  @finish
@@1:
  MOV  AX, -1
  JMP  @finish
@@2:
  MOV  AX, 1
@finish:
end;

procedure Border3d(Canvas: TCanvas; Width, Height: Integer);
begin
  with Canvas do
  begin
    Pen.Color := clBtnHighLight;
    MoveTo(0, Height);
    LineTo(0, 0);
    LineTo(Width, 0);
    Pen.Color := clBtnShadow;
    LineTo(Width, Height);
    LineTo(0, Height);
  end;
end;

procedure RecessBevel(Canvas: TCanvas; R: TRect);
begin
  Frame3D(Canvas, R, clBtnShadow, clBtnHighlight, 1);
end;

procedure ErrorMsg(const msg: string);
begin
  MsgDialog(msg, mtError, [mbOK], 0);
end;

procedure ErrorMsgRes(Ident: Word);
begin
  MsgDialog(LoadStr(Ident), mtError, [mbOK], 0);
end;

procedure ErrorMsgResFmt(Ident: Word; const Args : array of const);
begin
  MsgDialog(FmtLoadStr(Ident, Args), mtError, [mbOK], 0);
end;

procedure PlaySound(const filename: TFilename);
var
  s: TFilename;
begin
  if CompareText(filename, '<None>') <> 0 then
    SndPlaySound(StrPCopy(@s, filename), SND_ASYNC or SND_NODEFAULT);
end;

function Intersects(const R, S: TRect): Boolean;
var
  dummy: TRect;
begin
  Result := IntersectRect(dummy, R, S) <> 0;
end;

function NormalizeRect(p, q: TPoint): TRect; assembler;
asm
  MOV  AX, p.x
  MOV  BX, p.y
  MOV  CX, q.x
  MOV  DX, q.y
  CMP  AX, CX
  JLE  @@1
  XCHG AX, CX
@@1:
  CMP  BX, DX
  JLE  @@2
  XCHG BX, DX
@@2:
  LES  DI, @Result
  MOV  TRect(ES:[DI]).Left, AX
  MOV  TRect(ES:[DI]).Top, BX
  MOV  TRect(ES:[DI]).Right, CX
  MOV  TRect(ES:[DI]).Bottom, DX
end;

function TimeStampToDate(FileDate: Longint): TDateTime;
begin
  try
    Result := FileDateToDateTime(FileDate)
  except
    on EConvertError do Result := Date;
  end;
end;

function GetRadioIndex(const R: array of TRadioButton): Integer;
begin
  for Result := 0 to High(R) do
    if R[Result].Checked then Exit;
  Result := 0;
end;

procedure SetRadioIndex(const R: array of TRadioBUtton; index: Integer);
var
  i: Integer;
begin
  for i := 0 to High(R) do
    R[i].Checked := i = index;
end;

function GetMenuCheck(const M: array of TMenuItem): Integer;
begin
  for Result := 0 to High(M) do
    if M[Result].Checked then Exit;
  Result := 0;
end;

procedure SetMenuCheck(const M: array of TMenuItem; index: Integer);
var
  i: Integer;
begin
  for i := 0 to High(M) do
    M[i].Checked := i = index;
end;

function GetButtonState(const B: array of TSpeedButton): Integer;
begin
  for Result := 0 to High(B) do
    if B[Result].Down then Exit;
  Result := 0;
end;

procedure SetButtonState(const B: array of TSpeedButton; index: Integer);
var
  i: Integer;
begin
  for i := 0 to High(B) do
    B[i].Down := i = index;
end;

function GetFlag(const Data: Pointer; Flag: Longint): Boolean;
begin
  Result := Longint(Data) and Flag <> 0;
end;

function SetFlag(const Data: Pointer; Flag: Longint): Pointer;
begin
  Result := Pointer(Longint(Data) or Flag);
end;

function ClearFlag(const Data: Pointer; Flag: Longint): Pointer;
begin
  Result := Pointer(Longint(Data) and not Flag);
end;

function ToggleFlag(const Data: Pointer; Flag: Longint): Pointer;
begin
  Result := Pointer(Longint(Data) xor Flag);
end;

procedure RefreshCursor;
var
  p: TPoint;
begin
  GetCursorPos(p);
  SetCursorPos(p.x, p.y);
end;

function DoUpdateWindow(Wnd: HWND; lParam: Longint): Bool; export;
begin
  UpdateWindow(Wnd);
  Result := True;
end;

procedure UpdateScreen;
begin
  case RepaintBeforeHourglass of
    1: EnumTaskWindows(GetCurrentTask, @DoUpdateWindow, 0);
    2: EnumWindows(@DoUpdateWindow, 0);
  end;
end;

procedure ShowHourGlass;
begin
  UpdateScreen;
  SetCursor(LoadCursor(0, IDC_WAIT));
end;

procedure ShowArrow;
begin
  SetCursor(LoadCursor(0, IDC_ARROW));
end;

function AddHistory(Combo: TComboBox): Boolean;
var
  i: Integer;
  s: string;
begin
  Result := False;
  with Combo, Combo.Items do
    if Text <> '' then
    begin
      i := IndexOf(Text);
      if i = -1 then
      begin
        Result := True;
        Insert(0, Text)
      end
      else if i > 0 then
      begin
        { Same as Exchange(i, 0), but Exchange can clear the
          Text property if the text is the string at i }
        Result := True;
        s := Text;
        Delete(i);
        Insert(0, s);
        Text := s;
      end;
      while (Count > 0) and (Count > MaxHistorySize) do
      begin
        Result := True;
        Delete(Count - 1);
      end;
    end;
end;

procedure AssignHistoryText(Combo: TCombobox; const NewText: string);
begin
  with Combo do
  begin
    if NewText > '' then Text := NewText;
    if (Text = '') and (Items.Count >= 1) then Text := Items[0];
  end;
end;

function MsgDialog(const Msg: string; AType: TMsgDlgType;
  AButtons: TMsgDlgButtons; HelpCtx: Longint): Word;
const
  Sound: array[TMsgDlgType] of Word =
    (MB_ICONEXCLAMATION, MB_ICONHAND, MB_OK, MB_ICONQUESTION, 0);
begin
  if MsgDialogSounds and (AType < mtCustom) then MessageBeep(Sound[AType]);
  Result := MessageDlg(Msg, AType, AButtons, HelpCtx);
end;

function MsgDialogRes(Ident: Word; AType: TMsgDlgType;
  AButtons: TMsgDlgButtons; HelpCtx: Longint): Word;
begin
  Result := MsgDialog(LoadStr(Ident), AType, AButtons, HelpCtx);
end;

function MsgDialogResFmt(Ident: Word; const Args: array of const;
  AType: TMsgDlgType; AButtons: TMsgDlgButtons; HelpCtx: Longint): Word;
begin
  Result := MsgDialog(FmtLoadStr(Ident, Args), AType, AButtons, HelpCtx);
end;

function ShowModalDialog(FormClass: TFormClass): TModalResult;
begin
  ShowHourGlass;
  with FormClass.Create(Application) do
    try
      Result := ShowModal;
    finally
      Free;
    end;
end;

function InitBitmap(ABitmap: TBitmap; AWidth, AHeight: Integer;
  Color: TColor): TBitmap;
begin
  { initializes a bitmap with width, height and background colour }
  with ABitmap do
  begin
    Width := AWidth;
    Height := AHeight;
    Canvas.Brush.Color := Color;
    Canvas.FillRect(Rect(0, 0, Width, Height));
  end;
  Result := ABitmap;
end;

procedure ShrinkIcon(H: HIcon; Glyph: TBitmap; BGColor: TColor);
const
  DarkStretch: array[Boolean] of Integer =
    (STRETCH_DELETESCANS, STRETCH_ANDSCANS);
var
  bmp: TBitmap;
  a, i, j: Integer;
  src, dest: HDC;
  OldStretch: Integer;
  r, g, b: array [1..4] of Byte;
  p: array[1..4] of TColorRef;
  rdest, gdest, bdest: Byte;
  pdest: TColorRef;
begin
  bmp := InitBitmap(TBitmap.Create, 32, 32, BGColor);
  DrawIcon(bmp.Canvas.Handle, 0, 0, H);
  try
    with Glyph do
    begin
      Width := 16;
      Height := 16;
      src := bmp.Canvas.Handle;
      dest := Canvas.Handle;
      if HQIconStretch then
      begin
        { Routine using averaging/mixing algorithm, producing high
          quality 16x16 icons -- use only with 256 or more colors }
        for i := 0 to 15 do
          for j := 0 to 15 do
          begin
            p[1] := GetPixel(src, i shl 1, j shl 1);
            p[2] := GetPixel(src, (i shl 1) + 1, j shl 1);
            p[3] := GetPixel(src, i shl 1, (j shl 1) + 1);
            p[4] := GetPixel(src, (i shl 1) + 1, (j shl 1) + 1);
            for a := 1 to 4 do
            begin
              r[a] := GetRValue(p[a]);
              g[a] := GetGValue(p[a]);
              b[a] := GetBValue(p[a]);
            end;
            rdest := (r[1] + r[2] + r[3] + r[4]) div 4;
            gdest := (g[1] + g[2] + g[3] + g[4]) div 4;
            bdest := (b[1] + b[2] + b[3] + b[4]) div 4;
            pdest := RGB(rdest, gdest, bdest);
            SetPixel(dest, i, j, pdest);
          end;
      end
      else
      begin
        { Routine using Window's own StretchBlt function, which
          gives rather poor results, IMHO... }
        OldStretch := SetStretchBltMode(dest, DarkStretch[DarkIconStretch]);
        StretchBlt(dest, 0, 0, 16, 16, src, 0, 0, 32, 32, SRCCOPY);
        for i := 0 to 15 do
          for j := 0 to 15 do
           if GetPixel(dest, i, j) = clSilver then
             SetPixel(dest, i, j, GetPixel(src, i shl 1, j shl 1));
        SetStretchBltMode(dest, OldStretch);
      end;
      { Set the bottom left pixel to background color }
      Canvas.Pixels[0, 15] := BGColor;
    end;
  finally
    bmp.Free;
  end;
end;

function CreateBrushPattern(bgColor, fgColor: TColor): TBitmap;
var
  X, Y: Integer;
begin
  Result := TBitmap.Create;
  Result.Width := 8;
  Result.Height := 8;
  with Result.Canvas do
  begin
    Brush.Style := bsSolid;
    Brush.Color := bgColor;
    FillRect(Rect(0, 0, Result.Width, Result.Height));
    for Y := 0 to 7 do
      for X := 0 to 7 do
        if (Y mod 2) = (X mod 2) then  { toggles between even/odd pixles }
          Pixels[X, Y] := fgColor;     { on even/odd rows }
  end;
end;

procedure Sleep(MSec: Longint);
var
  Present, Future: Longint;
  Msg: TMsg;
begin
  Present := GetTimerCount;
  Future := Present + MSec;
  if Application = nil then
  repeat
    if PeekMessage(Msg, 0, 0, 0, PM_REMOVE) then
    begin
      TranslateMessage(Msg);
      DispatchMessage(Msg);
    end;
  until GetTimerCount >= Future
  else
  repeat
    Application.ProcessMessages;
  until GetTimerCount >= Future;
end;

function GetTimerCount: Longint;
var
  TI: TTimerInfo;
begin
  TI.dwSize := SizeOf(TI);
  TimerCount(@TI);
  Result := TI.dwmsThisVM;
end;

procedure CopyStringsToClipboard(strings: TStrings);
var
  P: PChar;
begin
  P := strings.GetText;
  Clipboard.SetTextBuf(P);
  StrDispose(P);
end;

function ShortTimeToStr(Time: TDateTime): string;
begin
  DateTimeToString(Result, ShortTimeFormat, Time);
end;

type
  TSurfaceWin = class(TWinControl);

procedure FreePageHandles(Notebook: TNotebook);
begin
  with Notebook do
  begin
    LockWindowUpdate(Handle);
    try
      TSurfaceWin(Pages.Objects[PageIndex]).DestroyHandle;
    finally
      LockWindowUpdate(0);
    end;
  end;
end;

function GetMinPosition(Wnd: HWND): TPoint;
var
  place: TWindowPlacement;
begin
  { Returns minimized icon coordinates.  Those which haven't been minimized
    before can have -1 values, in which case Windows picks a suitable
    position when required }
  place.Length := sizeof(place);
  GetWindowPlacement(Wnd, @place);
  Result := place.ptMinPosition;
end;

procedure MoveDesktopIcon(Wnd: HWND; pt: TPoint);
var
  place: TWindowPlacement;
begin
  { Repositions a window's icon.  If the window is minimized,
    it must be hidden before being moved to ensure that the
    desktop background is updated }
  place.Length := sizeof(place);
  GetWindowPlacement(Wnd, @place);
  with place.ptMinPosition do
    if (x = pt.x) and (y = pt.y) then Exit;
  place.ptMinPosition := pt;
  place.Flags := place.Flags or WPF_SETMINPOSITION;
  if IsIconic(Wnd) then
  begin
    ShowWindow(Wnd, SW_HIDE);
    place.ShowCmd := SW_SHOWMINNOACTIVE;
  end
  else place.ShowCmd := SW_SHOWNA;
  SetWindowPlacement(Wnd, @place);
end;

procedure GetHeaderDivisions(H: THeader; A: array of PInteger);
var
  i, w: Integer;
begin
  with H do
  begin
    i := 0;
    w := 0;
    while (i <= High(A)) and (i < Sections.Count) do
    begin
      Inc(w, SectionWidth[i]);
      if A[i] <> nil then A[i]^ := w;
      Inc(i);
    end;
  end;
end;

procedure EnableControlList(C: array of TControl; Enable: Boolean);
var
  i: Integer;
begin
  for i := 0 to High(C) do C[i].Enabled := Enable;
end;

procedure Beep;
begin
  MessageBeep(0);
end;

end.
