/******************************************************************************
 * Copyright (c) 2006-2008 Palm,Inc. or its subsidiaries.
 * All rights reserved.
 *****************************************************************************/

/**
 @ingroup CDMAStatus
 */
 
/**
 *  @brief 	This sample code shows how to retrieve radio connection status, operator name, and 
 * 			voicemail number on CDMA phones.
 * 			It also shows how to obtain the BaseStationIDInfo (CellId) information on 
 *			Centro CDMA devices.
 *
 *
 *
 *
 *  @version 1.0	- Initial Revision (03/25/08)
 *  
 *
 *  @file CDMAStatus.c
 */

 
#include <PalmOS.h>
#include <PalmOSGlue.h>
#include <HsNav.h>      // Include for HsExt.h
#include <HsExt.h>      // Include for HsGetPhoneLibrary()
#include <HsPhone.h>    // Include for Phone library API
#include <HsPhoneNetwork.h>
#include <HsNavCommon.h>
#include <palmOneResources.h>

#include "Common.h"
#include "CDMAStatus.h"
#include "CDMAStatus_Rsc.h"


/*********************************************************************
 * Global variables
 *********************************************************************/

UInt16 gLibRef = 0;


#define MAX_STR 80

void * GetObjectPtr(UInt16 objectID);
void MainFormInit(FormType *frmP, Boolean redraw);

/*********************************************************************
 * Internal Constants
 *********************************************************************/

/* Define the minimum OS version we support */
#define ourMinVersion    sysMakeROMVersion(3,0,0,sysROMStageDevelopment,0)
#define kPalmOS20Version sysMakeROMVersion(2,0,0,sysROMStageDevelopment,0)

/*********************************************************************
 * Internal Functions
 *********************************************************************/

/***********************************************************************
 *
 * FUNCTION:    PrvCommonInitGauge
 *
 * DESCRIPTION:	Call this function on form open to initialize the radio and
 *				 battery status gadgets.
 *			.
 *
 * PARAMETERS:  frm
 *
 * RETURNED:    nothing
 *
 *
 * REVISION HISTORY:
 *			Name	Date		Description
 *			----	----		-----------
 *			vmk		1/30/02		Created -- based on David Matiskella's code
 *								 in Soho.
 *
 ***********************************************************************/
static void PrvCommonInitGauge( FormPtr frm )
{
  UInt32 hsStatusVersion;
  
  if (FtrGet (hsFtrCreator, hsFtrIDStatusGadgetRev, &hsStatusVersion) == 0)
	  {
		HsStatusSetGadgetType (frm, NetworkBatteryGadget, hsStatusGadgetBattery);
		HsStatusSetGadgetType (frm, NetworkSignalGadget, hsStatusGadgetSignal);
	  }
} // CommonInitGauge

 
 

/*
 * FUNCTION: GetObjectPtr
 *
 * DESCRIPTION:
 *
 * This routine returns a pointer to an object in the current form.
 *
 * PARAMETERS:
 *
 * formId
 *     id of the form to display
 *
 * RETURNED:
 *     address of object as a void pointer
 */

static void * GetObjectPtr(UInt16 objectID)
{
	FormType * frmP;

	frmP = FrmGetActiveForm();
	return FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, objectID));
}



/**
 * @brief MainFormInit
 *	     This function initializes the main form 
 *
 *
 * @param frmP   IN:  	Pointer to the main form
 * @param redraw IN:   redraw the main form ?
 * @retval		void	None
 *
 *
 * @version This function was added in version 1.0
 *
 **/
void MainFormInit(FormType *frmP, Boolean redraw)
{
	Err err = errNone;
	
	Int16  bufSize       = 0;
	UInt16 signalQuality = 0;
	UInt32 deviceID      = 0;
	
	UInt16 base_id = 0; /* base station id */
	UInt16 sid = 0; /* station id */
	UInt16 nid = 0; /* network id */
	
	
	Char    tmp[MAX_STR];
	CharPtr tmpPtr = tmp;
	CharPtr operatorName = NULL;
	CharPtr voicemailNum = NULL;
    PhnPowerType power = phnPowerOff;
    PhnRadioStateType radioState;
  
	FieldType *ConnectionField;
	FieldType *OperatorField;
	FieldType *VoicemailField;
	FieldType *RadioField;
	FieldType *SigQualityField;
	
	FieldType *NIDField;
	FieldType *SIDField;
	FieldType *CellIdField;		//Base station id
	
	ConnectionField = (FieldType *) GetObjectPtr(MainConnectionField);
	OperatorField = (FieldType *) GetObjectPtr(MainOperatorField);
	VoicemailField = (FieldType *) GetObjectPtr(MainVoicemailField);
	RadioField = (FieldType *) GetObjectPtr(MainRadioField);
	SigQualityField = (FieldType *) GetObjectPtr(MainQualityField);
		
	NIDField = (FieldType *) GetObjectPtr(MainNIDField);	
	CellIdField = (FieldType *) GetObjectPtr(MainCellIDField);
	SIDField = (FieldType *) GetObjectPtr(MainSIDField);
	
    power = PhnLibModulePowered(gLibRef);
    PhnLibGetRadioState(gLibRef, &radioState);
    
    if (power == phnPowerOff)
    {
        DbgMessage ("Radio Off \n");
        SetFieldTextFromStr(RadioField, "OFF", redraw);
        
    }
    else
    {
        DbgMessage ("Radio On \n");
        SetFieldTextFromStr(RadioField, "ON", redraw);
        
    }
    
    /* Checks if the phone found any network cellular service. */
	if (PhnLibRegistered(gLibRef))
	{
		DbgMessage ("connectionfiled OK \n");
		SetFieldTextFromStr(ConnectionField, "OK", redraw);
		
				
		err = PhnLibCurrentOperator(gLibRef, NULL, &operatorName, NULL);
		if (!err)
		{
			DbgMessage ("operatorName OK \n");
			SetFieldTextFromStr(OperatorField, operatorName, redraw);
			
		}
		else // Note: See HsPhoneErrors.h for complete list of telephony error codes
		{	
			DbgMessage ("operatorName Notavailable \n");
			SetFieldTextFromStr(OperatorField, "Retrieving...", redraw);
			
		}
		
		DbgMessage ("Get Voicemail Number OK \n");
		err = PhnLibGetBoxNumber(gLibRef, kBoxVoice, radioState.activeLineNumber, &voicemailNum);
		
		/* If the radio is ready and the voicemail # is available */
		/* There is a known bug in Treo 680 of the PhnLibGetBoxNumber API. You need to check */
		/* both the return value and the voicemail number to make sure you actually retrieve the voicemail#*/
		if ((!err)&&(voicemailNum != NULL))
		{
			DbgMessage ("voiceNum OK \n");
			SetFieldTextFromStr(VoicemailField, voicemailNum, redraw);
			
		}
		else // Note: See HsPhoneErrors.h for complete list of telephony error codes
		{	
			DbgMessage ("voiceNum Notavailable \n");
			SetFieldTextFromStr(VoicemailField, "Retrieving...", redraw);
			
		}	
		
				
			
		DbgMessage ("Get SignalQuality OK \n");
		err = PhnLibSignalQuality(gLibRef, &signalQuality);
		if (!err)
		{
			DbgMessage ("SignalQuality OK \n");
			tmpPtr = StrIToA(tmpPtr, signalQuality);
			SetFieldTextFromStr(SigQualityField, tmpPtr, redraw);
			
		}
		else // Note: See HsPhoneErrors.h for complete list of telephony error codes
		{	
			DbgMessage ("SignalQuality Notavailable \n");
			SetFieldTextFromStr(SigQualityField, "Retrieving...", redraw);
			
		}	
		
		// Cell ID information -- specific to Centro CDMA devices
		FtrGet(sysFtrCreator, sysFtrNumOEMDeviceID, &deviceID);
		if (deviceID == kPalmOneDeviceIDGandalf) {
		
			DbgMessage ("Get CellInfo OK \n");
			
			//This API is CDMA specific
			err = PhnLibGetBaseStationIdInfo(gLibRef, &base_id, &sid, &nid );

			if (!err)
			{
				DbgMessage ("CellInfo OK \n");
				
				// Display base station id
				tmpPtr = StrIToA(tmpPtr, base_id);
				SetFieldTextFromStr(CellIdField, tmpPtr, redraw);
				
				// Display system id
				tmpPtr = StrIToA(tmpPtr, sid);
				SetFieldTextFromStr(SIDField, tmpPtr, redraw);
				
				// Display network id
				tmpPtr = StrIToA(tmpPtr, nid);
				SetFieldTextFromStr(NIDField, tmpPtr, redraw);
			}
			else // Note: See HsPhoneErrors.h for complete list of telephony error codes
			{	
				DbgMessage ("CellInfo Notavailable \n");
				SetFieldTextFromStr(CellIdField, "Retrieving...", redraw);
				SetFieldTextFromStr(SIDField, "Retrieving...", redraw);
				SetFieldTextFromStr(NIDField, "Retrieving...", redraw);
			}	
		} else { // Set base station id, system id and network id fields to "unsupported" on non-Gandalf devices
		   
			SetFieldTextFromStr(CellIdField, "[Unsupported]", redraw);
			SetFieldTextFromStr(SIDField, "[Unsupported]", redraw);	
			SetFieldTextFromStr(NIDField, "[Unsupported]", redraw);		
		}	
	}
	else
	{
		DbgMessage ("Service not found \n");
		SetFieldTextFromStr(ConnectionField, "Service not found", redraw);
		DbgMessage ("operator - \n");
		SetFieldTextFromStr(OperatorField, "-", redraw);
		SetFieldTextFromStr(VoicemailField, "-", redraw);
		
	}
	
	if (operatorName) MemPtrFree(operatorName);
	if (voicemailNum) MemPtrFree(voicemailNum);
}




/*
 * FUNCTION: MainFormDoCommand
 *
 * DESCRIPTION: This routine performs the menu command specified.
 *
 * PARAMETERS:
 *
 * command
 *     menu item id
 */

static Boolean MainFormDoCommand(UInt16 command)
{
	Boolean handled = false;

	UInt16  cardNo;
	LocalID dbID;
	DmSearchStateType searchState;

	DmGetNextDatabaseByTypeCreator(true, &searchState, sysFileTApplication,
									 appFileCreator, true, &cardNo, &dbID);

	switch (command)
	{
		case OptionsAbout:
			MenuEraseStatus(0);
			HsAboutHandspringApp(cardNo, dbID, "2008", "Palm DTS Team");
			handled = true;
		break;
	}

	return handled;
}

/*
 * FUNCTION: MainFormHandleEvent
 *
 * DESCRIPTION:
 *
 * This routine is the event handler for the "MainForm" of this 
 * application.
 *
 * PARAMETERS:
 *
 * eventP
 *     a pointer to an EventType structure
 *
 * RETURNED:
 *     true if the event was handled and should not be passed to
 *     FrmHandleEvent
 */

static Boolean MainFormHandleEvent(EventType * eventP)
{
	Boolean handled = false;
	FormType * frmP = FrmGetActiveForm();

	switch (eventP->eType) 
	{
		case menuEvent:
			return MainFormDoCommand(eventP->data.menu.itemID);

		case frmOpenEvent:
			DbgMessage ("frmOpenEvent \n");
			
			/* Set focus on Refresh button */
			FrmSetFocus(frmP, noFocus);
		  	FrmSetFocus(frmP, FrmGetObjectIndex(frmP, MainRefreshButton));
			
			FrmDrawForm(frmP);
			
			MainFormInit(frmP, true);
			
			PrvCommonInitGauge(frmP);
			
			handled = true;
			break;
            
        case frmUpdateEvent:
			/* 
			 * To do any custom drawing here, first call
			 * FrmDrawForm(), then do your drawing, and
			 * then set handled to true. 
			 */
			break;
			
		case nilEvent:
			
			DbgMessage ("nilEvent \n");
			MainFormInit(frmP, true);
		    handled = true;
		    
		    break;
		    	
		case ctlSelectEvent:
		{
			if ((eventP->data.ctlSelect.controlID == MainRefreshButton)||
				(eventP->data.keyDown.keyCode == vchrRockerCenter))
			{
			    PhnPowerType power = phnPowerOff;
			     
			    power = PhnLibModulePowered(gLibRef);
			    
			    if (power == phnPowerOff)   
			        PhnLibSetModulePower(gLibRef, true);
			        
			    DbgMessage ("Refresh \n");
				MainFormInit(frmP, true);
			}

			break;
		}
		
		default:
		    break;
	}
    
	return handled;
}

/*
 * FUNCTION: AppHandleEvent
 *
 * DESCRIPTION: 
 *
 * This routine loads form resources and set the event handler for
 * the form loaded.
 *
 * PARAMETERS:
 *
 * event
 *     a pointer to an EventType structure
 *
 * RETURNED:
 *     true if the event was handled and should not be passed
 *     to a higher level handler.
 */

static Boolean AppHandleEvent(EventType * eventP)
{
	UInt16 formId;
	FormType * frmP;

	if (eventP->eType == frmLoadEvent)
	{
		/* Load the form resource. */
		formId = eventP->data.frmLoad.formID;
		frmP = FrmInitForm(formId);
		FrmSetActiveForm(frmP);

		/* 
		 * Set the event handler for the form.  The handler of the
		 * currently active form is called by FrmHandleEvent each
		 * time is receives an event. 
		 */
		switch (formId)
		{
			case MainForm:
				FrmSetEventHandler(frmP, MainFormHandleEvent);
				break;

		}
		return true;
	}

	return false;
}

/*
 * FUNCTION: AppEventLoop
 *
 * DESCRIPTION: This routine is the event loop for the application.
 */

static void AppEventLoop(void)
{
	UInt16 error;
	EventType event;

	do 
	{
		/* change timeout if you need periodic nilEvents */
		EvtGetEvent(&event, evtWaitForever);

		if (! SysHandleEvent(&event))
		{
			if (! MenuHandleEvent(0, &event, &error))
			{
				if (! AppHandleEvent(&event))
				{
					FrmDispatchEvent(&event);
				}
			}
		}
	} while (event.eType != appStopEvent);
}

/*
 * FUNCTION: AppStart
 *
 * DESCRIPTION:  Get the current application's preferences.
 *
 * RETURNED:
 *     errNone - if nothing went wrong
 */

static Err AppStart(void)
{
	Err err=errNone;
	
	err=HsGetPhoneLibrary(&gLibRef);
	if(err==errNone) {
		PhnLibOpen(gLibRef);

	    /* Register for phone event:
    		 We're mainly interested in phnEvtRegistration indication */
		PhnLibRegister(gLibRef, appFileCreator, phnServiceAll);
	}
	return err;
}

/*
 * FUNCTION: AppStop
 *
 * DESCRIPTION: Save the current state of the application.
 */

static void AppStop(void)
{
        
	/* Close all the open forms. */
	FrmCloseAllForms();

}

/*
 * FUNCTION: RomVersionCompatible
 *
 * DESCRIPTION: 
 *
 * This routine checks that a ROM version is meet your minimum 
 * requirement.
 *
 * PARAMETERS:
 *
 * requiredVersion
 *     minimum rom version required
 *     (see sysFtrNumROMVersion in SystemMgr.h for format)
 *
 * launchFlags
 *     flags that indicate if the application UI is initialized
 *     These flags are one of the parameters to your app's PilotMain
 *
 * RETURNED:
 *     error code or zero if ROM version is compatible
 */

static Err RomVersionCompatible(UInt32 requiredVersion, UInt16 launchFlags)
{
	UInt32 romVersion;

	/* See if we're on in minimum required version of the ROM or later. */
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
	if (romVersion < requiredVersion)
	{
		if ((launchFlags & 
			(sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
			(sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
		{
			FrmAlert (RomIncompatibleAlert);

			/* Palm OS versions before 2.0 will continuously relaunch this
			 * app unless we switch to another safe one. */
			if (romVersion < kPalmOS20Version)
			{
				AppLaunchWithCommand(
					sysFileCDefaultApp, 
					sysAppLaunchCmdNormalLaunch, NULL);
			}
		}

		return sysErrRomIncompatible;
	}

	return errNone;
}

/*
 * FUNCTION: PilotMain
 *
 * DESCRIPTION: This is the main entry point for the application.
 * 
 * PARAMETERS:
 *
 * cmd
 *     word value specifying the launch code. 
 *
 * cmdPB
 *     pointer to a structure that is associated with the launch code
 *
 * launchFlags
 *     word value providing extra information about the launch.
 *
 * RETURNED:
 *     Result of launch, errNone if all went OK
 */

UInt32 PilotMain(UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags)
{
	Err error;

	error = RomVersionCompatible (ourMinVersion, launchFlags);
	if (error) return (error);

	switch (cmd)
	{
		case sysAppLaunchCmdNormalLaunch:
			error = AppStart();
			if (error) 
				return error;

			/* 
			 * start application by opening the main form
			 * and then entering the main event loop 
			 */
			FrmGotoForm(MainForm);
			AppEventLoop();

			AppStop();
			break;
	}

	return errNone;
}
