/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

var Cc = Components.classes;
var Ci = Components.interfaces;
var Cu = Components.utils;

Cu.import("resource://webapprt/modules/WebappRT.jsm");
Cu.import("resource://gre/modules/Services.jsm");
Cu.import("resource://gre/modules/XPCOMUtils.jsm");
Cu.import("resource://gre/modules/Task.jsm");

XPCOMUtils.defineLazyGetter(this, "gAppBrowser",
                            () => document.getElementById("content"));

//@line 22 "/Volumes/BruceDeuce/src/tenfourfox/webapprt/content/webapp.js"

var progressListener = {
  QueryInterface: XPCOMUtils.generateQI([Ci.nsIWebProgressListener,
                                         Ci.nsISupportsWeakReference]),
  onLocationChange: function onLocationChange(progress, request, location,
                                              flags) {

    // Close tooltip (code adapted from /browser/base/content/browser.js)
    let pageTooltip = document.getElementById("contentAreaTooltip");
    let tooltipNode = pageTooltip.triggerNode;
    if (tooltipNode) {
      // Optimise for the common case
      if (progress.isTopLevel) {
        pageTooltip.hidePopup();
      }
      else {
        for (let tooltipWindow = tooltipNode.ownerDocument.defaultView;
             tooltipWindow != tooltipWindow.parent;
             tooltipWindow = tooltipWindow.parent) {
          if (tooltipWindow == progress.DOMWindow) {
            pageTooltip.hidePopup();
            break;
          }
        }
      }
    }

    let isSameOrigin = (location.prePath === WebappRT.config.app.origin);

    // Set the title of the window to the name of the webapp, adding the origin
    // of the page being loaded if it's from a different origin than the app
    // (per security bug 741955, which specifies that other-origin pages loaded
    // in runtime windows must be identified in chrome).
    let title = WebappRT.localeManifest.name;
    if (!isSameOrigin) {
      title = location.prePath + " - " + title;
    }
    document.documentElement.setAttribute("title", title);

//@line 73 "/Volumes/BruceDeuce/src/tenfourfox/webapprt/content/webapp.js"
  },

  onStateChange: function onStateChange(aProgress, aRequest, aFlags, aStatus) {
    if (aRequest instanceof Ci.nsIChannel &&
        aFlags & Ci.nsIWebProgressListener.STATE_START &&
        aFlags & Ci.nsIWebProgressListener.STATE_IS_DOCUMENT) {
      updateCrashReportURL(aRequest.URI);
    }
  }
};

function onOpenWindow(event) {
  if (event.detail.name === "_blank") {
    let uri = Services.io.newURI(event.detail.url, null, null);

    // Prevent the default handler so nsContentTreeOwner.ProvideWindow
    // doesn't create the window itself.
    event.preventDefault();

    // Direct the URL to the browser.
    Cc["@mozilla.org/uriloader/external-protocol-service;1"].
    getService(Ci.nsIExternalProtocolService).
    getProtocolHandlerInfo(uri.scheme).
    launchWithURI(uri);
  }

  // Otherwise, don't do anything to make nsContentTreeOwner.ProvideWindow
  // create the window itself and return it to the window.open caller.
}

function onDOMContentLoaded() {
  window.removeEventListener("DOMContentLoaded", onDOMContentLoaded, false);
  // The initial window's app ID is set by Startup.jsm before the app
  // is loaded, so this code only handles subsequent windows that are opened
  // by the app via window.open calls.  We do this on DOMContentLoaded
  // in order to ensure it gets set before the window's content is loaded.
  if (gAppBrowser.docShell.appId === Ci.nsIScriptSecurityManager.NO_APP_ID) {
    // Set the principal to the correct app ID.  Since this is a subsequent
    // window, we know that WebappRT.configPromise has been resolved, so we
    // don't have to yield to it before accessing WebappRT.appID.
    gAppBrowser.docShell.setIsApp(WebappRT.appID);
  }
}
window.addEventListener("DOMContentLoaded", onDOMContentLoaded, false);

function onLoad() {
  window.removeEventListener("load", onLoad, false);

  gAppBrowser.addProgressListener(progressListener,
                                  Ci.nsIWebProgress.NOTIFY_LOCATION |
                                  Ci.nsIWebProgress.NOTIFY_STATE_DOCUMENT);

  updateMenuItems();

  gAppBrowser.addEventListener("mozbrowseropenwindow", onOpenWindow);
}
window.addEventListener("load", onLoad, false);

function onUnload() {
  gAppBrowser.removeProgressListener(progressListener);
  gAppBrowser.removeEventListener("mozbrowseropenwindow", onOpenWindow);
}
window.addEventListener("unload", onUnload, false);

// Fullscreen handling.

//@line 148 "/Volumes/BruceDeuce/src/tenfourfox/webapprt/content/webapp.js"

// On Mac, we dynamically create the label for the Quit menuitem, using
// a string property to inject the name of the webapp into it.
var updateMenuItems = Task.async(function*() {
//@line 153 "/Volumes/BruceDeuce/src/tenfourfox/webapprt/content/webapp.js"
  yield WebappRT.configPromise;

  let manifest = WebappRT.localeManifest;
  let bundle =
    Services.strings.createBundle("chrome://webapprt/locale/webapp.properties");
  let quitLabel = bundle.formatStringFromName("quitApplicationCmdMac.label",
                                              [manifest.name], 1);
  let hideLabel = bundle.formatStringFromName("hideApplicationCmdMac.label",
                                              [manifest.name], 1);
  document.getElementById("menu_FileQuitItem").setAttribute("label", quitLabel);
  document.getElementById("menu_mac_hide_app").setAttribute("label", hideLabel);
//@line 165 "/Volumes/BruceDeuce/src/tenfourfox/webapprt/content/webapp.js"
});

//@line 170 "/Volumes/BruceDeuce/src/tenfourfox/webapprt/content/webapp.js"

function updateEditUIVisibility() {
//@line 204 "/Volumes/BruceDeuce/src/tenfourfox/webapprt/content/webapp.js"
}

function updateCrashReportURL(aURI) {
//@line 222 "/Volumes/BruceDeuce/src/tenfourfox/webapprt/content/webapp.js"
}

// Context menu handling code.
// At the moment there isn't any built-in menu, we only support HTML5 custom
// menus.

var gContextMenu = null;

XPCOMUtils.defineLazyGetter(this, "PageMenu", function() {
  let tmp = {};
  Cu.import("resource://gre/modules/PageMenu.jsm", tmp);
  return new tmp.PageMenu();
});

function showContextMenu(aEvent, aXULMenu) {
  if (aEvent.target != aXULMenu) {
    return true;
  }

  gContextMenu = new nsContextMenu(aXULMenu);
  if (gContextMenu.shouldDisplay) {
    updateEditUIVisibility();
  }

  return gContextMenu.shouldDisplay;
}

function hideContextMenu(aEvent, aXULMenu) {
  if (aEvent.target != aXULMenu) {
    return;
  }

  gContextMenu = null;

  updateEditUIVisibility();
}

function nsContextMenu(aXULMenu) {
  this.initMenu(aXULMenu);
}

nsContextMenu.prototype = {
  initMenu: function(aXULMenu) {
    this.hasPageMenu = PageMenu.maybeBuildAndAttachMenu(document.popupNode,
                                                        aXULMenu);
    this.shouldDisplay = this.hasPageMenu;
  },
};
