/*
 * @(#)TreePath.java	1.15 98/02/05
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.tree;

import java.io.*;
import java.util.Vector;

/**
 * Represents a path to a node. TreePath is Serializable, but if any 
 * components of the path are not serializable, it will not be written 
 * out.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate 
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.15 02/05/98
 * @author Scott Violet
 */
public class TreePath extends Object implements Serializable
{
    /** Path this instance represents. */
    transient protected Object[]        path;

    /**
     * Constructs a path from an array of Objects, uniquely identifying 
     * the path from the root of the tree to a specific node, as returned
     * by the tree's data model.
     * <p>
     * The model is free to return an array of any Objects it needs to 
     * represent the path. The DefaultTreeModel returns an array of 
     * TreeNode objects. The first TreeNode in the path is the root of the
     * tree, the last TreeNode is the node identified by the path.
     *
     * @param path  an array of Objects representing the path to a node
     */
    public TreePath(Object[] path) {
        if(path == null)
            throw new IllegalArgumentException("path in TreePath must be non null.");
        this.path = path;
    }

    /**
     * Constructs a TreePath when there is only item in the path.
     * <p>
     * @param singlePath  an Object representing the path to a node
     * @see #TreePath(Object[])
     */
    public TreePath(Object singlePath) {
        if(singlePath == null)
            throw new IllegalArgumentException("path in TreePath must be non null.");
        path = new Object[1];
        path[0] = singlePath;
    }

    /**
     * Returns an array of Objects containing the components of this
     * TreePath.
     *
     * @return an array of Objects representing the TreePath
     * @see #TreePath(Object[])
     */
    public Object[] getPath() {
        int             pathLength = path.length;
        Object[]        retPath = new Object[pathLength];

        System.arraycopy(path, 0, retPath, 0, pathLength);
        return retPath;
    }

    /**
     * Returns the last component of this path. For a path
     * returned by the DefaultTreeModel, that is the TreeNode object
     * for the node specified by the path.
     *
     * @return the Object at the end of the path
     * @see #TreePath(Object[])
     */
    public Object getLastPathComponent() {
        if(path.length > 0)
            return path[path.length - 1];
        return null;
    }

    /**
     * Returns the number of elements in the path.
     *
     * @return an int giving a count of items the path
     */
    public int getPathCount() {
        return path.length;
    }

    /**
     * Returns the path component at the specified index.
     *
     * @param element  an int specifying an element in the path, where
     *                 0 is the first element in the path
     * @return the Object at that index location
     * @throws IllegalArgumentException if the index is beyond the length
     *         of the path
     * @see #TreePath(Object[])
     */
    public Object getPathComponent(int element) {
        if(element < 0 || element >= getPathCount())
            throw new IllegalArgumentException("Index " + element + " is greater than path length");
        return path[element];
    }

    /**
     * Tests two TreePaths for equality by checking each element of the
     * paths for equality.
     *
     * @param o the Object to compare
     */
    public boolean equals(Object o) {
        if(o instanceof TreePath) {
            TreePath            oTreePath = (TreePath)o;
            Object[]             oPath = oTreePath.path;

            if((oPath != null && path == null) ||
               (oPath == null && path != null))
                return false;
            if(path != null) {
                int        pathLength = path.length;

                if(pathLength != oPath.length)
                    return false;
                for(int counter = 0; counter < pathLength; counter++)
                    if(!path[counter].equals(oPath[counter]))
                        return false;
            }
            return true;
        }
        return false;
    }

    /**
     * Returns the hashCode for the object.  This must be defined
     * here to ensure 100% pure.
     *
     * @return the hashCode for the object
     */
    public int hashCode() { 
	return super.hashCode();
    }

    /**
     * Returns true if the specified node is a descendant of this
     * TreePath. A TreePath, child, is a descendent of another TreePath,
     * parent, if child contains all of the components that make up 
     * parent's path.
     *
     * @return true if aTreePath is a descendant of the receiver.
     */
    public boolean isDescendant(TreePath aTreePath) {
        if(path != null) {
            Object[]            otherPath = aTreePath.getPath();
            int                 pathLength = path.length;

            if(otherPath.length >= pathLength && pathLength > 0) {
                for(int counter = 0; counter < pathLength; counter++)
                    if(!path[counter].equals(otherPath[counter]))
                        return false;
                return true;
            }
        }
        return false;
    }

    /**
     * Returns a string that displays and identifies this
     * object's properties.
     *
     * @return a String representation of this object
     */
    public String toString() {
        StringBuffer tempSpot = new StringBuffer("[");

        for(int counter = 0; counter < path.length; counter++) {
            if(counter > 0)
                tempSpot.append(", ");
            tempSpot.append(path[counter]);
        }
        tempSpot.append("]");
        return tempSpot.toString();
    }

    // Serialization support.  
    private void writeObject(ObjectOutputStream s) throws IOException {
        s.defaultWriteObject();

        Vector      values = new Vector();
        boolean     writePath = true;

        for(int counter = 0, maxCounter = path.length; counter < maxCounter;
            counter++) {
            if(!(path[counter] instanceof Serializable)) {
                writePath = false;
                break;
            }
        }
        if(writePath) {
            values.addElement("path");
            values.addElement(path);
        }
        s.writeObject(values);
    }

    private void readObject(ObjectInputStream s) 
        throws IOException, ClassNotFoundException {
        s.defaultReadObject();

        Vector          values = (Vector)s.readObject();
        int             indexCounter = 0;
        int             maxCounter = values.size();

        if(indexCounter < maxCounter && values.elementAt(indexCounter).
           equals("path")) {
            path = (Object[])values.elementAt(++indexCounter);
            indexCounter++;
        }
        else
            path = new Object[0];
    }
}
