/*
 * @(#)StyleXlater.java	1.12 98/03/13
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */
package com.sun.java.swing.text.html;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.FontMetrics;
import java.util.Hashtable;
import java.util.StringTokenizer;
import com.sun.java.swing.*;
import com.sun.java.swing.text.*;


/**
 * A class to help translate cascading style definitions into 
 * paragraph & character styles.
 *
 * @author  Jill Nakata
 * @version 1.12 03/13/98
 */
class StyleXlater {

    //
    // Property Names in Cascading Style Sheets
    //
    public static final String BACKGROUND_COLOR = "background-color";
    public static final String BORDER_TOP = "border-top";
    public static final String BORDER_BOTTOM = "border-bottom";
    public static final String BORDER_LEFT = "border-left";
    public static final String BORDER_RIGHT = "border-right";
    public static final String COLOR = "color";
    public static final String FONT_FAMILY = "font-family";
    public static final String FONT_SIZE = "font-size";
    public static final String FONT_SIZES = "font-sizes";
    public static final String FONT_STYLE = "font-style";
    public static final String FONT_WEIGHT = "font-weight";
    public static final String LIST_STYLE_IMAGE = "list-style-image";
    public static final String LIST_STYLE_TYPE = "list-style-type";
    public static final String MARGIN_TOP = "margin-top";
    public static final String MARGIN_BOTTOM = "margin-bottom";
    public static final String MARGIN_LEFT = "margin-left";
    public static final String MARGIN_RIGHT = "margin-right";
    public static final String TEXT_DECORATION = "text-decoration";
    public static final String TEXT_ALIGN = "text-align";
    public static final String VERTICAL_ALIGN = "vertical-align";

    /**
     * Construct a Style Translator
     */
    public StyleXlater()
    {
    }

     /**
      * Translate the parser attribute name & value into a paragraph or
      * character style name & value.
      */
     public static void translateProperty(StyleSheet ss, Style s, 
					   String name, String value) {

	//
        // Remove begin and end quotes of the value "xxx" => xxx.
	//
	String val = Utilities.removeSurroundingQuotes(value);

	if (name.equalsIgnoreCase(FONT_FAMILY)) {
	    if (val.equalsIgnoreCase("monospace") ||
	        val.equalsIgnoreCase("monospaced") ||
	        val.equalsIgnoreCase("typewriter"))
	      StyleConstants.setFontFamily(s, "Monospaced");
	    else
	      StyleConstants.setFontFamily(s, val);
	}
	else if (name.equalsIgnoreCase(FONT_SIZE)) {
	    setFontSize(val, s, ss);
	}
	else if (name.equalsIgnoreCase(FONT_WEIGHT)) {
	    setFontStyle(val, s);
	}
	else if (name.equalsIgnoreCase(FONT_STYLE)) {
	    setFontStyle(val, s);
	}
	else if (name.equalsIgnoreCase(MARGIN_TOP)) {
	    StyleConstants.setSpaceAbove(s, Integer.valueOf(val).intValue());
	}
	else if (name.equalsIgnoreCase(MARGIN_BOTTOM)) {
	    StyleConstants.setSpaceBelow(s, Integer.valueOf(val).intValue());
	}
	else if (name.equalsIgnoreCase(MARGIN_LEFT)) {
	    StyleConstants.setLeftIndent(s, Integer.valueOf(val).intValue());
	}
	else if (name.equalsIgnoreCase(MARGIN_RIGHT)) {
	    StyleConstants.setRightIndent(s, Integer.valueOf(val).intValue());
	}
	else if (name.equalsIgnoreCase(TEXT_DECORATION)) {
	    setTextDecoration(val, s);
	}
	else if (name.equalsIgnoreCase(TEXT_ALIGN)) {
	    setTextAlignment(val, s);
	}
	else if (name.equalsIgnoreCase(COLOR)) {
	    setForegroundColor(val, s);
	}
	else if (name.equalsIgnoreCase(LIST_STYLE_TYPE) ||
	   	 name.equalsIgnoreCase(LIST_STYLE_IMAGE)) {
	    s.addAttribute(name.toLowerCase(), val);
	}
	else if (name.equalsIgnoreCase(FONT_SIZES)) {
	    ss.setFontSizes(val);
	}
	else {
	    s.addAttribute(name.toLowerCase(), val);
        }

     }

     /**
      * Convert this font-size to a value between -3..3
      *         xxs   xs   sm   med  lg   xlg  xxlg
      *        "-3", "-2", "-1", "0", "1", "2", "3"
      */
    static String convertFontSizeString(String val) {

	if (val.equalsIgnoreCase("xx-small") ||
	    val.equalsIgnoreCase("xxsmall"))
	  return "-3";
	else if (val.equalsIgnoreCase("x-small") ||
	         val.equalsIgnoreCase("xsmall"))
	  return "-2";
	else if (val.equalsIgnoreCase("small")) 
	  return "-1";
	else if (val.equalsIgnoreCase("medium"))
	  return "+0";
	else if (val.equalsIgnoreCase("large"))
	  return "+1";
	else if (val.equalsIgnoreCase("x-large") ||
	         val.equalsIgnoreCase("xlarge"))
	  return "+2";
	else if (val.equalsIgnoreCase("xx-large") ||
	         val.equalsIgnoreCase("xxlarge"))
	  return "+3";
	else {
	  System.out.println("StyleXlater.convertFontSizeString - " + 
			     "Warning!  Unimplemented font size value: " + val);
	  return "0";
	}
    }

    /**
     * Set the font style in the Style.
     */
    static void setFontStyle(String fontStyle, Style s) {

	if (fontStyle == null)
	  return;

	if (fontStyle.equalsIgnoreCase("bold") || fontStyle.equals("bolder")) {
	  StyleConstants.setBold(s, true);
        }
	else if (fontStyle.equalsIgnoreCase("italic")) {
	  StyleConstants.setItalic(s, true);
        }
	else if (fontStyle.equalsIgnoreCase("typewriter") ||
		 fontStyle.equalsIgnoreCase("monospace") ||
		 fontStyle.equalsIgnoreCase("monospaced")) {
	  StyleConstants.setFontFamily(s, "Monospaced");
        }
	else if (fontStyle.equalsIgnoreCase("bolditalic")) {
	  StyleConstants.setBold(s, true);
	  StyleConstants.setItalic(s, true);
        }
	else if (fontStyle.equalsIgnoreCase("normal")) {
	  ;  // plain, do nothing
	}
	// We do not have "light" fonts so use normal (do nothing).
	else if (fontStyle.equalsIgnoreCase("light") || fontStyle.equals("lighter") ||
		 fontStyle.equalsIgnoreCase("oblique")) {
	}
    }

    /**
     * Set the text decoration in the style, if supported.
     */
    static void setTextDecoration(String decor, Style s) {

	if (decor.equalsIgnoreCase("underline"))
	  StyleConstants.setUnderline(s, true);
/*
	else 
	  System.out.println("HTMXlater.translateProperty:  " + 
			     "Unsupported property value: " + decor);
*/
    }

    /**
     * Set the text alignment in the style.
     */
    static void setTextAlignment(String align, Style s) {

	if (align.equalsIgnoreCase("left"))
	  StyleConstants.setAlignment(s, StyleConstants.ALIGN_LEFT);
	else if (align.equalsIgnoreCase("center"))
	  StyleConstants.setAlignment(s, StyleConstants.ALIGN_CENTER);
	else if (align.equalsIgnoreCase("right"))
	  StyleConstants.setAlignment(s, StyleConstants.ALIGN_RIGHT);
	else if (align.equalsIgnoreCase("justify"))
	  StyleConstants.setAlignment(s, StyleConstants.ALIGN_JUSTIFIED);
	else 
	  System.out.println("HTMXlater.translateProperty:  " + 
			     "Unsupported property value: " + align);
    }

    /**
     * Translate the property value for "color".
     */
    static void setForegroundColor(String val, Style s) {

        Color color;

        // e.g., "#ff7bd4"
        if (val.startsWith("#")) {
	    color = Utilities.stringToColor(val);
	    StyleConstants.setForeground(s, color);
	}

	// "rgb(255, 125, 213)" or "rgb(100%, 49%, 84%)"
	else if (val.startsWith("rgb") || val.startsWith("RGB")) {
	    int index =  val.indexOf("(");
	    StringTokenizer st = new StringTokenizer(val.substring
					(index+1, val.length()-1), ", \t \n");
	    int n = 0;
	    int rgb[] = new int[3];
	    while (n < rgb.length && st.hasMoreTokens()) {
		String str = st.nextToken();
		if (str.endsWith("%"))
		  rgb[n] = (Integer.valueOf(str.substring(0, str.length()-1)).intValue() * 255) / 100;
	        else
		  rgb[n] = Integer.valueOf(str).intValue();
		n++;
            }
	    // Make sure we got three values for R, G, B.
	    if (n == 3) {
	        color = new Color(rgb[0], rgb[1], rgb[2]);
	        StyleConstants.setForeground(s, color);
	    }
	}

	// "red", "mauve"
	else {
	    color = Utilities.stringToColor(val);
	    StyleConstants.setForeground(s, color);
	}
    }

    /**
     * Translate the property value for "font-size".
     */
    static void setFontSize(String val, Style s, StyleSheet ss) {

        int size = 0;

        // If countTokens() == 0, then this value is all numeric so
        // just assume this number is setting pt size.
	//
        StringTokenizer st = new StringTokenizer(val, "0123456789");
        if (st.countTokens() == 0) {
	    size = Integer.valueOf(val).intValue();;
	    StyleConstants.setFontSize(s, size);
	}

	// "+2pt" or "-1pt"
	else if ((val.startsWith("+") || val.startsWith("-")) &&
	    val.endsWith("pt")) {
	    System.out.println("StyleXlater.translateAttribute: " +
		"Unimplemented font-size value " + val);
	}

	// e.g., "12pt" or "14pt"
	else if (val.toLowerCase().endsWith("pt")) {
	    size = Integer.valueOf(val.substring(0, val.length()-2)).intValue();
	    StyleConstants.setFontSize(s, size);
	}

	// "xx-small", "x-small", "small", "medium", "large", "x-large"
	else {
	    //
	    // Convert this font-size to a value between -3..3
	    //        xxs   xs   sm   med  lg   xlg  xxlg
	    // e.g., "-3" "-2", "-1", "0", "1", "2", "3"
	    String fontval = convertFontSizeString(val);
	    size = ss.getPtSize(fontval);
	    StyleConstants.setFontSize(s, size);
	}

	//
	// If is it the default style, then adjust the base
	// font size in the Style Sheet so that we know
	// what "medium" size maps to.
	//
	String styleName = (String)s.getAttribute(AttributeSet.NameAttribute);
	if (size != 0 && styleName != null &&
 	    (styleName.equals(Constants.BODY) || 
	     styleName.equals(StyleContext.DEFAULT_STYLE))) {
	    ss.setPtSize(size);
	}
    }

}


