/*
 * @(#)StyleSheet.java	1.6 98/03/13
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */
package com.sun.java.swing.text.html;

import java.util.*;
import java.awt.Font;
import java.awt.FontMetrics;

import com.sun.java.swing.text.*;

/**
 * Class that holds current style for a document.
 *
 * @author  Jill Nakata
 * @version 1.6 03/13/98
 */
class StyleSheet {

    public final static int DEFAULT_FONT_SIZE = 3;

    // Member Variables

    private int baseFontIndex;
    private int baseFontSize;
    private int    docPtSize;
    private int    docIndex;
    static int sizeMap[];

    static {
        // Initialize the sizeMap array from the fonts.sizes property.
        int map[] = new int[32];
	String sizes = "7,8,9,10,12,14,16,18,20,22,24,26,28";
//		        0 1 2  3  4  5  6  7  8  9 10 11 12
//docindex		                   DI
//current base font index                 BFI
//basefont values	             1  2  3  4  5  6  7
        StringTokenizer s = new StringTokenizer(sizes, ", \t\n");
        int n = 0;
        while ((n < map.length) && s.hasMoreTokens()) {
            map[n++] = Integer.valueOf(s.nextToken()).intValue();
        }
        sizeMap = new int[n];
        System.arraycopy(map, 0, sizeMap, 0, n);
    }

    /**
     * StyleSheet constructor originating from a style.
     */
    public StyleSheet(Style s) {

      docPtSize = StyleConstants.getFontSize(s);

      //
      // baseFontIndex is the index into sizeMap which
      // points to the current base font setting.
      // The default is 3 (DEFAULT_FONT_SIZE) until it is 
      // changed when a <BASEFONT> tag is reached.
      //
      // Here, we're setting base font 3 (DEFAULT) to 
      // be pt size docPtSize.
      //
      baseFontIndex = getIndexOfSize(docPtSize);
      baseFontSize = DEFAULT_FONT_SIZE;
      //
      // docIndex is the index into sizeMap which
      // points to the docPtSize setting.
      // This value changes when the docPtSize changes.
      // Calling this will set docIndex.
      //
      setPtSize(docPtSize);

    }

    /**
     * StyleSheet constructor originating from default.
     */
    public StyleSheet() {

      docPtSize = 12;  // should get this from DEFAULT_STYLE;

      //
      // baseFontIndex is the index into sizeMap which
      // points to the current base font setting.
      // The default is 3 (DEFAULT_FONT_SIZE) until it is 
      // changed when a <BASEFONT> tag is reached.
      //
      // Here, we're setting base font 3 (DEFAULT) to 
      // be pt size docPtSize.
      //
      baseFontIndex = getIndexOfSize(docPtSize);
      baseFontSize = DEFAULT_FONT_SIZE;
      //
      // docIndex is the index into sizeMap which
      // points to the docPtSize setting.
      // This value changes when the docPtSize changes.
      // Calling this will set docIndex.
      //
      setPtSize(docPtSize);

    }

    /**
     * Call with sizes string separated by '.'s.
     * Used in style sheet definitions.
     * e.g., sizes=" 7.8.9.10.12.14.16.18.20.22.24.26.28"
     */
    public void setFontSizes(String sizes) {
        int map[] = new int[32];
        StringTokenizer s = new StringTokenizer(sizes, ". \t\n");
        int n = 0;
        while ((n < map.length) && s.hasMoreTokens()) {
            map[n++] = Integer.valueOf(s.nextToken()).intValue();
        }
        sizeMap = new int[n];
        System.arraycopy(map, 0, sizeMap, 0, n);

	//
	// Recalculate index and docindex based on new sizes.
	//
	baseFontIndex = getIndexOfSize(docPtSize);
        //
        // docIndex is the index into sizeMap which
        // points to the docPtSize setting.
        // This value changes when the docPtSize changes.
        // Calling this will set docIndex.
        //
        setPtSize(docPtSize);

    }

    public void setBaseFontSize(int sz) {
	if (sz < 1)
	  baseFontSize = 0;
	else if (sz > 7)
	  baseFontSize = 7;
	else
	  baseFontSize = sz;
	
	int diff = baseFontSize - DEFAULT_FONT_SIZE;
        baseFontIndex = docIndex + diff;
	if (baseFontIndex > sizeMap.length - 1)
	  baseFontIndex = sizeMap.length - 1;
	else if (baseFontIndex < 0)
	  baseFontIndex = 0;
    }

    public void setBaseFontSize(String size) {

      int relSize, absSize, diff;

      if (size != null) {
        if (size.startsWith("+")) {
          relSize = Integer.valueOf(size.substring(1)).intValue();
	  diff = DEFAULT_FONT_SIZE - (baseFontSize + relSize);
	  baseFontIndex = docIndex + diff;
        } else if (size.startsWith("-")) {
          relSize = -Integer.valueOf(size.substring(1)).intValue();
	  diff = DEFAULT_FONT_SIZE - (baseFontSize + relSize);
	  baseFontIndex = docIndex + diff;
        } else {
          setBaseFontSize(Integer.valueOf(size).intValue());
	  diff = baseFontSize - DEFAULT_FONT_SIZE;
          baseFontIndex = docIndex + diff;
        }
      }

      if (baseFontIndex > sizeMap.length - 1)
        baseFontIndex = sizeMap.length - 1;
      else if (baseFontIndex < 0)
        baseFontIndex = 0;

    }
    

    public static int  getIndexOfSize(int pt) {
        for (int i = 0; i < sizeMap.length; i ++ )
                if (pt <= sizeMap[i])
                        return i;
        return sizeMap.length - 1;
    }
 

    /**
     * Return the point size.
     */
    public int getPtSize(int index) {
	if (index < 0)
	  return sizeMap[0];
	else if (index > sizeMap.length - 1)
	  return sizeMap[sizeMap.length - 1];
	else
	  return sizeMap[index];

    }

    /**
     *  Given a string "+2", "-2", "2".
     *  returns a pt size value
    **/
    public int getPtSize(String size) {
      int relSize, absSize, diff, index;

      if (size != null) {
        if (size.startsWith("+")) {
          relSize = Integer.valueOf(size.substring(1)).intValue();
	  diff = (baseFontSize + relSize) - DEFAULT_FONT_SIZE;
	  index = docIndex + diff;
        } else if (size.startsWith("-")) {
          relSize = -Integer.valueOf(size.substring(1)).intValue();
	  diff = (baseFontSize + relSize) - DEFAULT_FONT_SIZE;
	  index = docIndex + diff;
        } else {
          absSize = Integer.valueOf(size).intValue();
	  diff = absSize - DEFAULT_FONT_SIZE;
          index = docIndex + diff;
        }
	if (index > sizeMap.length - 1)
	  index = sizeMap.length - 1;
	else if (index < 0)
	  index = 0;

	return sizeMap[index];
      }
      return sizeMap[baseFontIndex];


    }

    /**
     * Sets the point size.
    **/
    public void setPtSize(int ptSize) {
      docPtSize = ptSize;
      // Update the docIndex
      docIndex = getIndexOfSize(docPtSize);
      // If docPtSize changed, then reset the
      // baseFontIndex.
      int diff = baseFontSize - DEFAULT_FONT_SIZE;
      baseFontIndex = docIndex + diff;
    }


    /**
     * Returns the next largest point size.
    **/
    public int getBigger(int ptSize) {
	int index = getIndexOfSize(ptSize);
	if (index + 1 > sizeMap.length - 1)
	  index = sizeMap.length - 1;
    else
	  index++;
	return sizeMap[index];
    }

    /**
     * Returns the next smallest point size.
    **/
    public int getSmaller(int ptSize) {
	int index = getIndexOfSize(ptSize);
	if (index - 1 < 0)
	  index = 0;
    else
	  index--;
	return sizeMap[index];
    }

    /**
     * Given a pt size, return the relative 
     * size value.
    **/
    public int getRelSize(int ptSize) {
	int index = getIndexOfSize(ptSize);
	return index - baseFontIndex;
    }

}

