/*
 * @(#)ListView.java	1.13 98/03/13
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */
package com.sun.java.swing.text.html;

import java.util.Enumeration;
import java.awt.*;
import java.net.URL;
import java.util.StringTokenizer;
import java.net.MalformedURLException;
import com.sun.java.swing.Icon;
import com.sun.java.swing.ImageIcon;
import com.sun.java.swing.border.*;
import com.sun.java.swing.text.*;

/**
 * A view implementation to display an html list
 *
 * @author  Timothy Prinzing
 * @author  Sara Swanson
 * @version 1.13 03/13/98
 */
class ListView extends HTMLBoxView  {

    private String type = null;
    private int start = 1;
    URL imageurl;
    Icon img = null;
    private int bulletgap = 5;

    /**
     * Creates a new view that represents a list element.
     *
     * @param elem the element to create a view for
     */
    public ListView(Element elem) {
	super(elem, View.Y_AXIS);
	setParagraphInsets(elem.getAttributes());
	AttributeSet attr = elem.getAttributes();

	if (attr != null) {
	    String bgstr = (String)attr.getAttribute("-bullet-gap");
	    if (bgstr != null) {
		try {
	            bulletgap = (Integer.valueOf(bgstr)).intValue();
		} catch (NumberFormatException e) {
		    bulletgap = 5;
		}
	    }

	    /* Get the image to use as a list bullet */
	    String imgstr = (String)attr.getAttribute("list-style-image");
	    if (imgstr == null) {
	        imgstr = (String)attr.getAttribute("list-style");
	    }
	    if (imgstr == null) {
		type = null;
	    } else if (imgstr.equalsIgnoreCase("none")) {
		type = new String("none");
	    } else {
                try {
		    String tmpstr = null;
		    StringTokenizer st = new StringTokenizer(imgstr, "()");
		    if (st.hasMoreTokens())
			tmpstr = st.nextToken();
		    if (st.hasMoreTokens())
			tmpstr = st.nextToken();
                    URL u = new URL(tmpstr);
                    img = new ImageIcon(u);
		    type = new String("html-image");
                } catch (MalformedURLException e) {
		    type = null;
                }
	    }

	    /* Get the type of bullet to use in the list */
	    String typestr = (String) attr.getAttribute("type");
	    if ((typestr == null) || (typestr == Constants.NULL_ATTRIBUTE)) {
	        if (type == null) {
		    String nm = elem.getName();
		    if ((nm != null) && nm.equalsIgnoreCase("ol")) {
			type = new String("1");
		    } else {
	                type = (String) attr.getAttribute("list-style-type");
	                if ((type == null) 
			    || (type == Constants.NULL_ATTRIBUTE)) {
	                    type = (String) attr.getAttribute("list-style");
	                    if ((type == null) 
				|| (type == Constants.NULL_ATTRIBUTE))
				type = new String("disc");
		        }
		    }
		}
	    } else {
		type = typestr;
	    }

	    /* Get the number to start the list with */
	    String startstr = (String)attr.getAttribute("start");
	    if (startstr != null) {
		try {
	            start = (Integer.valueOf(startstr)).intValue();
		} catch (NumberFormatException e) {
		}
	    }

	}
    }

    /**
     * Calculates the desired shape of the list.
     *
     * @return the desired span
     * @see View#getPreferredSpan
     */
    public float getAlignment(int axis) {
        switch (axis) {
        case View.X_AXIS:
            return 0.5f;
        case View.Y_AXIS:
            return 0.5f;
        default:
            throw new IllegalArgumentException("Invalid axis: " + axis);
        }
    }

    /**
     * Paints one of the children; called by paint().  By default
     * that is all it does, but a subclass can use this to paint 
     * things relative to the child.
     *
     * @param g the graphics context
     * @param alloc the allocation region
     * @param index the index of the child
     */
    protected void paintChild(Graphics g, Rectangle alloc, int index) {
	View v = getView(index);
	AttributeSet a = v.getElement().getAttributes();
	String childtype = (String) a.getAttribute("type");
	if (childtype == null) {
	    childtype = type;
	}

	if (childtype.equalsIgnoreCase("square")) {
    		drawShape(g, childtype, alloc.x, alloc.y, alloc.height, 
			v.getAlignment(View.Y_AXIS));
	} else if (childtype.equalsIgnoreCase("circle")) {
    		drawShape(g, childtype, alloc.x, alloc.y, alloc.height, 
			v.getAlignment(View.Y_AXIS));
	} else if (childtype.equalsIgnoreCase("1")
		|| childtype.equalsIgnoreCase("decimal")) {
		drawLetter(g, '1', alloc.x, alloc.y, alloc.height, index);
	} else if (childtype.equals("a")
		|| childtype.equalsIgnoreCase("lower-alpha")) {
		drawLetter(g, 'a', alloc.x, alloc.y, alloc.height, index);
	} else if (childtype.equals("A")
		|| childtype.equalsIgnoreCase("upper-alpha")) {
		drawLetter(g, 'A', alloc.x, alloc.y, alloc.height, index);
	} else if (childtype.equals("i")
		|| childtype.equalsIgnoreCase("lower-roman")) {
		drawLetter(g, 'i', alloc.x, alloc.y, alloc.height, index);
	} else if (childtype.equals("I")
		|| childtype.equalsIgnoreCase("upper-roman")) {
		drawLetter(g, 'I', alloc.x, alloc.y, alloc.height, index);
	} else if (childtype.equalsIgnoreCase("html-image")) {
    		drawIcon(g, alloc.x, alloc.y, alloc.height, 
			v.getAlignment(View.Y_AXIS));
	} else if (childtype.equals("none")) {
		;
	} else {
	//else if (childtype.equalsIgnoreCase("disc"))
    		drawShape(g, childtype, alloc.x, alloc.y, alloc.height, 
			v.getAlignment(View.Y_AXIS));
	}

	super.paintChild(g, alloc, index);
    }

    /**
     * Draws the bullet icon specified by the list-style-image argument.
     *
     * @param g     the graphics context
     * @param ax    x coordinate to place the bullet
     * @param ay    y coordinate to place the bullet
     * @param ah    height of the container the bullet is placed in
     * @param align preferred alignment factor for the child view
     */
    protected void drawIcon(Graphics g, int ax, int ay, int ah,
	float align) {
	g.setColor(Color.black);
	int x = ax - img.getIconWidth() - bulletgap;
	int y = ay + (int)(ah * align) - 3;

	img.paintIcon(getContainer(), g, x, y);
    }

    /**
     * Draws the graphical bullet item specified by the type argument.
     *
     * @param g     the graphics context
     * @param type  type of bullet to draw (circle, square, disc)
     * @param ax    x coordinate to place the bullet
     * @param ay    y coordinate to place the bullet
     * @param ah    height of the container the bullet is placed in
     * @param align preferred alignment factor for the child view
     */
    protected void drawShape(Graphics g, String type, int ax, int ay, int ah,
	float align) {
	g.setColor(Color.black);
	int x = ax - bulletgap - 7;
	int y = ay + (int)(ah * align) - 3;

	if (type.equalsIgnoreCase("square")) {
	    g.drawRect(x, y, 7, 7);
	} else if (type.equalsIgnoreCase("circle")) {
	    g.drawOval(x, y, 7, 7);
	} else {
	//else if (type.equalsIgnoreCase("disc")) 
	    g.fillOval(x, y, 7, 7);
	}
    }

    /**
     * Draws the letter or number for an ordered list.
     *
     * @param g     the graphics context
     * @param letter type of ordered list to draw
     * @param ax    x coordinate to place the bullet
     * @param ay    y coordinate to place the bullet
     * @param ah    height of the container the bullet is placed in
     * @param index position of the list item in the list
     */
    protected void drawLetter(Graphics g, char letter, int ax, int ay, int ah,
	int index) {
	g.setColor(Color.black);
	String str = formatItemNum(index + start, letter) + ".";
	FontMetrics fm = g.getFontMetrics();
	int stringwidth = fm.stringWidth(str);
	int x = ax - stringwidth - bulletgap;
	int y = ay + fm.getAscent() + fm.getLeading();
	g.drawString(str, x, y);
    }

    /**
     * Converts the item number into the ordered list number
     * (i.e.  1 2 3, i ii iii, a b c, etc.
     *
     * @param itemNum number to format
     * @param type    type of ordered list
     */
    protected String formatItemNum(int itemNum, char type) {
        String numStyle = "1";
 
        boolean uppercase = false;
 
        String formattedNum;
 
        switch (type) {
        case '1':
        default:
            formattedNum = String.valueOf(itemNum);
            break;
 
        case 'A':
            uppercase = true;
            // fall through
        case 'a':
            formattedNum = formatAlphaNumerals(itemNum);
            break;
 
        case 'I':
            uppercase = true;
            // fall through
        case 'i':
            formattedNum = formatRomanNumerals(itemNum);
        }
 
        if (uppercase) {
            formattedNum = formattedNum.toUpperCase();
        }
 
        return formattedNum;
    }
 
    /**
     * Converts the item number into an alphabetic character
     *
     * @param itemNum number to format
     */
    protected String formatAlphaNumerals(int itemNum) {
        String result = "";
 
        if (itemNum > 26) {
            result = formatAlphaNumerals(itemNum / 26) +
                formatAlphaNumerals(itemNum % 26);
        } else {
            // -1 because item is 1 based.
            result = String.valueOf((char)('a' + itemNum - 1));
        }
 
        return result;
    }

    /* list of roman numerals */
    protected static final char romanChars[][] = {
        {'i', 'v'},
        {'x', 'l' },
        {'c', 'd' },
        {'m', '?' },
        };
 
    /**
     * Converts the item number into a roman numeral
     *
     * @param num  number to format
     */
    protected String formatRomanNumerals(int num) {
        return formatRomanNumerals(0, num);
    }
 
    /**
     * Converts the item number into a roman numeral
     *
     * @param num  number to format
     */
    protected String formatRomanNumerals(int level, int num) {
        if (num < 10) {
            return formatRomanDigit(level, num);
        } else {
            return formatRomanNumerals(level + 1, num / 10) +
                formatRomanDigit(level, num % 10);
        }
    }
 
 
    /**
     * Converts the item number into a roman numeral
     *
     * @param level position
     * @param num   digit to format
     */
    protected String formatRomanDigit(int level, int digit) {
        String result = "";
        if (digit == 9) {
            result = result + romanChars[level][0];
            result = result + romanChars[level + 1][0];
            return result;
        } else if (digit == 4) {
            result = result + romanChars[level][0];
            result = result + romanChars[level][1];
            return result;
        } else if (digit >= 5) {
            result = result + romanChars[level][1];
            digit -= 5;
        }
 
        for (int i = 0; i < digit; i++) {
            result = result + romanChars[level][0];
        }
 
        return result;
    }

}
