/*
 * @(#)PlainDocument.java	1.23 98/04/09
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */
package com.sun.java.swing.text;

import java.util.Vector;
import com.sun.java.swing.event.*;

/**
 * A plain document that uses one font and color.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @author  Timothy Prinzing
 * @version 1.23 04/09/98
 * @see     Document
 * @see     AbstractDocument
 */
public class PlainDocument extends AbstractDocument {

    /**
     * Name of the attribute that specifies the tab
     * size for tabs contained in the content.  The
     * type for the value is Integer.
     */
    public static final String tabSizeAttribute = "tabSize";

    /**
     * Name of the attribute that specifies the maximum
     * length of a line, if there is a maximum length.
     * The type for the value is Integer.
     */
    public static final String lineLimitAttribute = "lineLimit";

    /**
     * Constructs a plain text document.  A default model using StringContent
     * is constructed and set.
     */
    public PlainDocument() {
	this(new StringContent());
    }

    /**
     * Constructs a plain text document.  A default root element is created,
     * and the tab size set to 8.
     *
     * @param c  the container for the content
     */
    protected PlainDocument(Content c) {
	super(c);
	putProperty(tabSizeAttribute, new Integer(8));
	defaultRoot = createDefaultRoot();
    }

    /**
     * Gets the default root element for the document model.
     *
     * @return the root
     * @see Document#getDefaultRootElement
     */
    public Element getDefaultRootElement() {
	return defaultRoot;
    }

    /**
     * Creates the root element to be used to represent the
     * default document structure.
     *
     * @return the element base
     */
    protected AbstractElement createDefaultRoot() {
	BranchElement map = new BranchElement(null, null);
	LeafElement line = new LeafElement(map, null, 0, 1);
	Element[] lines = new Element[1];
	lines[0] = line;
	map.replace(0, 0, lines);
	return map;
    }

    /**
     * Updates document structure as a result of text insertion.  This
     * will happen within a write lock.  Since this document simply
     * maps out lines, we refresh the line map.
     *
     * @param chng the change event describing the dit
     * @param attr the set of attributes for the inserted text
     */
    protected void insertUpdate(DefaultDocumentEvent chng, AttributeSet attr) {
	removed.removeAllElements();
	added.removeAllElements();
	BranchElement lineMap = (BranchElement) getDefaultRootElement();
	int offset = chng.getOffset();
	int length = chng.getLength();
	if (offset > 0) {
	  offset -= 1;
	  length += 1;
	}
	int index = lineMap.getElementIndex(offset);
	Element rmCandidate = lineMap.getElement(index);
	int rmOffs0 = rmCandidate.getStartOffset();
	int rmOffs1 = rmCandidate.getEndOffset();
	int lastOffset = rmOffs0;
	try {
	    String str = getText(offset, length);
	    boolean hasBreaks = false;
	    for (int i = 0; i < length; i++) {
		char c = str.charAt(i);
		if (c == '\n') {
		    int breakOffset = offset + i + 1;
		    added.addElement(new LeafElement(lineMap, null, lastOffset, breakOffset));
		    lastOffset = breakOffset;
		    hasBreaks = true;
		}
	    }
	    if (hasBreaks) {
		int rmCount = 1;
		removed.addElement(rmCandidate);
		if ((offset + length == rmOffs1) && (lastOffset != rmOffs1) &&
		    ((index+1) < lineMap.getElementCount())) {
		    rmCount += 1;
		    Element e = lineMap.getElement(index+1);
		    removed.addElement(e);
		    rmOffs1 = e.getEndOffset();
		}
		if (lastOffset < rmOffs1) {
		    added.addElement(new LeafElement(lineMap, null, lastOffset, rmOffs1));
		}

		Element[] aelems = new Element[added.size()];
		added.copyInto(aelems);
		Element[] relems = new Element[removed.size()];
		removed.copyInto(relems);
		ElementEdit ee = new ElementEdit(lineMap, index, relems, aelems);
		chng.addEdit(ee);
		lineMap.replace(index, relems.length, aelems);
	    }
	} catch (BadLocationException e) {
	    throw new Error("Internal error: " + e.toString());
	}
    }
    
    /**
     * Updates any document structure as a result of text removal.
     * This will happen within a write lock. Since the structure
     * represents a line map, this just checks to see if the 
     * removal spans lines.  If it does, the two lines outside
     * of the removal area are joined together.
     *
     * @param chng the change event describing the edit
     */
    protected void removeUpdate(DefaultDocumentEvent chng) {
	removed.removeAllElements();
	BranchElement map = (BranchElement) getDefaultRootElement();
	int offset = chng.getOffset();
	int length = chng.getLength();
	int line0 = map.getElementIndex(offset);
	int line1 = map.getElementIndex(offset + length);
	if (line0 != line1) {
	    // a line was removed
	    for (int i = line0; i <= line1; i++) {
		removed.addElement(map.getElement(i));
	    }
	    int p0 = map.getElement(line0).getStartOffset();
	    int p1 = map.getElement(line1).getEndOffset();
	    Element[] aelems = new Element[1];
	    aelems[0] = new LeafElement(map, null, p0, p1);
	    Element[] relems = new Element[removed.size()];
	    removed.copyInto(relems);
	    ElementEdit ee = new ElementEdit(map, line0, relems, aelems);
	    chng.addEdit(ee);
	    map.replace(line0, relems.length, aelems);
	}
	

    }

    private AbstractElement defaultRoot;
    private Vector added = new Vector();     // Vector<Element>
    private Vector removed = new Vector();   // Vector<Element>
}
