/*
 * @(#)DefaultHighlighter.java	1.22 98/04/09
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */
package com.sun.java.swing.text;

import java.util.Vector;
import java.awt.*;
import com.sun.java.swing.plaf.*;

/**
 * Implements the Highlighter interfaces.  Implements a simple highlight
 * painter that renders in a solid color.
 * 
 * @author  Timothy Prinzing
 * @version 1.22 04/09/98
 * @see     Highlighter
 */
public class DefaultHighlighter implements Highlighter {

    /**
     * Creates a new DefaultHighlighther object.
     */
    public DefaultHighlighter() {
    }

    // ---- Highlighter methods ----------------------------------------------

    /**
     * Renders the highlights.
     *
     * @param g the graphics context
     */
    public void paint(Graphics g) {
        // PENDING(prinz) - should cull ranges not visible
        Rectangle a = new Rectangle(component.getSize());
        Insets insets = component.getInsets();
        a.x += insets.left;
        a.y += insets.top;
        a.width -= insets.left + insets.right;
        a.height -= insets.top + insets.bottom;
        int len = highlights.size();
        for (int i = 0; i < len; i++) {
	    HighlightInfo info = (HighlightInfo) highlights.elementAt(i);
	    Highlighter.HighlightPainter p = info.getPainter();
	    p.paint(g, info.getStartOffset(), info.getEndOffset(), a, component);
        }
    }

    /**
     * Called when the UI is being installed into the
     * interface of a JTextComponent.  Installs the editor, and
     * removes any existing highlights.
     *
     * @param c the editor component
     * @see Highlighter#install
     */
    public void install(JTextComponent c) {
	component = c;
	removeAllHighlights();
    }

    /**
     * Called when the UI is being removed from the interface of
     * a JTextComponent.
     *
     * @param c the component
     * @see Highlighter#deinstall
     */
    public void deinstall(JTextComponent c) {
	component = null;
    }

    /**
     * Adds a highlight to the view.  Returns a tag that can be used 
     * to refer to the highlight.
     *
     * @param p0   the start offset of the range to highlight >= 0
     * @param p1   the end offset of the range to highlight >= p0
     * @param p    the painter to use to actually render the highlight
     * @returns    an object that can be used as a tag
     *   to refer to the highlight
     * @exception BadLocationException if the specified location is invalid
     */
    public Object addHighlight(int p0, int p1, Highlighter.HighlightPainter p) throws BadLocationException {
	Document doc = component.getDocument();
	TextUI mapper = component.getUI();
        HighlightInfo i = new HighlightInfo();
        i.painter = p;
	i.p0 = doc.createPosition(p0);
	i.p1 = doc.createPosition(p1);
        highlights.addElement(i);
	mapper.damageRange(p0, p1);
        return i;
    }

    /**
     * Removes a highlight from the view.
     *
     * @param tag the reference to the highlight
     */
    public void removeHighlight(Object tag) {
	TextUI mapper = component.getUI();
	HighlightInfo info = (HighlightInfo) tag;
	mapper.damageRange(info.p0.getOffset(), info.p1.getOffset());
	highlights.removeElement(tag);
    }

    /**
     * Removes all highlights.
     */
    public void removeAllHighlights() {
	TextUI mapper = component.getUI();
	if (mapper != null) {
	    int len = highlights.size();
	    if (len != 0) {
		int p0 = Integer.MAX_VALUE;
		int p1 = 0;
		for (int i = 0; i < len; i++) {
		    HighlightInfo info = (HighlightInfo) highlights.elementAt(i);
		    p0 = Math.min(p0, info.p0.getOffset());
		    p1 = Math.max(p1, info.p1.getOffset());
		}
		mapper.damageRange(p0, p1);
		highlights.removeAllElements();
	    }
	}
    }

    /**
     * Changes a highlight.
     *
     * @param tag the highlight tag
     * @param p0 the beginning of the range >= 0
     * @param p1 the end of the range >= p0
     * @exception BadLocationException if the specified location is invalid
     */
    public void changeHighlight(Object tag, int p0, int p1) throws BadLocationException {
	TextUI mapper = component.getUI();
	Document doc = component.getDocument();
	HighlightInfo info = (HighlightInfo) tag;
	int oldP0 = info.p0.getOffset();
	int oldP1 = info.p1.getOffset();
	if (p0 == oldP0) {
	    mapper.damageRange(Math.min(oldP1, p1), Math.max(oldP1, p1));
	} else if (p1 == oldP1) {
	    mapper.damageRange(Math.min(p0, oldP0), Math.max(p0, oldP0));
	} else {
	    mapper.damageRange(oldP0, oldP1);
	    mapper.damageRange(p0, p1);
	}
	info.p0 = doc.createPosition(p0);
	info.p1 = doc.createPosition(p1);
    }

    /**
     * Makes a copy of the highlights.  Does not actually clone each highlight,
     * but only makes references to them.
     *
     * @return the copy
     * @see Highlighter#getHighlights
     */
    public Highlighter.Highlight[] getHighlights() {
	Highlighter.Highlight[] h = new Highlighter.Highlight[highlights.size()];
	highlights.copyInto(h);
	return h;
    }

    // ---- member variables --------------------------------------------
    
    private Vector highlights = new Vector();  // Vector<HighlightInfo>
    private JTextComponent component;


    /**
     * Simple highlight painter that fills a highlighted area with
     * a solid color.
     */
    public static class DefaultHighlightPainter implements Highlighter.HighlightPainter {

        /**
         * Constructs a new highlight painter.
         *
         * @param c the color for the highlight
         */
        public DefaultHighlightPainter(Color c) {
	    color = c;
	}
	
        /**
         * Returns the color of the highlight.
         *
         * @return the color
         */
	public Color getColor() {
	    return color;
	}

	// --- HighlightPainter methods ---------------------------------------

        /**
         * Paints a highlight.
         *
         * @param g the graphics context
         * @param offs0 the starting model offset >= 0
         * @param offs1 the ending model offset >= offs1
         * @param bounds the bounding box for the highlight
         * @param c the editor
         */
        public void paint(Graphics g, int offs0, int offs1, Shape bounds, JTextComponent c) {
	    Rectangle alloc = bounds.getBounds();
	    try {
		// --- determine locations ---
		TextUI mapper = c.getUI();
		Rectangle p0 = mapper.modelToView(offs0);
		Rectangle p1 = mapper.modelToView(offs1);

		// --- render ---
		g.setColor(getColor());
		if (p0.y == p1.y) {
		    // same line, render a rectangle
		    Rectangle r = p0.union(p1);
		    g.fillRect(r.x, r.y, r.width, r.height);
		} else {
		    // different lines
		    int p0ToMarginWidth = alloc.x + alloc.width - p0.x;
		    g.fillRect(p0.x, p0.y, p0ToMarginWidth, p0.height);
		    if ((p0.y + p0.height) != p1.y) {
			g.fillRect(alloc.x, p0.y + p0.height, alloc.width, 
				   p1.y - (p0.y + p0.height));
		    }
		    g.fillRect(alloc.x, p1.y, (p1.x - alloc.x), p1.height);
		}
	    } catch (BadLocationException e) {
		// can't render
	    }
	}

	private Color color;

    }

    class HighlightInfo implements Highlighter.Highlight {

	public int getStartOffset() {
	    return p0.getOffset();
	}

	public int getEndOffset() {
	    return p1.getOffset();
	}

	public Highlighter.HighlightPainter getPainter() {
	    return painter;
	}

	Position p0;
	Position p1;
        Highlighter.HighlightPainter painter;
    }

}
