/*
 * @(#)ComponentView.java	1.18 98/04/09
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */
package com.sun.java.swing.text;

import java.awt.*;
import com.sun.java.swing.event.*;

/**
 * Component decorator that implements the view interface.  The 
 * entire element is used to represent the component.  This acts
 * as a gateway from the display-only View implementations to
 * interactive lightweight components (ie it allows components
 * to be embedded into the View hierarchy.  The parent of the component
 * is the container that is handed out by the associated view 
 * factory.
 *
 * @author Timothy Prinzing
 * @version 1.18 04/09/98
 */
public class ComponentView extends View  {

    /**
     * Creates a new ComponentView object.
     *
     * @param elem the element to decorate
     */
    public ComponentView(Element elem) {
	super(elem);
	AttributeSet attr = elem.getAttributes();
	c = StyleConstants.getComponent(attr);
	c.setVisible(false);
    }

    // --- View methods ---------------------------------------------

    /**
     * Paints a component view.
     * The real paint behavior occurs naturally from the association
     * that the component has with its parent container (the same
     * container hosting this view), so this simply allows us to 
     * position the component properly relative to the view.  Since
     * the coordinate system for the view is simply the parent 
     * containers, positioning the child component is easy.
     *
     * @param g the graphics context
     * @param a the shape
     * @see View#paint
     */
    public void paint(Graphics g, Shape a) {
	c.setBounds(a.getBounds());
	if (! c.isVisible()) {
	    c.setVisible(true);
	}
    }

    /**
     * Determines the preferred span for this view along an
     * axis.
     *
     * @param axis may be either View.X_AXIS or View.Y_AXIS
     * @returns  the span the view would like to be rendered into >= 0.
     *           Typically the view is told to render into the span
     *           that is returned, although there is no guarantee.  
     *           The parent may choose to resize or break the view.
     * @exception IllegalArgumentException for an invalid axis
     */
    public float getPreferredSpan(int axis) {
	Dimension size = c.getPreferredSize();
	switch (axis) {
	case View.X_AXIS:
	    return size.width;
	case View.Y_AXIS:
	    return size.height;
	default:
	    throw new IllegalArgumentException("Invalid axis: " + axis);
	}
    }

    /**
     * Determines the desired alignment for this view along an
     * axis.  This is implemented to give the alignment of the
     * embedded component.
     *
     * @param axis may be either View.X_AXIS or View.Y_AXIS
     * @returns the desired alignment.  This should be a value
     *   between 0.0 and 1.0 where 0 indicates alignment at the
     *   origin and 1.0 indicates alignment to the full span
     *   away from the origin.  An alignment of 0.5 would be the
     *   center of the view.
     */
    public float getAlignment(int axis) {
	switch (axis) {
	case View.X_AXIS:
	    return c.getAlignmentX();
	case View.Y_AXIS:
	    return c.getAlignmentY();
	default:
	    return super.getAlignment(axis);
	}
    }

    /**
     * Sets the size of the view.
     *
     * @param width the width >= 0
     * @param height the height >= 0
     */
    public void setSize(float width, float height) {
	c.setSize((int) width, (int) height);
    }

    /**
     * Sets the parent for a child view.
     * The parent calls this on the child to tell it who its
     * parent is.  If this is null, the view has been removed
     * and we need to remove the associated component from its
     * parent.
     *
     * @param p the parent
     */
    public void setParent(View p) {
	super.setParent(p);
	if (p == null) {
	    Container parent = c.getParent();
	    parent.remove(c);
	} else {
	    Container parent = getContainer();
	    parent.add(c);
	}
    }

    /**
     * Provides a mapping from the coordinate space of the model to
     * that of the view.
     *
     * @param pos the position to convert >= 0
     * @param a the allocated region to render into
     * @return the bounding box of the given position is returned
     * @exception BadLocationException  if the given position does not
     *   represent a valid location in the associated document
     * @see View#modelToView
     */
    public Shape modelToView(int pos, Shape a) throws BadLocationException {
	int p0 = getStartOffset();
	int p1 = getEndOffset();
	if ((pos >= p0) && (pos < p1)) {
	    Rectangle r = new Rectangle(a.getBounds());
	    r.width = 0;
	    return r;
	}
	return null;
    }

    /**
     * Provides a mapping from the view coordinate space to the logical
     * coordinate space of the model.
     *
     * @param x the X coordinate >= 0
     * @param y the Y coordinate >= 0
     * @param a the allocated region to render into
     * @return the location within the model that best represents
     *    the given point in the view
     * @see View#viewToModel
     */
    public int viewToModel(float x, float y, Shape a) {
	return getStartOffset();
    }

    // --- member variables ------------------------------------------------

    private Component c;
}

