/*
 * @(#)JFileChooser.java	1.28 98/04/23
 * 
 * Copyright (c) 1997,1998 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.preview;

import com.sun.java.swing.*;
import com.sun.java.swing.event.*;
import com.sun.java.swing.preview.filechooser.*;
import com.sun.java.swing.plaf.FileChooserUI;
import com.sun.java.accessibility.*;

import java.io.File;
import java.util.Vector;
import java.awt.Component;
import java.awt.Container;
import java.awt.BorderLayout;
import java.awt.Frame;
import java.awt.event.*;

/**
 * JFileChooser provides a simple mechanism for the user to chooser a file.
 *
 * The following pops up a file chooser in the users home directory that
 * only sees .jpg and .gif images:
 *    JFileChooser chooser = new JFileChooser();
 *    ExtensionFileFilter filter = new ExtensionFileFilter();
 *    filter.addExtension("jpg");
 *    filter.addExtension("gif");
 *    filter.setDescription("JPG & GIF Images");
 *    chooser.setFileFilter(filter);
 *    int returnVal = chooser.showOpenDialog(parent);
 *    if(returnVal == JFileChooser.APPROVE_OPTION) {
 *       System.out.println("You chose to open this file: " +
 *            chooser.getSelectedFile().getName());
 *    }
 *
 * @version 1.28 04/23/98
 * @author Jeff Dinkins
 *
 * // PENDING(jeff) add beaninfo
 *
 */
public class JFileChooser extends JComponent implements Accessible {

    // ************************
    // ***** Dialog Types *****
    // ************************

    /**
     * Type value indicating that the FileChooser supports an "Open"
     * file operation.
     */
    public static final int OPEN_DIALOG = 0;

    /**
     * Type value indicating that the FileChooser supports a "Save"
     * file operation.
     */
    public static final int SAVE_DIALOG = 1;

    /**
     * Type value indicating that the FileChooser supports a developer
     * sepcified file operation.
     */
    public static final int CUSTOM_DIALOG = 2;


    // ********************************
    // ***** Dialog Return Values *****
    // ********************************

    /**
     * Return value if cancel is chosen.
     */
    public static final int CANCEL_OPTION = 1;

    /**
     * Return value approve is chosen.
     */
    public static final int APPROVE_OPTION = 0;

    /**
     * Return value if an error occured.
     */
    public static final int ERROR_OPTION = 0;


    // **********************************
    // ***** FileChooser properties *****
    // **********************************


    public static final int FILES_ONLY = 0;
    public static final int DIRECTORIES_ONLY = 1;
    public static final int FILES_AND_DIRECTORIES = 2;

    public static final String CANCEL_SELECTION = "CancelSelection";
    public static final String APPROVE_SELECTION = "ApproveSelection";

    public static final String APPROVE_BUTTON_TEXT_CHANGED_PROPERTY = "ApproveButtonTextChangedProperty";
    public static final String APPROVE_BUTTON_TOOL_TIP_TEXT_CHANGED_PROPERTY = "ApproveButtonToolTipTextChangedProperty";

    public static final String DIRECTORY_CHANGED_PROPERTY = "directoryChanged";
    public static final String SELECTED_FILE_CHANGED_PROPERTY = "ApproveSelection";

    public static final String MULTI_SELECTION_ENABLED_CHANGED_PROPERTY = "fileFilterChanged";

    public static final String FILE_SYSTEM_VIEW_CHANGED_PROPERTY = "FileSystemViewChanged";

    public static final String FILE_VIEW_CHANGED_PROPERTY = "fileViewChanged";
    public static final String FILE_HIDING_CHANGED_PROPERTY = "FileHidingChanged";

    public static final String FILE_FILTER_CHANGED_PROPERTY = "fileFilterChanged";

    public static final String FILE_SELECTION_MODE_CHANGED_PROPERTY = "fileSelectionChanged";
    public static final String ACCESSORY_CHANGED_PROPERTY = "AccessoryChangedProperty";

    public static final String DIALOG_TYPE_CHANGED_PROPERTY = "DialogTypeChangedProperty";
    public static final String CHOOSABLE_FILE_FILTER_CHANGED_PROPERTY = "ChoosableFileFilterChangedProperty";

    // ******************************
    // ***** instance variables *****
    // ******************************

    private String dialogTitle = null;
    private String approveButtonText = null;
    private String approveButtonToolTipText = null;
    private ActionListener actionListener = null;

    private Vector filters = new Vector(5);
    private JDialog dialog = null;
    private int dialogType = OPEN_DIALOG;
    private int returnValue = ERROR_OPTION;
    private JComponent accessory = null;

    private FileView fileView = null;
    private FileView uiFileView = null;

    private boolean useFileHiding = true;

    private int fileSelectionMode = FILES_ONLY;

    private boolean multiSelectionEnabled = false;

    private FileFilter fileFilter = null;

    private FileSystemView fileSystemView = null;

    private File currentDirectory = null;
    private File selectedFile = null;
    private File[] selectedFiles;

    // *************************************
    // ***** JFileChooser Constructors *****
    // *************************************

    /**
     * Creates a JFileChooser pointing to the user's home directory.
     */
    public JFileChooser() {
        this((File) null);
    }
    
    /**
     * Creates a JFileChooser using the given path. Passing in a null
     * string causes the file chooser to point to the users home directory.
     *
     * @param path  a String giving the path to a file or directory
     */
    public JFileChooser(String path) {
	init();
	setCurrentDirectory(getFileSystemView().createFileObject(path));
	setFileFilter(getAcceptAllFileFilter());
    }

    /**
     * Creates a JFileChooser using the given File as the path. Passing
     * in a null file causes the file chooser to point to the users's
     * home directory.
     *
     * @param directory  a File object specifying the path to a file 
     *                   or directory
     */
    public JFileChooser(File directory) {
	init();
	setCurrentDirectory(directory);
	setFileFilter(getAcceptAllFileFilter());
    }

    // common initialization
    protected void init() {
	setFileSystemView(FileSystemView.getFileSystemView());
        updateUI();
    }

    // *****************************
    // ****** File Operations ******
    // *****************************

    /**
     * Returns the selected file. This can be set either by the
     * programmer via setFile() or by a user action, such as
     * either typing the filename int the UI or selecting the
     * file from a list in the UI.
     *
     * @return the selected file
     */
    public File getSelectedFile() {
	return selectedFile;
    }

    /**
     * Sets the selected file. If the file's parent directory is
     * not the current directory, it changed the current directory
     * to be the files parent directory.
     *
     * @see #getSelectedFile();
     *
     * @param selectedFile the selected file 
     */
    public void setSelectedFile(File selectedFile) {
	// PENDING(jeff) - make sure that the file's path is
	// in the current directory. If not, change the current
	// directory to the file's path. 
	File oldValue = this.selectedFile;
	this.selectedFile = selectedFile;
	ensureFileIsVisible(selectedFile);
	firePropertyChange(SELECTED_FILE_CHANGED_PROPERTY, oldValue, this.selectedFile);
    }

    /**
     * Returns a list of selected files if the filechooser is
     * set to allow multi-selection.
     */
    public File[] getSelectedFiles() {
	if(selectedFiles == null) {
	    return new File[0];
	} else {
	    return (File[]) selectedFiles.clone();
	}
    }

    /**
     * Sets the list of selected files if the filechooser is
     * set to allow multi-selection.
     */
    // PENDING(jeff) Add propertychanged event
    public void setSelectedFiles(File[] selectedFiles) {
	this.selectedFiles = selectedFiles;
    }

    /**
     * Returns the current directory. 
     *
     * @return the current directory
     * @see #setCurrentDirectory
     */
    public File getCurrentDirectory() {
	return currentDirectory;
    }

    /**
     * Sets the current directory. Passing in null sets the filechooser
     * to point to the users's home directory.
     *
     * If the file passed in as currentDirectory is not a directory, the
     * parent of the file will be used as the currentDirectory. If the
     * parent is not traversable, then it will walk up the parent tree
     * until it finds a traversable direcotry, or hits the root of the
     * file system.
     *
     * @param currentDirectory the current directory to point to
     * @see #getCurrentDirectory
     */
    public void setCurrentDirectory(File dir) {
	if((this.currentDirectory == dir) && (dir != null)) {
	    return;
	}

	File oldValue = this.currentDirectory;

	if(dir == null) {
	    this.currentDirectory = getFileSystemView().createFileObject(System.getProperty("user.home"));
	} else {
	    File prev = null;
	    while(!isTraversable(dir) && prev != dir && !getFileSystemView().isRoot(dir)) {
		prev = dir;
		dir = getFileSystemView().getParentDirectory(dir);
	    }
	    this.currentDirectory = dir;
	}

	firePropertyChange(DIRECTORY_CHANGED_PROPERTY, oldValue, this.currentDirectory);
    }

    /**
     * Changes the directory to be set to the parent of the
     * current directory. 
     *
     * @see #getCurrentDirectory
     */
    public void changeToParentDirectory() {
	File oldValue = getCurrentDirectory();
	setCurrentDirectory(getFileSystemView().getParentDirectory(oldValue));
    }

    /**
     * Tells the UI to rescan it's files list from the current directory.
     */
    public void rescanCurrentDirectory() {
        getUI().rescanCurrentDirectory();
    }

    public void ensureFileIsVisible(File f) {
        getUI().ensureFileIsVisible(f);
    }

    // **************************************
    // ***** FileChooser Dialog methods *****
    // **************************************

    /**
     * Pops up an "Open File" file chooser dialog.
     *
     * @return   the return state of the filechooser on popdown:
     *             CANCEL_OPTION, APPROVE_OPTION
     */
    public int showOpenDialog(Component parent) {
	setDialogType(OPEN_DIALOG);
	return showDialog(parent, null);
    }

    /**
     * Pops up a "Save File" file chooser dialog.
     *
     * @return   the return state of the filechooser on popdown:
     *             CANCEL_OPTION, APPROVE_OPTION
     */
    public int showSaveDialog(Component parent) {
	setDialogType(SAVE_DIALOG);
	return showDialog(parent, null);
    }

    /**
     * Pops a custom file chooser dialog with a custom ApproveButton.
     * e.g. filechooser.showDialog(parentWindow, "Run Application"
     * would pop up a filechooser with a "Run Application" button
     * instead of the normal "Save" or "Open" button.
     *
     * @param   approveButtonText the text of the ApproveButton
     * @return  the return state of the filechooser on popdown:
     *             CANCEL_OPTION, APPROVE_OPTION
     */
    public int showDialog(Component parent, String approveButtonText) {
	if(approveButtonText != null) {
	    setApproveButtonText(approveButtonText);
	}

        Frame frame = parent instanceof Frame ? (Frame) parent
              : (Frame)SwingUtilities.getAncestorOfClass(Frame.class, parent);

	// PENDING(jeff) - make sure that ui sets initial dialog title
        dialog = new JDialog(frame, getDialogTitle(), true);
        Container contentPane = dialog.getContentPane();
        contentPane.setLayout(new BorderLayout());
        contentPane.add(this, BorderLayout.CENTER);
 
        dialog.pack();
        dialog.setLocationRelativeTo(parent);
 
        dialog.show();
	return returnValue;
    }

    // **************************
    // ***** Dialog Options *****
    // **************************

    /**
     * Returns the type of this dialog.
     *
     * @return   the type of dialog to be displayed:
     *           OPEN_DIALOG, SAVE_DIALOG, CUSTOM_DIALOG
     *
     * @see #setDialogType
     */
    public int getDialogType() {
	return dialogType;
    }

    /**
     * Sets the type of this dialog. Use OPEN_DIALOG when you want to
     * bring up a filechooser that the user can use to open a file. Likewise,
     * use SAVE_DIALOG for letting the user choose a file for saving.
     *
     * Use CUSTOM_DIALOG when you want to use the filechooser in a context
     * other than "Open" or "Save". For instance, you might want to bring up
     * a filechooser that allows the user to choose a file to execute. Note that
     * you normally would not need to set the FileChooser to use CUSTOM_DIALOG
     * since a call to setApproveButtonText does this for you.
     *
     * @param dialogType the type of dialog to be displayed:
     *                   OPEN_DIALOG, SAVE_DIALOG, CUSTOM_DIALOG
     *
     * @see #getDialogType
     * @see #setApproveButtonText
     */ 
    // PENDING(jeff) - fire button text change property
    public void setDialogType(int dialogType) {
	if(this.dialogType == dialogType) {
	    return;
	}
	if(!(dialogType == OPEN_DIALOG || dialogType == SAVE_DIALOG || dialogType == CUSTOM_DIALOG)) {
	    throw new IllegalArgumentException("Incorrect Dialog Type: " + dialogType);
	}
	int oldValue = this.dialogType;
	this.dialogType = dialogType;
	if(dialogType == OPEN_DIALOG || dialogType == SAVE_DIALOG) {
	    setApproveButtonText(null);
	}
	firePropertyChange(DIALOG_TYPE_CHANGED_PROPERTY, oldValue, dialogType);
    }

    /**
     * Sets the string that goes in the FileChooser window's title bar.
     *
     * @see #getDialogTitle
     */
    public void setDialogTitle(String dialogTitle) {
	this.dialogTitle = dialogTitle;
	if(dialog != null) {
	    dialog.setTitle(dialogTitle);
	}
    }

    /**
     * Gets the string that goes in the FileChooser's titlebar.
     *
     * @see #setDialogTitle
     */
    public String getDialogTitle() {
	return dialogTitle;
    }

    // ************************************
    // ***** FileChooser View Options *****
    // ************************************



    /**
     * Sets the tooltip text used in the ApproveButton.
     * If null, the UI object will determine the button's text.
     *
     * @return the text used in the ApproveButton
     *
     * @see #setApproveButtonText
     * @see #setDialogType
     * @see #showDialog
     */ 
    public void setApproveButtonToolTipText(String toolTipText) {
	if(approveButtonToolTipText == toolTipText) {
	    return;
	}
	String oldValue = approveButtonToolTipText;
	approveButtonToolTipText = toolTipText;
	setDialogType(CUSTOM_DIALOG);
	firePropertyChange(APPROVE_BUTTON_TOOL_TIP_TEXT_CHANGED_PROPERTY, oldValue, approveButtonToolTipText);
    }

    /**
     * Returns the tooltip text used in the ApproveButton.
     * If null, the UI object will determine the button's text.
     *
     * @return the text used in the ApproveButton
     *
     * @see #setApproveButtonText
     * @see #setDialogType
     * @see #showDialog
     */ 
    public String getApproveButtonToolTipText() {
	return approveButtonToolTipText;
    }

    /**
     * Sets the text used in the ApproveButton in the FileChooserUI.
     *
     * @param approveButtonText the text used in the ApproveButton
     *
     * @see #getApproveButtonText
     * @see #setDialogType
     * @see #showDialog
     */ 
    // PENDING(jeff) - have ui set this on dialog type change
    public void setApproveButtonText(String approveButtonText) {
	if(this.approveButtonText == approveButtonText) {
	    return;
	}
	String oldValue = this.approveButtonText;
	this.approveButtonText = approveButtonText;
	setDialogType(CUSTOM_DIALOG);
	firePropertyChange(APPROVE_BUTTON_TEXT_CHANGED_PROPERTY, oldValue, approveButtonText);
    }

    /**
     * Returns the text used in the ApproveButton in the FileChooserUI.
     * If null, the UI object will determine the button's text.
     *
     * Typically, this would be "Open" or "Save".
     *
     * @return the text used in the ApproveButton
     *
     * @see #setApproveButtonText
     * @see #setDialogType
     * @see #showDialog
     */ 
    public String getApproveButtonText() {
	return approveButtonText;
    }

    /**
     * Gets the list of user choosable file filters
     *
     * @return a FileFilter array containing all the choosable
     *         file filters
     *
     * @ see #addChoosableFileFilter
     * @ see #removeChoosableFileFilter
     * @ see #resetChoosableFileFilter
     */ 
    public FileFilter[] getChoosableFileFilters() {
	FileFilter[] filterArray = new FileFilter[filters.size()];
	filters.copyInto(filterArray);
	return filterArray;
    }

    /**
     * Adds a filter to the list of user choosable file filters.
     * 
     * @param filter the FileFilter to add to the choosable file
     *               filter list
     *
     * @ see #getChoosableFileFilter
     * @ see #removeChoosableFileFilter
     * @ see #resetChoosableFileFilter
     */ 
    public void addChoosableFileFilter(FileFilter filter) {
	if(!filters.contains(filter)) {
	    FileFilter[] oldValue = getChoosableFileFilters();
	    filters.addElement(filter);
	    firePropertyChange(CHOOSABLE_FILE_FILTER_CHANGED_PROPERTY, oldValue, getChoosableFileFilters());
	} 
    }

    /**
     * Removes a filter from the list of user choosable file filters. Returns
     * true if the file filter was removed;
     *
     * @ see #addChoosableFileFilter
     * @ see #getChoosableFileFilter
     * @ see #resetChoosableFileFilter
     */ 
    public boolean removeChoosableFileFilter(FileFilter f) {
	if(filters.contains(f)) {
	    FileFilter[] oldValue = getChoosableFileFilters();
	    filters.removeElement(f);
	    firePropertyChange(CHOOSABLE_FILE_FILTER_CHANGED_PROPERTY, oldValue, getChoosableFileFilters());
	    return true;
	} else {
	    return false;
	}
    }

    /**
     * Resets the choosable file filter list to it's starting state. Normally,
     * this removes all added file filters while leaving the AcceptAll file filter.
     *
     * @see #addChoosableFileFilter
     * @see #getChoosableFileFilter
     * @see #removeChoosableFileFilter
     */
    public void resetChoosableFileFilters() {
	FileFilter[] oldValue = getChoosableFileFilters();
	filters.removeAllElements();
	filters.addElement(getAcceptAllFileFilter());
	firePropertyChange(CHOOSABLE_FILE_FILTER_CHANGED_PROPERTY, oldValue, getChoosableFileFilters());
    }

    /**
     * Returns the AcceptAll file filter (e.g. (All Files *.*) on windows).
     */
    public FileFilter getAcceptAllFileFilter() {
	FileFilter filter = null;
	if(getUI() != null) {
	    filter = getUI().getAcceptAllFileFilter();
	}
	return filter;
    }

    /**
     * Return the accessory component.
     *
     * @return this JFileChooser's accessory component, or null
     * @see #setAccessory
     */
    public JComponent getAccessory() {
        return accessory;
    }

    /**
     * Sets the accessory component. An accessory is often used to show a preview
     * image of the selected file; however, it can be used for anything that
     * the programmer wishes - such as extra custom file chooser controls.
     *
     * Note: if there was a previous accessory, you should unregister
     * any listeners that the accessory might have registered with the
     * file chooser.
     */
    public void setAccessory(JComponent newAccessory) {
        JComponent oldValue = accessory;
        accessory = newAccessory;
	firePropertyChange(ACCESSORY_CHANGED_PROPERTY, oldValue, accessory);
    }
    
    /**
     * Sets the FileChooser to allow the user to just select files, just select
     * directories, or select both files and directetories.
     *
     * @param dialogType the type of dialog to be displayed:
     *                   FILES_ONLY, DIRECTORIES_ONLY, FILES_AND_DIRECTORIES
     *
     * @see #getFileSelectionMode
     */
    public void setFileSelectionMode(int mode) {
	if(fileSelectionMode == mode) {
	    return;
	}
	int oldValue = fileSelectionMode;
	fileSelectionMode = mode;
	firePropertyChange(FILE_SELECTION_MODE_CHANGED_PROPERTY, oldValue, fileSelectionMode);
    }

    /**
     * @see #setFileSelectionMode
     */
    public int getFileSelectionMode() {
	return fileSelectionMode;
    }

    /**
     * Convenience call that determines if files are selectable based on the current
     * file selection mode
     *
     * @see #setFileSelectionMode
     * @see #getFileSelectionMode
     */
    public boolean isFileSelectionEnabled() {
	return ((fileSelectionMode == FILES_ONLY) || (fileSelectionMode == FILES_AND_DIRECTORIES));
    }

    /**
     * Convenience call that determines if directories are selectable based on the current
     * file selection mode
     *
     * @see #setFileSelectionMode
     * @see #getFileSelectionMode
     */
    public boolean isDirectorySelectionEnabled() {
	return ((fileSelectionMode == DIRECTORIES_ONLY) || (fileSelectionMode == FILES_AND_DIRECTORIES));
    }

    /**
     * Sets the filechooser to allow multiple file selections.
     * NOTE: this functionality is not yet implemented in the current L&Fs.
     *
     * @see #isMultiSelectionEnabled
     */
    public void setMultiSelectionEnabled(boolean b) {
	if(multiSelectionEnabled == b) {
	    return;
	}
	boolean oldValue = multiSelectionEnabled;
	multiSelectionEnabled = b;
	firePropertyChange(MULTI_SELECTION_ENABLED_CHANGED_PROPERTY, oldValue, multiSelectionEnabled);
    }

    /**
     * @see #setMultiSelectionEnabled
     */
    public boolean isMultiSelectionEnabled() {
	return multiSelectionEnabled;
    }

    
    /**
     * If true, hidden files are not shown in the filechooser
     *
     * @return the status of the file hiding property
     * @see #setFileHidingEnabled
     */
    public boolean isFileHidingEnabled() {
	return useFileHiding;
    }

    /**
     * Sets file hiding on or off. If true, hidden files are not shown
     * in the filechooser. The job of determining which files are
     * show is done by the FileView.
     *
     * @param b the boolean value that determines whether file hiding is
     *          turned on or not.
     * @see #isFileHidingEnabled
     */
    public void setFileHidingEnabled(boolean b) {
	boolean oldValue = useFileHiding;
	useFileHiding = b;
	firePropertyChange(FILE_HIDING_CHANGED_PROPERTY, oldValue, useFileHiding);
    }

    /**
     * Sets the current File Filter. The file filter is used by the
     * filechooser to filter out files from view from the user.
     *
     * @param filter the new current file filter to use
     * @see #getFilefilter
     */
    public void setFileFilter(FileFilter filter) {
	FileFilter oldValue = fileFilter;
	fileFilter = filter;
	firePropertyChange(FILE_FILTER_CHANGED_PROPERTY, oldValue, fileFilter);
    }
    

    /**
     * Returns the currently selected file filter.
     *
     * @return the current file filter.
     * @see #setFileFilter
     * @see #addChoosableFileFilter
     */
    public FileFilter getFileFilter() {
	return fileFilter;
    }

    /**
     * Sets the file view to used to retrieve UI information, such as
     * the icon that represents a file or the type description of a file.
     *
     * @see #getFileView
     */
    public void setFileView(FileView fileView) {
	FileView oldValue = this.fileView;
	this.fileView = fileView;
	firePropertyChange(FILE_VIEW_CHANGED_PROPERTY, oldValue, fileView);
    }

    /**
     * Returns the current file view.
     *
     * @see #setFileView
     */
    public FileView getFileView() {
	return fileView;
    }
    
    // ******************************
    // *****FileView delegation *****
    // ******************************

    // NOTE: all of the following methods attempt to delegate
    // first to the client set fileView, and if null is returned
    // (or there is now client defined fileView) then calls the
    // UI's default fileView.
    
    /**
     * @see FileView#getName
     */
    public String getName(File f) {
	String filename = null;
	if(getFileView() != null) {
	    filename = getFileView().getName(f);
	}
	if(filename == null && uiFileView != null) {
	    filename = uiFileView.getName(f);
	}
	return filename;
    }

    /**
     * @see FileView#getDescription
     */
    public String getDescription(File f) {
	String description = null;
	if(getFileView() != null) {
	    description = getFileView().getDescription(f);
	}
	if(description == null && uiFileView != null) {
	    description = uiFileView.getDescription(f);
	}
	return description;
    }

    /**
     * @see FileView#getTypeDescription
     */
    public String getTypeDescription(File f) {
	String typeDescription = null;
	if(getFileView() != null) {
	    typeDescription = getFileView().getTypeDescription(f);
	}
	if(typeDescription == null && uiFileView != null) {
	    typeDescription = uiFileView.getTypeDescription(f);
	}
	return typeDescription;
    }

    /**
     * @see FileView#getIconDescription
     */
    public Icon getIcon(File f) {
	Icon icon = null;
	if(getFileView() != null) {
	    icon = getFileView().getIcon(f);
	}
	if(icon == null && uiFileView != null) {
	    icon = uiFileView.getIcon(f);
	}
	return icon;
    }

    /**
     * @see FileView#isTraversable
     */
    public boolean isTraversable(File f) {
	Boolean traversable = null;
	if(getFileView() != null) {
	    traversable = getFileView().isTraversable(f);
	}
	if(traversable == null && uiFileView != null) {
	    traversable = uiFileView.isTraversable(f);
	}
	if(traversable == null && f != null) {
	    if(f.isDirectory()) {
		traversable = Boolean.TRUE;
	    } else {
		traversable = Boolean.FALSE;
	    }
	} else if(traversable == null) {
	    return false;
	}
	return traversable.booleanValue();
    }

    /**
     * @see FileFilter#accept
     */
    public boolean accept(File f) {
	boolean shown = true;
	if(fileFilter != null) {
	    shown = fileFilter.accept(f);
	}
	return shown;
    }

    /**
     * Sets the file system view which the JFileChooser uses to
     * access and create file system resouces, such as finding
     * the floppy drive and getting a list of root drives.
     *
     * @see FileSystemView
     */
    public void setFileSystemView(FileSystemView fileSystemView) {
	FileSystemView oldValue = this.fileSystemView;
	this.fileSystemView = fileSystemView;
	firePropertyChange(FILE_SYSTEM_VIEW_CHANGED_PROPERTY, oldValue, fileSystemView);
    }

    /**
     * @see setfileSystemView
     */
    public FileSystemView getFileSystemView() {
	return fileSystemView;
    }

    // **************************
    // ***** Event Handling *****
    // **************************

    /**
     * Called by the UI when the user hits the approve
     * (AKA "Open" or "Save") button. This can also by
     * called by the programmer.
     */
    public void approveSelection() {
	returnValue = APPROVE_OPTION;
	if(dialog != null) {
	    dialog.setVisible(false);
	}
	fireActionPerformed(APPROVE_SELECTION);
    }

    /**
     * Called by the UI when the user hits the cancel button.
     * This can also be called by the programmer.
     */
    public void cancelSelection() {
	returnValue = CANCEL_OPTION;
	if(dialog != null) {
	    dialog.setVisible(false);
	}
	fireActionPerformed(CANCEL_SELECTION);
    }

    /**
     * adds an ActionListener to the button
     */
    public void addActionListener(ActionListener l) {
        listenerList.add(ActionListener.class, l);
    }
 
    /**
     * removes an ActionListener from the button
     */
    public void removeActionListener(ActionListener l) {
        listenerList.remove(ActionListener.class, l);
    }
 
    /**
     * Notify all listeners that have registered interest for
     * notification on this event type. The event instance
     * is lazily created using the parameters passed into
     * the fire method.
     * @see EventListenerList
     */
    protected void fireActionPerformed(String command) {
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        ActionEvent e = null;
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==ActionListener.class) {
                // Lazily create the event:
                if (e == null) {
                    e = new ActionEvent(this,
                                        ActionEvent.ACTION_PERFORMED,
                                        command);
                }
                ((ActionListener)listeners[i+1]).actionPerformed(e);
            }
        }
    }

    // *********************************
    // ***** Pluggable L&F methods *****
    // *********************************

    /**
     * Notification from the UIFactory that the L&F
     * has changed.
     *
     * @see JComponent#updateUI
     */
    public void updateUI() {
        setUI((FileChooserUI)UIManager.getUI(this));

	uiFileView = getUI().getFileView();
	boolean useAcceptAllFileFilter = removeChoosableFileFilter(getAcceptAllFileFilter());
	if(useAcceptAllFileFilter) {
	    addChoosableFileFilter(getAcceptAllFileFilter());
	}
    }

    /**
     * Returns a string that specifies the name of the L&F class
     * that renders this component.
     *
     * @return "ButtonUI"
     * @see JComponent#getUIClassID
     * @see UIDefaults#getUI
     * @beaninfo
     *        expert: true
     *   description: A string that specifies the name of the L&F class.
     */
    public String getUIClassID() {
        return "FileChooserUI";
    }

    /**
     * Gets the UI object which implements the L&F for this component.
     *
     * @return the FileChooserUI object that implements the FileChooserUI L&F
     */
    public FileChooserUI getUI() {
        return (FileChooserUI) ui;
    }


}
