/*
 * @(#)JColorChooser.java	1.33 98/02/02
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not disclose
 * such Confidential Information and shall use it only in accordance with the
 * terms of the license agreement you entered into with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
 * LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR
 * ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.preview;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.beans.*;

import com.sun.java.swing.*;
import com.sun.java.swing.plaf.ColorChooserUI;
import com.sun.java.swing.event.*;

/**
 * JColorChooser provides a pane of controls designed to allow 
 * a user to manipulate and select a color.
 *
 * This class provides 3 levels of API:
 * <ol>
 * <li>A static convenience method which shows a modal color-chooser
 * dialog and returns the color selected by the user.
 * <li>A static convenience method for creating a color-chooser dialog
 * where ActionListeners can be specified to be invoked when
 * the user presses one of the dialog buttons.
 * <li>The ability to create instances of JColorChooser panes
 * directly (within any container). PropertyChange listeners
 * can be added to detect when the current "color" property changes.
 * </ol>
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate 
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate 
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.33 02/02/98
 * @author James Gosling
 * @author Amy Fowler
 */
public class JColorChooser extends JComponent implements Serializable {

    /**
     * The color property name.
     */
    public static final String      COLOR_PROPERTY = "color";

    /**
     * Shows a modal color-chooser dialog and blocks until the
     * dialog is hidden.  If the user presses the "OK" button, then
     * this method hides/disposes the dialog and returns the selected color.  
     * If the user presses the "Cancel" button or closes the dialog without 
     * pressing "OK", then this method hides/disposes the dialog and returns
     * null.
     *
     * @param component    the parent Component for the dialog
     * @param title        the String containing the dialog's title
     * @param initialColor the initial Color set when the color-chooser is shown
     */
    public static Color showDialog(Component component, 
                                   String title, Color initialColor) {
                                       
        final JColorChooser pane = new JColorChooser(initialColor != null? 
                                               initialColor : Color.white);

        ColorTracker ok = new ColorTracker(pane);
        JDialog dialog = createDialog(component, title, true, pane, ok, null);
        dialog.addWindowListener(new ColorChooserDialog.Closer());
        dialog.addComponentListener(new ColorChooserDialog.DisposeOnClose());

        dialog.show(); // blocks until user brings dialog down...

        return ok.getColor();
    }


    /**
     * Creates and returns a new dialog containing the specified
     * ColorChooser pane along with "OK", "Cancel", and "Reset" buttons. 
     * If the "OK" or "Cancel" buttons are pressed, the dialog is
     * automatically hidden (but not disposed).  If the "Reset" 
     * button is pressed, the color-chooser's color will be reset to the
     * color which was set the last time show() was invoked on the dialog
     * and the dialog will remain showing.
     *
     * @param c              the parent component for the dialog
     * @param title          the title for the dialog
     * @param modal          a boolean. When true, the remainder of the program
     *                       is inactive until the dialog is closed.
     * @param chooserPane    the color-chooser to be placed inside the dialog
     * @param okListener     the ActionListener invoked when "OK" is pressed
     * @param cancelListener the ActionListener invoked when "Cancel" is pressed
     */
    public static JDialog createDialog(Component c, String title, boolean modal,
                                       JColorChooser chooserPane, 
                                       ActionListener okListener,
                                       ActionListener cancelListener) {

        return new ColorChooserDialog(c, title, modal, chooserPane,
                                                okListener, cancelListener);
    }

    private Color color;

    /**
     * Creates a color chooser pane with an initial color of white.
     */
    public JColorChooser() {
        this(Color.white);
    }

    /**
     * Creates a color chooser pane with the specified initial color.
     *
     * @param initialColor the initial color set in the chooser
     */
    public JColorChooser(Color initialColor) {
        updateUI();
        setColor(initialColor);
    }

    /**
     * Returns the L&F object that renders this component.
     *
     * @return the ColorChooserUI object that renders this component
     */
    public ColorChooserUI getUI() {
        return (ColorChooserUI)ui;
    }

    /**
     * Sets the L&F object that renders this component.
     *
     * @param ui  the ColorChooserUI L&F object
     * @see UIDefaults#getUI
     */
    public void setUI(ColorChooserUI ui) {
        super.setUI(ui);
    }

    /**
     * Notification from the UIManager that the L&F has changed. 
     * Replaces the current UI object with the latest version from the 
     * UIManager.
     *
     * @see JComponent#updateUI
     */
    public void updateUI() {
        setUI((ColorChooserUI)UIManager.getUI(this));
    }

    /**
     * Returns the name of the L&F class that renders this component.
     *
     * @return "ColorChooserUI"
     * @see JComponent#getUIClassID
     * @see UIDefaults#getUI
     */
    public String getUIClassID() {
        return "ColorChooserUI";
    }

    /** 
     * Gets the current color value from the color chooser.
     * By default, this delegates to the model.
     *
     * @return the current color value of the color chooser
     */
    public Color getColor() {
        return color;
    }

    /**
     * Sets the current color of the color chooser to the
     * specified color.
     * This will fire a PropertyChangeEvent for the property
     * named "color".
     *
     * @param color the color to be set in the color chooser
     * @see JComponent#addPropertyChangeListener
     */
    public void setColor(Color color) {
        if (!color.equals(this.color)) {
            Color old = this.color;
            this.color = color;
            firePropertyChange(JColorChooser.COLOR_PROPERTY, old, color);
        }
    }

    /**
     * Sets the current color of the color chooser to the
     * specified RGB color.
     *
     * @param r   an int specifying the amount of Red
     * @param g   an int specifying the amount of Green
     * @param b   an int specifying the amount of Blue
     */
    public void setColor(int r, int g, int b) {
        setColor(new Color(r,g,b));     
    }

    /**
     * Sets the current color of the color chooser to the
     * specified color.
     *
     * @param c an int value that sets the current color in the chooser
     *          where the low-order 8 bits specify the Blue value,
     *          the next 8 bits specify the Green value, and the 8 bits
     *          above that specify the Red value.
     */
    public void setColor(int c) {
        setColor((c >> 16) & 0xFF, (c >> 8) & 0xFF, c & 0xFF);
    }

    /**
     * Adds a color chooser panel to the color chooser.
     *
     * @param name   a string that specifies the name of the panel
     */
    public void addChooserPanel( String name, ColorChooserPanel panel ) {
        getUI().addChooserPanel( name, panel );
    }

    /**
     * Removes the ColorChooserPanel specified by "name".
     *
     * @param name   a string that specifies the panel to be removed
     */
    public ColorChooserPanel removeChooserPanel( String name ) {
        return getUI().removeChooserPanel( name );
    }
}

/*
 * Class which builds a color chooser dialog consisting of
 * a JColorChooser with "Ok", "Cancel", and "Reset" buttons.
 *
 * Note: This needs to be fixed to deal with localization!
 */
class ColorChooserDialog extends JDialog {
    private Color initialColor;
    private JColorChooser chooserPane;

    public ColorChooserDialog(Component c, String title, boolean modal,
                              JColorChooser chooserPane,
                              ActionListener okListener, ActionListener cancelListener) {
        super(JOptionPane.getFrameForComponent(c), title, modal);
        //setResizable(false);

        this.chooserPane = chooserPane;

        Container contentPane = getContentPane();        
        contentPane.setLayout(new BorderLayout());
        contentPane.add(chooserPane, BorderLayout.CENTER);

        /*
         * Create Lower button panel
         */
        JPanel buttonPane = new JPanel();
        buttonPane.setLayout(new FlowLayout(FlowLayout.CENTER));
        JButton okButton = new JButton("OK");
        okButton.setActionCommand("OK");
        if (okListener != null) {
            okButton.addActionListener(okListener);
        }
        okButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                hide();
            }
        });
        buttonPane.add(okButton);

        JButton cancelButton = new JButton("Cancel");
        cancelButton.setActionCommand("cancel");
        if (cancelListener != null) {
            cancelButton.addActionListener(cancelListener);
        }
        cancelButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                hide();
            }
        });
        buttonPane.add(cancelButton);

        JButton resetButton = new JButton("Reset");
        resetButton.addActionListener(new ActionListener() {
           public void actionPerformed(ActionEvent e) {               
               reset();
           }
        });
        buttonPane.add(resetButton);
        contentPane.add(buttonPane, BorderLayout.SOUTH);

        pack();
        setLocationRelativeTo(c);

    }

    public void show() {
        initialColor = chooserPane.getColor();
        super.show();
    }

    public void reset() {
        chooserPane.setColor(initialColor);
    }
            
    static class Closer extends WindowAdapter implements Serializable{        
        public void windowClosing(WindowEvent e) {
            Window w = e.getWindow();
            w.hide();
        }
    }

    static class DisposeOnClose extends ComponentAdapter implements Serializable{
        public void componentHidden(ComponentEvent e) {
            Window w = (Window)e.getComponent();
            w.dispose();
        }
    }

}

class ColorTracker implements ActionListener, Serializable {
    JColorChooser chooser;
    Color color;

    public ColorTracker(JColorChooser c) {
        chooser = c;
    }
 
    public void actionPerformed(ActionEvent e) {
        color = chooser.getColor();
    }

    public Color getColor() {
        return color;
    }
}



