/*
 * @(#)MacMenuUtilities.java	1.2 98/01/30
 *
 * Copyright (c) 1998 Sun Microsystems, Inc. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 */

package com.sun.java.swing.plaf.mac;

import com.sun.java.swing.*;
import com.sun.java.swing.plaf.UIResource;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Rectangle;
import java.awt.Component;
import java.awt.Insets;
import java.io.Serializable;

/**
 * @version @(#)MacMenuUtils.java	1.0 01/20/98
 * @author Symantec
 */

class MacMenuUtilities
{
	static final Insets menuItemMargin = new Insets(0, 2, 0, 2);
	static final Insets menuBarItemMargin = new Insets(2, 4, 2, 4);

	static final int widestCommandKeyW = 20;
	static final int menuItemGap = 3;

	/* MacComboBoxArrowConstants */
	final static int arrowIconWidth = 7;

	final static int anArrowHeight = 4;
	final static int arrowVGap = 2;
	final static int arrowIconHeight = anArrowHeight + arrowVGap + anArrowHeight;
	
	private static final int CHECK = 0;
	private static final int TEXT = 1;
	private static final int ICON = 2;
	private static final int COMMAND = 3;
	private static final int SHORTCUT = 4;
	private static final int ARROW = 5;
	
	private static Icon popupMenuDownScrollerArrowIcon = null;
	private static Icon popupMenuUpScrollerArrowIcon = null;
	private static Icon comboBoxPopupArrowIcon = null;


	private static abstract class MenuItemIcon implements Icon, UIResource, Serializable {
		final static int menuItemIconWidth = 11;
		final static int menuItemIconHeight = 11;

		public int getIconWidth() {
			return menuItemIconWidth;
		}
		
		public int getIconHeight() {
			return menuItemIconHeight;
		}
	}
	
	static Icon getCheckBoxMenuItemIcon() {
		return new MenuItemIcon() {
			public void paintIcon(Component c, Graphics g, int x, int y) {
				if (((AbstractButton) c).getModel().isSelected()) {
					g.translate(x, y);
	
					g.drawLine(1, 5, 2, 5);		// Draw the down stroke
					g.drawLine(1, 6, 5, 6);
					g.drawLine(2, 7, 4, 7);
					g.drawLine(3, 8, 3, 8);
	
					g.drawLine(5, 5, 9, 1);		// Draw the up stroke
					g.drawLine(6, 5, 9, 2);
	
					g.translate(-x, -y);
				}
			}
		};
	}

	static Icon getMenuItemCheckIcon() {
		return new MenuItemIcon() {
			public void paintIcon(Component c, Graphics g, int x, int y) {}
		};
	}

	static Icon getMenuArrowIcon() {
		return new MenuItemIcon() {
			public void paintIcon(Component c, Graphics g, int x, int y) {
				g.translate(x, y);
	
				g.drawLine(0, 0, 0, 10);
				g.drawLine(1, 1, 1, 9);
				g.drawLine(2, 2, 2, 8);
				g.drawLine(3, 3, 3, 7);
				g.drawLine(4, 4, 4, 6);
				g.drawLine(5, 5, 5, 5);
	
				g.translate(-x, -y);
			}
		};
	}

	static Icon menuCommandSymbolIcon = new MenuItemIcon() {
			public void paintIcon(Component c, Graphics g, int x, int y) {
				g.translate(x, y);
	
				// Draw the outside edges
				g.drawLine(2, 1, 3, 1);		// Top of tl cloverleaf
				g.drawLine(7, 1, 8, 1);		// Top of tr cloverleaf

				g.drawLine(1, 2, 1, 3);		// Left of tl cloverleaf
				g.drawLine(9, 2, 9, 3);		// Right of tr cloverleaf

				g.drawLine(1, 7, 1, 8);		// Left of bl cloverleaf
				g.drawLine(9, 7, 9, 8);		// Right of br cloverleaf

				g.drawLine(2, 9, 3, 9);		// Bottom of bl cloverleaf
				g.drawLine(7, 9, 8, 9);		// Bottom of br cloverleaf

				// Draw the inside edges
				g.drawLine(4, 2, 4, 8);		// Right of left cloverleaves
				g.drawLine(6, 2, 6, 8);		// Left of right cloverleaves

				g.drawLine(2, 4, 8, 4);		// Bottom of top cloverleaves
				g.drawLine(2, 6, 8, 6);		// Top of bottom cloverleaves
	
				g.translate(-x, -y);
			}
		};


	static Icon getPopupMenuDownScrollerArrowIcon() {
		if (popupMenuDownScrollerArrowIcon == null)
			popupMenuDownScrollerArrowIcon = new MenuItemIcon() {
				public void paintIcon(Component c, Graphics g, int x, int y) {
					g.translate(x, y);
		
					g.drawLine(0, 3, 10, 3);
					g.drawLine(1, 4, 9, 4);
					g.drawLine(2, 5, 8, 5);
					g.drawLine(3, 6, 7, 6);
					g.drawLine(4, 7, 6, 7);
					g.drawLine(5, 8, 5, 8);
		
					g.translate(-x, -y);
				}
			};
		
		return popupMenuDownScrollerArrowIcon;
	}

	static Icon getPopupMenuUpScrollerArrowIcon() {
		if (popupMenuUpScrollerArrowIcon == null)
			popupMenuUpScrollerArrowIcon = new MenuItemIcon() {
				public void paintIcon(Component c, Graphics g, int x, int y) {
					g.translate(x, y);
		
					g.drawLine(5, 3, 5, 3);
					g.drawLine(4, 4, 6, 4);
					g.drawLine(3, 5, 7, 5);
					g.drawLine(2, 6, 8, 6);
					g.drawLine(1, 7, 9, 7);
					g.drawLine(0, 8, 10, 8);
		
					g.translate(-x, -y);
				}
			};
		
		return popupMenuUpScrollerArrowIcon;
	}

	static Icon getComboBoxPopupArrowIcon() {
		if (comboBoxPopupArrowIcon == null)
			comboBoxPopupArrowIcon = new MenuItemIcon() {
				public void paintIcon(Component c, Graphics g, int x, int y) {
					Rectangle bounds = c.getBounds();
			
					int xOrigin = (bounds.width - arrowIconWidth) / 2;
					int yOrigin = (bounds.height - arrowIconHeight) / 2;
					
					g.translate(xOrigin, yOrigin);

					// Paint the upper arrow
					g.drawLine(3, 0, 3, 0);
					g.drawLine(2, 1, 4, 1);
					g.drawLine(1, 2, 5, 2);
					g.drawLine(0, 3, 6, 3);
			
					// Paint the lower arrow
					g.drawLine(0, 6, 6, 6);
					g.drawLine(1, 7, 5, 7);
					g.drawLine(2, 8, 4, 8);
					g.drawLine(3, 9, 3, 9);
			
					g.translate(-xOrigin, -yOrigin);
				}
			};
		
		return comboBoxPopupArrowIcon;
	}
	
	static String calculateMenuItemSizes(	JMenuItem menuItem,
											FontMetrics fm,
											Icon icon,
											Icon checkIcon,
											Icon arrowIcon,
											KeyStroke shortcut,
											Rectangle viewR, 
											Rectangle[] itemRects
											)
	{
		if ((menuItem.getParent() instanceof JMenuBar) )
		{
			arrowIcon = null;
			shortcut = null;
		}
		else if (arrowIcon != null || (shortcut != null && shortcut.getKeyCode() == 0))
		{
			shortcut = null;
		}
		itemRects[TEXT] = new Rectangle();
		itemRects[ICON] = new Rectangle();

		Rectangle allowableViewR = new Rectangle(viewR);

		/* Initialize the width and height of the checkIcon bounds rectangle itemRects[CHECK].
		 *		adjust the allowableViewR width to compensate
		*/
		if (checkIcon != null) {
			itemRects[CHECK] = new Rectangle(checkIcon.getIconWidth(), checkIcon.getIconHeight());
			allowableViewR.width -= (itemRects[CHECK].width + menuItemGap);
		}

		/* Initialize the width and height of the arrowIcon bounds rectangle itemRects[ARROW].
		 *		adjust the allowableViewR width to compensate
		*/
		if (arrowIcon != null) {
			itemRects[ARROW] = new Rectangle(widestCommandKeyW + arrowIcon.getIconWidth(), arrowIcon.getIconHeight());
			allowableViewR.width -= (itemRects[ARROW].width + menuItemGap);
		}

		/* Initialize the width of the shortcut bounds rectangles itemRects[COMMAND, SHORTCUT].
		 *		adjust the allowableViewR width to compensate
		*/
		if (shortcut != null) {
			itemRects[COMMAND] = new Rectangle(MenuItemIcon.menuItemIconWidth, MenuItemIcon.menuItemIconHeight);
			itemRects[SHORTCUT] = new Rectangle(widestCommandKeyW, 0);
			allowableViewR.width -= (itemRects[COMMAND].width + itemRects[SHORTCUT].width + menuItemGap);
		}

		// layout the text and icon
		String text = SwingUtilities.layoutCompoundLabel(fm, menuItem.getText(), icon,
												menuItem.getVerticalAlignment(), 
												menuItem.getHorizontalAlignment(),
												menuItem.getVerticalTextPosition(), 
												menuItem.getHorizontalTextPosition(),
												allowableViewR, itemRects[ICON], itemRects[TEXT], 
												menuItemGap);

		return text;
	}


	/** 
	* Compute and return the location of the icons origin, the 
	* location of origin of the text baseline, and a possibly clipped
	* version of the compound labels string.  Locations are computed
	* relative to the viewR rectangle. 
	*/
	static String layoutMenuItem(
									JMenuItem menuItem,
									FontMetrics fm,
									Icon icon,
									Icon checkIcon,
									Icon arrowIcon,
									KeyStroke shortcut,
									Rectangle viewR,
									Rectangle[] itemRects
									)
	{
		Insets i = menuItem.getInsets();
		viewR.x += i.left;
		viewR.y += i.top;
		viewR.width -= (i.right + viewR.x);
		viewR.height -= (i.bottom + viewR.y);
		
		String text = MacMenuUtilities.calculateMenuItemSizes(	menuItem,
																fm,
																icon,
																checkIcon,
																arrowIcon,
																shortcut,
																viewR, 
																itemRects
																);
		int verticalCenter = (itemRects[ICON].union(itemRects[TEXT])).height / 2;

		/* Adjust the location of the checkIcon bounds rectangle itemRects[CHECK].
		 *	(Also adjust the starting locations of the TEXT and ICON bounds rectangle itemRects.
		*/
		if (itemRects[CHECK] != null) {
			itemRects[CHECK].y = verticalCenter - (itemRects[CHECK].height/2);			// Center it vertically
			itemRects[TEXT].x += (itemRects[CHECK].width + menuItemGap);				// Adjust the TEXT location
			itemRects[ICON].x += (itemRects[CHECK].width + menuItemGap);				// Adjust the ICON location
		}

		/* Adjust the location of the arrowIcon bounds rectangle itemRects[ARROW].
		*/
		if (itemRects[ARROW] != null) {
			itemRects[ARROW].x = (viewR.x + viewR.width) - itemRects[ARROW].width;
			itemRects[ARROW].y = verticalCenter - (itemRects[ARROW].height/2);			// Center it vertically
		}

		/* Adjust the shortcut bounds rectangles itemRects[COMMAND, SHORTCUT].
		*/
		if (itemRects[SHORTCUT] != null) {
			itemRects[COMMAND].x = (viewR.x + viewR.width) - (itemRects[COMMAND].width + itemRects[SHORTCUT].width);
			itemRects[COMMAND].y = verticalCenter - (itemRects[COMMAND].height/2);		// Center it vertically

			itemRects[SHORTCUT].x = itemRects[COMMAND].x + itemRects[COMMAND].width;	// Locate it to the right of the command key
			itemRects[SHORTCUT].y = itemRects[TEXT].y;									// Position it vertically with the text
			itemRects[SHORTCUT].height = itemRects[TEXT].height;
		}
		
		return text;
	}


	static void paintMenuItem(Graphics g, JMenuItem menuItem,
								Icon checkIcon, Icon arrowIcon,
								Color background, Color foreground)
	{
		Font font = menuItem.getFont();
		FontMetrics fm = menuItem.getToolkit().getFontMetrics(font);

		Rectangle viewRect = new Rectangle(menuItem.getSize());
		Rectangle itemRects[] = new Rectangle[6];

		Color holdc = g.getColor();

		// Paint the background
		if (background != null) {
			g.setColor(background);
			g.fillRect(0,0, viewRect.width, viewRect.height);
		} else if (menuItem.isOpaque()) {
			g.setColor(menuItem.getBackground());
			g.fillRect(0,0, viewRect.width, viewRect.height);
		}

		
		KeyStroke shortcut = menuItem.getAccelerator();

		// layout the text and icon
		String text = MacMenuUtilities.layoutMenuItem(
												menuItem, fm,
												menuItem.getIcon(), checkIcon, arrowIcon, shortcut,
												viewRect, itemRects
												);

		if (foreground != null) {
			g.setColor(foreground);
		} else {
			g.setColor(menuItem.getForeground());
		}

		// Paint the Icon
		if (menuItem.getIcon() != null) { 
			ButtonModel model = menuItem.getModel();
			Icon icon;
			if (!model.isEnabled()) {
				icon = (Icon) menuItem.getDisabledIcon();
			} else if (model.isPressed() && model.isArmed()) {
				icon = (Icon) menuItem.getPressedIcon();
				if(icon == null) {
					// Use default icon
					icon = (Icon) menuItem.getIcon();
				} 
			} else {
				icon = (Icon) menuItem.getIcon();
			}

			icon.paintIcon(menuItem, g, itemRects[ICON].x, itemRects[ICON].y);
		}

		// Paint the Check
		if (itemRects[CHECK] != null) {
			checkIcon.paintIcon(menuItem, g, itemRects[CHECK].x, itemRects[CHECK].y);
		}
		
		// Draw the Text
		if (!itemRects[TEXT].isEmpty()) {
			g.drawString(text, itemRects[TEXT].x, itemRects[TEXT].y + fm.getAscent());
		}
		
		// Paint the Arrow
		if (itemRects[ARROW] != null) {
			arrowIcon.paintIcon(menuItem, g, itemRects[ARROW].x + MenuItemIcon.menuItemIconWidth, itemRects[ARROW].y);
		}

		// Draw the shortcut
		if (itemRects[SHORTCUT] != null) {
			menuCommandSymbolIcon.paintIcon(menuItem, g, itemRects[COMMAND].x, itemRects[COMMAND].y);

			char data[] = new char[] { (char) shortcut.getKeyCode() };
			g.drawChars(data, 0, 1, itemRects[SHORTCUT].x, itemRects[SHORTCUT].y + fm.getAscent());
		}

		g.setColor(holdc);
	}


	static Dimension getPreferredMenuItemSize(JMenuItem menuItem,
												Icon checkIcon,
												Icon arrowIcon)
	{
		Font font = menuItem.getFont();
		FontMetrics fm = menuItem.getToolkit().getFontMetrics(font);
		
		Rectangle viewR = new Rectangle(Short.MAX_VALUE, Short.MAX_VALUE);
		Rectangle itemRects[] = new Rectangle[6];

		MacMenuUtilities.calculateMenuItemSizes(
											menuItem, fm,
											menuItem.getIcon(), checkIcon, arrowIcon, menuItem.getAccelerator(),
											viewR, itemRects
											);

		// find the union of the icon and text rects
		Rectangle r = itemRects[ICON].union(itemRects[TEXT]);
		
		// Add in the checkIcon
		if (itemRects[CHECK] != null) {
			r.width += (itemRects[CHECK].width + menuItemGap);
		}
		
		// Add in the arrowIcon
		if (itemRects[ARROW] != null) {
			r.width += (itemRects[ARROW].width + menuItemGap);
		}
		
		// Add in the shortcut
		if (itemRects[SHORTCUT] != null) {
			r.width += (itemRects[COMMAND].width + itemRects[SHORTCUT].width + menuItemGap);
		}

		// Add in the insets
		Insets insets = menuItem.getInsets();
		r.width += insets.left + insets.right;
		r.height += insets.top + insets.bottom;

		return r.getSize();
	}
}
