/*
 * @(#)BasicTreeCellEditorContainer.java	1.15 98/02/05
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.plaf.basic;

import com.sun.java.swing.*;
import java.awt.*;
import java.io.*;
import java.util.EventObject;
import java.util.Vector;
import com.sun.java.swing.tree.*;
import com.sun.java.swing.event.*;

/**
 * Use this object when you have a TreeCellEditor that you wish to use
 * in a tree that uses a BasicTreeCellRenderer.  The editor you pass in
 * will get all the TreeCellEditor messages and will be added as a
 * child.  This will then draw the appropriate image and position the
 * editor at the appropriate place.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.15 02/05/98
 * @author Scott Violet
 */
public class BasicTreeCellEditorContainer extends Container implements
					     TreeCellEditor
{
    /** Renderer used to draw the corresponding component. */
    protected BasicTreeCellRenderer       renderer;
    /** Component used in editing.  Will be valid for the life of the editing
      * session. */
    transient protected Component         editor;
    /** The real TreeCellEditor that this Container is wrapping. */
    transient protected TreeCellEditor    realEditor;

    /** Icon for the current item in the current editing session. */
    transient protected Icon                        editingIcon;
    /** Offset, in the x direction, where to start editing from. */
    protected int                         editingOffset;

    /** Is the current item being edited a leaf? */
    protected boolean                     editingLeaf;
    /** Is the current item being edited expanded? */
    protected boolean                     editingExpanded;

    public BasicTreeCellEditorContainer(TreeCellEditor realEditor,
					BasicTreeCellRenderer renderer) {
	this.realEditor = realEditor;
	this.renderer = renderer;
	setLayout(null);
    }

    /**
      * Returns the value currently being edited.
      */
    public Object getCellEditorValue() {
	return realEditor.getCellEditorValue();
    }

    /**
      * If the realEditor returns true to this message, prepareForEditing
      * is messaged and true is returned.
      */
    public boolean isCellEditable(EventObject event) {
	if(realEditor.isCellEditable(event)) {
	    prepareForEditing();
	    return true;
	}
	return false;
    }

    /**
      * Messages the realEditor for the return value.
      */
    public boolean shouldSelectCell(EventObject event) {
	return realEditor.shouldSelectCell(event);
    }

    /**
      * Sets the editor, adds it to this Container and updates the editingIcon
      * and editingOffset.
      */
    protected void prepareForEditing() {
	this.add(editor);
	if(renderer != null) {
	    if(editingLeaf)
		editingIcon = renderer.getLeafIcon();
	    else if(editingExpanded)
		editingIcon = renderer.getOpenIcon();
	    else
		editingIcon = renderer.getClosedIcon();
	    if(editingIcon == null)
		editingOffset = 0;
	    else
		editingOffset = renderer.getIconTextGap() +
		    editingIcon.getIconWidth();
	}
	else {
	    editingOffset = 0;
	    editingIcon = null;
	}
    }

    /**
      * If the realEditor will allow editing to stop, the realEditor is
      * removed and true is returned, otherwise false is returned.
      */
    public boolean stopCellEditing() {
	if(realEditor.stopCellEditing()) {
	    if(editor != null)
		this.remove(editor);
	    editingIcon = null;
	    return true;
	}
	return false;
    }

    /**
      * Messages cancelCellEditing to the realEditor and removes it from this
      * instance.
      */
    public void cancelCellEditing() {
	realEditor.cancelCellEditing();
	if(editor != null)
	    this.remove(editor);
	editingIcon = null;
    }

    /**
      * Adds the CellEditorListener.
      */
    public void addCellEditorListener(CellEditorListener l) {
	realEditor.addCellEditorListener(l);
    }

    /**
      * Removes the previously added CellEditorListener l.
      */
    public void removeCellEditorListener(CellEditorListener l) {
	realEditor.removeCellEditorListener(l);
    }

    /**
      * Lays out this Container.  If editing, the editor will be placed at
      * editingOffset in the x direction and 0 for y.
      */
    public void doLayout() {
	if(editor != null) {
	    Dimension             cSize = getSize();

	    editor.getPreferredSize();
	    editor.setLocation(editingOffset, 0);
	    editor.setBounds(editingOffset, 0, cSize.width - editingOffset,
			     cSize.height);
	}
    }

    /**
      * Configures the editor.  Passed onto the realEditor.
      */
    public Component getTreeCellEditorComponent(JTree tree, Object value,
						boolean isSelected,
						boolean expanded,
						boolean leaf, int row) {
	editingLeaf = leaf;
	editingExpanded = expanded;
	editor = realEditor.getTreeCellEditorComponent(tree, value, isSelected,
					      expanded,leaf, row);
	return this;
    }

    /**
      * If editing, the icon is painted.
      */
    public void paint(Graphics g) {
	if(editingIcon != null) {
	    int       yLoc = Math.max(0, (getSize().height - editingIcon.getIconHeight()) / 2);

	    editingIcon.paintIcon(this, g, 0, yLoc);
	}
	super.paint(g);
    }

    /**
      * Returns the preferred size for the Container.  This will be
      * the preferred size of the editor offset by editingOffset.
      */
    public Dimension getPreferredSize() {
	if(editor != null) {
	    Dimension         pSize = editor.getPreferredSize();

	    pSize.width += editingOffset + 5;
	    return pSize;
	}
	return new Dimension(0, 0);
    }

    public void requestFocus() {
	if(editor != null)
	    editor.requestFocus();
	else
	    super.requestFocus();
    }

    // Serialization support.  
    private void writeObject(ObjectOutputStream s) throws IOException {
	Vector      values = new Vector();

	s.defaultWriteObject();
	// Save the editingIcon, if its Serializable.
	if(editingIcon != null && editingIcon instanceof Serializable) {
	    values.addElement("editingIcon");
	    values.addElement(editingIcon);
	}
	// Save the realEditor, if its Serializable.
	if(realEditor != null && realEditor instanceof Serializable) {
	    values.addElement("realEditor");
	    values.addElement(realEditor);
	}
	s.writeObject(values);
    }

    private void readObject(ObjectInputStream s) 
	throws IOException, ClassNotFoundException {
	s.defaultReadObject();

	Vector          values = (Vector)s.readObject();
	int             indexCounter = 0;
	int             maxCounter = values.size();

	if(indexCounter < maxCounter && values.elementAt(indexCounter).
	   equals("editingIcon")) {
	    editingIcon = (Icon)values.elementAt(++indexCounter);
	    indexCounter++;
	}
	if(indexCounter < maxCounter && values.elementAt(indexCounter).
	   equals("realEditor")) {
	    realEditor = (TreeCellEditor)values.elementAt(++indexCounter);
	    indexCounter++;
	}
    }

}
