/*
 * @(#)BasicToolBarUI.java	1.39 98/04/20
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.plaf.basic;

import com.sun.java.swing.*;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.*;
import java.beans.*;
import java.io.Serializable;

import com.sun.java.swing.border.*;
import com.sun.java.swing.plaf.*;

/**
 * A Windows L&F implementation of ToolBarUI.  This implementation 
 * is a "combined" view/controller.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.39 04/20/98
 * @author Georges Saab
 */
public class BasicToolBarUI extends ToolBarUI implements Serializable {
    protected JToolBar toolBar;
    private boolean floating;
    private boolean floatable;
    private int floatingX;
    private int floatingY;
    private JFrame floatingFrame;
    protected DragWindow dragWindow;
    private Container dockingSource;
    private DockingListener dockingListener;
    private int dockingSensitivity = 0;

    protected Color dockingColor = null;
    protected Color floatingColor = null;
    protected Color dockingBorderColor = null;
    protected Color floatingBorderColor = null;

    public static final int HORIZONTAL = 0;
    public static final int VERTICAL = 1;

    public static ComponentUI createUI(JComponent x) {
	return new BasicToolBarUI();
    }

    public void installUI(JComponent c) {
	toolBar = (JToolBar) c;

	// Set defaults
        installDefaults(c);

        // Initialize instance vars
        dockingSensitivity = 0;
        floating = floatable = false;
        floatingX = floatingY = 0;
 	floatingFrame = createFloatingFrame(toolBar);

	setOrientation(HORIZONTAL);

        // Set-up docking behavior
        installListeners(c);

	c.setOpaque(true);
    }
    
    public void uninstallUI(JComponent c) {

        // Clear defaults
        uninstallDefaults(c);

        // Clear instance vars
	if (isFloating() == true)
	    setFloating(false, null);
        floatingFrame = null;
        dragWindow = null;
        dockingSource = null;

        // Remove docking behavior
        uninstallListeners(c);
    }

    protected void installDefaults(JComponent c) {
 	LookAndFeel.installBorder(c,"ToolBar.border");	
	LookAndFeel.installColorsAndFont(c,
					      "ToolBar.background",
					      "ToolBar.foreground",
					      "ToolBar.font");
	// Toolbar specific defaults
	if ( dockingColor == null || dockingColor instanceof UIResource )
	    dockingColor = UIManager.getColor("ToolBar.dockingColor");
	if ( floatingColor == null || floatingColor instanceof UIResource )
	    floatingColor = UIManager.getColor("ToolBar.floatingColor");
	if ( dockingBorderColor == null || 
	     dockingBorderColor instanceof UIResource )
	    dockingBorderColor = UIManager.getColor("ToolBar.dockingBorderColor");
	if ( floatingBorderColor == null || 
	     floatingBorderColor instanceof UIResource )
	    floatingBorderColor = UIManager.getColor("ToolBar.floatingBorderColor");
    }

    protected void uninstallDefaults(JComponent c) {
	LookAndFeel.uninstallBorder(c);
        dockingColor = null;
        floatingColor = null;
        dockingBorderColor = null;
        floatingBorderColor = null;
    }

    protected void installListeners(JComponent c) {
        dockingListener = createDockingListener((JToolBar)c);
        setFloatable(true);	
    }

    protected void uninstallListeners(JComponent c) {
        setFloatable(false);
        if (dockingListener != null) {
            dockingListener = null;
        }
    }

    protected JFrame createFloatingFrame(JToolBar toolbar) {
	JFrame frame = new JFrame(toolbar.getName());
	WindowListener wl = createFrameListener();
	frame.addWindowListener(wl);
        return frame;
    }

    protected DragWindow createDragWindow(JToolBar toolbar) {
	Frame frame = null;
	if(toolBar != null) {
	    Container p;
	    for(p = toolBar.getParent() ; p != null && !(p instanceof Frame) ;
		p = p.getParent());
	    if(p != null && p instanceof Frame)
		frame = (Frame) p;
	}
	if(frame == null) {
	    floatingFrame = createFloatingFrame(toolBar);
	    frame = floatingFrame;
	}

	DragWindow dragWindow = new DragWindow(frame);
	return dragWindow;
    }

    public Dimension getMinimumSize(JComponent c) {
        return getPreferredSize(c);
    }

    public Dimension getPreferredSize(JComponent c) {
	return null;
    }

    public Dimension getMaximumSize(JComponent c) {
        return getPreferredSize(c);
    }

    public void setFloatingLocation(int x, int y) {
	floatingX = x;
	floatingY = y;
    }
    
    public boolean isFloating() {
	return floating;
    }

    public void setFloating(boolean b, Point p) {
 	if (toolBar.isFloatable() == true) {
	    if (dragWindow != null)
		dragWindow.setVisible(false);
	    this.floating = b;
	    if (b == true) {
		if (dockingSource == null) {
		    dockingSource = toolBar.getParent();
		    dockingSource.remove(toolBar);
		}
		if (floatingFrame == null)
		    floatingFrame = createFloatingFrame(toolBar);
		floatingFrame.getContentPane().add(toolBar,BorderLayout.CENTER);
		setOrientation(HORIZONTAL);
		floatingFrame.pack();
		floatingFrame.setLocation(floatingX, floatingY);
		floatingFrame.show();
	    } else {
		floatingFrame.setVisible(false);
		floatingFrame.getContentPane().remove(toolBar);
		String constraint = getDockingConstraint(dockingSource,
							 p);
		int orientation = mapConstraintToOrientation(constraint);
		setOrientation(orientation);
		if (dockingSource== null)
		    dockingSource = toolBar.getParent();
		dockingSource.add(constraint, toolBar);
	    }
 	    dockingSource.invalidate();
 	    Container dockingSourceParent = dockingSource.getParent();
	    if (dockingSourceParent != null) 
		dockingSourceParent.validate();
	    dockingSource.repaint();
	}
    }

    private int mapConstraintToOrientation(String constraint) {
	int orientation = HORIZONTAL;
	if ((constraint != null) &&
	    (constraint.equals(BorderLayout.EAST) ||
	     constraint.equals(BorderLayout.WEST))) {
	    orientation = VERTICAL;
	}
	return orientation;
    }
    
    public void setOrientation(int orientation) {	
	if (orientation == VERTICAL) {
	    toolBar.setLayout(new BoxLayout(toolBar, BoxLayout.Y_AXIS));
	} else {
	    toolBar.setLayout(new BoxLayout(toolBar, BoxLayout.X_AXIS));
	}
	if (dragWindow !=null)
	    dragWindow.setOrientation(orientation);
    }
    
    /**
     * Gets the color displayed when over a docking area
     */
    public Color getDockingColor() {
	return dockingColor;
    }
    
    /**
     * Sets the color displayed when over a docking area
     */
   public void setDockingColor(Color c) {
	this.dockingColor = c;
    }

    /**
     * Gets the color displayed when over a floating area
     */
    public Color getFloatingColor() {
	return floatingColor;
    }

    /**
     * Sets the color displayed when over a floating area
     */
    public void setFloatingColor(Color c) {
	this.floatingColor = c;
    }

    public void propertyChange(PropertyChangeEvent e) {
	String propertyName = e.getPropertyName();
	if (e.getPropertyName().equals("floatable")) {
	    Boolean b = (Boolean) e.getNewValue();
	    setFloatable(b.booleanValue());
	}
    }
    
    public void setFloatable(boolean b) {
	if (b == true) {
	    toolBar.addMouseMotionListener(dockingListener);
	    toolBar.addMouseListener(dockingListener);
	} else {
	    toolBar.removeMouseMotionListener(dockingListener);
	    toolBar.removeMouseListener(dockingListener);
	}
    }
    
    public boolean canDock(Component c, Point p) {
	// System.out.println("Can Dock: " + p);
	boolean b = false;
	if (c.contains(p)) {
	    if (dockingSensitivity == 0)
		dockingSensitivity = toolBar.getSize().height;
	    // North
	    if (p.y < dockingSensitivity)
		b = true;
	    // South
	    if (p.y > c.getSize().height-dockingSensitivity)
		b = true;
	    // West  (Base distance on height for now!)
	    if (p.x < dockingSensitivity)
		b = true;
	    // East  (Base distance on height for now!)
	    if (p.x > c.getSize().width-dockingSensitivity)
		b = true;
	}
	return b;
    }

    private String getDockingConstraint(Component c, Point p) {
	// System.out.println("Docking Constraint: " + p);
	String s = BorderLayout.NORTH;
	if ((p != null) && (c.contains(p))) {
	    if (dockingSensitivity == 0)
		dockingSensitivity = toolBar.getSize().height;
	    if (p.y > c.getSize().height-dockingSensitivity)
		s = BorderLayout.SOUTH;
	    // West  (Base distance on height for now!)
	    if (p.x < dockingSensitivity)
		s = BorderLayout.WEST;
	    // East  (Base distance on height for now!)
	    if (p.x > c.getSize().width-dockingSensitivity)
		s = BorderLayout.EAST;
	    // North  (Base distance on height for now!)
	    if (p.y < dockingSensitivity)
		s = BorderLayout.NORTH;
	}
	return s;
    }

    protected void dragTo(Point position, Point origin) {
	if (toolBar.isFloatable() == true) {
	    if (dragWindow == null)
		dragWindow = createDragWindow(toolBar);
	    Point offset = dragWindow.getOffset();
	    if (offset == null) {
		Dimension size = toolBar.getPreferredSize();
		offset = new Point(size.width/2, size.height/2);
		dragWindow.setOffset(offset);
	    }
	    Point global = new Point(origin.x+ position.x,
				     origin.y+position.y);
	    Point dragPoint = new Point(global.x- offset.x, 
					global.y- offset.y);
	    if (dockingSource == null)
		dockingSource = toolBar.getParent();
	    
	    Point dockingPosition = dockingSource.getLocationOnScreen();
	    Point comparisonPoint = new Point(global.x-dockingPosition.x,
					      global.y-dockingPosition.y);
	    if (canDock(dockingSource, comparisonPoint)) {
		dragWindow.setBackground(getDockingColor());	
		String constraint = getDockingConstraint(dockingSource,
							 comparisonPoint);
		int orientation = mapConstraintToOrientation(constraint);
		dragWindow.setOrientation(orientation);
		dragWindow.setBorderColor(dockingBorderColor);
	    } else {
		dragWindow.setBackground(getFloatingColor());
		dragWindow.setOrientation(HORIZONTAL);
		dragWindow.setBorderColor(floatingBorderColor);
	    }
	    
	    dragWindow.setLocation(dragPoint.x, dragPoint.y);
	    if (dragWindow.isVisible() == false) {
		Dimension size = toolBar.getPreferredSize();
		dragWindow.setSize(size.width, size.height);
		dragWindow.show();
	    }
	}
    }

    protected void floatAt(Point position, Point origin) {
	if(toolBar.isFloatable() == true) {
	    Point offset = dragWindow.getOffset();
	    if (offset == null) {
		offset = position;
		dragWindow.setOffset(offset);
	    }
	    Point global = new Point(origin.x+ position.x,
				     origin.y+position.y);
	    setFloatingLocation(global.x-offset.x, 
				global.y-offset.y);
	    if (dockingSource != null) { 
		Point dockingPosition = dockingSource.getLocationOnScreen();
		Point comparisonPoint = new Point(global.x-dockingPosition.x,
						  global.y-dockingPosition.y);
		if (canDock(dockingSource, comparisonPoint)) {
		    setFloating(false, comparisonPoint);
		} else {
		    setFloating(true, null);
		}
	    } else {
		setFloating(true, null);
	    }
	    dragWindow.setOffset(null);
	}
    }
    
    protected DockingListener createDockingListener(JToolBar toolbar) {
	return new DockingListener(toolbar);
    }
    
    protected WindowListener createFrameListener() {
	return new FrameListener();
    }

    class FrameListener extends WindowAdapter {
	public void windowClosing(WindowEvent w) {	    
	    setFloating(false, null);
	}

    } 
    public class DockingListener implements MouseMotionListener, MouseListener,
    Serializable {
	protected JToolBar toolBar;
	protected boolean isDragging = false;
	protected Point origin = null;

	public DockingListener(JToolBar t) {
	    this.toolBar = t;
	} 

	public void mouseClicked(MouseEvent e) {}
	public void mousePressed(MouseEvent e) { 
	    isDragging = false;
	}
	public void mouseReleased(MouseEvent e) {
	    if (isDragging == true) {
		Point position = e.getPoint();
		if (origin == null)
		    origin = e.getComponent().getLocationOnScreen();
		floatAt(position, origin);
	    }
	    origin = null;
	    isDragging = false;
	}
	public void mouseEntered(MouseEvent e) { }
	public void mouseExited(MouseEvent e) { }

	public void mouseDragged(MouseEvent e) {
	    isDragging = true;
	    Point position = e.getPoint();
	    if (origin == null)
		origin = e.getComponent().getLocationOnScreen();
	    dragTo(position, origin);
	}
	public void mouseMoved(MouseEvent e) {
	}
    }

    static class DragWindow extends Window {
	Color borderColor = Color.gray;
	int orientation = HORIZONTAL;
	Point offset; // offset of the mouse cursor inside the DragWindow

	DragWindow(Frame f) {
	    super(f);
	}

	public void setOrientation(int o) {
	    if(isShowing()) {
		if (o == this.orientation)
		    return;	    
		this.orientation = o;
		Dimension size = getSize();
		setSize(new Dimension(size.height, size.width));
		if (offset!=null)
		    setOffset(new Point(offset.y, offset.x));
		repaint();
	    }
	}

	public Point getOffset() {
	    return offset;
	}

	public void setOffset(Point p) {
	    this.offset = p;
	}
	
	public void setBorderColor(Color c) {
	    if (this.borderColor == c)
		return;
	    this.borderColor = c;
	    repaint();
	}

	public Color getBorderColor() {
	    return this.borderColor;
	}

	public void paint(Graphics g) {
	    Color temp = g.getColor();
	    g.setColor(getBackground());	    
	    Dimension size = getSize();
	    g.fillRect(0,0,size.width, size.height);	    
	    g.setColor(getBorderColor());
	    g.drawRect(0,0,size.width-1, size.height-1);	    
	    g.setColor(temp);
	    super.paint(g);
	}
	public Insets getInsets() {
	    return new Insets(1,1,1,1);
	}
    }
}







