/*
 * @(#)BasicSplitPaneUI.java	1.29 98/04/08
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.plaf.basic;

import com.sun.java.swing.*;
import com.sun.java.swing.border.Border;
import com.sun.java.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.peer.ComponentPeer;
import java.beans.*;
import java.util.*;
import java.io.Serializable;
import com.sun.java.swing.plaf.SplitPaneUI;
import com.sun.java.swing.plaf.ComponentUI;

/**
 * Standard split pane.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.29 04/08/98
 * @author Scott Violet
 * @author Steve Wilson
 */
public class BasicSplitPaneUI extends SplitPaneUI implements
       PropertyChangeListener, Serializable
{
    /** The divider used for non-continuous layout is added to the split pane
     * with this object. */
    protected static final String NON_CONTINUOUS_DIVIDER = "nonContinuousDivider";

    /** JSplitPane instance this instance is providing the look and feel for. */
    protected JSplitPane splitPane;

    /** How the views are split. */
    private int orientation;

    public int getOrientation() {
	return orientation;
    }

    public void setOrientation(int orientation) {
	this.orientation = orientation;
    }

    /** LayoutManager that is created and placed into the split pane. */
    protected BasicHorizontalLayoutManager layoutManager;

    /** Responsible for handling the current dragging session. */
    private boolean continuousLayout;

    public boolean isContinuousLayout() {
	return continuousLayout;
    }
    
    public void setContinuousLayout(boolean b) {
	continuousLayout = b;
    }

    /** Divides the two views. */
    protected BasicSplitPaneDivider divider;

    /** Last location the divider was at. */
    private int lastDragLocation;

    public int getLastDragLocation() {
	return lastDragLocation;
    }

    public void setLastDragLocation(int l) {
	lastDragLocation = l;
    }

    /** The size of the divider while the dragging session is valid. */
    protected int dividerSize;

    /** Activated during a dragging session when the */
    protected Component nonContinuousLayoutDivider;

    /** Set to true in startDragging if any of the children
     * (not including the nonContinuousLayoutDivider) are heavy
     * weights. */
    protected boolean draggingHW;

    /** Location of the divider when the dragging session began. */
    protected int beginDragDividerLocation;

    /**
      * Creates a new BasicSplitPaneUI instance
      */
    public static ComponentUI createUI(JComponent x) {
	return new BasicSplitPaneUI();
    }

    /**
      * Creates a new instance of SplitPaneUI.
      */
    public BasicSplitPaneUI() {
	setLastDragLocation(-1);
    }

    /**
      * Installs the reciever as the L&F for the passed in split pane.
      */
    public void installUI(JComponent c) {
	installDefaults(c);
	installListeners(c);
	splitPane.layout();
    }

    protected void installDefaults(JComponent c){ 
	splitPane = (JSplitPane)c;



	LookAndFeel.installBorder(c, "SplitPane.border");

	if(divider == null)
	    divider = createDefaultDivider();
	divider.setBasicSplitPaneUI(this);

	splitPane.setDividerSize( ((Integer)(UIManager.get("SplitPane.dividerSize"))).intValue()  );

       	divider.setDividerSize(splitPane.getDividerSize());
	splitPane.add(divider, JSplitPane.DIVIDER);

	setOrientation(splitPane.getOrientation());
	setContinuousLayout(splitPane.isContinuousLayout());

	resetLayoutManager();
	/* Install the nonContinuousLayoutDivider here to avoid having to
	   add/remove everything later. */
	if(nonContinuousLayoutDivider == null) {
	    setNonContinuousLayoutDivider(createDefaultNonContinuousLayoutDivider(),
					  true);
	} else {
	    setNonContinuousLayoutDivider(nonContinuousLayoutDivider, true);
	}
    }

    protected void installListeners(JComponent c) {
	splitPane.addPropertyChangeListener(this);
    }

    /**
      * Removes the receiver from the L&F controller of the passed in split
      * pane.
      */
    public void uninstallUI(JComponent c) {
	uninstallListeners(c);
	uninstallDefaults(c);
    }

    protected void uninstallListeners(JComponent c) {
	splitPane.removePropertyChangeListener(this);
    }

    protected void uninstallDefaults(JComponent c) {
	if(splitPane.getLayout() == layoutManager) {
	    splitPane.setLayout(null);
	}


	if(nonContinuousLayoutDivider != null) {
	    splitPane.remove(nonContinuousLayoutDivider);
	}

	splitPane.remove(divider);
	divider.setBasicSplitPaneUI(null);
	splitPane = null;
	layoutManager = null;

	setNonContinuousLayoutDivider(null);
    }

    /**
      * Messaged from the JSplitPane the reciever is contained in.
      * May potentially reset the layout manager and cause a
      * validate() to be sent.
      */
    public void propertyChange(PropertyChangeEvent e) {
	if(e.getSource() == splitPane) {
	    String changeName = e.getPropertyName();

	    if(changeName.equals(JSplitPane.ORIENTATION_PROPERTY)) {
		orientation = splitPane.getOrientation();
		resetLayoutManager();
	    } else if(changeName.equals(JSplitPane.CONTINUOUS_LAYOUT_PROPERTY)) {
		setContinuousLayout(splitPane.isContinuousLayout());
		if(!isContinuousLayout()) {
		    if(nonContinuousLayoutDivider == null) {
		      setNonContinuousLayoutDivider(createDefaultNonContinuousLayoutDivider(), true);
		    } else if(nonContinuousLayoutDivider.getParent() == null) {
		      setNonContinuousLayoutDivider(
			  nonContinuousLayoutDivider, true);
		    }
		}
	    } else if(changeName.equals(JSplitPane.DIVIDER_SIZE_PROPERTY)) {
		divider.setDividerSize(splitPane.getDividerSize());
		layoutManager.resetSizeAt(2);
		splitPane.invalidate();
		splitPane.validate();
	    }
	}
    }

    /**
      * Sets the Component used to divider the two Components.
      */
  // PENDING (Steve) - remove this method as unused?
/*    public void setDivider(BasicSplitPaneDivider newD) {
	if(newD != null) {
	    if(splitPane != null) {
		if(divider != null) {
		    splitPane.remove(divider);
		}
		divider = newD;
		splitPane.add(divider, JSplitPane.DIVIDER);
		layoutManager.resetSizeAt(2);
		splitPane.invalidate();
		splitPane.validate();
	    } else {
		divider = newD;
	    }
	    divider.setBasicSplitPaneUI(this);
	}
    }*/

    /**
      * Returns the divider between the top Components.
      */
    public BasicSplitPaneDivider getDivider() {
	return divider;
    }

    /**
     * Returns the default non continuous layout divider, which is an
     * instanceof Canvas that fills the background in dark gray.
     */
    protected Component createDefaultNonContinuousLayoutDivider() {
	return new Canvas() {
	    public void paint(Graphics g) {
		if(!isContinuousLayout() && getLastDragLocation() != -1) {
		    Dimension      size = splitPane.getSize();

		    g.setColor(Color.darkGray);
		    if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
			g.fillRect(0, 0, dividerSize - 1, size.height - 1);
		    } else {
			g.fillRect(0, 0, size.width - 1, dividerSize - 1);
		    }
		}
	    }
	};
    }

    /**
     * Sets the divider to use when the splitPane is configured to
     * not continuously layout. This divider will only be used during a
     * dragging session. It is recommended that the passed in component
     * be a heavy weight.
     */
    protected void setNonContinuousLayoutDivider(Component newDivider) {
	setNonContinuousLayoutDivider(newDivider, true);
    }

    protected void setNonContinuousLayoutDivider(Component newDivider,
						 boolean rememberSizes) {
	if(nonContinuousLayoutDivider != null && splitPane != null) {
	    splitPane.remove(nonContinuousLayoutDivider);
	}
	nonContinuousLayoutDivider = newDivider;
	if(nonContinuousLayoutDivider != null && splitPane != null) {
	    nonContinuousLayoutDivider.setLocation(-1000, -1000);

	    /* Needs to remove all the components and readd them!
	       YECK! */
	    Component             leftC = splitPane.getLeftComponent();
	    Component             rightC = splitPane.getRightComponent();
	    int[]                 sizes = layoutManager.getSizes();

	    if(leftC != null)
		splitPane.setLeftComponent(null);
	    if(rightC != null)
		splitPane.setRightComponent(null);
	    splitPane.remove(divider);
	    splitPane.add(nonContinuousLayoutDivider, BasicSplitPaneUI.
			  NON_CONTINUOUS_DIVIDER,
			  splitPane.getComponentCount());
	    splitPane.setLeftComponent(leftC);
	    splitPane.setRightComponent(rightC);
	    splitPane.add(divider, JSplitPane.DIVIDER);
	    if(rememberSizes)
		layoutManager.setSizes(sizes);
	    splitPane.validate();
	    splitPane.paintImmediately(splitPane.getX(),
				       splitPane.getY(),
				       splitPane.getWidth(),
				       splitPane.getHeight());
	}
    }

    /**
     * Returns the divider to use when the splitPane is configured to
     * not continuously layout. This divider will only be used during a
     * dragging session.
     */
    public Component getNonContinuousLayoutDivider() {
	return nonContinuousLayoutDivider;
    }

    /**
      * Returns the splitpane this instance is currently contained
      * in.
      */
    public JSplitPane getSplitPane() {
	return splitPane;
    }

    /**
      * Creates the default divider.
      */
    public BasicSplitPaneDivider createDefaultDivider() {
	return new BasicSplitPaneDivider(this);
    }

    /**
      * Messaged to reset the preferred sizes.
      */
    public void resetToPreferredSizes() {
	if(splitPane != null) {
	    layoutManager.resetToPreferredSizes();
	    //	    splitPane.invalidate();
	    splitPane.validate();
	    layoutManager.layoutContainer(splitPane);
	}
    }

    /**
      * Sets the location of the divider to location.
      */
    public void setDividerLocation(int location) {

	Component                 leftC = splitPane.getLeftComponent();
	Component                 rightC = splitPane.getRightComponent();

	if(leftC != null && rightC != null) {
	    Insets                insets = splitPane.getInsets();

	    if(insets != null) {
		if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
		    leftC.setSize(Math.max(0, location - insets.left), 10);
		} else {
		    leftC.setSize(10, Math.max(0, location - insets.top));
		}
	    } else {
		if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
		    leftC.setSize(Math.max(0, location), 10);
		} else {
		    leftC.setSize(10, Math.max(0, location));
		}
	    }
	    //	    splitPane.invalidate();
	    splitPane.validate();
	    splitPane.repaint();
	}
    }

    /**
      * Returns the location of the divider.
      */
    public int getDividerLocation() {
	if(orientation == JSplitPane.HORIZONTAL_SPLIT)
	    return divider.getLocation().x;
	return divider.getLocation().y;
    }

    /**
      * Gets the minimum location of the divider.
      */
    public int getMinimumDividerLocation() {
	int       minLoc = 0;
	Component leftC = splitPane.getLeftComponent();
	if (leftC != null) {
	    Insets    insets = splitPane.getInsets();
            Dimension minSize = leftC.getMinimumSize();
	    if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
		minLoc = minSize.width;
	    } else {
		minLoc = minSize.height;
	    }
	    minLoc += getDividerBorderSize();
	    if(insets != null) {
		if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
		    minLoc += insets.left;
		} else {
		    minLoc += insets.top;
		}
	    }
	}
	return minLoc;
    }

    /**
      * Gets the maximum location of the divider.
      */
    public int getMaximumDividerLocation() {
	Dimension splitPaneSize = splitPane.getSize();
	int       maxLoc = 0;
	Component rightC = splitPane.getRightComponent();
	if (rightC != null) {
	    Insets    insets = splitPane.getInsets();
            Dimension minSize = rightC.getMinimumSize();
	    if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
		maxLoc = splitPaneSize.width - minSize.width;
	    } else {
		maxLoc = splitPaneSize.height - minSize.height; 
	    }
	    maxLoc -= (dividerSize + getDividerBorderSize());
	    if(insets != null) {
		if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
		    maxLoc += insets.left;
		} else {
		    maxLoc += insets.top;
		}
	    }
	}
	return Math.max(getMinimumDividerLocation(), maxLoc);
    }

    /**
     * Messaged after the JSplitPane the receiver is providing the look
     * and feel for paints its children.
     */
    public void finishedPaintingChildren(JSplitPane jc, Graphics g) {
	if(jc == splitPane && getLastDragLocation() != -1 &&
	    !isContinuousLayout() && !draggingHW) {
	    Dimension      size = splitPane.getSize();

	    g.setColor(Color.darkGray);
	    if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
		g.fillRect(getLastDragLocation(), 0, dividerSize - 1, size.height - 1);
	    } else {
		g.fillRect(0, lastDragLocation, size.width - 1, dividerSize - 1);
	    }
	}
    }

    /**
      * Messaged to paint the look and feel.
      */
    public void paint(Graphics g, JComponent jc) {
    }

    /**
      * Returns the preferred size for the passed in component,
      * This is passed off to the current layoutmanager.
      */
    public Dimension getPreferredSize(JComponent jc) {
	if(splitPane != null)
	    return layoutManager.preferredLayoutSize(splitPane);
	return new Dimension(0, 0);
    }

    /**
      * Returns the minimum size for the passed in component,
      * This is passed off to the current layoutmanager.
      */
    public Dimension getMinimumSize(JComponent jc) {
	if(splitPane != null)
	    return layoutManager.minimumLayoutSize(splitPane);
	return new Dimension(0, 0);
    }

    /**
      * Returns the maximum size for the passed in component,
      * This is passed off to the current layoutmanager.
      */
    public Dimension getMaximumSize(JComponent jc) {
	if(splitPane != null)
	    return layoutManager.maximumLayoutSize(splitPane);
	return new Dimension(0, 0);
    }

    /**
     * Returns the insets. The insets are returned from the broder insets
     * of the current border.
     */
    public Insets getInsets(JComponent jc) {
	return null;
    }

    /**
      * Resets the layout manager based on orientation and messages it
      * with invalidateLayout to pull in appropriate Components.
      */
    protected void resetLayoutManager() {
	if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
	    layoutManager = new BasicHorizontalLayoutManager();
	} else {
	    layoutManager = new BasicVerticalLayoutManager();
	}
	splitPane.setLayout(layoutManager);
	layoutManager.updateComponents();
       	splitPane.validate();
	splitPane.repaint();
    }

    /**
      * Should be messaged before the dragging session starts, resets
      * lastDragLocation and dividerSize.
      */
    protected void startDragging() {
	Component                 leftC = splitPane.getLeftComponent();
	Component                 rightC = splitPane.getRightComponent();
	ComponentPeer             cPeer;

	beginDragDividerLocation = getDividerLocation();
	draggingHW = false;
	if(leftC != null && (cPeer = leftC.getPeer()) != null &&
	   !(cPeer instanceof java.awt.peer.LightweightPeer)) {
	    draggingHW = true;
	} else if(rightC != null && (cPeer = rightC.getPeer()) != null
		  && !(cPeer instanceof java.awt.peer.LightweightPeer)) {
	    draggingHW = true;
	}
	if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
	    setLastDragLocation(divider.getBounds().x);
	    dividerSize = divider.getSize().width + 2 * getDividerBorderSize();
	    if(!isContinuousLayout() && draggingHW) {
		nonContinuousLayoutDivider.setBounds
		    (getLastDragLocation(), 0, dividerSize, splitPane.getHeight());
	    }
	} else {
	    setLastDragLocation(divider.getBounds().y);
	    dividerSize = divider.getSize().height + 2 *
		getDividerBorderSize();
	    if(!isContinuousLayout() && draggingHW) {
		nonContinuousLayoutDivider.setBounds
		    (0, getLastDragLocation(), splitPane.getWidth(), dividerSize);
	    }
	}
    }

    /**
      * Messaged during a dragging session to move the divider to the
      * passed in location. If continuousLayout is true the location is
      * reset and the splitPane validated.
      */
    protected void dragDividerTo(int location) {
	if(getLastDragLocation() != location) {
	    if(isContinuousLayout()) {
		Component          leftC = splitPane.getLeftComponent();
		Rectangle          leftBounds = leftC.getBounds();

		if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
		    leftC.setSize(location - leftBounds.x,
				  leftBounds.height);
		} else {
		    leftC.setSize(leftBounds.width, location - leftBounds.y);
		}
		splitPane.validate();
		splitPane.repaint();
		setLastDragLocation(location);
	    } else {
		int lastLoc = getLastDragLocation();

		setLastDragLocation(location);
		if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
		    int      splitHeight = splitPane.getSize().height;

		    if(draggingHW) {
			nonContinuousLayoutDivider.setLocation(getLastDragLocation(),0);
		    } else {
			splitPane.repaint(lastLoc, 0, dividerSize, splitHeight);
			splitPane.repaint(location, 0, dividerSize, splitHeight);
		    }
		} else {
		    int      splitWidth = splitPane.getSize().width;

		    if(draggingHW) {
			nonContinuousLayoutDivider.setLocation(0, getLastDragLocation());
		    } else {
			  splitPane.repaint(0, lastLoc, splitWidth, dividerSize);
			  splitPane.repaint(0, location, splitWidth, dividerSize);
		    }
		}
	    }
	}
    }

    /**
      * Messaged to finish the dragging session. If not continuous display
      * the dividers location will be reset.
      */
    protected void finishDraggingTo(int location) {
	dragDividerTo(location);
	setLastDragLocation(-1);
	if(!isContinuousLayout()) {
	    Component          leftC = splitPane.getLeftComponent();
	    Rectangle          leftBounds = leftC.getBounds();

	    if(orientation == JSplitPane.HORIZONTAL_SPLIT) {
		int      splitHeight = splitPane.getSize().height;

		leftC.setSize(location - leftBounds.x,
			      leftBounds.height);
		if(draggingHW)
		    nonContinuousLayoutDivider.setLocation(-dividerSize, 0);
		splitPane.paintImmediately(location, 0, dividerSize,
					   splitHeight);
	    } else {
		int      splitWidth = splitPane.getSize().width;

		leftC.setSize(leftBounds.width, location - leftBounds.y);
		if(draggingHW)
		    nonContinuousLayoutDivider.setLocation(0, -dividerSize);
		splitPane.paintImmediately(0, location, splitWidth,
					   dividerSize);
	    }
	    /* Do the layout. */
	    splitPane.validate();
	    splitPane.repaint();
	}
	splitPane.setLastDividerLocation(beginDragDividerLocation);
    }

    /**
     * Returns the width of one side of the divider border.
     */
    protected int getDividerBorderSize() {
	return 1;
    }


    /**
     * LayoutManager for JSplitPanes that have an orientation of
     * HORIZONTAL_SPLIT.
     * <p>
     * Warning: serialized objects of this class will not be compatible with
     * future swing releases.  The current serialization support is appropriate
     * for short term storage or RMI between Swing1.0 applications.  It will
     * not be possible to load serialized Swing1.0 objects with future releases
     * of Swing.  The JDK1.2 release of Swing will be the compatibility
     * baseline for the serialized form of Swing objects.
      */
    public class BasicHorizontalLayoutManager extends Object implements
	      LayoutManager2, Serializable
    {
	/* left, right, divider. */
	protected int[]                   sizes;
	protected Component[]             components;
	protected int                     dividerLocation;


	BasicHorizontalLayoutManager() {
	    components = new Component[3];
	    components[0] = components[1] = components[2] = null;
	    sizes = new int[3];
	    dividerLocation = -1;
	}

	/**
	 * Resets the size of the Component at the passed in location.
	 */
	protected void resetSizeAt(int index) {
	    sizes[index] = -1;
	}

	/**
	 * Sets the sizes to <code>newSizes</code>.
	 */
	protected void setSizes(int[] newSizes) {
	    System.arraycopy(newSizes, 0, sizes, 0, 3);
	}

	/**
	 * Returns the sizes of the components.
	 */
	protected int[] getSizes() {
	    int[]         retSizes = new int[3];

	    System.arraycopy(sizes, 0, retSizes, 0, 3);
	    return retSizes;
	}

	/**
	  * Returns the width of the passed in Components preferred size.
	  */
	protected int getPreferredSizeOfComponent(Component c) {
	    return c.getPreferredSize().width;
	}

	/**
	  * Returns the width of the passed in component.
	  */
	protected int getSizeOfComponent(Component c) {
	    return c.getSize().width;
	}

	/**
	  * Returns the available width based on the container size and
	  * Insets.
	  */
	protected int getAvailableSize(Dimension containerSize,
				       Insets insets) {
	    if(insets == null)
		return containerSize.width;
	    return (containerSize.width - (insets.left + insets.right + 2 *
					      getDividerBorderSize()));
	}

	/**
	  * Returns the left inset, unless the Insets or null in which case
	  * 0 is returned.
	  */
	protected int getInitialLocation(Insets insets) {
	    if(insets != null)
		return insets.left;
	    return 0;
	}

	/**
	  * Sets the width of the component c to be c, placing its
	  * x location at location, y to the insets.top and height
	  * to the containersize.height less the top and bottom insets.
	  */
	protected void setComponentToSize(Component c, int size,
					  int location, Insets insets,
					  Dimension containerSize) {
	    if(insets != null) {
		c.setBounds(location, insets.top, size, containerSize.height -
			    (insets.top + insets.bottom));
	    } else {
		c.setBounds(location, 0, size, containerSize.height);
	    }
	}

	public void layoutContainer(Container container) {


	    Dimension         containerSize = container.getSize();
	    // if the splitpane has a zero size then no op out of here.
	    // if we execute this function now, we're going to cause ourselves
	    // much grief.
	    if (containerSize.height == 0 || containerSize.width == 0 ) {
	        return;
	    }

	    int               counter;
	    int               newSize;
	    int               totalSize = 0;


	    Insets            insets = splitPane.getInsets();
	    int               availableSize = getAvailableSize(containerSize,
							       insets);
	    int               beginLocation;

	    /* Set the last location. */
	    beginLocation = splitPane.getDividerLocation();

	    /* Check to see if the size of something has changed and
	       adjust the other Component if it has. */
	    for(counter = 0; counter < 3; counter++) {
		if(components[counter] != null) {
		    /* -1 signifies this is the first time this component
		       is being layed out and that the preferred size should
		       be asked for. */
		    if(sizes[counter] == -1) {
			sizes[counter] = Math.min(availableSize,
			     getPreferredSizeOfComponent(components[counter]));
		    } else {
			newSize = getSizeOfComponent(components[counter]);
			if(sizes[counter] != newSize) {
			    if(counter == 0) {
				if(components[1] != null) {
				    sizes[1] = Math.max(0, sizes[1] +
						(sizes[counter] - newSize));
				}
			    } else if(counter == 1) {
				if(components[0] != null) {
				    sizes[0] = Math.max(0, sizes[0] +
						(sizes[counter] - newSize));
				}
			    }
			    sizes[counter] = newSize;
			    /* Do this to not adjust the next one, but adjust
			       the divider. */
			    if(counter == 0)
				counter = 1;
			}
		    }
		}
	    }

	    for(counter = 0; counter < 3; counter++)
		totalSize += sizes[counter];

	    /* If the width has changed, adjust the right, and then left
	       component (if necessary). */
	    if(totalSize != availableSize) {
		int           toDiff = (availableSize - totalSize);

		if(components[1] != null) {
		    newSize = Math.max(0, sizes[1] + toDiff);
		    if(newSize == 0) {
			toDiff += sizes[1];
			sizes[1] = 0;
			if(components[0] != null)
			    sizes[0] = Math.max(0, sizes[0] + toDiff);
		    } else {
			sizes[1] = newSize;
		    }
		} else if(components[0] != null) {
		    sizes[0] = Math.max(0, sizes[0] + toDiff);
		}
	    }

	    /* Reset the bounds of each component. */
	    int                  nextLocation = getInitialLocation(insets);
	    int                  bdSize = getDividerBorderSize();

	    counter = 0;
	    while(counter < 3) {
		if(components[counter] != null) {
		    setComponentToSize(components[counter], sizes[counter],
				       nextLocation, insets, containerSize);
		    nextLocation += sizes[counter];
		}
		switch(counter) {
		case 0:
		    counter = 2;
		    nextLocation += bdSize;
		    break;
		case 2:
		    counter = 1;
		    nextLocation += bdSize;
		    break;
		case 1:
		    counter = 3;
		    break;
		}
	    }

	    if(beginLocation != splitPane.getDividerLocation()) {
		splitPane.setLastDividerLocation(beginLocation);
	    }

	}

	/**
	  * Adds the component at place.  Place must be one of
	  * JSplitPane.LEFT, RIGHT, TOP, BOTTOM, or null (for the
	  * divider).
	  */
	public void addLayoutComponent(String place, Component component) {
	    boolean         isValid = true;

	    if(place != null) {
		if(place.equals(JSplitPane.DIVIDER)) {
		    /* Divider. */
		    components[2] = component;
		    sizes[2] = -1;
		} else if(place.equals(JSplitPane.LEFT) ||
			place.equals(JSplitPane.TOP)) {
		    components[0] = component;
		    sizes[0] = -1;
		} else if(place.equals(JSplitPane.RIGHT) ||
			place.equals(JSplitPane.BOTTOM)) {
		    components[1] = component;
		    sizes[1] = -1;
		} else if(!place.equals(BasicSplitPaneUI.NON_CONTINUOUS_DIVIDER))
		    isValid = false;
	    } else {
		isValid = false;
	    }
	    if(!isValid)
		throw new IllegalArgumentException("cannot add to layout: unknown constraint: " + place);
	}

	/**
	  * Returns the minimum size needed to contain the children.
	  * The width is the sum of all the childrens min widths and
	  * the height is the largest of the childrens minimum heights.
	  */
	public Dimension minimumLayoutSize(Container container) {
	    Dimension              minSize;
	    int                    minX, minY;
	    Insets                 insets = splitPane.getInsets();

	    minX = minY = 0;

	    for(int counter = 0; counter < 3; counter++) {
		if(components[counter] != null) {
		    minSize = components[counter].getMinimumSize();
		    minX += minSize.width;
		    if(minSize.height > minY)
			minY = minSize.height;
		}
	    }
	    if(insets != null) {
		minX += insets.left + insets.right;
		minY += insets.bottom + insets.top;
	    }
	    return new Dimension(minX, minY);
	}

	/**
	  * Returns the preferred size needed to contain the children.
	  * The width is the sum of all the childrens preferred widths and
	  * the height is the largest of the childrens preferred heights.
	  */
	public Dimension preferredLayoutSize(Container container) {
	    Dimension              preSize;
	    int                    preX, preY;
	    Insets                 insets = splitPane.getInsets();

	    preX = preY = 0;

	    for(int counter = 0; counter < 3; counter++) {
		if(components[counter] != null) {
		    preSize = components[counter].getPreferredSize();
		    preX += preSize.width;
		    if(preSize.height > preY)
			preY = preSize.height;
		}
	    }
	    if(insets != null) {
		preX += insets.left + insets.right;
		preY += insets.bottom + insets.top;
	    }
	    return new Dimension(preX, preY);
	}

	/**
	  * Removes the specified component from our knowledge.
	  */
	public void removeLayoutComponent(Component component) {
	    for(int counter = 0; counter < 3; counter++) {
		if(components[counter] == component) {
		    components[counter] = null;
		    sizes[counter] = 0;
		}
	    }
	}

	//
	// LayoutManager2
	//
	/**
	  * Adds the specified component to the layout, using the specified
	  * constraint object.
	  * @param comp the component to be added
	  * @param constraints  where/how the component is added to the layout.
	  */
	public void addLayoutComponent(Component comp, Object constraints) {
	    if ((constraints == null) || (constraints instanceof String)) {
		addLayoutComponent((String)constraints, comp);
	    } else {
		throw new IllegalArgumentException("cannot add to layout: constraint must be a string (or null)");
	    }
	}

	/**
	  * Returns the alignment along the x axis.  This specifies how
	  * the component would like to be aligned relative to other 
	  * components.  The value should be a number between 0 and 1
	  * where 0 represents alignment along the origin, 1 is aligned
	  * the furthest away from the origin, 0.5 is centered, etc.
	  */
	public float getLayoutAlignmentX(Container target) {
	    return 0.0f;
	}

	/**
	  * Returns the alignment along the y axis.  This specifies how
	  * the component would like to be aligned relative to other 
	  * components.  The value should be a number between 0 and 1
	  * where 0 represents alignment along the origin, 1 is aligned
	  * the furthest away from the origin, 0.5 is centered, etc.
	  */
	public float getLayoutAlignmentY(Container target) {
	    return 0.0f;
	}

	/**
	  * Does nothing. If the developer really wants to change the
	  * size of one of the views JSplitPane.resetToPreferredSizes should
	  * be messaged.
	  */
	public void invalidateLayout(Container c) {
	}

	/**
	  * Returns the maximum layout size, which is Integer.MAX_VALUE
	  * in both directions.
	  */
	public Dimension maximumLayoutSize(Container target) {
	    return new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);
	}

	/**
	  * Resets the cached sizes so that next time this instance is
	  * layed out it'll ask for the preferred sizes.
	  */
	public void resetToPreferredSizes() {
	    for(int counter = 0; counter < 3; counter++)
		sizes[counter] = -1;
	}

	/**
	  * Determines the components. This should be called whenever
	  * a new instance of this is installed into an existing
	  * SplitPane.
	  */
	protected void updateComponents() {
	    Component              comp;

	    comp = splitPane.getLeftComponent();
	    if(components[0] != comp) {
		components[0] = comp;
		if(comp == null) {
		    sizes[0] = 0;
		} else {
		    sizes[0] = -1;
		}
	    }

	    comp = splitPane.getRightComponent();
	    if(components[1] != comp) {
		components[1] = comp;
		if(comp == null) {
		    sizes[1] = 0;
		} else {
		    sizes[1] = -1;
		}
	    }

	    /* Find the divider. */
	    Component[]           children = splitPane.getComponents();
	    Component             oldDivider = components[2];

	    components[2] = null;
	    for(int counter = children.length - 1; counter >= 0; counter--) {
		if(children[counter] != components[0] &&
		   children[counter] != components[1] &&
		    children[counter] != nonContinuousLayoutDivider) {
		    if(oldDivider != children[counter]) {
			components[2] = children[counter];
			if(children[counter] == null) {
			    sizes[2] = 0;
			} else {
			    sizes[2] = -1;
			}
		    } else {
			components[2] = oldDivider;
		    }
		    break;
		}
	    }
	    if(components[2] == null)
		sizes[2] = 0;
	}
    } // End of class BasicSplitPaneUI.BasicHorizontalLayoutManager


    /**
     * LayoutManager used for JSplitPanes with an orientation of
     * VERTICAL_SPLIT.
     * <p>
     * Warning: serialized objects of this class will not be compatible with
     * future swing releases.  The current serialization support is appropriate
     * for short term storage or RMI between Swing1.0 applications.  It will
     * not be possible to load serialized Swing1.0 objects with future releases
     * of Swing.  The JDK1.2 release of Swing will be the compatibility
     * baseline for the serialized form of Swing objects.
     */
    public class BasicVerticalLayoutManager extends
		    BasicHorizontalLayoutManager implements Serializable
    {
	/**
	  * Returns the width of the passed in Components preferred size.
	  */
	protected int getPreferredSizeOfComponent(Component c) {
	    return c.getPreferredSize().height;
	}

	/**
	  * Returns the width of the passed in component.
	  */
	protected int getSizeOfComponent(Component c) {
	    return c.getSize().height;
	}

	/**
	  * Returns the available width based on the container size and
	  * Insets.
	  */
	protected int getAvailableSize(Dimension containerSize,
				       Insets insets) {
	    if(insets == null)
		return containerSize.height;
	    return (containerSize.height - (insets.bottom + insets.top + 2 *
					       getDividerBorderSize()));
	}

	/**
	  * Returns the left inset, unless the Insets or null in which case
	  * 0 is returned.
	  */
	protected int getInitialLocation(Insets insets) {
	    if(insets != null)
		return insets.top;
	    return 0;
	}

	/**
	  * Sets the width of the component c to be c, placing its
	  * x location at location, y to the insets.top and height
	  * to the containersize.height less the top and bottom insets.
	  */
	protected void setComponentToSize(Component c, int size,
					  int location, Insets insets,
					  Dimension containerSize) {
	    if(insets != null) {
		c.setBounds(insets.left, location, containerSize.width -
		    (insets.left + insets.right), size);
	    } else {
		c.setBounds(0, location, containerSize.width, size);
	    }
	}

	/**
	  * Returns the minimum size needed to contain the children.
	  * The height is the sum of all the childrens min heights and
	  * the width is the largest of the childrens minimum widths.
	  */
	public Dimension minimumLayoutSize(Container container) {
	    Dimension              minSize;
	    int                    minX, minY;
	    Insets                 insets = splitPane.getInsets();

	    minX = minY = 0;

	    for(int counter = 0; counter < 3; counter++) {
		if(components[counter] != null) {
		    minSize = components[counter].getMinimumSize();
		    minY += minSize.height;
		    if(minSize.width > minX)
			minX = minSize.width;
		}
	    }
	    if(insets != null) {
		minX += insets.left + insets.right;
		minY += insets.bottom + insets.top;
	    }
	    return new Dimension(minX, minY);
	}

	/**
	  * Returns the preferred size needed to contain the children.
	  * The width is the sum of all the childrens preferred widths and
	  * the height is the largest of the childrens preferred heights.
	  */
	public Dimension preferredLayoutSize(Container container) {
	    Dimension              preSize;
	    int                    preX, preY;
	    Insets                 insets = splitPane.getInsets();

	    preX = preY = 0;
	    for(int counter = 0; counter < 3; counter++) {
		if(components[counter] != null) {
		    preSize = components[counter].getPreferredSize();
		    preY += preSize.height;
		    if(preSize.width > preX)
			preX = preSize.width;
		}
	    }
	    if(insets != null) {
		preX += insets.left + insets.right;
		preY += insets.bottom + insets.top;
	    }
	    return new Dimension(preX, preY);
	}

    } // End of class BasicSplitPaneUI.BasicHorizontalLayoutManager

    /**
     * This border draws the borders around both of the contained components in the
     * the splitter.  It is a very odd border.
     * <p>
     * Warning: serialized objects of this class will not be compatible with
     * future swing releases.  The current serialization support is appropriate
     * for short term storage or RMI between Swing1.0 applications.  It will
     * not be possible to load serialized Swing1.0 objects with future releases
     * of Swing.  The JDK1.2 release of Swing will be the compatibility
     * baseline for the serialized form of Swing objects.
     */
    public static class SplitBorder implements Border, Serializable
    {
        private Color highlightColor;
        private Color shadowColor;

        public SplitBorder(Color highlight, Color shadow) {
	    highlightColor = highlight;
	    shadowColor = shadow;
	}

	public void paintBorder(Component c, Graphics g, int x, int y,
				int width, int height) {
	    Component          child;
	    Rectangle          cBounds;

	    JSplitPane splitPane = (JSplitPane)c;
	    
	    child = splitPane.getLeftComponent();
	    if(splitPane.getOrientation() == JSplitPane.HORIZONTAL_SPLIT) {
		if(child != null) {
		    cBounds = child.getBounds();
		    g.setColor(shadowColor);
		    g.drawLine(0, 0, cBounds.width + 1, 0);
		    g.drawLine(0, 1, 0, cBounds.height + 2);

		    g.setColor(highlightColor);
		    g.drawLine(1, cBounds.height + 1, cBounds.width + 2,
			       cBounds.height + 1);
		    g.drawLine(cBounds.width + 1, 1, cBounds.width + 1, cBounds.height + 2);
		}
		child = splitPane.getRightComponent();
		if(child != null) {
		    cBounds = child.getBounds();

		    int             maxX = cBounds.x + cBounds.width;
		    int             maxY = cBounds.y + cBounds.height;
		    
		    g.setColor(shadowColor);
		    g.drawLine(cBounds.x - 1, 0, maxX, 0);
		    g.drawLine(cBounds.x - 1, maxY, cBounds.x, maxY);
		    g.drawLine(cBounds.x - 1, 0, cBounds.x -1 , maxY);
		    g.setColor(highlightColor);
		    g.drawLine(cBounds.x, maxY, maxX, maxY);
		    g.drawLine(maxX, 0, maxX, maxY + 1);
		}
	    } else {
		if(child != null) {
		    cBounds = child.getBounds();
		    g.setColor(shadowColor);
		    g.drawLine(0, 0, cBounds.width + 1, 0);
		    g.drawLine(0, 1, 0, cBounds.height + 1);
		    g.setColor(highlightColor);
		    g.drawLine(1 + cBounds.width, 0, 1 + cBounds.width,
			       cBounds.height + 1);
		    g.drawLine(0, cBounds.height + 1, cBounds.width, cBounds.height + 1);
		}
		child = splitPane.getRightComponent();
		if(child != null) {
		    cBounds = child.getBounds();

		    int             maxX = cBounds.x + cBounds.width;
		    int             maxY = cBounds.y + cBounds.height;
		    
		    g.setColor(shadowColor);
		    g.drawLine(0, cBounds.y - 1, 0, maxY);
		    g.drawLine(maxX, cBounds.y - 1, maxX, cBounds.y);
		    g.drawLine(0, cBounds.y - 1, cBounds.width, cBounds.y - 1);
		    g.setColor(highlightColor);
		    g.drawLine(0, maxY, cBounds.width + 1, maxY);
		    g.drawLine(maxX, cBounds.y, maxX, maxY);
		}
	    }
	}
	public Insets getBorderInsets(Component c) {
	    return new Insets(1, 1, 1, 1);
	}
	public boolean isBorderOpaque() { return true; }
    }
}
